/*********************************************************************
*    Project: MG82F6D64-DEMO
*			MG82F6D64 LQFP64_V10 EV Board (TH210A)
*			CpuCLK=12MHz, SysCLK=12MHz
*    Description:
*			RTC Clock source: XTAL 32.768KHz
*			RTC Clock prescale: 32768
*			RTC Reload Value: (64-1)=63
*			RTC Overflow freq: 32.768K/32768/1 = 1Hz
*    Note:
*
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F6D64.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6D64.H"
#include ".\include\API_Uart_BRGRL_MG82F6D64.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection: 
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	33170000,36000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P55
#define LED_R		P56
#define LED_G_1		P57

/***********************************************************************************
*Function:   	void INT_SF(void)
*Description:	SF(system flag)Interrupt handler
		 		RTC,WDTF,BOD0F,BOD1F
*Input:   
*Output:     
*************************************************************************************/
void INT_SF(void) interrupt INT_VECTOR_SF
{
	if((PCON1&WDTF)!=0)
	{
		PCON1=WDTF;
	}
	if((PCON1&BOF0)!=0)
	{
		PCON1=BOF0;
	}
	if((PCON1&BOF1)!=0)
	{
		PCON1=BOF1;
	}
	if((PCON1&RTCF)!=0)
	{
		PCON1=RTCF;
		LED_R=!LED_R;
	}
}

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:		u8 IAP_ReadPPage(u8 PsfrAddr)
*Description:	Read P page sfr
*Input:   
*		 		u8 PsfrAddr: sfr Address
*Output:     
*		 		u8 : sfr data
*************************************************************************************/
u8 IAP_ReadPPage(u8 PsfrAddr)
{
	bit bEA=EA;
	EA = 0; 					//Turn off Interrupt
	IFADRH = 0; 				//IFADRH must be 0
	IFADRL= PsfrAddr;			//sfr Address
	IFMT = ISP_READ_P;
	ISPCR = 0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR = 0;					//clear
	EA = bEA;					//restore interrupt
	return IFD;
}

/***********************************************************************************
*Function:		void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
*Description:	write P page sfr
*Input:   
*		 		u8 PsfrAddr: sfr Address
*		 		u8 PsfrData: sfr data
*Output:     
*************************************************************************************/
void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
{
	bit bEA=EA;
	EA = 0; 					//Turn off Interrupt
	IFADRH = 0; 				//IFADRH must be 0
	IFADRL= PsfrAddr;			//sfr Address
	IFD= PsfrData;				//sfr data
	IFMT = ISP_WRITE_P;
	ISPCR = 0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR = 0;					//clear
	EA = bEA;					//restore interrupt
}

/***********************************************************************************
*Function:   	void InitRTC(void)
*Description:	Initialize RTC 
*		RTC Clock source: XTAL 32.768KHz
*		RTC Clock prescale: 32768
*		RTC Reload Value: (64-1)=63
*		RTC Overflow freq: 32.768K/32768/1 = 1Hz
*Input:   
*Output:     
*************************************************************************************/
void InitRTC(void)
{
	u8 x;
	x=IAP_ReadPPage(CKCON2_P);
	x=(x&0x3F)|(ENABLE_XTAL|GAIN_FOR_32K);
	IAP_WritePPage(CKCON2_P,x);
	while(CKCON1 & XTOR == 0x00);		// wait for XTAL stable
	
	RTC_SetClock_XTAL2_ECKI_P60();		// RTC Clock source: XTAL 32.768KHz
	RTC_SetClock_Div_32768();			// RTC Clock prescale: 32768
	RTC_SetReload(64-1);				// RTC Reload Value: (64-1)=63
	RTC_SetCounter(64-1);	
	
	RTC_ClearRTCF();					// clear RTCF
	RTC_Enable();						// Enable RTC
	
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnSF_RTC();			// Enable RTC interrupt, must enble SF interrupt
	INT_EnSF();				// Enable SF interrupt

	
}

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP5PushPull(BIT5|BIT6|BIT7);					// Set P55,P56,P57 as Push Pull(LED)
}

/***********************************************************************************
*Function:       void InitSystem(void)
*Description:    Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitRTC();			
	InitInterrupt();
	
	INT_EnAll();			// Enable global interrupt

}

void main()
{
    InitSystem();
	
 	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
 	LED_G_0=1;LED_G_1=1;LED_R=1;
					
    while(1)
    {
    	DelayXms(100);
    	LED_G_0=!LED_G_0;
    }

}


