/* Includes ------------------------------------------------------------------*/
#include "stdio.h"
#include "mg32f157.h"
#include "bsp_usart1.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
SPI_InitTypeDef  SPI_InitStructure;
GPIO_InitTypeDef GPIO_InitStructure;
NVIC_InitTypeDef NVIC_InitStructure;

/* SPI Master Send Data */
const uint8_t MasterWriteData[] = 
{
  0x52,0x79,0x0C,0xAF,0xC0,0x99,0x44,0x71,0x9D,0x30,0xAD,0x68,0xF7,0x95,0xF8,0x38,
};

/* SPI Slave Send Data */
const uint8_t SlaveWriteData[] = 
{
  0x70,0x42,0x5C,0x40,0xAC,0xC6,0x4B,0x5B,0xB6,0x63,0xE3,0x8A,0xAA,0x99,0xDD,0x31,
};

/* SPI Master and Slave Receive buf Data */
uint8_t MasterReadData[sizeof(MasterWriteData)], SlaveReadData[sizeof(SlaveWriteData)];
uint16_t MasterTxIndex = 0, MasterRxIndex = 0, MasterRxFlag = 0;
uint16_t SlaveTxIndex = 0, SlaveRxIndex = 0, SlaveRxFlag = 0;
uint8_t ret = 0;

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/
void SPI_MasterConfig(void);
void SPI_SlaveConfig(void);

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
  /* Updating the system clock */
  SystemCoreClockUpdate();

  /* Configure the interrupt priority group */
  NVIC_PriorityGroupConfig(NVIC_PriorityGroup_2);

  /* Init UART1 to 115200-8n1 for print message */
  usart1_init(96000000, 115200);
  
  printf("SPI Slave Example \r\n");

  /* Configure the SPI Master and Slave */
  SPI_SlaveConfig();
  SPI_MasterConfig();
  
  /* CS Enable */
  GPIO_ResetBits(GPIOA, GPIO_Pin_4);

  /* SPI Master Polling Send and Receive Data */
  for (uint16_t i = 0; i < sizeof(MasterWriteData); i++)
  {
    while (SPI_I2S_GetFlagStatus(SPI1 , SPI_I2S_FLAG_TXE) == RESET);
  
    SPI_I2S_SendData(SPI1, MasterWriteData[i]);
  
    while (SPI_I2S_GetFlagStatus(SPI1, SPI_I2S_FLAG_RXNE) == RESET);
  
    MasterReadData[i] = SPI_I2S_ReceiveData(SPI1);
  }
  
  while(!SlaveRxFlag);
  
  /* Print SPI Master communication data */
  printf("SPI1 Send and Recv Data \r\n");
  printf("Index | Send | Recv \r\n");
  for(uint16_t i = 0; i < sizeof(MasterWriteData); i++)
  {
    printf(" %3d  | 0x%02X | 0x%02X \r\n", i, MasterWriteData[i], MasterReadData[i]);
    if(MasterReadData[i] != SlaveWriteData[i])
      ret = 1;
  }

  /* Print SPI Slave communication data */
  printf("SPI2 Send and Recv Data \r\n");
  printf("Index | Send | Recv \r\n");
  for(uint16_t i = 0; i < sizeof(SlaveWriteData); i++)
  {
    printf(" %3d  | 0x%02X | 0x%02X \r\n", i, SlaveWriteData[i], SlaveReadData[i]);
    if(SlaveReadData[i] != MasterWriteData[i])
      ret = 1;
  }

  if (ret)
    printf("SPI Slave Error \r\n");
  else
    printf("SPI Slave Success \r\n");

  while (1)
  {

  }
}

/**
 * @brief  Configures SPI Master.
 * @param  None
 * @retval None
 */
void SPI_MasterConfig(void)
{
  /* Enable SPI_Master Periph and GPIO clock */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOA| RCC_APB2Periph_SPI1, ENABLE);

  /* 
    SPI_CS      PA4
    SPI_CLK     PA5
    SPI_MISO    PA6
    SPI_MOSI    PA7
   */
  /* Configure SPI Master pins: CS, SCK, MISO and MOSI */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_4;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOA, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_5| GPIO_Pin_7;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOA, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_6;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOA, &GPIO_InitStructure);
  
  /* SPI_Master configuration ------------------------------------------------*/
  SPI_InitStructure.SPI_Direction = SPI_Direction_2Lines_FullDuplex;
  SPI_InitStructure.SPI_Mode = SPI_Mode_Master;
  SPI_InitStructure.SPI_DataSize = SPI_DataSize_8b;
  SPI_InitStructure.SPI_CPOL = SPI_CPOL_Low;
  SPI_InitStructure.SPI_CPHA = SPI_CPHA_2Edge;
  SPI_InitStructure.SPI_FirstBit = SPI_FirstBit_MSB;
  SPI_InitStructure.SPI_NSS = SPI_NSS_Soft;
  SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_32;
  SPI_InitStructure.SPI_CRCPolynomial = 7;
  SPI_Init(SPI1 , &SPI_InitStructure);

  /* Enable SPI_Master */
  SPI_Cmd(SPI1 , ENABLE);
}

/**
 * @brief  Configures SPI Slave.
 * @param  None
 * @retval None
 */
void SPI_SlaveConfig(void)
{
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOB, ENABLE);
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_SPI2, ENABLE);

  /*
    SPI2_CS      PB12
    SPI2_CLK     PB13
    SPI2_MISO    PB14
    SPI2_MOSI    PB15
   */
  /* Configure SPI Slave pins: CS, SCK, MISO and MOSI */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_12;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOB, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_14;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOB, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_13| GPIO_Pin_15;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOB, &GPIO_InitStructure);

  /* Configure and enable SPI_Slave interrupt -------------------------------*/
  NVIC_InitStructure.NVIC_IRQChannel = SPI2_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 1;
  NVIC_Init(&NVIC_InitStructure);

  /* SPI_Slave configuration ------------------------------------------------*/
  SPI_InitStructure.SPI_Direction = SPI_Direction_2Lines_FullDuplex;
  SPI_InitStructure.SPI_Mode = SPI_Mode_Slave;
  SPI_InitStructure.SPI_DataSize = SPI_DataSize_8b;
  SPI_InitStructure.SPI_CPOL = SPI_CPOL_Low;
  SPI_InitStructure.SPI_CPHA = SPI_CPHA_2Edge;
  SPI_InitStructure.SPI_FirstBit = SPI_FirstBit_MSB;
  SPI_InitStructure.SPI_NSS = SPI_NSS_Hard;
  SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_4;
  SPI_InitStructure.SPI_CRCPolynomial = 7;
  SPI_Init(SPI2 , &SPI_InitStructure);

  /* Enable SPI_Slave TXE interrupt */
  SPI_I2S_ITConfig(SPI2, SPI_I2S_IT_TXE, ENABLE);
  SPI_I2S_ITConfig(SPI2, SPI_I2S_IT_RXNE, ENABLE);

  /* Enable SPI_Slave */
  SPI_Cmd(SPI2 , ENABLE);
}

/* Slave Interrupt */
void SPI2_IRQHandler(void)
{
  if (SPI_I2S_GetITStatus(SPI2, SPI_I2S_IT_TXE) != RESET)
  {
    /* Send SPI_Master data */
    SPI_I2S_SendData(SPI2, SlaveWriteData[SlaveTxIndex++]);

    if (SlaveTxIndex >= sizeof(SlaveWriteData))
    {
      SPI_I2S_ITConfig(SPI1, SPI_I2S_IT_TXE, DISABLE);
    }
  }

  if (SPI_I2S_GetITStatus(SPI2, SPI_I2S_IT_RXNE) != RESET)
  {
    SlaveReadData[SlaveRxIndex++] = SPI_I2S_ReceiveData(SPI2);
    if(SlaveRxIndex >= sizeof(MasterWriteData))
    {
      SPI_I2S_ITConfig(SPI2, SPI_I2S_IT_RXNE, DISABLE);
      SPI_I2S_DeInit(SPI2);
      SlaveRxFlag = 1;
    }
  }
}

#ifdef  USE_FULL_ASSERT

/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t* file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  while (1)
  {

  }
}

#endif
