/* Includes ------------------------------------------------------------------*/
#include "stdio.h"
#include "mg32f157.h"
#include "bsp_usart1.h"
#include "gpio_bit_band.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define KEY0 1
#define KEY1 2

#define KEY0_State PAin(0)
#define KEY1_State !PAin(1)

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
GPIO_InitTypeDef GPIO_InitStructure;
CAN_InitTypeDef CAN_InitStructure;
CAN_FilterInitTypeDef CAN_FilterInitStructure;
NVIC_InitTypeDef NVIC_InitStructure;

CAN_Msg tx_message, rx_message;

/* Private function prototypes -----------------------------------------------*/
void Key_Config(void);
uint8_t kEY_Scan(void);
uint32_t RNG_Get(void);
void CAN_SendMessage(void);

/* Private functions ---------------------------------------------------------*/

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
  uint8_t key;

  /* Updating the system clock */
  SystemCoreClockUpdate();

  /* Configure the interrupt priority group */
  NVIC_PriorityGroupConfig(NVIC_PriorityGroup_2);

  /* Turn on the peripheral clock */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOA, ENABLE);
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_CAN1, ENABLE);
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_RNG, ENABLE);

  TRNG_Cmd(ENABLE);
  Key_Config();
  usart1_init(96000000, 115200);
  printf(">>>>>>>>>>  CAN Loop Back Communication  <<<<<<<<<<\r\n");

  /* Enable the CAN1 Interrupt Channel */
  NVIC_InitStructure.NVIC_IRQChannel = USB_DMA_CAN1_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 1;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 1;
  NVIC_Init(&NVIC_InitStructure);

  /* Set the CAN peripheral to reset mode */
  CAN_Reset(CAN1, ENABLE);

  /* CAN1 Class mode configuration */
  CAN_InitStructure.CAN_Mode = CAN_Mode_LoopBack_Silent;
  CAN_InitStructure.CAN_Sample = CAN_Sample_Once;
  CAN_InitStructure.CAN_Prescaler = 6;
  CAN_InitStructure.CAN_SJW = CAN_SJW_2tq;
  CAN_InitStructure.CAN_SEG1 = CAN_SEG1_5tq;
  CAN_InitStructure.CAN_SEG2 = CAN_SEG2_2tq;
  CAN_Init(CAN1, &CAN_InitStructure);

  /* CAN1 Filter configuration */
  CAN_FilterInitStructure.Filter_Mode = CAN_Filter_Mode_Single;
  CAN_FilterInitStructure.Filter_FrameFormat = CAN_Filter_FrameFormat_Extended;
  CAN_FilterInitStructure.Filter1_Type = CAN_Filter_Type_Data;
  CAN_FilterInitStructure.Filter1_RTRCmd = DISABLE;
  CAN_FilterInitStructure.Filter1_Code = 0xFA37B58;
  CAN_FilterInitStructure.Filter1_Mask = 0xFFFFFFFF;
  CAN_FilterInit(CAN1, &CAN_FilterInitStructure);

  CAN_ITConfig(CAN1, CAN_IT_RI, ENABLE);

  /* Set peripheral CAN to exit the reset mode */
  CAN_Reset(CAN1, DISABLE);

  while (1)
  {
    key = kEY_Scan();

    if (KEY0 == key)
      /* Press KEY1 to send a frame message */
      CAN_SendMessage();
  }
}

/**
 * @brief  Configure key input.
 * @param  None
 * @retval None
 */
void Key_Config(void)
{
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOA, ENABLE);

  /* KEY0 */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IPD;
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_0;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOA, &GPIO_InitStructure);

  /* KEY1 */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IPU;
  GPIO_InitStructure.GPIO_Pin =  GPIO_Pin_1;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOA, &GPIO_InitStructure);
}

/**
 * @brief  Scan key input.
 * @param  None
 * @retval None
 */
uint8_t kEY_Scan(void)
{
  uint16_t i = 0;
  static uint8_t key_up = 1;

  if(key_up & (KEY0_State || KEY1_State))
  {
    i = 0xFFF;
    while (i--);
    key_up = 0;
    if (KEY0_State)  return KEY0;
    else if (KEY1_State) return KEY1;
  }
  if (!(KEY0_State || KEY1_State))
    key_up = 1;
  return 0;
}

/**
 * @brief  Use a random number generator to generate a 32-bit piece of data.
 * @param  None
 * @retval 32-bit random data
 */
uint32_t RNG_Get(void)
{
  while (TRNG_GetFlagStatus(TRNG_FLAG_DRDY) == RESET);

  return TRNG_GetData();
}

/**
 * @brief  Generate A frame message and send it through the peripheral CAN.
 * @param  None
 * @retval None
 */
void CAN_SendMessage(void)
{
  uint32_t data;

  data = RNG_Get();

  if (data % 2 == 0)
  {
    tx_message.IDType = CAN_IDType_Extended;
    tx_message.Identifier = (data & 0x7FF);
  }
  else
  {
    tx_message.IDType = CAN_Filter_Type_Data;
    tx_message.Identifier = (data & 0x3FFFFFFF);
  }
  tx_message.DataLength = (data % 8);
  for (uint8_t i = 0; i < tx_message.DataLength; i++)
  {
    tx_message.Data[i] = (RNG_Get() & 0xFF);
  }
  
  while (CAN_GetFlagStatus(CAN1, CAN_Status_TBS) == RESET);
  CAN_Transmit(CAN1, &tx_message, ENABLE);
}

void USB_DMA_CAN1_IRQHandler(void)
{
  static uint32_t recv_count;

  if (CAN_GetITStatus(CAN1, CAN_Flag_RI) == SET)
  {
    CAN_Receive(CAN1, &rx_message);

    recv_count++;

    printf(">>>>>>>>>>  Interrupt recv message num is [%d]  <<<<<<<<<< \r\n", recv_count);
    if (rx_message.IDType == CAN_IDType_Extended)
      printf("  Extern ID is 0x%08X\r\n", rx_message.Identifier);
    else
      printf("  Std ID is 0x%04X\r\n", rx_message.Identifier);

    if(rx_message.FrameType == CAN_FrameType_Data)
      printf("  Data Frame \r\n");
    else
      printf("  Remote Frame \r\n");

    printf("  Recv Data Length %02d \r\n", rx_message.DataLength);

    for (uint8_t i = 0; i < rx_message.DataLength; i++)
    {
      printf("    data[%d] = 0x%02X\r\n", i, rx_message.Data[i]);
    }

    CAN_ClearITPendingBit(CAN1, CAN_Flag_RI);
  }
}

#ifdef  USE_FULL_ASSERT

/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t* file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  while (1)
  {
  }
}

#endif
