/**
 * @file    mg32f157_sdio.c
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file provides all the SDIO firmware functions.
 */

/* Includes ------------------------------------------------------------------*/
#include "mg32f157_sdio.h"
#include "mg32f157_rcc.h"

/** @addtogroup MG32F157_StdPeriph_Driver
  * @{
  */

/** @defgroup SDIO
  * @brief SDIO driver modules
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/** @defgroup SDIO_StdPeriph_Functions
  * @{
  */

/**
 * @brief  Deinitializes the SDIOx peripheral registers to their default reset values.
 * @param  SDIOx: Pointer to selected SDIO peripheral.
 *         This parameter can be one of the following values:
 *         SDIO.
 * @return None
 */
void SDIO_DeInit(SDIO_TypeDef *SDIOx)
{
  SDIOx->CTRL = 0x0;
  SDIOx->CLKDIV = 0x0;
  SDIOx->CLKENA = 0x0;
  SDIOx->INTMASK = 0x0;
  SDIOx->BLKSIZ = 0x200;;
  SDIOx->BYTCNT = 0x200;
  SDIOx->CMDARG = 0x0;
  SDIOx->CMD = 0x0;
  SDIOx->MINTSTS = 0x0;
  SDIOx->RINTSTS = 0x0;
  SDIOx->STATUS = 0x0;
  SDIOx->FIFOTH = 0x0;
  SDIOx->TCBCNT = 0x0;
}

/**
 * @brief  Initializes the SDIO according to the specified
 *         parameters in the SDIO_InitTypeDef and create the associated handle.
 * @param  SDIOx: Pointer to SDIO register base.
 * @param  SDIO_InitStruct: SDIO initialization structure.
 * @return None
 */
void SDIO_Init(SDIO_TypeDef *SDIOx, SDIO_InitTypeDef *SDIO_InitStruct)
{
  uint32_t tmpreg = 0U;

  /* Check the parameters */
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));
  assert_param(IS_SDIO_CLOCK_POWER_SAVE(SDIO_InitStruct->ClockPowerSave));
  assert_param(IS_SDIO_BUS_WIDE(SDIO_InitStruct->BusWide));
  assert_param(IS_SDIO_CLKDIV(SDIO_InitStruct->ClockDiv));
  
  tmpreg = SDIO_InitStruct->ClockDiv;
  tmpreg <<= 8;
  tmpreg |= SDIO_InitStruct->ClockDiv;
  tmpreg <<= 8;
  tmpreg |= SDIO_InitStruct->ClockDiv;
  tmpreg <<= 8;
  tmpreg |= SDIO_InitStruct->ClockDiv;
  SDIOx->CLKDIV = tmpreg; 
  SDIOx->CLKENA |= (SDIO_InitStruct->ClockPowerSave << 16);

  if( SDIO_InitStruct->BusWide == SDIO_BUS_WIDE_4B)
    SDIOx->CTYPE = 0x00000001;
  else
    SDIOx->CTYPE = 0x00000000;
}

/**
 * @brief  Configure the width of the SDIO data bus.
 * @param  SDIOx: Pointer to SDIO register base
 * @param  BusWidth: SDIO data bus width
 * @return None
 */
void SDIO_ConfigBusWidth(SDIO_TypeDef *SDIOx, uint32_t BusWidth)
{
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));
  assert_param(IS_SDIO_BUS_WIDE(BusWidth));

  if (SDIO_BUS_WIDE_1B == BusWidth)
  {
    SDIOx->CTYPE = 0x0;
  }
  else if (SDIO_BUS_WIDE_4B == BusWidth)
  {
    SDIOx->CTYPE = ((0x01 << 0));
  }
  else if (SDIO_BUS_WIDE_8B == BusWidth)
  {
    SDIOx->CTYPE = ((0x01 << 0) | (0x01 << 16));
  }
}

/**
 * @brief  Read data (word) from Rx FIFO in blocking mode (polling)
 * @param  SDIOx: Pointer to SDIO register base
 * @return SDIO FIFO data
 */
uint32_t SDIO_ReadFIFO(SDIO_TypeDef *SDIOx)
{
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  /* Read data from Rx FIFO */
  return (SDIOx->FIFO);
}

/**
 * @brief  Write data (word) to Tx FIFO in blocking mode (polling)
 * @param  SDIOx: Pointer to SDIO register base
 * @param  pWriteData: pointer to data to write
 * @return None
 */
void SDIO_WriteFIFO(SDIO_TypeDef *SDIOx, uint32_t *pWriteData)
{ 
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  /* Write data to FIFO */ 
  SDIOx->FIFO = *pWriteData;
}

/**
 * @brief  Set SDIO Power state to ON.
 * @param  SDIOx: Pointer to SDIO register base
 * @return HAL status
 */
void SDIO_PowerON(SDIO_TypeDef *SDIOx)
{  
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  /* Set power state to ON */ 
  SDIOx->PWREN = 0x3FFFFFFF;
}

/**
 * @brief  Set SDIO Power state to OFF.
 * @param  SDIOx: Pointer to SDIO register base
 * @return HAL status
 */
void SDIO_PowerOFF(SDIO_TypeDef *SDIOx)
{
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  /* Set power state to OFF */
  SDIOx->PWREN &= (~0x3FFFFFFF);
}

/**
 * @brief  Get SDIO Power state.
 * @param  SDIOx: Pointer to SDIO register base
 * @return Power status of the controller. The returned value can be one of the 
 *         following values:
 *            - 0x00: Power OFF
 *            - 0x02: Power UP
 *            - 0x03: Power ON 
 */
uint32_t SDIO_GetPower(SDIO_TypeDef *SDIOx)  
{
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  if (SDIOx->PWREN & 0x3FFFFFFF)
  {
    return SDIO_PowerState_ON;
  }
  return SDIO_PowerState_OFF;
}

/**
 * @brief  Enables or disables the SDIO Clock.
 * @param  SDIOx: Pointer to SDIO register base
 * @param  NewState: new state of the SDIO Clock. This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void SDIO_ClockCmd(SDIO_TypeDef *SDIOx, FunctionalState NewState)
{
  /* Check the parameters */
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));
  assert_param(IS_FUNCTIONAL_STATE(NewState));
  
  if (NewState != DISABLE)
  {
    SDIOx->CLKENA |= SDIO_CLKENA_CLK_ENABLE;
  }
  else
  {
    SDIOx->CLKENA &= (~SDIO_CLKENA_CLK_ENABLE);
  }
}

/**
 * @brief  Update the SDIO Clock.
 * @param  SDIOx: Pointer to SDIO register base
 * @return None
 */
void SDIO_ClockUpdate(SDIO_TypeDef *SDIOx)
{
  /* Check the parameters */
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  SDIOx->CMD = SDIO_CMD_CLK_UPDATE;
  /*  wait start_cmd cleared */
  while (SDIOx->CMD & (SDIO_CMD_COMPLATE_FLAG))
  {
  }
}

/**
 * @brief  Reset the SDIO Controller.
 * @param  SDIOx: Pointer to SDIO register base
 * @return None
 */
void SDIO_ControlReset(SDIO_TypeDef *SDIOx)
{
  /* Check the parameters */
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  SDIOx->CTRL = SDIO_CTRL_RESET;

  while (SDIOx->CTRL & SDIO_CTRL_RESET)
  {
  }
}

/**
 * @brief  Enables or disables the SDIO DMA request.
 * @param  SDIOx: Pointer to SDIO register base
 * @param  NewState: new state of the selected SDIO DMA request.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void SDIO_DMACmd(SDIO_TypeDef *SDIOx, FunctionalState NewState)
{
  /* Check the parameters */
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));
  assert_param(IS_FUNCTIONAL_STATE(NewState));
  
  if (NewState != DISABLE)
  {
    SDIO->BMOD |= SDIO_BMOD_DE;
  }
  else
  {
    SDIO->BMOD &= (~SDIO_BMOD_DE);
  }
}

/**
 * @brief  Enables or disables the SDIO interrupts.
 * @param  SDIOx: Pointer to SDIO register base
 * @param  SDIO_IT: specifies the SDIO interrupt sources to be enabled or disabled.
 *         This parameter can be one or a combination of the following values:
 *         @arg SDIO_IT_CCRCFAIL: Command response received (CRC check failed) interrupt.
 *         @arg SDIO_IT_DCRCFAIL: Data block sent/received (CRC check failed) interrupt.
 *         @arg SDIO_IT_CTIMEOUT: Command response timeout interrupt.
 *         @arg SDIO_IT_DTIMEOUT: Data timeout interrupt.
 *         @arg SDIO_IT_TXUNDERR: Transmit FIFO underrun error interrupt.
 *         @arg SDIO_IT_RXOVERR:  Received FIFO overrun error interrupt.
 *         @arg SDIO_IT_CMDREND:  Command response received (CRC check passed) interrupt.
 *         @arg SDIO_IT_CMDSENT:  Command sent (no response required) interrupt.
 *         @arg SDIO_IT_DATAEND:  Data end (data counter, SDIDCOUNT, is zero) interrupt.
 *         @arg SDIO_IT_DBCKEND:  Data block sent/received (CRC check passed) interrupt.
 *         @arg SDIO_IT_CMDACT:   Command transfer in progress interrupt.
 *         @arg SDIO_IT_TXACT:    Data transmit in progress interrupt.
 *         @arg SDIO_IT_RXACT:    Data receive in progress interrupt.
 *         @arg SDIO_IT_TXFIFOHE: Transmit FIFO Half Empty interrupt.
 *         @arg SDIO_IT_RXFIFOHF: Receive FIFO Half Full interrupt.
 *         @arg SDIO_IT_TXFIFOF:  Transmit FIFO full interrupt.
 *         @arg SDIO_IT_RXFIFOF:  Receive FIFO full interrupt.
 *         @arg SDIO_IT_TXFIFOE:  Transmit FIFO empty interrupt.
 *         @arg SDIO_IT_RXFIFOE:  Receive FIFO empty interrupt.
 *         @arg SDIO_IT_TXDAVL:   Data available in transmit FIFO interrupt.
 *         @arg SDIO_IT_RXDAVL:   Data available in receive FIFO interrupt.
 *         @arg SDIO_IT_SDIOIT:   SD I/O interrupt received interrupt.
 * @param  NewState: new state of the selected SDIO DMA request.
 *         This parameter can be: ENABLE or DISABLE.
 * @return None
 */
void SDIO_ITConfig(SDIO_TypeDef *SDIOx, uint32_t SDIO_IT, FunctionalState NewState)
{
  /* Check the parameters */
  // assert_param(IS_SDIO_IT(SDIO_IT));
  assert_param(IS_FUNCTIONAL_STATE(NewState));
  
  if (NewState != DISABLE)
  {
    /* Enable the SDIO interrupts */
    SDIO->INTMASK |= SDIO_IT;
  }
  else
  {
    /* Disable the SDIO interrupts */
    SDIO->INTMASK &= ~SDIO_IT;
  } 
}


/**
 * @brief  Checks whether the specified SDIO flag is set or not.
 * @param  SDIOx: Pointer to SDIO register base
 * @param  SDIO_FLAG: specifies the flag to check.
 *         This parameter can be one of the following values:
 *         @arg SDIO_FLAG_CCRCFAIL: Command response received (CRC check failed).
 *         @arg SDIO_FLAG_DCRCFAIL: Data block sent/received (CRC check failed).
 *         @arg SDIO_FLAG_CTIMEOUT: Command response timeout.
 *         @arg SDIO_FLAG_DTIMEOUT: Data timeout.
 *         @arg SDIO_FLAG_TXUNDERR: Transmit FIFO underrun error.
 *         @arg SDIO_FLAG_RXOVERR: Received FIFO overrun error.
 *         @arg SDIO_FLAG_CMDREND: Command response received (CRC check passed).
 *         @arg SDIO_FLAG_CMDSENT: Command sent (no response required).
 *         @arg SDIO_FLAG_DATAEND: Data end (data counter, SDIDCOUNT, is zero).
 *         @arg SDIO_FLAG_DBCKEND: Data block sent/received (CRC check passed).
 *         @arg SDIO_FLAG_CMDACT: Command transfer in progress.
 *         @arg SDIO_FLAG_TXACT: Data transmit in progress.
 *         @arg SDIO_FLAG_RXACT: Data receive in progress.
 *         @arg SDIO_FLAG_TXFIFOHE: Transmit FIFO Half Empty.
 *         @arg SDIO_FLAG_RXFIFOHF: Receive FIFO Half Full.
 *         @arg SDIO_FLAG_TXFIFOF: Transmit FIFO full.
 *         @arg SDIO_FLAG_RXFIFOF: Receive FIFO full.
 *         @arg SDIO_FLAG_TXFIFOE: Transmit FIFO empty.
 *         @arg SDIO_FLAG_RXFIFOE: Receive FIFO empty.
 *         @arg SDIO_FLAG_TXDAVL: Data available in transmit FIFO.
 *         @arg SDIO_FLAG_RXDAVL: Data available in receive FIFO.
 *         @arg SDIO_FLAG_SDIOIT: SD I/O interrupt received.
 * @return The new state of SDIO_FLAG (SET or RESET).
 */
FlagStatus SDIO_GetFlagStatus(SDIO_TypeDef *SDIOx, uint32_t SDIO_FLAG)
{
  FlagStatus bitstatus = RESET;

  if ((SDIO->RINTSTS & SDIO_FLAG) != (uint32_t)RESET)
  {
    bitstatus = SET;
  }
  else
  {
    bitstatus = RESET;
  }
  return bitstatus;
}

/**
 * @brief  Clears the SDIO's pending flags.
 * @param  SDIOx: Pointer to SDIO register base
 * @param  SDIO_FLAG: specifies the flag to clear.
 *         This parameter can be one or a combination of the following values:
 *         @arg SDIO_FLAG_CCRCFAIL: Command response received (CRC check failed).
 *         @arg SDIO_FLAG_DCRCFAIL: Data block sent/received (CRC check failed).
 *         @arg SDIO_FLAG_CTIMEOUT: Command response timeout.
 *         @arg SDIO_FLAG_DTIMEOUT: Data timeout.
 *         @arg SDIO_FLAG_TXUNDERR: Transmit FIFO underrun error.
 *         @arg SDIO_FLAG_RXOVERR: Received FIFO overrun error.
 *         @arg SDIO_FLAG_CMDREND: Command response received (CRC check passed).
 *         @arg SDIO_FLAG_CMDSENT: Command sent (no response required).
 *         @arg SDIO_FLAG_DATAEND: Data end (data counter, SDIDCOUNT, is zero).
 *         @arg SDIO_FLAG_DBCKEND: Data block sent/received (CRC check passed).
 *         @arg SDIO_FLAG_SDIOIT: SD I/O interrupt received.
 * @return None
 */
void SDIO_ClearFlagStatus(SDIO_TypeDef *SDIOx, uint32_t SDIO_FLAG)
{
  SDIO->RINTSTS = SDIO_FLAG;
}

/**
 * @brief  Checks whether the specified SDIO interrupt has occurred or not.
 * @param  SDIOx: Pointer to SDIO register base
 * @param  SDIO_IT: specifies the SDIO interrupt source to check.
 *         This parameter can be one of the following values:
 *         @arg SDIO_IT_CCRCFAIL: Command response received (CRC check failed) interrupt.
 *         @arg SDIO_IT_DCRCFAIL: Data block sent/received (CRC check failed) interrupt.
 *         @arg SDIO_IT_CTIMEOUT: Command response timeout interrupt.
 *         @arg SDIO_IT_DTIMEOUT: Data timeout interrupt.
 *         @arg SDIO_IT_TXUNDERR: Transmit FIFO underrun error interrupt.
 *         @arg SDIO_IT_RXOVERR: Received FIFO overrun error interrupt.
 *         @arg SDIO_IT_CMDREND: Command response received (CRC check passed) interrupt.
 *         @arg SDIO_IT_CMDSENT: Command sent (no response required) interrupt.
 *         @arg SDIO_IT_DATAEND: Data end (data counter, SDIDCOUNT, is zero) interrupt.
 *         @arg SDIO_IT_DBCKEND: Data block sent/received (CRC check passed) interrupt.
 *         @arg SDIO_IT_CMDACT: Command transfer in progress interrupt.
 *         @arg SDIO_IT_TXACT: Data transmit in progress interrupt.
 *         @arg SDIO_IT_RXACT: Data receive in progress interrupt.
 *         @arg SDIO_IT_TXFIFOHE: Transmit FIFO Half Empty interrupt.
 *         @arg SDIO_IT_RXFIFOHF: Receive FIFO Half Full interrupt.
 *         @arg SDIO_IT_TXFIFOF: Transmit FIFO full interrupt.
 *         @arg SDIO_IT_RXFIFOF: Receive FIFO full interrupt.
 *         @arg SDIO_IT_TXFIFOE: Transmit FIFO empty interrupt.
 *         @arg SDIO_IT_RXFIFOE: Receive FIFO empty interrupt.
 *         @arg SDIO_IT_TXDAVL: Data available in transmit FIFO interrupt.
 *         @arg SDIO_IT_RXDAVL: Data available in receive FIFO interrupt.
 *         @arg SDIO_IT_SDIOIT: SD I/O interrupt received interrupt.
 * @return The new state of SDIO_IT (SET or RESET).
 */
ITStatus SDIO_GetITStatus(SDIO_TypeDef *SDIOx, uint32_t SDIO_IT)
{ 
  ITStatus bitstatus = RESET;
  
  /* Check the parameters */
  assert_param(IS_SDIO_IT(SDIO_IT));

  if ((SDIO->RINTSTS & SDIO_IT) != (uint32_t)RESET)  
  {
    bitstatus = SET;
  }
  else
  {
    bitstatus = RESET;
  }
  return bitstatus;
}

/**
 * @brief  Clears the SDIO's interrupt pending bits.
 * @param  SDIOx: Pointer to SDIO register base.
 * @param  SDIO_IT: specifies the interrupt pending bit to clear.
 *         This parameter can be one or a combination of the following values:
 *         @arg SDIO_IT_CCRCFAIL: Command response received (CRC check failed) interrupt.
 *         @arg SDIO_IT_DCRCFAIL: Data block sent/received (CRC check failed) interrupt.
 *         @arg SDIO_IT_CTIMEOUT: Command response timeout interrupt.
 *         @arg SDIO_IT_DTIMEOUT: Data timeout interrupt.
 *         @arg SDIO_IT_TXUNDERR: Transmit FIFO underrun error interrupt.
 *         @arg SDIO_IT_RXOVERR: Received FIFO overrun error interrupt.
 *         @arg SDIO_IT_CMDREND: Command response received (CRC check passed) interrupt.
 *         @arg SDIO_IT_CMDSENT: Command sent (no response required) interrupt.
 *         @arg SDIO_IT_DATAEND: Data end (data counter, SDIO_DCOUNT, is zero) interrupt.
 *         @arg SDIO_IT_SDIOIT: SD I/O interrupt received interrupt.
 * @return None
 */
void SDIO_ClearITPendingBit(SDIO_TypeDef *SDIOx, uint32_t SDIO_IT)
{
  /* Check the parameters */
  assert_param(IS_SDIO_IT(SDIO_IT));

  SDIO->RINTSTS = SDIO_IT;
}

/**
 * @brief  Configure the SDIO command path according to the specified parameters in
 *         SDIO_CmdInitTypeDef structure and send the command.
 * @param  SDIOx: Pointer to SDIO register base.
 * @param  Command: pointer to a SDIO_CmdInitTypeDef structure that contains
 *         the configuration information for the SDIO command.
 * @return None
 */
void SDIO_SendCommand(SDIO_TypeDef *SDIOx, SDIO_CmdInitTypeDef *Command)
{
  uint32_t tmpreg = 0U;

  /* Check the parameters */
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));
  assert_param(IS_SDIO_CMD_INDEX(Command->CmdIndex));
  assert_param(IS_SDIO_RESPONSE(Command->Response));
  assert_param(IS_SDIO_CPSM(Command->CPSM));

  /* Set the SDIO Argument value */
  SDIOx->CMDARG = Command->Argument;

  /* Set SDIO command parameters */
  tmpreg |= (uint32_t)(0xA0000000 |\
                       Command->CmdIndex |\
                       Command->Response |\
                       Command->CPSM);
  
  /* Write to SDIO CMD register */
  SDIOx->CMD = ((SDIOx->CMD & (~0x0020FFFF)) | tmpreg);
   
  tmpreg = SDIOx->CMD;
  while((tmpreg & 0x80000000)!= 0x0) 
  {
    tmpreg = SDIOx->CMD;
  };
}

/**
 * @brief  Return the command index of last command for which response received
 * @param  SDIOx: Pointer to SDIO register base
 * @return Command index of the last command response received
 */
uint8_t SDIO_GetCommandResponse(SDIO_TypeDef *SDIOx)
{
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  return (uint8_t)((SDIOx->STATUS>>11)&0x0000003F); //bit 16:11
}

/**
 * @brief  Return the response received from the card for the last command
 * @param  SDIOx: Pointer to SDIO register base
 * @param  Response: Specifies the SDIO response register.
 *         This parameter can be one of the following values:
 *         @arg SDIO_RESP1: Response Register 0
 *         @arg SDIO_RESP2: Response Register 1
 *         @arg SDIO_RESP3: Response Register 2
 *         @arg SDIO_RESP4: Response Register 3
 * @return The Corresponding response register value
 */
uint32_t SDIO_GetResponse(SDIO_TypeDef *SDIOx, uint32_t Response)
{
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  __IO uint32_t tmp = 0U;

  /* Check the parameters */
  assert_param(IS_SDIO_RESP(Response));

  /* Get the response */
  tmp = (uint32_t)&(SDIOx->RESP0) + Response;

  return (*(__IO uint32_t *) tmp);
}

/**
 * @brief  Configure the SDIO data path according to the specified
 *         parameters in the SDIO_DataInitTypeDef.
 * @param  SDIOx: Pointer to SDIO register base
 * @param  Data: pointer to a SDIO_DataInitTypeDef structure
 *         that contains the configuration information for the SDIO data.
 * @return None
 */
void SDIO_ConfigData(SDIO_TypeDef *SDIOx, SDIO_DataInitTypeDef* Data)
{
  assert_param(IS_SDIO_ALL_PERIPH(SDIOx));

  /* Check the parameters */
  assert_param(IS_SDIO_DATA_LENGTH(Data->DataLength));
  assert_param(IS_SDIO_BLOCK_SIZE(Data->DataBlockSize));
  assert_param(IS_SDIO_TRANSFER_DIR(Data->TransferDir));
  assert_param(IS_SDIO_TRANSFER_MODE(Data->TransferMode));
  assert_param(IS_SDIO_DPSM(Data->DPSM));
  /* Set the SDIO Data TimeOut value */
  SDIO->TMOUT = (Data->DataTimeOut);

  /* Set the SDIO DataLength value */
  SDIO->BYTCNT = Data->DataLength;

  SDIO->BLKSIZ = Data->DataBlockSize;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
