/**
 * @file    bsp_sdio_sdcard.c
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file provides all the RCC firmware functions.
 */

/* Includes ------------------------------------------------------------------*/
#include "bsp_sdio_sdcard.h"
#include "mg32f157.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/** 
  * @brief SDIO Static flags, TimeOut, FIFO Address.
  */
#define NULL 0
#define SDIO_CMD0TIMEOUT                ((uint32_t)0x00010000)

/** 
  * @brief  Mask for errors Card Status R1 (OCR Register) 
  */
#define SD_OCR_ADDR_OUT_OF_RANGE        ((uint32_t)0x80000000)
#define SD_OCR_ADDR_MISALIGNED          ((uint32_t)0x40000000)
#define SD_OCR_BLOCK_LEN_ERR            ((uint32_t)0x20000000)
#define SD_OCR_ERASE_SEQ_ERR            ((uint32_t)0x10000000)
#define SD_OCR_BAD_ERASE_PARAM          ((uint32_t)0x08000000)
#define SD_OCR_WRITE_PROT_VIOLATION     ((uint32_t)0x04000000)
#define SD_OCR_LOCK_UNLOCK_FAILED       ((uint32_t)0x01000000)
#define SD_OCR_COM_CRC_FAILED           ((uint32_t)0x00800000)
#define SD_OCR_ILLEGAL_CMD              ((uint32_t)0x00400000)
#define SD_OCR_CARD_ECC_FAILED          ((uint32_t)0x00200000)
#define SD_OCR_CC_ERROR                 ((uint32_t)0x00100000)
#define SD_OCR_GENERAL_UNKNOWN_ERROR    ((uint32_t)0x00080000)
#define SD_OCR_STREAM_READ_UNDERRUN     ((uint32_t)0x00040000)
#define SD_OCR_STREAM_WRITE_OVERRUN     ((uint32_t)0x00020000)
#define SD_OCR_CID_CSD_OVERWRIETE       ((uint32_t)0x00010000)
#define SD_OCR_WP_ERASE_SKIP            ((uint32_t)0x00008000)
#define SD_OCR_CARD_ECC_DISABLED        ((uint32_t)0x00004000)
#define SD_OCR_ERASE_RESET              ((uint32_t)0x00002000)
#define SD_OCR_AKE_SEQ_ERROR            ((uint32_t)0x00000008)
#define SD_OCR_ERRORBITS                ((uint32_t)0xFDFFE008)

/** 
  * @brief  Masks for R6 Response
  */
#define SD_R6_GENERAL_UNKNOWN_ERROR     ((uint32_t)0x00002000)
#define SD_R6_ILLEGAL_CMD               ((uint32_t)0x00004000)
#define SD_R6_COM_CRC_FAILED            ((uint32_t)0x00008000)

#define SD_VOLTAGE_WINDOW_SD            ((uint32_t)0x80100000)
#define SD_HIGH_CAPACITY                ((uint32_t)0x40000000)
#define SD_STD_CAPACITY                 ((uint32_t)0x00000000)
#define SD_CHECK_PATTERN                ((uint32_t)0x000001AA)

#define SD_MAX_VOLT_TRIAL               ((uint32_t)0x0000FFFF)
#define SD_ALLZERO                      ((uint32_t)0x00000000)

#define SD_WIDE_BUS_SUPPORT             ((uint32_t)0x00040000)
#define SD_SINGLE_BUS_SUPPORT           ((uint32_t)0x00010000)
#define SD_CARD_LOCKED                  ((uint32_t)0x02000000)

#define SD_DATATIMEOUT                 ((uint32_t)0xFFFFFFFF)
#define SD_0TO7BITS                     ((uint32_t)0x000000FF)
#define SD_8TO15BITS                    ((uint32_t)0x0000FF00)
#define SD_16TO23BITS                   ((uint32_t)0x00FF0000)
#define SD_24TO31BITS                   ((uint32_t)0xFF000000)
#define SD_MAX_DATA_LENGTH              ((uint32_t)0x01FFFFFF)

#define SD_HALFFIFO                     ((uint32_t)0x00000008)
#define SD_HALFFIFOBYTES                ((uint32_t)0x00000020)

/** 
  * @brief  Command Class Supported 
  */
#define SD_CCCC_LOCK_UNLOCK             ((uint32_t)0x00000080)
#define SD_CCCC_WRITE_PROT              ((uint32_t)0x00000040)
#define SD_CCCC_ERASE                   ((uint32_t)0x00000020)

/** 
  * @brief  Following commands are SD Card Specific commands.
  *         SDIO_APP_CMD should be sent before sending these commands. 
  */
#define SDIO_SEND_IF_COND               ((uint32_t)0x00000008)

/* Private variables ---------------------------------------------------------*/ 
static uint32_t CardType = SDIO_STD_CAPACITY_SD_CARD_V1_1;
static uint32_t CSD_Tab[4], CID_Tab[4], RCA = 0;
static uint8_t SDSTATUS_Tab[16];
__IO uint32_t StopCondition = 0;
__IO uint32_t TransferEnd = 0;
__IO SD_Error TransferError = SD_OK;
SD_CardInfo SDCardInfo;

SDIO_InitTypeDef SDIO_InitStructure;
SDIO_CmdInitTypeDef SDIO_CmdInitStructure;
SDIO_DataInitTypeDef SDIO_DataInitStructure;

/* Private function prototypes -----------------------------------------------*/
static SD_Error SDIO_CmdError(void);
static SD_Error SDIO_CmdResp1Error(uint8_t SD_CMD);
static SD_Error SDIO_CmdResp2Error(void);
static SD_Error SDIO_CmdResp3Error(void);
static SD_Error SDIO_CmdResp7Error(void);
static SD_Error SDIO_CmdResp6Error(uint8_t SD_CMD, uint16_t *pRCA);

static SD_Error FindSCR(uint16_t rca, uint32_t *pscr);
static SD_Error SDEnWideBus(FunctionalState NewState);
static SD_Error IsCardProgramming(uint8_t *pstatus);

static void GPIO_Configuration(void);

/* Private functions ---------------------------------------------------------*/
void SD_DeInit(void)
{
  GPIO_InitTypeDef  GPIO_InitStructure;
  
  /*!< Disable SDIO Clock */
  SDIO_ClockCmd(SDIO, DISABLE);
  SDIO_ClockUpdate(SDIO);
  
  /*!< Set Power State to OFF */
  SDIO_PowerOFF(SDIO);

  /*!< Disable the SDIO AHB Clock */
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_SDIO, DISABLE);

  /*!< Configure PC.08, PC.09, PC.10, PC.11, PC.12 pin: D0, D1, D2, D3, CLK pin */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_8 | GPIO_Pin_9 | GPIO_Pin_10 | GPIO_Pin_11 | GPIO_Pin_12;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
  GPIO_Init(GPIOC, &GPIO_InitStructure);

  /*!< Configure PD.02 CMD line */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_2;
  GPIO_Init(GPIOD, &GPIO_InitStructure); 
}

/**
  * @brief  Initializes the SD according to the specified parameters in the 
            SD_HandleTypeDef and create the associated handle.
  * @param  None
  * @retval HAL status
  */
SD_Error SD_Init(void)
{
  SD_Error errorstatus = SD_OK;
  /*!< Enable the SDIO AHB Clock */
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_SDIO, ENABLE);
  
  GPIO_Configuration();

  SDIO_DeInit(SDIO);

  errorstatus = SD_PowerON();
  if(errorstatus != SD_OK)
  {
    return errorstatus;
  }

  /* Initialize the Card parameters */
  errorstatus = SD_InitializeCards();
  if (errorstatus != SD_OK)
  {
    /*!< CMD Response TimeOut (wait for CMDSENT flag) */
    return(errorstatus);
  }

  SDIO_InitStructure.ClockPowerSave = SDIO_CLOCK_POWER_SAVE_DISABLE;
  SDIO_InitStructure.BusWide = SDIO_BUS_WIDE_1B;
  SDIO_InitStructure.ClockDiv = SDIO_TRANSFER_CLK_DIV;
  SDIO_Init(SDIO, &SDIO_InitStructure);

  /* update card clock configuration only to card clock domain */
  SDIO_ClockUpdate(SDIO);

  if (errorstatus == SD_OK)
  {
    errorstatus = SD_GetCardInfo(&SDCardInfo);
  }

  if (errorstatus == SD_OK)
  {
    /* code */
    errorstatus = SD_SelectDeselect((uint32_t)(SDCardInfo.RCA << 16));
  }

  if (errorstatus == SD_OK)
  {
    errorstatus = SD_EnableWideBusOperation(SDIO_BUS_WIDE_4B);
  }

  return errorstatus;
}

/**
  * @brief  Returns the current card's state.
  * @param  None
  * @retval SDCardState: SD Card Error or SD Card Current State.
  */
SDCardState SD_GetState(void)
{
  SD_Error errorstatus = SD_OK;
  uint32_t resp1 = 0;

  /*-------------- CMD13 -------------------*/
  SDIO_CmdInitStructure.Argument = (uint32_t)RCA << 16;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SEND_STATUS;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;   
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SEND_STATUS);

  if (SD_OK != errorstatus)
  {
    return SD_CARD_ERROR;
  }

  resp1 = SDIO_GetResponse(SDIO, SDIO_RESP1);
  return (SDCardState)((resp1 >> 9) & 0x0F);
}

/**
  * @brief  Gets the cuurent sd card data transfer status.
  * @param  None
  * @retval SDTransferState: Data Transfer state.
  *         This value can be:
  *        - SD_TRANSFER_OK: No data transfer is acting
  *        - SD_TRANSFER_BUSY: Data transfer is acting
  */
SDTransferState SD_GetStatus(void)
{
  SDCardState cardstate =  SD_CARD_TRANSFER;

  cardstate = SD_GetState();
  
  if (cardstate == SD_CARD_TRANSFER)
  {
    return(SD_TRANSFER_OK);
  }
  else if(cardstate == SD_CARD_ERROR)
  {
    return (SD_TRANSFER_ERROR);
  }
  else
  {
    return(SD_TRANSFER_BUSY);
  }
}

/**
  * @brief  Enquires cards about their operating voltage and configures clock
  *         controls and stores SD information that will be needed in future
  *         in the SD handle.
  * @param  None
  * @retval error state
  */
SD_Error SD_PowerON(void)
{
  SD_Error errorstatus = SD_OK;
  uint32_t SDType = SD_STD_CAPACITY;
  uint32_t response = 0, count = 0, validvoltage = 0;

  SDIO_InitStructure.ClockPowerSave = SDIO_CLOCK_POWER_SAVE_DISABLE;
  SDIO_InitStructure.BusWide = SDIO_BUS_WIDE_1B;
  SDIO_InitStructure.ClockDiv = SDIO_INIT_CLK_DIV;

  SDIO_ControlReset(SDIO);

  SDIO_Init(SDIO, &SDIO_InitStructure);

  /* Disable SDIO Clock */
  SDIO_ClockCmd(SDIO, DISABLE);

  /* Set Power State to ON */
  SDIO_PowerON(SDIO);

  SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_ALL);
  SDIO_ITConfig(SDIO, SDIO_IT_ALL, DISABLE);

  SDIO_ClockCmd(SDIO, ENABLE);

  /* update card clock configuration only to card clock domain */
  SDIO_ClockUpdate(SDIO);

  SDIO_CmdInitStructure.Argument = 0x0;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_GO_IDLE_STATE;
  SDIO_CmdInitStructure.Response = SDIO_Response_No;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdError();
  if (errorstatus != SD_OK)
  {
    /* CMD Response TimeOut (wait for CMDSENT flag) */
    return errorstatus;
  }

  SDIO_CmdInitStructure.Argument = SD_CHECK_PATTERN;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_HS_SEND_EXT_CSD;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp7Error();
  if (errorstatus == SD_OK)
  {
    CardType = SDIO_STD_CAPACITY_SD_CARD_V2_0;

    SDType = SD_HIGH_CAPACITY;
  }

  SDIO_CmdInitStructure.Argument = 0x00;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_CMD;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_CMD);

  if (errorstatus == SD_OK)
  {
    while ((!validvoltage) && (count < SD_MAX_VOLT_TRIAL))
    {
      SDIO_CmdInitStructure.Argument = 0x00;
      SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_CMD;
      SDIO_CmdInitStructure.Response = SDIO_Response_Short;
      SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
      SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

      errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_CMD);
      
      if (errorstatus != SD_OK)
      {
        return(errorstatus);
      }

      SDIO_CmdInitStructure.Argument = SD_VOLTAGE_WINDOW_SD | SDType;
      SDIO_CmdInitStructure.CmdIndex = SD_CMD_SD_APP_OP_COND;
      SDIO_CmdInitStructure.Response = SDIO_Response_Short;
      SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
      SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

      errorstatus = SDIO_CmdResp3Error();

      if (errorstatus != SD_OK)
      {
        return(errorstatus); 
      }

      response = SDIO_GetResponse(SDIO, SDIO_RESP1);

      validvoltage = (((response >> 31) == 1) ? 1 : 0);
      count++;
    }

    if (count >= SD_MAX_VOLT_TRIAL)
    {
      errorstatus = SD_INVALID_VOLTRANGE;
      return(errorstatus);
    }

    if (response &= SD_HIGH_CAPACITY)
    {
      CardType = SDIO_HIGH_CAPACITY_SD_CARD;
    }
  }/* else MMC Card */
  return errorstatus;
}

/**
  * @brief  Disable the SDIO card clock output
  * @param  None
  * @retval error state
  */
SD_Error SD_PowerOFF(void)
{
  SD_Error errorstatus = SD_OK;

  SDIO_ControlReset(SDIO);

  /* Disable SDIO Clock */
  SDIO_ClockCmd(SDIO, DISABLE);

  /* update card clock configuration only to card clock domain */
  SDIO_ClockUpdate(SDIO);

  return errorstatus;
}

/**
  * @brief  Initializes the SD Card.
  * @param  None.
  * @note   This function initializes the SD card. It could be used when a card 
            re-initialization is needed.
  * @retval HAL status
  */
SD_Error SD_InitializeCards(void)
{
  SD_Error errorstatus = SD_OK;
  uint16_t rca = 0x01;

  if (SDIO_PowerState_OFF == SDIO_GetPower(SDIO))
  {
    errorstatus = SD_REQUEST_NOT_APPLICABLE;
    return(errorstatus);
  }

  if (SDIO_SECURE_DIGITAL_IO_CARD != CardType)
  {
    SDIO_CmdInitStructure.Argument = 0x0;
    SDIO_CmdInitStructure.CmdIndex = SD_CMD_ALL_SEND_CID;
    SDIO_CmdInitStructure.Response = SDIO_Response_Long;
    SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
    SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

    errorstatus = SDIO_CmdResp2Error();

    if (SD_OK != errorstatus)
    {
      return(errorstatus);
    }

    CID_Tab[0] = SDIO_GetResponse(SDIO, SDIO_RESP1);
    CID_Tab[1] = SDIO_GetResponse(SDIO, SDIO_RESP2);
    CID_Tab[2] = SDIO_GetResponse(SDIO, SDIO_RESP3);
    CID_Tab[3] = SDIO_GetResponse(SDIO, SDIO_RESP4);
  }
  
  if ((SDIO_STD_CAPACITY_SD_CARD_V1_1 == CardType) 
    ||(SDIO_STD_CAPACITY_SD_CARD_V2_0 == CardType) 
    ||(SDIO_SECURE_DIGITAL_IO_COMBO_CARD == CardType)
    ||(SDIO_HIGH_CAPACITY_SD_CARD == CardType) )
  {
    SDIO_CmdInitStructure.Argument = 0x00;
    SDIO_CmdInitStructure.CmdIndex = SD_CMD_SET_REL_ADDR;    
    SDIO_CmdInitStructure.Response = SDIO_Response_Short;
    SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
    SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

    errorstatus = SDIO_CmdResp6Error(SD_CMD_SET_REL_ADDR, &rca);

    if (SD_OK != errorstatus)
    {
      return(errorstatus);
    }
  }
  if (SDIO_SECURE_DIGITAL_IO_CARD != CardType)
  {
    RCA = rca;

    SDIO_CmdInitStructure.Argument = (uint32_t)(rca << 16);
    SDIO_CmdInitStructure.CmdIndex = SD_CMD_SEND_CSD;
    SDIO_CmdInitStructure.Response = SDIO_Response_Long;
    SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
    SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

    errorstatus = SDIO_CmdResp2Error();

    if (SD_OK != errorstatus)
    {
      return(errorstatus);
    }

    CSD_Tab[0] = SDIO_GetResponse(SDIO, SDIO_RESP1);
    CSD_Tab[1] = SDIO_GetResponse(SDIO, SDIO_RESP2);
    CSD_Tab[2] = SDIO_GetResponse(SDIO, SDIO_RESP3);
    CSD_Tab[3] = SDIO_GetResponse(SDIO, SDIO_RESP4);
  }
  errorstatus = SD_OK;

  return(errorstatus);
}

SD_Error SD_GetCardInfo(SD_CardInfo *cardinfo)
{
  SD_Error errorstatus = SD_OK;
  uint8_t tmp = 0;

  cardinfo->CardType = (uint8_t)CardType;
  cardinfo->RCA = (uint16_t)RCA;

  /*!< Byte 0 */
  tmp = (uint8_t)((CSD_Tab[0] & 0xFF000000) >> 24);
  cardinfo->SD_csd.CSDStruct = (tmp & 0xC0) >> 6;
  cardinfo->SD_csd.SysSpecVersion = (tmp & 0x3C) >> 2;
  cardinfo->SD_csd.Reserved1 = tmp & 0x03;

  /*!< Byte 1 */
  tmp = (uint8_t)((CSD_Tab[0] & 0x00FF0000) >> 16);
  cardinfo->SD_csd.TAAC = tmp;

  /*!< Byte 2 */
  tmp = (uint8_t)((CSD_Tab[0] & 0x0000FF00) >> 8);
  cardinfo->SD_csd.NSAC = tmp;

  /*!< Byte 3 */
  tmp = (uint8_t)(CSD_Tab[0] & 0x000000FF);
  cardinfo->SD_csd.MaxBusClkFrec = tmp;

  /*!< Byte 4 */
  tmp = (uint8_t)((CSD_Tab[1] & 0xFF000000) >> 24);
  cardinfo->SD_csd.CardComdClasses = tmp << 4;

  /*!< Byte 5 */
  tmp = (uint8_t)((CSD_Tab[1] & 0x00FF0000) >> 16);
  cardinfo->SD_csd.CardComdClasses |= (tmp & 0xF0) >> 4;
  cardinfo->SD_csd.RdBlockLen = tmp & 0x0F;

  /*!< Byte 6 */
  tmp = (uint8_t)((CSD_Tab[1] & 0x0000FF00) >> 8);
  cardinfo->SD_csd.PartBlockRead = (tmp & 0x80) >> 7;
  cardinfo->SD_csd.WrBlockMisalign = (tmp & 0x40) >> 6;
  cardinfo->SD_csd.RdBlockMisalign = (tmp & 0x20) >> 5;
  cardinfo->SD_csd.DSRImpl = (tmp & 0x10) >> 4;
  cardinfo->SD_csd.Reserved2 = 0; /*!< Reserved */

  if ((CardType == SDIO_STD_CAPACITY_SD_CARD_V1_1) || (CardType == SDIO_STD_CAPACITY_SD_CARD_V2_0))
  {
    cardinfo->SD_csd.DeviceSize = (tmp & 0x03) << 10;

    /*!< Byte 7 */
    tmp = (uint8_t)(CSD_Tab[1] & 0x000000FF);
    cardinfo->SD_csd.DeviceSize |= (tmp) << 2;

    /*!< Byte 8 */
    tmp = (uint8_t)((CSD_Tab[2] & 0xFF000000) >> 24);
    cardinfo->SD_csd.DeviceSize |= (tmp & 0xC0) >> 6;

    cardinfo->SD_csd.MaxRdCurrentVDDMin = (tmp & 0x38) >> 3;
    cardinfo->SD_csd.MaxRdCurrentVDDMax = (tmp & 0x07);

    /*!< Byte 9 */
    tmp = (uint8_t)((CSD_Tab[2] & 0x00FF0000) >> 16);
    cardinfo->SD_csd.MaxWrCurrentVDDMin = (tmp & 0xE0) >> 5;
    cardinfo->SD_csd.MaxWrCurrentVDDMax = (tmp & 0x1C) >> 2;
    cardinfo->SD_csd.DeviceSizeMul = (tmp & 0x03) << 1;
    /*!< Byte 10 */
    tmp = (uint8_t)((CSD_Tab[2] & 0x0000FF00) >> 8);
    cardinfo->SD_csd.DeviceSizeMul |= (tmp & 0x80) >> 7;
    
    cardinfo->CardCapacity = (cardinfo->SD_csd.DeviceSize + 1) ;
    cardinfo->CardCapacity *= (1 << (cardinfo->SD_csd.DeviceSizeMul + 2));
    cardinfo->CardBlockSize = 1 << (cardinfo->SD_csd.RdBlockLen);
    cardinfo->CardCapacity *= cardinfo->CardBlockSize;
  }
  else if (CardType == SDIO_HIGH_CAPACITY_SD_CARD)
  {
    /*!< Byte 7 */
    tmp = (uint8_t)(CSD_Tab[1] & 0x000000FF);
    cardinfo->SD_csd.DeviceSize = (tmp & 0x3F) << 16;

    /*!< Byte 8 */
    tmp = (uint8_t)((CSD_Tab[2] & 0xFF000000) >> 24);

    cardinfo->SD_csd.DeviceSize |= (tmp << 8);

    /*!< Byte 9 */
    tmp = (uint8_t)((CSD_Tab[2] & 0x00FF0000) >> 16);

    cardinfo->SD_csd.DeviceSize |= (tmp);

    /*!< Byte 10 */
    tmp = (uint8_t)((CSD_Tab[2] & 0x0000FF00) >> 8);
    
    cardinfo->CardCapacity = (uint64_t)(cardinfo->SD_csd.DeviceSize + 1) * 512 * 1024;
    cardinfo->CardBlockSize = 512;    
  }


  cardinfo->SD_csd.EraseGrSize = (tmp & 0x40) >> 6;
  cardinfo->SD_csd.EraseGrMul = (tmp & 0x3F) << 1;

  /*!< Byte 11 */
  tmp = (uint8_t)(CSD_Tab[2] & 0x000000FF);
  cardinfo->SD_csd.EraseGrMul |= (tmp & 0x80) >> 7;
  cardinfo->SD_csd.WrProtectGrSize = (tmp & 0x7F);

  /*!< Byte 12 */
  tmp = (uint8_t)((CSD_Tab[3] & 0xFF000000) >> 24);
  cardinfo->SD_csd.WrProtectGrEnable = (tmp & 0x80) >> 7;
  cardinfo->SD_csd.ManDeflECC = (tmp & 0x60) >> 5;
  cardinfo->SD_csd.WrSpeedFact = (tmp & 0x1C) >> 2;
  cardinfo->SD_csd.MaxWrBlockLen = (tmp & 0x03) << 2;

  /*!< Byte 13 */
  tmp = (uint8_t)((CSD_Tab[3] & 0x00FF0000) >> 16);
  cardinfo->SD_csd.MaxWrBlockLen |= (tmp & 0xC0) >> 6;
  cardinfo->SD_csd.WriteBlockPaPartial = (tmp & 0x20) >> 5;
  cardinfo->SD_csd.Reserved3 = 0;
  cardinfo->SD_csd.ContentProtectAppli = (tmp & 0x01);

  /*!< Byte 14 */
  tmp = (uint8_t)((CSD_Tab[3] & 0x0000FF00) >> 8);
  cardinfo->SD_csd.FileFormatGrouop = (tmp & 0x80) >> 7;
  cardinfo->SD_csd.CopyFlag = (tmp & 0x40) >> 6;
  cardinfo->SD_csd.PermWrProtect = (tmp & 0x20) >> 5;
  cardinfo->SD_csd.TempWrProtect = (tmp & 0x10) >> 4;
  cardinfo->SD_csd.FileFormat = (tmp & 0x0C) >> 2;
  cardinfo->SD_csd.ECC = (tmp & 0x03);

  /*!< Byte 15 */
  tmp = (uint8_t)(CSD_Tab[3] & 0x000000FF);
  cardinfo->SD_csd.CSD_CRC = (tmp & 0xFE) >> 1;
  cardinfo->SD_csd.Reserved4 = 1;

  /*!< Byte 0 */
  tmp = (uint8_t)((CID_Tab[0] & 0xFF000000) >> 24);
  cardinfo->SD_cid.ManufacturerID = tmp;

  /*!< Byte 1 */
  tmp = (uint8_t)((CID_Tab[0] & 0x00FF0000) >> 16);
  cardinfo->SD_cid.OEM_AppliID = tmp << 8;

  /*!< Byte 2 */
  tmp = (uint8_t)((CID_Tab[0] & 0x000000FF00) >> 8);
  cardinfo->SD_cid.OEM_AppliID |= tmp;

  /*!< Byte 3 */
  tmp = (uint8_t)(CID_Tab[0] & 0x000000FF);
  cardinfo->SD_cid.ProdName1 = tmp << 24;

  /*!< Byte 4 */
  tmp = (uint8_t)((CID_Tab[1] & 0xFF000000) >> 24);
  cardinfo->SD_cid.ProdName1 |= tmp << 16;

  /*!< Byte 5 */
  tmp = (uint8_t)((CID_Tab[1] & 0x00FF0000) >> 16);
  cardinfo->SD_cid.ProdName1 |= tmp << 8;

  /*!< Byte 6 */
  tmp = (uint8_t)((CID_Tab[1] & 0x0000FF00) >> 8);
  cardinfo->SD_cid.ProdName1 |= tmp;

  /*!< Byte 7 */
  tmp = (uint8_t)(CID_Tab[1] & 0x000000FF);
  cardinfo->SD_cid.ProdName2 = tmp;

  /*!< Byte 8 */
  tmp = (uint8_t)((CID_Tab[2] & 0xFF000000) >> 24);
  cardinfo->SD_cid.ProdRev = tmp;

  /*!< Byte 9 */
  tmp = (uint8_t)((CID_Tab[2] & 0x00FF0000) >> 16);
  cardinfo->SD_cid.ProdSN = tmp << 24;

  /*!< Byte 10 */
  tmp = (uint8_t)((CID_Tab[2] & 0x0000FF00) >> 8);
  cardinfo->SD_cid.ProdSN |= tmp << 16;

  /*!< Byte 11 */
  tmp = (uint8_t)(CID_Tab[2] & 0x000000FF);
  cardinfo->SD_cid.ProdSN |= tmp << 8;

  /*!< Byte 12 */
  tmp = (uint8_t)((CID_Tab[3] & 0xFF000000) >> 24);
  cardinfo->SD_cid.ProdSN |= tmp;

  /*!< Byte 13 */
  tmp = (uint8_t)((CID_Tab[3] & 0x00FF0000) >> 16);
  cardinfo->SD_cid.Reserved1 |= (tmp & 0xF0) >> 4;
  cardinfo->SD_cid.ManufactDate = (tmp & 0x0F) << 8;

  /*!< Byte 14 */
  tmp = (uint8_t)((CID_Tab[3] & 0x0000FF00) >> 8);
  cardinfo->SD_cid.ManufactDate |= tmp;

  /*!< Byte 15 */
  tmp = (uint8_t)(CID_Tab[3] & 0x000000FF);
  cardinfo->SD_cid.CID_CRC = (tmp & 0xFE) >> 1;
  cardinfo->SD_cid.Reserved2 = 1;
  
  return(errorstatus);
}

/**
  * @brief  Enables wide bus opeartion for the requeseted card if supported by 
  *         card.
  * @param  WideMode: Specifies the SD card wide bus mode. 
  *   This parameter can be one of the following values:
  *     @arg SDIO_BusWide_8b: 8-bit data transfer (Only for MMC)
  *     @arg SDIO_BusWide_4b: 4-bit data transfer
  *     @arg SDIO_BusWide_1b: 1-bit data transfer
  * @retval SD_Error: SD Card Error code.
  */
SD_Error SD_GetCardStatus(SD_CardStatus *cardstatus)
{
  SD_Error errorstatus = SD_OK;
  uint8_t tmp = 0;

  errorstatus = SD_SendSDStatus((uint32_t *)SDSTATUS_Tab);

  if (errorstatus  != SD_OK)
  {
    return(errorstatus);
  }

  /*!< Byte 0 */
  tmp = (uint8_t)((SDSTATUS_Tab[0] & 0xC0) >> 6);
  cardstatus->DAT_BUS_WIDTH = tmp;

  /*!< Byte 0 */
  tmp = (uint8_t)((SDSTATUS_Tab[0] & 0x20) >> 5);
  cardstatus->SECURED_MODE = tmp;

  /*!< Byte 2 */
  tmp = (uint8_t)((SDSTATUS_Tab[2] & 0xFF));
  cardstatus->SD_CARD_TYPE = tmp << 8;

  /*!< Byte 3 */
  tmp = (uint8_t)((SDSTATUS_Tab[3] & 0xFF));
  cardstatus->SD_CARD_TYPE |= tmp;

  /*!< Byte 4 */
  tmp = (uint8_t)(SDSTATUS_Tab[4] & 0xFF);
  cardstatus->SIZE_OF_PROTECTED_AREA = tmp << 24;

  /*!< Byte 5 */
  tmp = (uint8_t)(SDSTATUS_Tab[5] & 0xFF);
  cardstatus->SIZE_OF_PROTECTED_AREA |= tmp << 16;

  /*!< Byte 6 */
  tmp = (uint8_t)(SDSTATUS_Tab[6] & 0xFF);
  cardstatus->SIZE_OF_PROTECTED_AREA |= tmp << 8;

  /*!< Byte 7 */
  tmp = (uint8_t)(SDSTATUS_Tab[7] & 0xFF);
  cardstatus->SIZE_OF_PROTECTED_AREA |= tmp;

  /*!< Byte 8 */
  tmp = (uint8_t)((SDSTATUS_Tab[8] & 0xFF));
  cardstatus->SPEED_CLASS = tmp;

  /*!< Byte 9 */
  tmp = (uint8_t)((SDSTATUS_Tab[9] & 0xFF));
  cardstatus->PERFORMANCE_MOVE = tmp;

  /*!< Byte 10 */
  tmp = (uint8_t)((SDSTATUS_Tab[10] & 0xF0) >> 4);
  cardstatus->AU_SIZE = tmp;

  /*!< Byte 11 */
  tmp = (uint8_t)(SDSTATUS_Tab[11] & 0xFF);
  cardstatus->ERASE_SIZE = tmp << 8;

  /*!< Byte 12 */
  tmp = (uint8_t)(SDSTATUS_Tab[12] & 0xFF);
  cardstatus->ERASE_SIZE |= tmp;

  /*!< Byte 13 */
  tmp = (uint8_t)((SDSTATUS_Tab[13] & 0xFC) >> 2);
  cardstatus->ERASE_TIMEOUT = tmp;

  /*!< Byte 13 */
  tmp = (uint8_t)((SDSTATUS_Tab[13] & 0x3));
  cardstatus->ERASE_OFFSET = tmp;
 
  return(errorstatus);
}

SD_Error SD_EnableWideBusOperation(uint32_t WideMode)
{
  SD_Error errorstatus = SD_OK;

  /*!< MMC Card doesn't support this feature */
  if (SDIO_MULTIMEDIA_CARD == CardType)
  {
    errorstatus = SD_UNSUPPORTED_FEATURE;
    return(errorstatus);
  }
  else if ((SDIO_STD_CAPACITY_SD_CARD_V1_1 == CardType) || (SDIO_STD_CAPACITY_SD_CARD_V2_0 == CardType) || (SDIO_HIGH_CAPACITY_SD_CARD == CardType))
  {                             
    if (SDIO_BUS_WIDE_8B == WideMode)
    {
      errorstatus = SD_UNSUPPORTED_FEATURE;
      return(errorstatus);
    }
    else if (SDIO_BUS_WIDE_4B == WideMode)
    {
      errorstatus = SDEnWideBus(ENABLE);

      if (SD_OK == errorstatus)
      {
        SDIO_ConfigBusWidth(SDIO, WideMode);
      }
    }
    else
    {
      errorstatus = SDEnWideBus(DISABLE);

      if (SD_OK == errorstatus)
      {
        SDIO_ConfigBusWidth(SDIO, WideMode);
      }
    }
  }

  return errorstatus;
}

/**
  * @brief  Send the Select Deselect command and check the response.
  * @param  SDIOx: Pointer to SDIO register base 
  * @param  addr: Address of the card to be selected  
  * @retval HAL status
  */
SD_Error SD_SelectDeselect(uint32_t Addr)
{
  SD_Error errorstatus = SD_OK;

  /* Send CMD7 SDIO_SEL_DESEL_CARD */
  SDIO_CmdInitStructure.Argument         = (uint32_t)Addr;
  SDIO_CmdInitStructure.CmdIndex         = SD_CMD_SEL_DESEL_CARD;
  SDIO_CmdInitStructure.Response         = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM             = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);
  
  /* Check for error conditions */
  errorstatus = SDIO_CmdResp1Error(SD_CMD_SEL_DESEL_CARD);

  return errorstatus;
}

/**
  * @brief  Allows to read one block from a specified address in a card. The Data
  *         transfer can be managed by DMA mode or Polling mode. 
  * @note   This operation should be followed by two functions to check if the 
  *         DMA Controller and SD Card status.
  *          - SD_ReadWaitOperation(): this function insure that the DMA
  *            controller has finished all data transfer.
  *          - SD_GetStatus(): to check that the SD Card has finished the 
  *            data transfer and it is ready for data.            
  * @param  readbuff: pointer to the buffer that will contain the received data
  * @param  ReadAddr: Address from where data are to be read.  
  * @param  BlockSize: the SD card Data block size. The Block size should be 512.
  * @retval SD_Error: SD Card Error code.
  */
SD_Error SD_ReadBlock(uint8_t *readbuff, uint64_t ReadAddr, uint16_t BlockSize)
{
  SD_Error errorstatus = SD_OK;
#if defined (SD_POLLING_MODE) 
  uint32_t *tempbuff = (uint32_t *)readbuff;
#endif

  TransferError = SD_OK;
  TransferEnd = 0;
  StopCondition = 0;  

  SDIO->CTRL = 0x06;
  while(SDIO->CTRL&0x00000006);    
  
  if (CardType == SDIO_HIGH_CAPACITY_SD_CARD)
  {
    BlockSize = 512;
    ReadAddr /= 512;
  }

  SDIO_CmdInitStructure.Argument = (uint32_t) BlockSize;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SET_BLOCKLEN;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;   //r1
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SET_BLOCKLEN);

  if (SD_OK != errorstatus)
  {
    return(errorstatus);
  }
 /*********************************************************************************/
  SDIO_DataInitStructure.DataTimeOut = SD_DATATIMEOUT;
  SDIO_DataInitStructure.DataLength = BlockSize;
  SDIO_DataInitStructure.DataBlockSize = SDIO_DATABLOCK_SIZE_512B;
  SDIO_DataInitStructure.TransferDir = SDIO_TransferDir_ToSDIO;
  SDIO_DataInitStructure.TransferMode = SDIO_TransferMode_Block;
  SDIO_DataInitStructure.DPSM = SDIO_DPSM_Enable;
  SDIO_ConfigData(SDIO, &SDIO_DataInitStructure);

  /*!< Send CMD17 READ_SINGLE_BLOCK */
  SDIO_CmdInitStructure.Argument = (uint32_t)ReadAddr;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_READ_SINGLE_BLOCK;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CMD_CHECK_RESP_CRC | SDIO_CMD_DATA_EXPECT | SDIO_CMD_WAIT_PRV_COMPLT;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_READ_SINGLE_BLOCK);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

#if defined (SD_POLLING_MODE)  
  /*!< In case of single block transfer, no need of stop transfer at all.*/
  /*!< Polling mode */
  while (!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_RXOVERR | SDIO_FLAG_DCRCFAIL | SDIO_FLAG_DTIMEOUT | SDIO_FLAG_DBCKEND | SDIO_FLAG_STBITERR))
  {
    if ((SDIO->STATUS & (1 << 2)) != (1 << 2))
    {
      *tempbuff =  SDIO_ReadFIFO(SDIO);
      tempbuff++;
    }
  }

  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DTIMEOUT) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_DATA_TIMEOUT;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DCRCFAIL) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_DATA_CRC_FAIL;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_RXOVERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_RX_OVERRUN;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_STBITERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_START_BIT_ERR;
    return(errorstatus);
  }
  while ((SDIO->STATUS & (1 << 2)) != (1 << 2))
  {
    *tempbuff = SDIO_ReadFIFO(SDIO);
    tempbuff++;
  }
  
  /*!< Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);

#elif defined (SD_DMA_MODE)
    SDIO_ITConfig(SDIO, SDIO_IT_DATAEND, ENABLE);
    SDIO_DMACmd(ENABLE);
    SD_DMA_RxConfig((uint32_t *)readbuff, BlockSize);
#endif

  return(errorstatus);
}

/**
  * @brief  Allows to read blocks from a specified address  in a card.  The Data
  *         transfer can be managed by DMA mode or Polling mode.
  * @note   This operation should be followed by two functions to check if the 
  *         DMA Controller and SD Card status.
  *          - SD_ReadWaitOperation(): this function insure that the DMA
  *            controller has finished all data transfer. 
  *          - SD_GetStatus(): to check that the SD Card has finished the 
  *            data transfer and it is ready for data.   
  * @param  readbuff: pointer to the buffer that will contain the received data.
  * @param  ReadAddr: Address from where data are to be read.
  * @param  BlockSize: the SD card Data block size. The Block size should be 512.
  * @param  NumberOfBlocks: number of blocks to be read.
  * @retval SD_Error: SD Card Error code.
  */
SD_Error SD_ReadMultiBlocks(uint8_t *readbuff, uint64_t ReadAddr, uint16_t BlockSize, uint32_t NumberOfBlocks)
{
  SD_Error errorstatus = SD_OK;
  TransferError = SD_OK;
  TransferEnd = 0;
  StopCondition = 1;
#if defined (SD_POLLING_MODE) 
  uint32_t *tempbuff = (uint32_t *)readbuff;
#endif

  SDIO->CTRL = 0x06;
  while(SDIO->CTRL&0x00000006);

  if (CardType == SDIO_HIGH_CAPACITY_SD_CARD)
  {
    BlockSize = 512;
    ReadAddr /= 512;
  }

  SDIO_CmdInitStructure.Argument = (uint32_t) BlockSize;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SET_BLOCKLEN;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;   //r1
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SET_BLOCKLEN);

  if (SD_OK != errorstatus)
  {
    return(errorstatus);
  }
    
  SDIO_DataInitStructure.DataTimeOut = SD_DATATIMEOUT;
  SDIO_DataInitStructure.DataLength = NumberOfBlocks * BlockSize;
  SDIO_DataInitStructure.DataBlockSize = SDIO_DATABLOCK_SIZE_512B;
  SDIO_DataInitStructure.TransferDir = SDIO_TransferDir_ToSDIO;
  SDIO_DataInitStructure.TransferMode = SDIO_TransferMode_Block;
  SDIO_DataInitStructure.DPSM = SDIO_DPSM_Enable;
  SDIO_ConfigData(SDIO, &SDIO_DataInitStructure);

  /*!< Send CMD18 READ_MULT_BLOCK with argument data address */
  SDIO_CmdInitStructure.Argument = (uint32_t)ReadAddr;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_READ_MULT_BLOCK;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short; //r1
  SDIO_CmdInitStructure.CPSM = SDIO_CMD_CHECK_RESP_CRC | SDIO_CMD_DATA_EXPECT | SDIO_CMD_WAIT_PRV_COMPLT;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_READ_MULT_BLOCK);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

#if defined (SD_POLLING_MODE)  
  /*!< In case of single block transfer, no need of stop transfer at all.*/
  /*!< Polling mode */
  while (!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_RXOVERR | SDIO_FLAG_DCRCFAIL | SDIO_FLAG_DTIMEOUT | SDIO_FLAG_DBCKEND | SDIO_FLAG_STBITERR))
  {
    if ((SDIO->STATUS & (1 << 2)) != (1 << 2))
    {
      *tempbuff =  SDIO_ReadFIFO(SDIO);
      tempbuff++;
    }
  }

  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DBCKEND))
  {
    if (StopCondition == 1)
    {
      errorstatus = SD_StopTransfer();
      if (errorstatus != SD_OK)
      {
        SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
        return errorstatus;
      }
    }
  }
  
  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DTIMEOUT) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_DATA_TIMEOUT;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DCRCFAIL) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_DATA_CRC_FAIL;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_RXOVERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_RX_OVERRUN;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_STBITERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_START_BIT_ERR;
    return(errorstatus);
  }
  while ((SDIO->STATUS & (1 << 2)) != (1 << 2))
  {
    *tempbuff = SDIO_ReadFIFO(SDIO);
    tempbuff++;
  }
  
  /*!< Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
#endif

  return(errorstatus);
}

/**
  * @brief  Allows to write one block starting from a specified address in a card.
  *         The Data transfer can be managed by DMA mode or Polling mode.
  * @note   This operation should be followed by two functions to check if the 
  *         DMA Controller and SD Card status.
  *          - SD_ReadWaitOperation(): this function insure that the DMA
  *            controller has finished all data transfer.
  *          - SD_GetStatus(): to check that the SD Card has finished the 
  *            data transfer and it is ready for data.      
  * @param  writebuff: pointer to the buffer that contain the data to be transferred.
  * @param  WriteAddr: Address from where data are to be read.   
  * @param  BlockSize: the SD card Data block size. The Block size should be 512.
  * @retval SD_Error: SD Card Error code.
  */
SD_Error SD_WriteBlock(uint8_t *writebuff, uint64_t WriteAddr, uint16_t BlockSize)
{
  SD_Error errorstatus = SD_OK;

#if defined (SD_POLLING_MODE)
  uint32_t *tempbuff = (uint32_t *)writebuff;
#endif

  TransferError = SD_OK;
  TransferEnd = 0;
  StopCondition = 0;

  SDIO->CTRL = 0x06;
  while(SDIO->CTRL&0x00000006);    

  if (CardType == SDIO_HIGH_CAPACITY_SD_CARD)
  {
    BlockSize = 512;
    WriteAddr /= 512;
  }

  SDIO_CmdInitStructure.Argument = (uint32_t) BlockSize;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SET_BLOCKLEN;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SET_BLOCKLEN);

  if (SD_OK != errorstatus)
  {
    return(errorstatus);
  }

  SDIO_DataInitStructure.DataTimeOut = SD_DATATIMEOUT;
  SDIO_DataInitStructure.DataLength = BlockSize;
  SDIO_DataInitStructure.DataBlockSize = SDIO_DATABLOCK_SIZE_512B;
  SDIO_DataInitStructure.TransferDir = SDIO_TransferDir_ToCard;
  SDIO_DataInitStructure.TransferMode = SDIO_TransferMode_Block;
  SDIO_DataInitStructure.DPSM = SDIO_DPSM_Enable;
  SDIO_ConfigData(SDIO, &SDIO_DataInitStructure);
  SDIO->FIFOTH |= 127;

  while (SDIO->STATUS & 0x02)
  {
    SDIO->FIFO = *tempbuff;
    tempbuff++;
  }
  
  /*!< Send CMD24 WRITE_SINGLE_BLOCK */
  SDIO_CmdInitStructure.Argument = WriteAddr;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_WRITE_SINGLE_BLOCK;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;   //r1
  SDIO_CmdInitStructure.CPSM = SDIO_CMD_CHECK_RESP_CRC | SDIO_CMD_DATA_EXPECT | SDIO_CMD_WAIT_PRV_COMPLT | SDIO_CMD_READ_WRITE;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_WRITE_SINGLE_BLOCK);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  /*!< In case of single data block transfer no need of stop command at all */
#if defined (SD_POLLING_MODE)
  while (!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_RXOVERR | SDIO_FLAG_DCRCFAIL | SDIO_FLAG_DTIMEOUT | SDIO_FLAG_DBCKEND | SDIO_FLAG_STBITERR))
  {
    if (SDIO->STATUS & 0x02)
    {
      if (SDIO->TBBCNT < SDIO->BYTCNT)
      {
        SDIO->FIFO = *tempbuff++;
      }
      else
      {
        SDIO_ClearFlagStatus(SDIO, SDIO_IT_TXDAVL);
      }
    }
  }

  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DTIMEOUT) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_DATA_TIMEOUT;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_STATIC_FLAGS) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_DCRCFAIL);
    errorstatus = SD_DATA_CRC_FAIL;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_TXUNDERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_TX_UNDERRUN;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_STBITERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_START_BIT_ERR;
    return(errorstatus);
  }

    /*!< Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
  
#elif defined (SD_DMA_MODE)
  SDIO_ITConfig(SDIO, SDIO_IT_DATAEND, ENABLE);
  SD_DMA_TxConfig((uint32_t *)writebuff, BlockSize);
  SDIO_DMACmd(ENABLE);
#endif

  return(errorstatus);
}

SD_Error SD_WriteMultiBlocks(uint8_t *writebuff, uint64_t WriteAddr, uint16_t BlockSize, uint32_t NumberOfBlocks)
{
  SD_Error errorstatus = SD_OK;
  __IO uint32_t count = 0;

  TransferError = SD_OK;
  TransferEnd = 0;
  StopCondition = 1;
  
  uint32_t *tempbuff = (uint32_t *)writebuff;

  SDIO->CTRL = 0x06;
  while(SDIO->CTRL&0x00000006);

  if (CardType == SDIO_HIGH_CAPACITY_SD_CARD)
  {
    BlockSize = 512;
    WriteAddr /= 512;
  }

  SDIO_CmdInitStructure.Argument = (uint32_t) BlockSize;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SET_BLOCKLEN;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;   //r1
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SET_BLOCKLEN);

  if (SD_OK != errorstatus)
  {
    return(errorstatus);
  }
 /*********************************************************************************/

  /*!< To improve performance  */
  SDIO_CmdInitStructure.Argument = (uint32_t) (RCA << 16);
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_CMD;  // cmd55
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_CMD);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  SDIO_CmdInitStructure.Argument = (uint32_t)BlockSize;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SET_BLOCK_COUNT;   //cmd23
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SET_BLOCK_COUNT);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  SDIO_DataInitStructure.DataTimeOut = SD_DATATIMEOUT;
  SDIO_DataInitStructure.DataLength = NumberOfBlocks * BlockSize;
  SDIO_DataInitStructure.DataBlockSize = SDIO_DATABLOCK_SIZE_512B;
  SDIO_DataInitStructure.TransferDir = SDIO_TransferDir_ToCard;
  SDIO_DataInitStructure.TransferMode = SDIO_TransferMode_Block;
  SDIO_DataInitStructure.DPSM = SDIO_DPSM_Enable;
  SDIO_ConfigData(SDIO, &SDIO_DataInitStructure);

  SDIO->FIFOTH |= 127;
  while (SDIO->STATUS & 0x02)
  {
    SDIO->FIFO = *tempbuff;
    tempbuff++;
  }

  /*!< Send CMD25 WRITE_MULT_BLOCK with argument data address */
  SDIO_CmdInitStructure.Argument = (uint32_t)WriteAddr;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_WRITE_MULT_BLOCK;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CMD_CHECK_RESP_CRC | SDIO_CMD_DATA_EXPECT | SDIO_CMD_WAIT_PRV_COMPLT | SDIO_CMD_READ_WRITE;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_WRITE_MULT_BLOCK);

  if (SD_OK != errorstatus)
  {
    return(errorstatus);
  }

  while (!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_TXUNDERR | SDIO_FLAG_DCRCFAIL | SDIO_FLAG_DTIMEOUT | SDIO_FLAG_DBCKEND))
  {
    if (SDIO->STATUS & 0x02)
    {
      if (SDIO->TBBCNT < SDIO->BYTCNT)
      {
        SDIO->FIFO = *tempbuff;
        tempbuff++;
      }
      else
      {
        SDIO_ClearFlagStatus(SDIO, SDIO_IT_TXDAVL);
      }
    }
  }

  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DBCKEND))
  {
    if (StopCondition == 1)
    {
      errorstatus = SD_StopTransfer();
      if (errorstatus != SD_OK)
      {
        SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
        return errorstatus;
      }
    }
  }

  while (SDIO->STATUS & (0x01 << 9))
  {
    /* code */
  }
  
  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DTIMEOUT) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_DATA_TIMEOUT;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DCRCFAIL) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_DATA_CRC_FAIL;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_TXUNDERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
    errorstatus = SD_TX_UNDERRUN;
    return(errorstatus);
  }

  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);

  return(errorstatus);
}

/**
  * @brief  Aborts an ongoing data transfer.
  * @param  None
  * @retval SD_Error: SD Card Error code.
  */
SD_Error SD_StopTransfer(void)
{
  SD_Error errorstatus = SD_OK;

  /*-------------- CMD13 -------------------*/
  SDIO_CmdInitStructure.Argument = 0x0;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_STOP_TRANSMISSION;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SEND_STATUS);
  return errorstatus;
}

/**
  * @brief  Allows to erase memory area specified for the given card.
  * @param  startaddr: the start address.
  * @param  endaddr: the end address.
  * @retval SD_Error: SD Card Error code.
  */
SD_Error SD_Erase(uint32_t startaddr, uint32_t endaddr)
{
  SD_Error errorstatus = SD_OK;
  uint32_t delay = 0;
  __IO uint32_t maxdelay = 0;
  uint8_t cardstate = 0;

  /*!< Check if the card coomnd class supports erase command */
  if (((CSD_Tab[1] >> 20) & SD_CCCC_ERASE) == 0)
  {
    errorstatus = SD_REQUEST_NOT_APPLICABLE;
    return(errorstatus);
  }

  maxdelay = 120000;

  if (SDIO_GetResponse(SDIO, SDIO_RESP1) & SD_CARD_LOCKED)
  {
    errorstatus = SD_LOCK_UNLOCK_FAILED;
    return(errorstatus);
  }

  if (CardType == SDIO_HIGH_CAPACITY_SD_CARD)
  {
    startaddr /= 512;
    endaddr /= 512;
  }
  
  /*!< According to sd-card spec 1.0 ERASE_GROUP_START (CMD32) and erase_group_end(CMD33) */
  if ((SDIO_STD_CAPACITY_SD_CARD_V1_1 == CardType) || (SDIO_STD_CAPACITY_SD_CARD_V2_0 == CardType) || (SDIO_HIGH_CAPACITY_SD_CARD == CardType))
  {
    /*!< Send CMD32 SD_ERASE_GRP_START with argument as addr  */
    SDIO_CmdInitStructure.Argument = startaddr;
    SDIO_CmdInitStructure.CmdIndex = SD_CMD_SD_ERASE_GRP_START;
    SDIO_CmdInitStructure.Response = SDIO_Response_Short;  //R1
    SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
    SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

    errorstatus = SDIO_CmdResp1Error(SD_CMD_SD_ERASE_GRP_START);
    if (errorstatus != SD_OK)
    {
      return(errorstatus);
    }

    /*!< Send CMD33 SD_ERASE_GRP_END with argument as addr  */
    SDIO_CmdInitStructure.Argument = endaddr;
    SDIO_CmdInitStructure.CmdIndex = SD_CMD_SD_ERASE_GRP_END;
    SDIO_CmdInitStructure.Response = SDIO_Response_Short;
    SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
    SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

    errorstatus = SDIO_CmdResp1Error(SD_CMD_SD_ERASE_GRP_END);
    if (errorstatus != SD_OK)
    {
      return(errorstatus);
    }
  }

  /*!< Send CMD38 ERASE */
  SDIO_CmdInitStructure.Argument = 0;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_ERASE;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_ERASE);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  for (delay = 0; delay < maxdelay; delay++)
  {}

  /*!< Wait till the card is in programming state */
  errorstatus = IsCardProgramming(&cardstate);

  while ((errorstatus == SD_OK) && ((SD_CARD_PROGRAMMING == cardstate) || (SD_CARD_RECEIVING == cardstate)))
  {
    errorstatus = IsCardProgramming(&cardstate);
  }

  return(errorstatus);
}

static SD_Error IsCardProgramming(uint8_t *pstatus)
{
  SD_Error errorstatus = SD_OK;
  __IO uint32_t respR1 = 0;

  SDIO_CmdInitStructure.Argument = (uint32_t) RCA << 16;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SEND_STATUS;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  while (!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL | SDIO_FLAG_CMDREND | SDIO_FLAG_CTIMEOUT))
  {
  }

  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT))
  {
    errorstatus = SD_CMD_RSP_TIMEOUT;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT);
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL))
  {
    errorstatus = SD_CMD_CRC_FAIL;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL);
    return(errorstatus);
  }

  /*!< Check response received is of desired command */
  if (SDIO_GetCommandResponse(SDIO) != SD_CMD_SEND_STATUS)
  {
    errorstatus = SD_ILLEGAL_CMD;
    return(errorstatus);
  }

  /*!< Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);


  /*!< We have received response, retrieve it for analysis  */
  respR1 = SDIO_GetResponse(SDIO, SDIO_RESP1);

  /*!< Find out card status */
  *pstatus = (uint8_t) ((respR1 >> 9) & 0x0000000F);   //status[12:9] :cardstate 

  if ((respR1 & SD_OCR_ERRORBITS) == SD_ALLZERO)
  {
    return(errorstatus);
  }

  if (respR1 & SD_OCR_ADDR_OUT_OF_RANGE)
  {
    return(SD_ADDR_OUT_OF_RANGE);
  }

  if (respR1 & SD_OCR_ADDR_MISALIGNED)
  {
    return(SD_ADDR_MISALIGNED);
  }

  if (respR1 & SD_OCR_BLOCK_LEN_ERR)
  {
    return(SD_BLOCK_LEN_ERR);
  }

  if (respR1 & SD_OCR_ERASE_SEQ_ERR)
  {
    return(SD_ERASE_SEQ_ERR);
  }

  if (respR1 & SD_OCR_BAD_ERASE_PARAM)
  {
    return(SD_BAD_ERASE_PARAM);
  }

  if (respR1 & SD_OCR_WRITE_PROT_VIOLATION)
  {
    return(SD_WRITE_PROT_VIOLATION);
  }

  if (respR1 & SD_OCR_LOCK_UNLOCK_FAILED)
  {
    return(SD_LOCK_UNLOCK_FAILED);
  }

  if (respR1 & SD_OCR_COM_CRC_FAILED)
  {
    return(SD_COM_CRC_FAILED);
  }

  if (respR1 & SD_OCR_ILLEGAL_CMD)
  {
    return(SD_ILLEGAL_CMD);
  }

  if (respR1 & SD_OCR_CARD_ECC_FAILED)
  {
    return(SD_CARD_ECC_FAILED);
  }

  if (respR1 & SD_OCR_CC_ERROR)
  {
    return(SD_CC_ERROR);
  }

  if (respR1 & SD_OCR_GENERAL_UNKNOWN_ERROR)
  {
    return(SD_GENERAL_UNKNOWN_ERROR);
  }

  if (respR1 & SD_OCR_STREAM_READ_UNDERRUN)
  {
    return(SD_STREAM_READ_UNDERRUN);
  }

  if (respR1 & SD_OCR_STREAM_WRITE_OVERRUN)
  {
    return(SD_STREAM_WRITE_OVERRUN);
  }

  if (respR1 & SD_OCR_CID_CSD_OVERWRIETE)
  {
    return(SD_CID_CSD_OVERWRITE);
  }

  if (respR1 & SD_OCR_WP_ERASE_SKIP)
  {
    return(SD_WP_ERASE_SKIP);
  }

  if (respR1 & SD_OCR_CARD_ECC_DISABLED)
  {
    return(SD_CARD_ECC_DISABLED);
  }

  if (respR1 & SD_OCR_ERASE_RESET)
  {
    return(SD_ERASE_RESET);
  }

  if (respR1 & SD_OCR_AKE_SEQ_ERROR)
  {
    return(SD_AKE_SEQ_ERROR);
  }

  return(errorstatus);
}

/**
  * @brief  Returns the current SD card's status.
  * @param  psdstatus: pointer to the buffer that will contain the SD card status 
  *         (SD Status register).
  * @retval SD_Error: SD Card Error code.
  */
SD_Error SD_SendSDStatus(uint32_t *psdstatus)
{
  SD_Error errorstatus = SD_OK;

  if (SDIO_GetResponse(SDIO, SDIO_RESP1) & SD_CARD_LOCKED)
  {
    errorstatus = SD_LOCK_UNLOCK_FAILED;
    return(errorstatus);
  }

  /*!< Set block size for card if it is not equal to current block size for card. */
  SDIO_CmdInitStructure.Argument = 64;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SET_BLOCKLEN;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SET_BLOCKLEN);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  /*!< CMD55 */
  SDIO_CmdInitStructure.Argument = (uint32_t) RCA << 16;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_CMD;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);
  errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_CMD);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  SDIO_DataInitStructure.DataTimeOut = SD_DATATIMEOUT;
  SDIO_DataInitStructure.DataLength = 64;
  SDIO_DataInitStructure.DataBlockSize = SDIO_DATABLOCK_SIZE_64B;
  SDIO_DataInitStructure.TransferDir = SDIO_TransferDir_ToSDIO;
  SDIO_DataInitStructure.TransferMode = SDIO_TransferMode_Block;
  SDIO_DataInitStructure.DPSM = SDIO_DPSM_Enable;
  SDIO_ConfigData(SDIO,&SDIO_DataInitStructure);

  /*!< Send ACMD13 SD_APP_STAUS  with argument as card's RCA.*/
  SDIO_CmdInitStructure.Argument = 0;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SD_APP_STAUS;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CMD_CHECK_RESP_CRC | SDIO_CMD_DATA_EXPECT | SDIO_CMD_WAIT_PRV_COMPLT;;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);
  errorstatus = SDIO_CmdResp1Error(SD_CMD_SD_APP_STAUS);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  while (!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_RXOVERR | SDIO_FLAG_DCRCFAIL | SDIO_FLAG_DTIMEOUT | SDIO_FLAG_DBCKEND | SDIO_FLAG_STBITERR))
  {
    if ((SDIO->STATUS & (1 << 2)) != (1 << 2))
    {
      *psdstatus =  SDIO_ReadFIFO(SDIO);
      psdstatus++;
    }
  }

  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DTIMEOUT) != RESET)
  {
    errorstatus = SD_DATA_TIMEOUT;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_DTIMEOUT);
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DCRCFAIL) != RESET)
  {
    errorstatus = SD_DATA_CRC_FAIL;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_DCRCFAIL);
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_RXOVERR) != RESET)
  {
    errorstatus = SD_RX_OVERRUN;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_RXOVERR);
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_STBITERR) != RESET)
  {
    errorstatus = SD_START_BIT_ERR;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_STBITERR);
    return(errorstatus);
  }

  while ((SDIO->STATUS & (1 << 2)) != (1 << 2))
  {
    *psdstatus = SDIO_ReadFIFO(SDIO);
    psdstatus++;
  }

  /*!< Clear all the static status flags*/
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);

  return(errorstatus);
}

static void GPIO_Configuration(void)
{ 
  GPIO_InitTypeDef GPIO_InitStructure;
  /**
    SDIO GPIO Configuration
      SDIO_D0  PC8 
      SDIO_D1  PC9 
      SDIO_D2  PC10
      SDIO_D3  PC11
      SDIO_CK  PC12
      SDIO_CMD PD2 
  */
  /*!< GPIOC and GPIOD Periph clock enable */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOC | RCC_APB2Periph_GPIOD , ENABLE);
  
  /*!< Configure PC.08, PC.09, PC.10, PC.11, PC.12 pin: D0, D1, D2, D3, CLK pin */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_8 | GPIO_Pin_9 | GPIO_Pin_10 | GPIO_Pin_11 | GPIO_Pin_12;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
  GPIO_Init(GPIOC, &GPIO_InitStructure);

  /*!< Configure PD.02 CMD line */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_2;
  GPIO_Init(GPIOD, &GPIO_InitStructure); 
}

static SD_Error FindSCR(uint16_t rca, uint32_t *pscr)
{ 
  uint32_t index = 0;
  SD_Error errorstatus = SD_OK;
  uint32_t tempscr[2] = {0, 0};  

  /*!< Set Block Size To 8 Bytes */ 
  SDIO_CmdInitStructure.Argument = (uint32_t)8;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SET_BLOCKLEN; //   cmd16
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;  //r1
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SET_BLOCKLEN);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  /*!< Send CMD55 APP_CMD with argument as card's RCA */
  SDIO_CmdInitStructure.Argument = (uint32_t) RCA << 16; 
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_CMD;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;
  SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_CMD);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  SDIO_DataInitStructure.DataTimeOut = SD_DATATIMEOUT;
  SDIO_DataInitStructure.DataLength = 8U;
  SDIO_DataInitStructure.DataBlockSize = SDIO_DATABLOCK_SIZE_8B;
  SDIO_DataInitStructure.TransferDir = SDIO_TransferDir_ToSDIO;
  SDIO_DataInitStructure.TransferMode = SDIO_TransferMode_Block;
  SDIO_DataInitStructure.DPSM = SDIO_DPSM_Enable;
  SDIO_ConfigData(SDIO, &SDIO_DataInitStructure);       

  /*!< Send ACMD51 SD_APP_SEND_SCR with argument as 0 */
  SDIO_CmdInitStructure.Argument = 0x0;
  SDIO_CmdInitStructure.CmdIndex = SD_CMD_SD_APP_SEND_SCR;
  SDIO_CmdInitStructure.Response = SDIO_Response_Short;  //r1
  SDIO_CmdInitStructure.CPSM = SDIO_CMD_CHECK_RESP_CRC | SDIO_CMD_DATA_EXPECT | SDIO_CMD_WAIT_PRV_COMPLT;
  SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

  errorstatus = SDIO_CmdResp1Error(SD_CMD_SD_APP_SEND_SCR);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }         

  while (!SDIO_GetFlagStatus(SDIO, (SDIO_FLAG_RXOVERR | SDIO_FLAG_DCRCFAIL | SDIO_FLAG_DTIMEOUT | SDIO_FLAG_DBCKEND| SDIO_FLAG_STBITERR)))
  {  
    if(SDIO->STATUS & (0x1FFF<<17)) // FIFO count
    {
      *(tempscr + index) = SDIO_ReadFIFO(SDIO);  
      index++;  

      if(index > 1 )
      break;
    }
  }
  
  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DTIMEOUT) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_DTIMEOUT);
    errorstatus = SD_DATA_TIMEOUT;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_DCRCFAIL) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_DCRCFAIL);
    errorstatus = SD_DATA_CRC_FAIL;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_RXOVERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_RXOVERR);
    errorstatus = SD_RX_OVERRUN;
    return(errorstatus);
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_STBITERR) != RESET)
  {
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_STBITERR);
    errorstatus = SD_START_BIT_ERR;
    return(errorstatus);
  }

  /*!< Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);

  *(pscr + 1) = ((tempscr[0] & SD_0TO7BITS) << 24) | ((tempscr[0] & SD_8TO15BITS) << 8) | ((tempscr[0] & SD_16TO23BITS) >> 8) | ((tempscr[0] & SD_24TO31BITS) >> 24);

  *(pscr) = ((tempscr[1] & SD_0TO7BITS) << 24) | ((tempscr[1] & SD_8TO15BITS) << 8) | ((tempscr[1] & SD_16TO23BITS) >> 8) | ((tempscr[1] & SD_24TO31BITS) >> 24);

  return(errorstatus);
}

static SD_Error SDEnWideBus(FunctionalState NewState)
{
  SD_Error errorstatus = SD_OK;

  uint32_t scr[2] = {0, 0};

  if (SDIO_GetResponse(SDIO, SDIO_RESP1) & SD_CARD_LOCKED)
  {
    errorstatus = SD_LOCK_UNLOCK_FAILED;
    return(errorstatus);
  }

  /*!< Get SCR Register */
  errorstatus = FindSCR(RCA, scr);

  if (errorstatus != SD_OK)
  {
    return(errorstatus);
  }

  /*!< If wide bus operation to be enabled */
  if (NewState == ENABLE)
  {
    /*!< If requested card supports wide bus operation */
    if ((scr[1] & SD_WIDE_BUS_SUPPORT) != SD_ALLZERO)
    {
      /*!< Send CMD55 APP_CMD with argument as card's RCA.*/
      SDIO_CmdInitStructure.Argument = (uint32_t) RCA << 16;
      SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_CMD;
      SDIO_CmdInitStructure.Response = SDIO_Response_Short;
      SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
      SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

      errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_CMD);

      if (errorstatus != SD_OK)
      {
        return(errorstatus);
      }

      /*!< Send ACMD6 APP_CMD with argument as 2 for wide bus mode */
      SDIO_CmdInitStructure.Argument = 0x2;
      SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_SD_SET_BUSWIDTH;
      SDIO_CmdInitStructure.Response = SDIO_Response_Short;
      SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
      SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

      errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_SD_SET_BUSWIDTH);

      if (errorstatus != SD_OK)
      {
        return(errorstatus);
      }
      return(errorstatus);
    }
    else
    {
      errorstatus = SD_REQUEST_NOT_APPLICABLE;
      return(errorstatus);
    }
  }   /*!< If wide bus operation to be disabled */
  else
  {
    /*!< If requested card supports 1 bit mode operation */
    if ((scr[1] & SD_SINGLE_BUS_SUPPORT) != SD_ALLZERO)
    {
      /*!< Send CMD55 APP_CMD with argument as card's RCA.*/
      SDIO_CmdInitStructure.Argument = (uint32_t) RCA << 16;
      SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_CMD;
      SDIO_CmdInitStructure.Response = SDIO_Response_Short;
      SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
      SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

      errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_CMD);

      if (errorstatus != SD_OK)
      {
        return(errorstatus);
      }

      /*!< Send ACMD6 APP_CMD with argument as 0 for single bus mode */
      SDIO_CmdInitStructure.Argument = 0x00;
      SDIO_CmdInitStructure.CmdIndex = SD_CMD_APP_SD_SET_BUSWIDTH;
      SDIO_CmdInitStructure.Response = SDIO_Response_Short;
      SDIO_CmdInitStructure.CPSM = SDIO_CPSM_Enable;
      SDIO_SendCommand(SDIO, &SDIO_CmdInitStructure);

      errorstatus = SDIO_CmdResp1Error(SD_CMD_APP_SD_SET_BUSWIDTH);

      if (errorstatus != SD_OK)
      {
        return(errorstatus);
      }

      return(errorstatus);
    }
    else
    {
      errorstatus = SD_REQUEST_NOT_APPLICABLE;
      return(errorstatus);
    }
  }
}

/**
  * @brief  Checks for error conditions for CMD0.
  * @param  None
  * @retval SD Card error state
  */
static SD_Error SDIO_CmdError(void)
{
  SD_Error errorstatus = SD_OK;

  uint32_t timeout = SDIO_CMD0TIMEOUT;
  
  while(!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CMDSENT) && (timeout > 0))
  {
    timeout--;
  }
  
  if (timeout == 0)
  {
    errorstatus = SD_CMD_RSP_TIMEOUT;
    return(errorstatus);
  }
  
  /* Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
  
  return errorstatus;
}

/**
  * @brief  Checks for error conditions for R1 response.
  * @param  SD_CMD: The sent command index  SD_CHECK_PATTERN
  * @retval SD Card error state
  */
static SD_Error SDIO_CmdResp1Error(uint8_t SD_CMD)
{
  uint32_t response_r1;
  
  while(!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL | SDIO_FLAG_CMDREND | SDIO_FLAG_CTIMEOUT))
  {
  }
  
  /* Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
  
  /* We have received response, retrieve it for analysis  */
  response_r1 = SDIO_GetResponse(SDIO, SDIO_RESP1);
  
  return (SD_Error)(response_r1 & SD_OCR_ERRORBITS);
}

/**
  * @brief  Checks for error conditions for R2 (CID or CSD) response.
  * @retval SD Card error state
  */
static SD_Error SDIO_CmdResp2Error(void)
{
  SD_Error errorstatus = SD_OK;

  while(!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL | SDIO_FLAG_CMDREND | SDIO_FLAG_CTIMEOUT))
  {
  }
    
  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT))
  {
    errorstatus = SD_CMD_RSP_TIMEOUT;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT);
    return errorstatus;
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL))
  {
    errorstatus = SD_DATA_CRC_FAIL;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL);
    return errorstatus;
  }

  /* Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);

  return SD_OK;
}

/**
  * @brief  Checks for error conditions for R3 (OCR) response.
  * @retval SD Card error state
  */
static SD_Error SDIO_CmdResp3Error(void)
{
  SD_Error errorstatus = SD_OK;

  while(!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL | SDIO_FLAG_CMDREND | SDIO_FLAG_CTIMEOUT))
  {
  }
  
  if(SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT))
  {
    errorstatus = SD_CMD_RSP_TIMEOUT;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT);
    return errorstatus;
  }

  /* Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
  return errorstatus;
}

/**
  * @brief  Checks for error conditions for R6 (RCA) response.
  * @param  SD_CMD: The sent command index
  * @param  pRCA: Pointer to the variable that will contain the SD card relative
  *         address RCA
  * @retval SD Card error state
  */
static SD_Error SDIO_CmdResp6Error(uint8_t SD_CMD, uint16_t *pRCA)
{
  SD_Error errorstatus = SD_OK;
  uint32_t response_r1;

  while(!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL | SDIO_FLAG_CMDREND | SDIO_FLAG_CTIMEOUT))
  {
  }
  
  if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT))
  {
    errorstatus = SD_CMD_RSP_TIMEOUT;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT);
    return errorstatus;
  }
  else if (SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL))
  {
    errorstatus = SD_DATA_CRC_FAIL;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL);
    return errorstatus;
  }

  /* Check response received is of desired command */
  if(SDIO_GetCommandResponse(SDIO) != SD_CMD)
  {
    errorstatus = SD_ILLEGAL_CMD;
    return errorstatus;
  }

  /* Clear all the static flags */
  SDIO_ClearFlagStatus(SDIO, SDIO_STATIC_FLAGS);
  
  /* We have received response, retrieve it.  */
  response_r1 = SDIO_GetResponse(SDIO, SDIO_RESP1);
  
  if(SD_ALLZERO == (response_r1 & (SD_R6_GENERAL_UNKNOWN_ERROR | SD_R6_ILLEGAL_CMD | SD_R6_COM_CRC_FAILED)))
  {
    *pRCA = (uint16_t) (response_r1 >> 16);
    
    return errorstatus;
  }

  if(response_r1 & SD_R6_ILLEGAL_CMD)
  {
    return SD_ILLEGAL_CMD;
  }
  
  if(response_r1 & SD_R6_GENERAL_UNKNOWN_ERROR)
  {
    return SD_GENERAL_UNKNOWN_ERROR;
  }

  if(response_r1 & SD_R6_COM_CRC_FAILED)
  {
    return SD_COM_CRC_FAILED;
  }

  return errorstatus;
}

/**
  * @brief  Checks for error conditions for R7 response.
  * @retval SD Card error state
  */
static SD_Error SDIO_CmdResp7Error(void)
{
  SD_Error errorstatus = SD_OK;
  uint32_t timeout = SDIO_CMD0TIMEOUT;
  
  while((!SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CCRCFAIL | SDIO_FLAG_CMDREND | SDIO_FLAG_CTIMEOUT)) && (timeout > 0))
  {
    timeout--;
  }

  if((timeout == 0) || SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT))
  {
    /* Card is SD V2.0 compliant */
    errorstatus = SD_CMD_RSP_TIMEOUT;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CTIMEOUT);
    return errorstatus;
  }
  
  if(SDIO_GetFlagStatus(SDIO, SDIO_FLAG_CMDREND))
  {
    /* Card is SD V2.0 compliant */
    errorstatus = SD_OK;
    SDIO_ClearFlagStatus(SDIO, SDIO_FLAG_CMDREND);
    return errorstatus;
  }
  return errorstatus;
}
