/*
  EEPROM.h -originally written by Ivan Grokhotkov as part of the MG32x02z 
            core for Arduino environment.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _EEPROM_H_
#define _EEPROM_H_


/**************************************************************************************************
* Flash and EEPROM support unlimited reading of their data, 
* but limit the number of times data can be written.
* EEPROM has a limit on the number of write operations, 
* so be careful not to place the write operation in the loop or other loop executing functions.
**************************************************************************************************/
/**************************************************************************************************
*      The TH244A(mcu :MG32F02U128) internal 128K bytes of FLASH program memory,
* built-in control logic that simulates FLASH as EEPROM
* (It can be set range from 0.5K bytes to 8K bytes ), 
* part of the program FLASH can be divided into data FLASH space according to application needs, 
* and EEPROM-like interface access is realized through the EEPROM controller.
*       When you include library - EEPROM.h in your ino project,
*  as default,you can use 512 bytes flash space as EEPROM fuction.
*/
/**************************************************************************************************
* If you need more EEPROM space, you need to modify the definition of 
* a specified macro: EEPROM_SIZE in library EEPROM.h file. 
* Can't support define EEPROM_SIZE in Arduino sketch .ino file.
* (EEPROM.h file PATH refer to:
*  ......\Arduino15\packages\megawin\hardware\MG32x02z\2.3.0\libraris\EEPROM\src)
* Support set EEPROM_SIZE range from 0.5K to 8K in increments of 0.5K bytes.
* Here the default setting is 512 bytes. You can modify the value to other as following list.
* Unspecify comments when you need to compile.
**************************************************************************************************/
//====Need to modify the definition of a specified macro: EEPROM_SIZE in library //EEPROM.h file.
// Can't  support define EEPROM_SIZE in Arduino sketch .ino file.
//================Support EEPROM_SIZE list begin. =======================
#define EEPROM_SIZE 512   //default set EEPROM 0.5K   512 bytes
//#define EEPROM_SIZE 1024  // set EEPROM 1K   1024 bytes
//#define EEPROM_SIZE 1536  // set EEPROM 1.5K   1536 bytes
//#define EEPROM_SIZE 2048  // set EEPROM 2K   2048 bytes
//#define EEPROM_SIZE 2560  // set EEPROM 2.5K   2560 bytes
//#define EEPROM_SIZE 3072  // set EEPROM 3K   3072 bytes
//#define EEPROM_SIZE 3584  // set EEPROM 3.5K   3584 bytes
//#define EEPROM_SIZE 4096  // set EEPROM 4K   4096 bytes
//#define EEPROM_SIZE 4608  // set EEPROM 4.5K   4608 bytes
//#define EEPROM_SIZE 5120  // set EEPROM 5K   5120 bytes
//#define EEPROM_SIZE 5632  // set EEPROM 5.5K   5632 bytes
//#define EEPROM_SIZE 6144  // set EEPROM 6K   6144 bytes
//#define EEPROM_SIZE 6656  // set EEPROM 6.5K   6656 bytes
//#define EEPROM_SIZE 7168  // set EEPROM 7K   7168 bytes
//#define EEPROM_SIZE 7680  // set EEPROM 7.5K   7680 bytes
//#define EEPROM_SIZE 8192  // set EEPROM 8K   8192 bytes
//===========Support EEPROM_SIZE list end============





#ifndef EEPROM_FLASH_PARTITION_NAME
  #define EEPROM_FLASH_PARTITION_NAME "eeprom"
#endif

#include <Arduino.h>




//if no defined.set default size
#ifndef EEPROM_SIZE
#define EEPROM_SIZE 512
#endif

class EEPROMClass 
{
public:
    EEPROMClass(uint32_t sector);
    EEPROMClass(void);
    ~EEPROMClass(void);

    bool begin(size_t size = EEPROM_SIZE);
    bool fromNVS();
    bool toNVS();
    bool wasRestored();
    uint8_t read(int address);
    void write(int address, uint8_t val);
    uint16_t length();
    bool commit();
    void end();

    uint8_t * getDataPtr();

    template<typename T> T &get(int address, T &t) 
    {
        if( (address < 0) 
         || (((size_t)address + (size_t)sizeof(T)) > _size) )
        {
            return t;
        }
        memcpy((uint8_t*) &t, _data + address, sizeof(T));
        return t;
    }

    template<typename T> const T &put(int address, const T &t)
    {
        if( (address < 0) 
         || (((size_t)address + (size_t)sizeof(T)) > _size))
        {
            return t;
        }
        memcpy(_data + address, (const uint8_t*) &t, sizeof(T));
        return t;
    }

    uint8_t readByte(int address);
    int8_t readChar(int address);
    uint8_t readUChar(int address);
    int16_t readShort(int address);
    uint16_t readUShort(int address);
    int32_t readInt(int address);
    uint32_t readUInt(int address);
    int32_t readLong(int address);
    uint32_t readULong(int address);
    int64_t readLong64(int address);
    uint64_t readULong64(int address);
    float_t readFloat(int address);
    double_t readDouble(int address);
    bool readBool(int address);
    size_t readString(int address, char* value, size_t maxLen);
    String readString(int address);
    size_t readBytes(int address, void * value, size_t maxLen);
    template <class T> T readAll (int address, T &);

    size_t writeByte(int address, uint8_t value);
    size_t writeChar(int address, int8_t value);
    size_t writeUChar(int address, uint8_t value);
    size_t writeShort(int address, int16_t value);
    size_t writeUShort(int address, uint16_t value);
    size_t writeInt(int address, int32_t value);
    size_t writeUInt(int address, uint32_t value);
    size_t writeLong(int address, int32_t value);
    size_t writeULong(int address, uint32_t value);
    size_t writeLong64(int address, int64_t value);
    size_t writeULong64(int address, uint64_t value);
    size_t writeFloat(int address, float_t value);
    size_t writeDouble(int address, double_t value);
    size_t writeBool(int address, bool value);
    size_t writeString(int address, const char* value);
    size_t writeString(int address, String value);
    size_t writeBytes(int address, const void* value, size_t len);
    template <class T> T writeAll (int address, const T &);

protected:
    uint32_t  _sector;
    size_t    _size;
    uint8_t   _data[EEPROM_SIZE];
    bool      _restored;  
};

//#if !defined(NO_GLOBAL_INSTANCES) && !defined(NO_GLOBAL_EEPROM)
extern EEPROMClass EEPROM;
//#endif

#endif
