/*
 Copyright (c) 2011 Arduino.  All right reserved.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include  "pwm.h"

#define  PWMPIN_AFS_CFG(a, num, afs)      .pin=MCU_##a##num, .pinAFS=a##num##_AFS_##afs

#ifdef  MG32F02U128
static pwm_config_str  pwm_cfg[NB_PWM_CHANNELS] = 
{
    {   //D42
        PWMPIN_AFS_CFG(PA, 12, TM26_OC00),
        .timStruct     = TM26,
        .timChannel    = MID_TM_Channel0,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },

    {   //D43
        PWMPIN_AFS_CFG(PA, 13, TM26_OC10),
        .timStruct     = TM26,
        .timChannel    = MID_TM_Channel1,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },

    
    {   //D3
        PWMPIN_AFS_CFG(PB, 10, TM20_OC11),
        .timStruct     = TM20,
        .timChannel    = MID_TM_Channel1,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },
    
    

    {   //D5
        PWMPIN_AFS_CFG(PB, 11, TM36_OC12),
        .timStruct     = TM36,
        .timChannel    = MID_TM_Channel1,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },
    
    {   //D6
        PWMPIN_AFS_CFG(PC, 0, TM20_OC00),
        .timStruct     = TM20,
        .timChannel    = MID_TM_Channel0,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },
    
    

    {   //D8
        PWMPIN_AFS_CFG(PC, 12, TM36_OC3),
        .timStruct     = TM36,
        .timChannel    = MID_TM_Channel3,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },
    
    {   //D9
        PWMPIN_AFS_CFG(PD, 0, TM36_OC2),
        .timStruct     = TM36,
        .timChannel    = MID_TM_Channel2,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },
    
    {   //D11
        PWMPIN_AFS_CFG(PD, 2, TM26_OC00),
        .timStruct     = TM26,
        .timChannel    = MID_TM_Channel0,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },
    
    {   //D10
        PWMPIN_AFS_CFG(PD, 9, TM26_OC11),
        .timStruct     = TM26,
        .timChannel    = MID_TM_Channel1,
        .clock_freq  = PWM_FREQUENCY,
        .doInit      = 0,
    },
};


#endif


int8_t get_pwm_instance( uint32_t pin )
{
    int8_t i;

    for(i = 0; i < NB_PWM_CHANNELS; i++) 
    {
        if( ( pwm_cfg[i].pin == pin) ) 
        {
            return i;
        }
    }
    return -1;
}


/**
  * @brief PWM_MspInit Initialization
  *        This function configures the hardware resources used in this example:
  *           - Peripheral's clock enable
  *           - Peripheral's GPIO Configuration
  * @param pin : Arduino pin
  * @retval None
  */
void PWM_MspInit(  uint32_t pin )
{
    int8_t idx = -1;
    
    Pin_Struct*  pPinx;		
    GPIO_InitTypeDef GPIOX;

    idx = get_pwm_instance( pin );
    if( ( idx >= NB_PWM_CHANNELS ) ||(idx < 0 ) )
    {
        return;
    }
    
    ///Enable GPIO Channels Clock
    /// Enable GPIO clock
    if( g_APinDescription[pin].ulPort == GPIOA ) 
    {
        __DRV_CSC_GPIOA_CLK_ENABLE();
    }
    else if( g_APinDescription[pin].ulPort == GPIOB )
    {
        __DRV_CSC_GPIOB_CLK_ENABLE();
    }
    else if( g_APinDescription[pin].ulPort == GPIOC )
    {
        __DRV_CSC_GPIOC_CLK_ENABLE();
    }


    GPIOX.Mode = GPIO_MODE_PUSHPULL_O;
    GPIOX.Pull = GPIO_NOPULLUP;

    GPIOX.Pin            = g_APinDescription[pin].GioPin;
    GPIOX.Speed          = GPIO_SPEED_HIGH;
    GPIOX.OUTDrive       = GPIO_OUTDRIVE_LEVEL0;
    GPIOX.FilterDivider  = GPIO_FILTERDIVIDER_BYPASS;
    GPIOX.Inverse        = GPIO_INVERSE_DISABLE;
    GPIOX.Alternate      = pwm_cfg[idx].pinAFS;
    pPinx = __GetPinxx_byGPIOx( g_APinDescription[pin].ulPort,  g_APinDescription[pin].PinNum );
    MID_GPIO_Pin_Init(  pPinx,  &GPIOX);

}


/**
  * @brief  DeInitializes TIM PWM_MspDeInit MSP.
  * @param  pin : Arduino pin
  * @retval None
  */
void PWM_MspDeInit(  uint32_t pin  )
{
    int8_t idx = -1;

    TM_HandleTypeDef        timHandle;

    idx = get_pwm_instance( pin );
    if( ( idx >= NB_PWM_CHANNELS ) ||(idx < 0 ) )
    {
        return;
    }
    
    timHandle.Instance                  = pwm_cfg[idx].timStruct;
    timHandle.Init.TM_CounterMode       = TM_SEPARATE_UP;
    timHandle.Init.TM_Period            = 1024;
    timHandle.Init.TM_Prescaler         = 0;

    pwm_cfg[idx].doInit      = 0;
    MID_TM_PWM_DeInit( &timHandle   );
}

/**
  * @brief  This function will set the PWM frequency value
  * @param  pin : the gpio pin to use
  * @param  clock_freq : frequency of the tim clock
  * @retval None
  */
void pwm_setFrequency(uint32_t ulPin,  uint32_t freq)
{
    int  i = 0;

    //find the instance in the global
    i = get_pwm_instance( ulPin );
    if( ( i >= NB_PWM_CHANNELS ) ||(i < 0 ) )
    {
        return;
    }

    pwm_cfg[i].clock_freq = freq;
    pwm_cfg[i].doInit     = 0;

}

/**
  * @brief  This function will set the PWM to the required value
  * @param  pin : the gpio pin to use
  * @param  period : period of the tim counter
  * @param  value : the value to push on the PWM output
  * @param  do_init : if set to 1 the initialization of the PWM is done
  * @retval None
  */
void pwm_start( uint32_t pin,
                uint32_t period,  uint32_t value,        uint8_t do_init )
{
    uint32_t Prescaler = 0;
    uint32_t fcpuDiv = 1;
    uint32_t Cnt = (SystemCoreClock/PWM_FREQUENCY)-1;
    
    uint32_t FrqTemp;

    int  i = 0;
    Pin_Struct*  pPinx;		
    GPIO_InitTypeDef GPIOX;
    
    do_init = do_init;

    TM_HandleTypeDef        timHandle;
    TM_OC_InitTypeDef       sConfig;
    TM_ClockConfigTypeDef   CKConfig;

    //find the instance in the global
    i = get_pwm_instance( pin );
    if( ( i >= NB_PWM_CHANNELS ) ||(i < 0 ) )
    {
        return;
    }
    
    
    FrqTemp = (pwm_cfg[i].clock_freq)/(Prescaler+1);
    
#if (F_CPU == 12000000) 
    fcpuDiv = 1;
#elif (F_CPU == 36000000)
    fcpuDiv = 2;
#else
    #error F_CPU frequency have to define 12MHz or 36MHz
#endif

    if( pwm_cfg[i].clock_freq >= 300 )
    {
        Prescaler = 0;
        Cnt = ((SystemCoreClock/fcpuDiv)/(pwm_cfg[i].clock_freq))-1;

    }
    else if( pwm_cfg[i].clock_freq >= 100 )
    {
        Prescaler = 3;
    }
    else if( pwm_cfg[i].clock_freq >= 30 )
    {
        Prescaler = 15;
    }
    else if( pwm_cfg[i].clock_freq >= 10 )
    {
        Prescaler = 63;
    }
    else if( pwm_cfg[i].clock_freq >= 3 )
    {
        Prescaler = 255;
    }
    else if( pwm_cfg[i].clock_freq >= 1 )
    {
        Prescaler = 1023;
    }
    else 
    {
        pwm_stop(pin);
        return;
    }
    
    FrqTemp = (pwm_cfg[i].clock_freq)*(Prescaler+1);
    Cnt = ((SystemCoreClock/fcpuDiv)/(FrqTemp))-1;

    timHandle.Instance  = pwm_cfg[i].timStruct;
    // ------------------------------------------------------------------------
    // 1.Initial for 10-bit PWM.
    // ------------------------------------------------------------------------
    if( Prescaler == 0 )
    {
        timHandle.Init.TM_CounterMode       = TM_SEPARATE_UP;
    }
    else 
    {
        timHandle.Init.TM_CounterMode       = TM_CASCADE_UP;
    }
    timHandle.Init.TM_Period            = Cnt;
    timHandle.Init.TM_Prescaler         = Prescaler;

    if( pwm_cfg[i].doInit == 0 )
    {
        GPIOX.Mode = GPIO_MODE_PUSHPULL_O;
        GPIOX.Pull = GPIO_PULLUP;
        GPIOX.Pin            = g_APinDescription[ pwm_cfg[i].pin ].GioPin;
        GPIOX.Speed          = GPIO_SPEED_LOW;
        GPIOX.OUTDrive       = GPIO_OUTDRIVE_LEVEL0;
        GPIOX.FilterDivider  = GPIO_FILTERDIVIDER_BYPASS;
        GPIOX.Inverse        = GPIO_INVERSE_DISABLE;
        GPIOX.Alternate      = pwm_cfg[i].pinAFS;
        pPinx = __GetPinxx_byGPIOx( g_APinDescription[pwm_cfg[i].pin].ulPort, g_APinDescription[pwm_cfg[i].pin].PinNum );
        MID_GPIO_Pin_Init(  pPinx,  &GPIOX);     

        // ------------------------------------------------------------------------
        // 1.Initial for 10-bit PWM.
        // ------------------------------------------------------------------------
        timHandle.Instance  = pwm_cfg[i].timStruct;
        if( Prescaler == 0 )
        {
            timHandle.Init.TM_CounterMode       = TM_SEPARATE_UP;
        }
        else 
        {
            timHandle.Init.TM_CounterMode       = TM_CASCADE_UP;
        }
        timHandle.Init.TM_Period            = Cnt;
        timHandle.Init.TM_Prescaler         = Prescaler;
        MID_TM_PWM_Init( &timHandle );
        
        // ------------------------------------------------------------------------
        // 2.Initial clock mode.
        // ------------------------------------------------------------------------
        CKConfig.TM_ClockSource         = TM_INTERNAL_CLOCK;
        CKConfig.TM_ExternalClockSource = 0;

#if (F_CPU == 12000000) 
        CKConfig.TM_INTClockDivision    = TM_INTERNALCLOCK_DIVDER_DIV1;
#elif (F_CPU == 36000000)
        CKConfig.TM_INTClockDivision    = TM_INTERNALCLOCK_DIVDER_DIV2;
#else
    #error F_CPU frequency have to define 12MHz or 36MHz        
        //CKConfig.TM_INTClockDivision    = TM_INTERNALCLOCK_DIVDER_DIV1;
#endif


        CKConfig.TM_InternalClockSource = TM_INTERNALCLOCK_PROC;
        
        MID_TM_ConfigClockSource(&timHandle, &CKConfig);  

        // ------------------------------------------------------------------------
        // 3.PWM output channel configuration.
        // ------------------------------------------------------------------------
        MID_TM_OC_Struct_Init(&sConfig);               // default initial (output state)
        sConfig.OCMode = TM_CH_16bit_PWM;
        
        sConfig.Pulse = ((Cnt*value)/period);
        MID_TM_PWM_ConfigChannel(&timHandle, &sConfig, pwm_cfg[i].timChannel);

        TM_OC1zOutput_Cmd( pwm_cfg[i].timStruct, ENABLE);

        // ------------------------------------------------------------------------
        // 4.Start PWM output.
        // ------------------------------------------------------------------------
        MID_TM_PWM_Start( &timHandle, pwm_cfg[i].timChannel);    // Channel0 be 16bit PWM output (Blocking mode: Polling) 

    }

  	// ------------------------------------------------------------------------
  	// 5.Update duty cycle.
  	// ------------------------------------------------------------------------
  	__DRV_TM_SET_COMPARE_B( &timHandle, pwm_cfg[i].timChannel, ((Cnt*value)/period) );
    pwm_cfg[i].doInit = 1;
    return;

}


/**
  * @brief  This function will disable the PWM
  * @param  pin : the gpio pin to use
  * @retval None
  */
void pwm_stop(  uint32_t pin )
{
    int8_t idx = -1;
    TM_HandleTypeDef        timHandle;

    //find the instance in the global
    idx = get_pwm_instance( pin );
    if( ( idx >= NB_PWM_CHANNELS ) ||(idx < 0 ) )
    {
        return;
    }
    
    
    timHandle.Instance                  = pwm_cfg[idx].timStruct;
    timHandle.Init.TM_CounterMode       = TM_SEPARATE_UP;
    timHandle.Init.TM_Period            = 1024;
    timHandle.Init.TM_Prescaler         = 0;

    pwm_cfg[idx].doInit     = 0;
    MID_TM_PWM_Stop(  &timHandle,   pwm_cfg[idx].timChannel );

}


