/*
  Copyright (c) 2011 Arduino.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "Arduino.h"
#include "variant.h"
#include "HardwareSerial.h"
#include "chip.h"
#include "uart.h"

#include "MG32x02z_GPIO_MID.h"
#include "MG32x02z_CFG.h"
#include "pins_arduino.h"



/*
 * UART objects
 */
 #if defined(HWSERIAL0)
HardwareSerial  Serial( &mURT0,  
                         ARDUINO_PIN_UART0_RX,
                         ARDUINO_PIN_UART0_TX, 
                         URT0_IRQn );
#endif


#if defined(HWSERIAL1)
HardwareSerial Serial1( &mURT1, 
              ARDUINO_PIN_UART1_RX,
              ARDUINO_PIN_UART1_TX, 
              URT123_IRQn );
#endif
#if defined(HWSERIAL2)
HardwareSerial Serial2( &mURT2,  
              ARDUINO_PIN_UART2_RX,
              ARDUINO_PIN_UART2_TX, 
              URT123_IRQn );
#endif
#if defined(URT3)&defined(HWSERIAL3)
HardwareSerial Serial3( &mURT3,
              ARDUINO_PIN_UART3_RX,
              ARDUINO_PIN_UART3_TX, 
              URT123_IRQn );
#endif
#if defined(HWSERIAL4)
HardwareSerial Serial4( &mURT4,  
              ARDUINO_PIN_UART4_RX,
              ARDUINO_PIN_UART4_TX, 
              URT4x_IRQn );
#endif
#if defined(HWSERIAL5)
HardwareSerial Serial5( &mURT5,  
              ARDUINO_PIN_UART5_RX,
              ARDUINO_PIN_UART5_TX, 
              URT4x_IRQn );
#endif
#if defined(HWSERIAL6)
HardwareSerial Serial6( &mURT6, 
              ARDUINO_PIN_UART6_RX,
              ARDUINO_PIN_UART6_TX, 
              URT4x_IRQn );
#endif
#if defined(HWSERIAL7)
HardwareSerial Serial7( &mURT7,  
              ARDUINO_PIN_UART7_RX,
              ARDUINO_PIN_UART7_TX, 
              URT4x_IRQn );
#endif



typedef struct
{
    uint32_t                pin;
    uint16_t                pinAFS;
    uint16_t                padding;
    URT_HandleTypeDef       *pURT;
}Uart_config_str;

#define  URTPIN_AFS_CFG(a, num, Ur, dr)      {.pin= (MCU_##a##num), .pinAFS=a##num##_AFS_##Ur##_##dr,  .padding=0,  .pURT=&m##Ur }

#ifdef    MG32F02U128
#define   NB_UART_CFG     63
static const Uart_config_str  uar_cfg[NB_UART_CFG] = 
{
    URTPIN_AFS_CFG(PA,8,URT4,TX ),
    URTPIN_AFS_CFG(PA,9,URT5,TX),
    URTPIN_AFS_CFG(PA,10,URT4,RX),
    URTPIN_AFS_CFG(PA,11,URT5,RX ),
    URTPIN_AFS_CFG(PA,12,URT6,TX),
    URTPIN_AFS_CFG(PA,13,URT6,RX ),
    URTPIN_AFS_CFG(PA,14,URT7,TX ),
    URTPIN_AFS_CFG(PA,15,URT7,RX ),
    URTPIN_AFS_CFG(PB,0,URT6,TX ),
    URTPIN_AFS_CFG(PB,1,URT1,RX),
    URTPIN_AFS_CFG(PB,1,URT6,RX),
    URTPIN_AFS_CFG(PB,2,URT0,TX),
    URTPIN_AFS_CFG(PB,2,URT7,TX ),
    URTPIN_AFS_CFG(PB,2,URT2,TX),
    URTPIN_AFS_CFG(PB,3,URT0,RX),
    URTPIN_AFS_CFG(PB,3,URT7,RX ),
    URTPIN_AFS_CFG(PB,3,URT2,RX),
    URTPIN_AFS_CFG(PB,3,URT1,TX),
    URTPIN_AFS_CFG(PB,6,URT2,TX ),
    URTPIN_AFS_CFG(PB,7,URT2,RX ),
    URTPIN_AFS_CFG(PB,8,URT0,TX),
    URTPIN_AFS_CFG(PB,8,URT4,TX),
    URTPIN_AFS_CFG(PB,9,URT0,RX),
    URTPIN_AFS_CFG(PB,9,URT4,RX),
    URTPIN_AFS_CFG(PB,10,URT1,TX),
    URTPIN_AFS_CFG(PB,11,URT1,RX),
    URTPIN_AFS_CFG(PB,12,URT5,TX),
    URTPIN_AFS_CFG(PB,13, URT4,RX),
    URTPIN_AFS_CFG(PB,14,URT4,TX),
    URTPIN_AFS_CFG(PB,15,URT5,RX),
    URTPIN_AFS_CFG(PC,0,URT5,TX),
    URTPIN_AFS_CFG(PC,1,URT0,RX),
    URTPIN_AFS_CFG(PC,1,URT5,RX),
    URTPIN_AFS_CFG(PC,4,URT0,RX),
    URTPIN_AFS_CFG(PC,4,URT1,RX),
    URTPIN_AFS_CFG(PC,4,URT6,RX),
    URTPIN_AFS_CFG(PC,5,URT0,TX),
    URTPIN_AFS_CFG(PC,5,URT1,TX),
    URTPIN_AFS_CFG(PC,5,URT6,TX),
    URTPIN_AFS_CFG(PC,8, URT1,TX),
    URTPIN_AFS_CFG(PC,8,URT6,TX ),
    URTPIN_AFS_CFG(PC,9,URT1,RX),
    URTPIN_AFS_CFG(PC,9,URT6,RX),
    URTPIN_AFS_CFG(PC,10,URT0,TX),
    URTPIN_AFS_CFG(PC,10,URT1,TX),
    URTPIN_AFS_CFG(PC,10,URT2,TX),
    URTPIN_AFS_CFG(PC,10,URT7,TX),
    URTPIN_AFS_CFG(PC,11,URT0,RX),
    URTPIN_AFS_CFG(PC,11,URT1,RX),
    URTPIN_AFS_CFG(PC,11,URT2,RX),
    URTPIN_AFS_CFG(PC,11,URT7,RX ),
    URTPIN_AFS_CFG(PC,13,URT2,RX),
    URTPIN_AFS_CFG(PC,13,URT6,RX),
    URTPIN_AFS_CFG(PC,14, URT2,TX),
    URTPIN_AFS_CFG(PC,14,URT6,TX),
    URTPIN_AFS_CFG(PD,2,URT2,TX),
    URTPIN_AFS_CFG(PD,3,URT2,RX),
    URTPIN_AFS_CFG(PA,0,URT4,TX ),
    URTPIN_AFS_CFG(PA,1,URT4,RX),
    URTPIN_AFS_CFG(PA,2,URT5,TX),
    URTPIN_AFS_CFG(PA,3,URT5,RX),
    URTPIN_AFS_CFG(PA,4,URT0,TX),
    URTPIN_AFS_CFG(PA,5,URT0,RX),
};

#endif



HardwareSerial::HardwareSerial(void)
{
    m_usart_typedef = NULL;
    m_irqtype = URT4x_IRQn;

    m_tx_pin = ((uint16_t)-1);
    m_rx_pin = ((uint16_t)-1);

    m_tx_pinAFS = ((uint16_t)-1);
    m_rx_pinAFS = ((uint16_t)-1);
            
    m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;
    
    m_available = 0;
}

HardwareSerial::HardwareSerial( URT_HandleTypeDef *uartdef)
{
    m_usart_typedef = uartdef;
    m_usart_typedef->Init.BaudRate = DEFAULT_INIT_BAUDRATE;

    m_irqtype = URT4x_IRQn;
    
    m_tx_pin = ((uint16_t)-1);
    m_rx_pin = ((uint16_t)-1);

    m_tx_pinAFS = ((uint16_t)-1);
    m_rx_pinAFS = ((uint16_t)-1);

    m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;
    
    m_available = 0;
}

HardwareSerial::HardwareSerial( URT_HandleTypeDef *uartdef,  uint16_t  RxPin,  uint16_t  TxPin,  IRQn_Type  irqn)
{
    m_usart_typedef = uartdef;
    m_usart_typedef->Init.BaudRate = DEFAULT_INIT_BAUDRATE;
    m_irqtype = irqn;
    
    m_tx_pin = TxPin;
    m_rx_pin = RxPin;

    m_tx_pinAFS = ((uint16_t)-1);
    m_rx_pinAFS = ((uint16_t)-1);

    m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;
    
    m_available = 0;
}


int HardwareSerial::begin(  uint16_t   ArdRxPin,      uint16_t   ArdTxPin,   unsigned long baudRate)
{  
    return begin( m_usart_typedef,
                  ArdRxPin,       ArdTxPin,
                  m_irqtype,      baudRate  );
} 

int HardwareSerial::begin( URT_HandleTypeDef *uartdef,
                           uint16_t   ArdRxPin,       uint16_t   ArdTxPin,
                           IRQn_Type      irqn,       unsigned long baudRate  )
{
    Pin_Struct*  pPinx;		
    GPIO_InitTypeDef PINX_InitStruct;

    uint32_t  i;

    uint16_t  Tx_pinAFS = -1;
    uint16_t  Rx_pinAFS = -1;

    if( uartdef != NULL ) 
    {
        m_usart_typedef = uartdef;
        m_usart_typedef->Init.BaudRate = baudRate;
    }

    if( m_usart_typedef == NULL ) 
    {
        return -1;
    }

    m_irqtype = irqn;

    m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;


    if( ( ArdRxPin >= MAX_DIGITAL_IOS )  )
    {
        ArdRxPin =  m_rx_pin;
    }

    if( ( ArdTxPin >= MAX_DIGITAL_IOS )  )
    {
        ArdTxPin =  m_tx_pin;
    }

    if( ( ArdTxPin < MAX_DIGITAL_IOS )  )
    {
         for( i=0; i<NB_UART_CFG; i++  )
        {
            if(  ( uar_cfg[i].pin == ArdTxPin )  && (uartdef == uar_cfg[i].pURT)  )
            {
                Tx_pinAFS = uar_cfg[i].pinAFS;
                break;
            }
        }

        if( i==NB_UART_CFG )
        {
            Tx_pinAFS = (uint16_t)-1;
        }
        else
        {
            m_tx_pin    = ArdTxPin;
            m_tx_pinAFS = Tx_pinAFS;
        }
    }

    if( ( ArdRxPin < MAX_DIGITAL_IOS )  )
    {
        for( i=0; i<NB_UART_CFG; i++  )
        {
            if(  ( uar_cfg[i].pin == ArdRxPin )  && (uartdef == uar_cfg[i].pURT)  )
            {
                Rx_pinAFS = uar_cfg[i].pinAFS;
                break;
            }
        }
        if( i==NB_UART_CFG )
        {
            Rx_pinAFS = ((uint16_t)-1);
        }
        else
        {
            m_rx_pin    = ArdRxPin;
            m_rx_pinAFS = Rx_pinAFS;
        }
    } 

    //GPIO Inital :   PINX_Alternate_Function
    PINX_InitStruct.Speed          = GPIO_SPEED_LOW;
    PINX_InitStruct.OUTDrive       = GPIO_OUTDRIVE_LEVEL0;
    PINX_InitStruct.FilterDivider  = GPIO_FILTERDIVIDER_BYPASS;
    PINX_InitStruct.Inverse        = GPIO_INVERSE_DISABLE;

    ///cfg rx pin
    if(  ( m_rx_pinAFS != ((uint16_t)-1) )  
      && ( m_rx_pin < MAX_DIGITAL_IOS)   )
    {
        PINX_InitStruct.Mode	      = GPIO_MODE_DIGITAL_I;
        PINX_InitStruct.Pull          = GPIO_PULLUP;
        PINX_InitStruct.Alternate     = m_rx_pinAFS;
        pPinx = __GetPinxx_byGPIOx( g_APinDescription[m_rx_pin].ulPort,  g_APinDescription[m_rx_pin].PinNum );
        MID_GPIO_Pin_Init(  pPinx,  &PINX_InitStruct);
    }

    if(  ( m_tx_pinAFS != ((uint16_t)-1) ) 
      && ( m_tx_pin < MAX_DIGITAL_IOS)  )
    {
        ///cfg tx pin
        PINX_InitStruct.Mode	       = GPIO_MODE_PUSHPULL_O;
        PINX_InitStruct.Pull           = GPIO_PULLUP;
        PINX_InitStruct.Alternate      = m_tx_pinAFS;
        pPinx = __GetPinxx_byGPIOx( g_APinDescription[m_tx_pin].ulPort, g_APinDescription[m_tx_pin].PinNum );
        MID_GPIO_Pin_Init( pPinx, &PINX_InitStruct);
    }

    NVIC_DisableIRQ( m_irqtype );
    m_usart_typedef->Init.BaudRate = baudRate;

    MID_URT_Init( m_usart_typedef );

    m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;

    //Enable NVIC IRQ :
    NVIC_SetPriority( m_irqtype,   1);
    NVIC_EnableIRQ( m_irqtype );

    //flush();

    MID_URT_Receive_IT( m_usart_typedef,  &m_rxTemp, 1 );
    
    m_available = 1;

    //write(" \b\b");
    return 0;
}


void HardwareSerial::begin()
{
    if( m_usart_typedef != NULL )
    {
        begin(m_usart_typedef->Init.BaudRate );
    }
    return;
}

// Public Methods //////////////////////////////////////////////////////////////
void HardwareSerial::begin(unsigned long baudRate)
{
    begin( m_usart_typedef,
           m_rx_pin,        m_tx_pin,
           m_irqtype,       baudRate  );
    return;
}

void HardwareSerial::rs485Begin(unsigned long baudRate)
{
    rs485Begin( m_usart_typedef,
           m_rx_pin,        m_tx_pin,
           m_irqtype,       baudRate  );
    return;
}

int HardwareSerial::rs485Begin(  URT_HandleTypeDef *uartdef,
                  uint16_t   ArdRxPin,       uint16_t   ArdTxPin,
                  IRQn_Type      irqn,       unsigned long baudRate  )
{
    Pin_Struct*  pPinx;		
    GPIO_InitTypeDef PINX_InitStruct;

    uint32_t  i;

    uint16_t  Tx_pinAFS = -1;
    uint16_t  Rx_pinAFS = -1;

    if( uartdef != NULL ) 
    {
        m_usart_typedef = uartdef;
        m_usart_typedef->Init.BaudRate = baudRate;
    }

    if( m_usart_typedef == NULL ) 
    {
        return -1;
    }

    m_irqtype = irqn;

    m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;


    if( ( ArdRxPin >= MAX_DIGITAL_IOS )  )
    {
        ArdRxPin =  m_rx_pin;
    }

    if( ( ArdTxPin >= MAX_DIGITAL_IOS )  )
    {
        ArdTxPin =  m_tx_pin;
    }

    if( ( ArdTxPin < MAX_DIGITAL_IOS )  )
    {
         for( i=0; i<NB_UART_CFG; i++  )
        {
            if(  ( uar_cfg[i].pin == ArdTxPin )  && (uartdef == uar_cfg[i].pURT)  )
            {
                Tx_pinAFS = uar_cfg[i].pinAFS;
                break;
            }
        }

        if( i==NB_UART_CFG )
        {
            Tx_pinAFS = (uint16_t)-1;
        }
        else
        {
            m_tx_pin    = ArdTxPin;
            m_tx_pinAFS = Tx_pinAFS;
        }
    }

    if( ( ArdRxPin < MAX_DIGITAL_IOS )  )
    {
        for( i=0; i<NB_UART_CFG; i++  )
        {
            if(  ( uar_cfg[i].pin == ArdRxPin )  && (uartdef == uar_cfg[i].pURT)  )
            {
                Rx_pinAFS = uar_cfg[i].pinAFS;
                break;
            }
        }
        if( i==NB_UART_CFG )
        {
            Rx_pinAFS = ((uint16_t)-1);
        }
        else
        {
            m_rx_pin    = ArdRxPin;
            m_rx_pinAFS = Rx_pinAFS;
        }
    } 

    //GPIO Inital :   PINX_Alternate_Function
    PINX_InitStruct.Speed          = GPIO_SPEED_LOW;
    PINX_InitStruct.OUTDrive       = GPIO_OUTDRIVE_LEVEL0;
    PINX_InitStruct.FilterDivider  = GPIO_FILTERDIVIDER_BYPASS;
    PINX_InitStruct.Inverse        = GPIO_INVERSE_DISABLE;

    ///cfg rx pin
    if(  ( m_rx_pinAFS != ((uint16_t)-1) )  
      && ( m_rx_pin < MAX_DIGITAL_IOS)   )
    {
        PINX_InitStruct.Mode	      = GPIO_MODE_DIGITAL_I;
        PINX_InitStruct.Pull          = GPIO_PULLUP;
        PINX_InitStruct.Alternate     = m_rx_pinAFS;
        pPinx = __GetPinxx_byGPIOx( g_APinDescription[m_rx_pin].ulPort,  g_APinDescription[m_rx_pin].PinNum );
        MID_GPIO_Pin_Init(  pPinx,  &PINX_InitStruct);
    }

    if(  ( m_tx_pinAFS != ((uint16_t)-1) ) 
      && ( m_tx_pin < MAX_DIGITAL_IOS)  )
    {
        ///cfg tx pin
        PINX_InitStruct.Mode	       = GPIO_MODE_PUSHPULL_O;
        PINX_InitStruct.Pull           = GPIO_PULLUP;
        PINX_InitStruct.Alternate      = m_tx_pinAFS;
        pPinx = __GetPinxx_byGPIOx( g_APinDescription[m_tx_pin].ulPort, g_APinDescription[m_tx_pin].PinNum );
        MID_GPIO_Pin_Init( pPinx, &PINX_InitStruct);
    } 

    
    NVIC_DisableIRQ( m_irqtype );
    m_usart_typedef->Init.BaudRate = baudRate;


    MID_URT_RS485_Init( m_usart_typedef, URT_DE_POLARITY_HIGH , URT_DEGUARDTIME_2 );

    m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;

    NVIC_SetPriority( m_irqtype,   1);
    //==============================================================================
    //Enable NVIC IRQ :
    NVIC_EnableIRQ( m_irqtype );

    //flush();

    MID_URT_Receive_IT( m_usart_typedef,  &m_rxTemp, 1 );
    m_available = 1;

    return 0;
}

void HardwareSerial::end( void )
{
    Pin_Struct*  pPinx;		
    GPIO_InitTypeDef PINX_InitStruct;

    m_available = 0;
    if( NULL == m_usart_typedef )
    {
        return;
    }

    MID_URT_DeInit( m_usart_typedef );

    //GPIO Inital :   PINX_Alternate_Function
    PINX_InitStruct.Speed          = GPIO_SPEED_LOW;
    PINX_InitStruct.OUTDrive       = GPIO_OUTDRIVE_LEVEL0;
    PINX_InitStruct.FilterDivider  = GPIO_FILTERDIVIDER_BYPASS;
    PINX_InitStruct.Inverse        = GPIO_INVERSE_DISABLE;

    ///cfg rx pin
    if(  ( m_rx_pinAFS != ((uint16_t)-1) )  
      && ( m_rx_pin < MAX_DIGITAL_IOS)   )
    {
        PINX_InitStruct.Mode	      = GPIO_MODE_DIGITAL_I; //GPIO_MODE_ANALOG_IO;  //GPIO_MODE_DIGITAL_I;
        PINX_InitStruct.Pull          = GPIO_PULLUP;
        PINX_InitStruct.Alternate     = 0; //m_rx_pinAFS;
        pPinx = __GetPinxx_byGPIOx( g_APinDescription[m_rx_pin].ulPort,  g_APinDescription[m_rx_pin].PinNum );
        MID_GPIO_Pin_Init(  pPinx,  &PINX_InitStruct);
    }

    if(  ( m_tx_pinAFS != ((uint16_t)-1) ) 
      && ( m_tx_pin < MAX_DIGITAL_IOS)  )
    {
        ///cfg tx pin
        PINX_InitStruct.Mode	       = GPIO_MODE_DIGITAL_I; //GPIO_MODE_ANALOG_IO;  //GPIO_MODE_DIGITAL_I;
        PINX_InitStruct.Pull           = GPIO_PULLUP;
        PINX_InitStruct.Alternate      = 0; //m_tx_pinAFS;
        pPinx = __GetPinxx_byGPIOx( g_APinDescription[m_tx_pin].ulPort, g_APinDescription[m_tx_pin].PinNum );
        MID_GPIO_Pin_Init( pPinx, &PINX_InitStruct);
    } 
    
    //MID_URT_Abort( m_usart_typedef );
    //MID_URT_Abort_IT( m_usart_typedef );   
    NVIC_DisableIRQ( m_irqtype );

  	m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;
}

int HardwareSerial::available( void )
{
    //return the amount of data available
    //return m_rxData_Count;
    //return (m_available != 0);
    return m_rxData_Count;
}


int HardwareSerial::availableForWrite( void )
{
    MID_URT_StateTypeDef state;
    
    if( NULL == m_usart_typedef )
    {
        return 0;
    }
    
    //return the amount of data available
    state = MID_URT_GetState( m_usart_typedef );
    return (state & MID_URT_STATE_READY);
}

int HardwareSerial::peek( void )
{
    int8_t data = -1;
    
    if( NULL == m_usart_typedef )
    {
        return -1;
    }

    if( m_rxData_Count > 0)
    {
        data = m_rxpData[ m_rxBegin ];
    }
    return data;
}

int HardwareSerial::read( void )
{
    char data = -1;

    if( NULL == m_usart_typedef )
    {
        return -1;
    }

    NVIC_DisableIRQ( m_irqtype );
    if( m_rxData_Count > 0)
    {
        data = m_rxpData[ m_rxBegin ];
        m_rxBegin++;

        m_rxBegin = (m_rxBegin % SERIAL_BUFFER_SIZE);
        m_rxData_Count--;
        NVIC_EnableIRQ( m_irqtype );
        return (0x000000FF & data);
    }
    else
    {
        m_rxBegin = m_rxEnd;
        m_rxData_Count = 0;
        NVIC_EnableIRQ( m_irqtype );
        return 0xFFFFFFFF;
    }

    return (0x000000FF & data);
}
    
void HardwareSerial::flush( void )
{
    if( NULL == m_usart_typedef )
    {
        return;
    }
    NVIC_DisableIRQ( m_irqtype );
    m_rxData_Count = 0;
    m_rxBegin = 0;
    m_rxEnd   = 0;
    NVIC_EnableIRQ( m_irqtype );
}


size_t HardwareSerial::write( uint8_t *buffer, size_t size)
{
    if( NULL == m_usart_typedef )
    {
        return 0;
    }
    
#if !defined(BLOCKING_FOREVER)
    MID_URT_Transmit( m_usart_typedef, buffer,   size,  1000 );
#else
    MID_URT_Transmit_WithoutTimer(m_usart_typedef, buffer,   size);
#endif

    return size;

}

size_t HardwareSerial::write( const uint8_t uc_data )
{
    if( NULL == m_usart_typedef )
    {
        return 0;
    }
    m_txdata = uc_data;

#if  !defined(BLOCKING_FOREVER)
    __IO MID_URT_StateTypeDef  TxState;
    MID_URT_Transmit( m_usart_typedef, &m_txdata,   1,  5 );
    do
    {
        TxState = m_usart_typedef->gState;
        delay(1);
    }
    while( TxState != MID_URT_STATE_READY );
#else
    MID_URT_Transmit_WithoutTimer(m_usart_typedef, &m_txdata,   1);
#endif
    return 1;
}

int HardwareSerial::printHex(  uint8_t uc_data   )
{
    int i;
    
    uint8_t u8_data;
    
    for( i=2; i>0; i-- )
    {
        u8_data = ( (uc_data >> ((i-1)*4)) & 0x0F);
        if( u8_data>9 )
        {
            write( ('a'+ u8_data -10) );
        }
        else
        {
            write( ('0'+ u8_data) );
        }
    }

    return 0;
}
int HardwareSerial::printHex(  uint16_t uc_data   )
{
    int i;
    
    uint8_t u8_data;
    
    for( i=4; i>0; i-- )
    {
        u8_data = ( (uc_data >> ((i-1)*4)) & 0x000F);
        if( u8_data>9 )
        {
            write( ('a'+ u8_data -10) );
        }
        else
        {
            write( ('0'+ u8_data) );
        }
    }

    return 0;
}

int HardwareSerial::printHex(  uint32_t uc_data   )
{
    int i;
    
    uint8_t u8_data;
    
    for( i=8; i>0; i-- )
    {
        u8_data = ( (uc_data >> ((i-1)*4)) & 0x0000000F);
        if( u8_data>9 )
        {
            write( ('a'+ u8_data -10) );
        }
        else
        {
            write( ('0'+ u8_data) );
        }
    }

    return 0;
}

int HardwareSerial::printDec(  uint32_t uc_data   )
{
    int i;
    uint32_t dev = 1; 
    uint32_t writeData = 0; 
    
    for( i=0; (i<16); i++ )
    {
        dev = dev*10;
        if(  (uc_data/dev) == 0  )
        {
            break;
        }
    }
    
    for( i=0; (i<16); i++ )
    {
        dev = dev/10;
        if( dev > 0 )
        {
            writeData = uc_data/dev;
            write( ('0'+ writeData) );

            uc_data   =  uc_data%dev;
        } 
        else
        {
            break;
        }
    }

    return 0;
}


void HardwareSerial::IrqTransHandler(void)
{
    ;
}

void HardwareSerial::IrqRecvHandler(void)
{
    __IO uint8_t temp;
    temp = m_rxTemp;
    MID_URT_Receive_IT( m_usart_typedef,  &m_rxTemp,  1 );

    if( m_rxData_Count < SERIAL_BUFFER_SIZE )
    {
        m_rxpData[ m_rxEnd ] = temp;
        m_rxEnd++;
        m_rxEnd = (m_rxEnd % SERIAL_BUFFER_SIZE);
        m_rxData_Count++;
    }
    else
    {
        m_rxData_Count = SERIAL_BUFFER_SIZE;
    }

    return;
}




/**
 *******************************************************************************
 * @brief	 Replace printf fputc function.
 * @details
 * @return
 * @note
 *******************************************************************************
 */
 /*
int fputc(int ch, FILE *f __attribute__((unused)))
{
    DEBUG_URTn->TDAT.B[0] = (uint8_t)ch;
    while((DEBUG_URTn->STA.W & URT_STA_TXF_mask_w)==0);

    return(ch);
}
*/

size_t uart_write( uart_id_e uart_id, uint8_t data )
{

    switch( uart_id )
    {
        case USART0_E:
            #if defined(HWSERIAL0)
            Serial.write( data );
            #endif
            break;

        case USART1_E:
            #if defined(HWSERIAL1)
            Serial1.write( data );
            #endif
            break;

        case USART2_E:
            #if defined(HWSERIAL2)
            Serial2.write( data );
            #endif
            break;

        #if defined(URT3)
        case USART3_E:
            #if defined(HWSERIAL3)
            Serial3.write( data );
            #endif
            break;
        #endif

        case USART4_E:
            #if defined(HWSERIAL4)
            Serial4.write( data );
            #endif
            break;

        case USART5_E:
            #if defined(HWSERIAL5)
            Serial5.write( data );
            #endif
            break;

        case USART6_E:
            #if defined(HWSERIAL6)
            Serial6.write( data );
            #endif
            break;

        case USART7_E:
            #if defined(HWSERIAL7)
            Serial7.write( data );
            #endif
            break;

        default:
            break;
    }

    return 1;
}



/**
 *******************************************************************************
 * @brief	    Tx Transfer completed callback.
 * @details     
 * @param[in]   MURT       : URT handle. 
 * @return      No.
 * @exception   No
 * @note        No
 *******************************************************************************
 */
 
void MID_URT_TxCpltCallback(URT_HandleTypeDef *MURT)
{
    if( MURT == &mURT0 )
    {
        #if defined(HWSERIAL0)
        Serial.IrqTransHandler();
        #endif
    }

    if( MURT == &mURT1 )
    {
        #if defined(HWSERIAL1)
        Serial1.IrqTransHandler();
        #endif
    }

    if( MURT == &mURT2 )
    {
        #if defined(HWSERIAL2)
        Serial2.IrqTransHandler();
        #endif
    }
		
#if defined(URT3)
    if( MURT == &mURT3 )
    {
        #if defined(HWSERIAL3)
        Serial3.IrqTransHandler();
        #endif
    }
#endif
		
    if( MURT == &mURT4 )
    {
        #if defined(HWSERIAL4)
        Serial4.IrqTransHandler();
        #endif
    }

    if( MURT == &mURT5 )
    {
        #if defined(HWSERIAL5)
        Serial5.IrqTransHandler();
        #endif
    }

    if( MURT == &mURT6 )
    {
        #if defined(HWSERIAL6)
        Serial6.IrqTransHandler();
        #endif
    }

    if( MURT == &mURT7 )
    {
        #if defined(HWSERIAL7)
        Serial7.IrqTransHandler();
        #endif
    }

}

/**
 *******************************************************************************
 * @brief	    Rx Receive completed callback.
 * @details     
 * @param[in]   MURT       : URT handle. 
 * @return      No.
 * @exception   No
 * @note        No
 *******************************************************************************
 */
void MID_URT_RxCpltCallback( URT_HandleTypeDef *MURT)
{
    if( MURT == &mURT0 )
    {
        #if defined(HWSERIAL0)
        Serial.IrqRecvHandler();
        #endif
    }

    if( MURT == &mURT1 )
    {
        #if defined(HWSERIAL1)
        Serial1.IrqRecvHandler();
        #endif
    }

    if( MURT == &mURT2 )
    {
        #if defined(HWSERIAL2)
        Serial2.IrqRecvHandler();
        #endif
    }

#if defined(URT3)
    if( MURT == &mURT3 )
    {
        #if defined(HWSERIAL3)
        Serial3.IrqRecvHandler();
        #endif
    }
#endif

    if( MURT == &mURT4 )
    {
        #if defined(HWSERIAL4)
        Serial4.IrqRecvHandler();
        #endif
    }

    if( MURT == &mURT5 )
    {
        #if defined(HWSERIAL5)
        Serial5.IrqRecvHandler();
        #endif
    }

    if( MURT == &mURT6 )
    {
        #if defined(HWSERIAL6)
        Serial6.IrqRecvHandler();
        #endif
    }

    if( MURT == &mURT7 )
    {
        #if defined(HWSERIAL7)
        Serial7.IrqRecvHandler();
        #endif
    }
}


extern "C" {
    
    
#if defined(DEBUG_PRTINT) 
int DEG_printHex(  uint32_t uc_data   )
{
    return Serial.printHex(uc_data);
}

int DEG_printDec(  uint32_t uc_data   )
{
    return Serial.printDec(uc_data);
}
int DEG_print(  const char* str   )
{
    return Serial.print(str);
}


int DEG_write(  const char* str,  size_t size)
{
    return Serial.write(  (uint8_t*)str, size);
}
int DEG_putc(  char  c   )
{
    return Serial.write(c);
}


#endif


#if defined(DEBUG_PRTINT)&&defined(REGISTER_DEBUG)
#define   MAX_REGISTER_CNT    16

uint32_t   g_TRegStrPoint = 0;
uint32_t   g_TRegEndPoint = 0;
uint32_t   g_TRegCnt = 0;

uint32_t   g_TRegister[ MAX_REGISTER_CNT ] = {0};


void TestReg_clear()
{
    uint32_t   i;
    
    g_TRegStrPoint = 0;
    g_TRegEndPoint = 0;
    g_TRegCnt = 0;

    for( i=0;  i<MAX_REGISTER_CNT;  i++ )
    {
        g_TRegister[ (g_TRegStrPoint+i)%MAX_REGISTER_CNT ] = 0;
    }
}

int TestReg_push( uint32_t data )
{
    g_TRegister[ g_TRegEndPoint ] = data;

    g_TRegEndPoint++;
    g_TRegEndPoint = (g_TRegEndPoint % MAX_REGISTER_CNT);
    g_TRegCnt++;

    if( g_TRegCnt > MAX_REGISTER_CNT  )
    {
        g_TRegStrPoint++;
        g_TRegStrPoint  = (g_TRegStrPoint % MAX_REGISTER_CNT);
    }
    return g_TRegEndPoint;
}

int TestReg_print()
{
    uint32_t   iloop;
    uint32_t   i;

    DEG_print( (char*)"\r\n" );

    DEG_print( (char*)"Register Cnt");
    DEG_print( (char*)".[");
    DEG_printDec( g_TRegCnt );
    DEG_print( (char*)"].");
    DEG_print( (char*)"\r\n");
        
    DEG_print( (char*)"\r\n" );
    if( MAX_REGISTER_CNT < g_TRegCnt )
    {
        iloop = MAX_REGISTER_CNT;
    }
    else 
    {
        iloop = g_TRegCnt;
    }
    
    for( i=0;  i<iloop;  i++ )
    {
        DEG_print( (char*)"Register ");
        DEG_printDec( i );
        DEG_print( (char*)".[0x");
        DEG_printHex( g_TRegister[ ((g_TRegStrPoint+i)%MAX_REGISTER_CNT) ] );
        DEG_print( (char*)"].");
        DEG_print( (char*)"\r\n");
    }
    
    DEG_print( (char*)"\r\n");
    
//    g_TRegCnt = 0;

    return g_TRegStrPoint;
}
#endif



}  //extern "C" {

