

/**
 ******************************************************************************
 *
 * @file       Sample_URT0_LINSlave.c
 * @brief      LIN slave sample code
 *
 * @par        Project
 *             MG32 M0
 * @version    V1.01
 * @date       2025/06/10
 * @author     Megawin Software Center
 * @copyright  Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *             All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_URT_DRV.h"
#include "MG32_GPIO_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define LIN_URTX             URT0
#define LIN_URTX_IRQn        URT0_IRQn
#define LIN_URTX_IRQHandler  URT0_IRQHandler
                             
#define LIN_URTX_AUTOCAL_EN  0
                             
#define LIN_URTX_TXPINX      PINB(8)
#define LIN_URTX_RXPINX      PINB(9)
                             
#define LIN_URTX_TXAFS       3
#define LIN_URTX_RXAFS       3
                             
#define LIN_URTX_TX          PB8
#define LIN_URTX_RX          PB9

#define LIN_SLAVE_PID        0xB1   /*!<The LIN Slave sample code address*/


/* Private typedef -----------------------------------------------------------*/
typedef enum
{
    LIN_IDLE_BREAK = 0,
    LIN_SYNC,
    LIN_PID,
    LIN_SLAVE_INIT_DATA,
    LIN_MASTER_RX_DATA,
    LIN_MASTER_RX_CHECKSUM,
    LIN_MASTER_RX_DONE,
    LIN_MASTER_TX_DATA,
    LIN_MASTER_TX_CHECKSUM,
    LIN_MASTER_TX_DONE,
    LIN_ERR
}LIN_StateTypeDef;

typedef struct
{
    LIN_StateTypeDef volatile State;
    uint8_t                   PID;
    uint8_t                   Data[8];
    uint8_t                   DataLength;
    uint8_t                   DataIndex;
    uint8_t                   CheckSum;
}LIN_FrameTypeDef;


/* Private macro -------------------------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
extern  LIN_FrameTypeDef LIN_Frame;

/* Exported functions --------------------------------------------------------*/
extern void Sample_URT_LINSlaveMode_Init(void);
extern void Sample_URT_LINSlave_main(void);

extern void LIN_ErrorCallback(void);
extern uint8_t Sample_URT_LINSlave_CalcParity(uint8_t Frame_ID);
extern uint8_t Sample_URT_LINSlave_CalcChecksum( uint8_t LIN_Frame_ID, uint8_t *LIN_Data, uint8_t LIN_Data_Length);

/* External variables --------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
LIN_FrameTypeDef  LIN_Frame;

static uint8_t LIN_RX_BreakDetectflag=0;


/* Private function prototypes -----------------------------------------------*/
void LIN_URTX_IRQHandler(void);



/**
 *******************************************************************************
 * @brief	   
 * @details 
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
void LIN_URTX_IRQHandler(void)
{
    uint32_t LIN_IRQHandlerTmp;
    uint8_t  LIN_IRQHandlerRxData;
    
    LIN_IRQHandlerTmp = URT_GetITAllFlagStatus( LIN_URTX) & URT_GetITStatus( LIN_URTX);
    
    if((LIN_IRQHandlerTmp & ( URT_IT_LS | URT_IT_BK))!= ( URT_IT_LS | URT_IT_BK))
    {
        
        LIN_RX_BreakDetectflag  = 1;
        LIN_Frame.State         = LIN_SYNC;
        
        URT_ClearITFlag(LIN_URTX, (URT_IT_LS | URT_IT_BK));
    }
    
    if(LIN_IRQHandlerTmp & URT_IT_RX)
    {
        LIN_IRQHandlerRxData = (uint8_t)URT_GetRXData( LIN_URTX);
        
        if( LIN_RX_BreakDetectflag == 1)
        {
            if( LIN_Frame.State  == LIN_SYNC)
            {
                if( LIN_IRQHandlerRxData == 0x55)
                {
                    LIN_Frame.State++;
                    return;
                }
                else
                {
                    LIN_Frame.State = LIN_ERR;
                    LIN_ErrorCallback();
                    return;
                }
            }
            if( LIN_Frame.State == LIN_PID)
            {
                if( LIN_IRQHandlerRxData == LIN_SLAVE_PID)
                {
                    LIN_Frame.State = LIN_SLAVE_INIT_DATA;
                }
            }
//            else if(LIN_Frame.State == LIN_MASTER_RX_DATA)
//            {
//                LIN_Frame.Data[LIN_Frame.DataIndex] = LIN_IRQHandlerRxData;
//                LIN_Frame.DataIndex++;
//                if( LIN_Frame.DataIndex >= LIN_Frame.DataLength)
//                {
//                    LIN_Frame.State++;
//                }
//            }
//            else if(LIN_Frame.State == LIN_MASTER_RX_CHECKSUM)
//            {
//                LIN_Frame.CheckSum = LIN_IRQHandlerRxData;
//                LIN_Frame.State++;
//            }
//            else if( LIN_Frame.State == LIN_MASTER_RX_DONE)
//            {
//                //RX done
//            }
        }
        
    }
}


/**
 *******************************************************************************
 * @brief	   
 * @details 
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_URT_LINSlaveMode_Init(void)
{
    URT_Data_TypeDef URT_LIN_DataDef;
    URT_BRG_TypeDef  URT_LIN_BRG;
    
    /*URT_TX / URT_RX pin can using MG32_GPIO_xxx_Init.h to configure or 
    following code to configure.*/
    PIN_InitTypeDef    URT_LIN_PINXInitStruct;

    URT_LIN_PINXInitStruct.PINX_Mode               = PINX_Mode_PushPull_O;
    URT_LIN_PINXInitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;
    URT_LIN_PINXInitStruct.PINX_Speed              = PINX_Speed_Low;
    URT_LIN_PINXInitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    URT_LIN_PINXInitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    URT_LIN_PINXInitStruct.PINX_Inverse            = PINX_Inverse_Disable;
    URT_LIN_PINXInitStruct.PINX_Alternate_Function = LIN_URTX_TXAFS;
    GPIO_PinMode_Config(LIN_URTX_TXPINX,&URT_LIN_PINXInitStruct);
    
    URT_LIN_PINXInitStruct.PINX_Mode               = PINX_Mode_Digital_I;
    URT_LIN_PINXInitStruct.PINX_Alternate_Function = LIN_URTX_RXAFS;
    GPIO_PinMode_Config(LIN_URTX_RXPINX,&URT_LIN_PINXInitStruct);    
    
    /*URT data character format*/
    URT_LIN_DataDef.URT_TX_DataLength  = URT_DataLength_8;
    URT_LIN_DataDef.URT_RX_DataLength  = URT_DataLength_8;
    URT_LIN_DataDef.URT_TX_DataOrder   = URT_DataTyped_LSB;
    URT_LIN_DataDef.URT_RX_DataOrder   = URT_DataTyped_LSB;
    URT_LIN_DataDef.URT_TX_Parity      = URT_Parity_No;
    URT_LIN_DataDef.URT_RX_Parity      = URT_Parity_No;
    URT_LIN_DataDef.URT_TX_StopBits    = URT_StopBits_1_0;
    URT_LIN_DataDef.URT_RX_StopBits    = URT_StopBits_1_0;
    URT_LIN_DataDef.URT_TX_DataInverse = DISABLE;
    URT_LIN_DataDef.URT_RX_DataInverse = DISABLE;
    URT_DataCharacter_Config(LIN_URTX, &URT_LIN_DataDef);
    
    /*URT mode configure*/
    URT_Mode_Select(LIN_URTX, URT_URT_mode);
    URT_DataLine_Select(LIN_URTX, URT_DataLine_2);
    URT_HalfDuplexMode_Cmd(LIN_URTX, ENABLE);
    
    /*URT data control related configure*/
    URT_RXShadowBufferThreshold_Select(LIN_URTX, URT_RXTH_1BYTE);
    URT_IdlehandleMode_Select(LIN_URTX, URT_IDLEMode_No);
    URT_TXGuardTime_Select(LIN_URTX, 0);
    
    /*TX error mode configure*/
    URT_TXErrorDetectMode_Select(LIN_URTX,URT_TXErrorDetect_TX);
    URT_TXErrorResendTime_Select(LIN_URTX,URT_TXErrorResend_0);
    
    /*Auto calibration related configure */
    URT_CalibrationMode_Select(LIN_URTX,URT_CALMode_Edge);
    #if LIN_URTX_AUTOCAL_EN == 0
        URT_AutoCalibration_Cmd(LIN_URTX,DISABLE);
    #else
        URT_AutoCalibration_Cmd(LIN_URTX,ENABLE);
    #endif
    
    /*URT baud-rate related configure*/
    URT_LIN_BRG.URT_InternalClockSource    = URT_BDClock_PROC;
    URT_LIN_BRG.URT_BaudRateMode           = URT_BDMode_Separated;
    URT_LIN_BRG.URT_PrescalerCounterReload = 5;	                  //Set PSR
    URT_LIN_BRG.URT_BaudRateCounterReload  = 9;  	              //Set RLR
    URT_BaudRateGenerator_Config(LIN_URTX, &URT_LIN_BRG);   		  
    URT_BaudRateGenerator_Cmd(LIN_URTX, ENABLE);	              //Enable BaudRateGenerator
    //---TX/RX Clock---//
    URT_TXClockSource_Select(LIN_URTX, URT_TXClock_Internal);	  //URT_TX use BaudRateGenerator
    URT_RXClockSource_Select(LIN_URTX, URT_RXClock_Internal);	  //URT_RX use BaudRateGenerator
    URT_TXOverSamplingSampleNumber_Select(LIN_URTX, 9);	          //Set TX OS_NUM (20 kbps in APB = 12MHz)
    URT_RXOverSamplingSampleNumber_Select(LIN_URTX, 9);	          //Set RX OS_NUM (20 kbps in APB = 12MHz)
    URT_RXOverSamplingMode_Select(LIN_URTX, URT_RXSMP_3TIME);
    URT_TX_Cmd(LIN_URTX, ENABLE);	                              //Enable TX
    URT_RX_Cmd(LIN_URTX, ENABLE);	                              //Enable RX
    
    /*URT interrupt related configure*/
    URT_ClearITFlag(LIN_URTX, (URT_IT_LS | URT_IT_BK | URT_IT_RX));
    URT_IT_Config(LIN_URTX, (URT_IT_LS | URT_IT_BK | URT_IT_RX), ENABLE);
    URT_ITEA_Cmd(LIN_URTX, ENABLE);
    NVIC_EnableIRQ(LIN_URTX_IRQn);
    
    /*Parameter initial*/
    LIN_RX_BreakDetectflag = 0;
    LIN_Frame.State        = LIN_IDLE_BREAK;
    LIN_Frame.DataIndex    = 0;
    
    URT_Cmd(LIN_URTX,ENABLE);
    
}

/**
 *******************************************************************************
 * @brief	   
 * @details 
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
uint8_t Sample_URT_LINSlave_CalcParity(uint8_t LIN_Frame_ID)
{
    uint8_t parity, P0, P1;
    
    parity = LIN_Frame_ID;
    
    P0 = ((((parity >> 0) & 0x01) ^ \
           ((parity >> 1) & 0x01) ^ \
           ((parity >> 2) & 0x01) ^ \
           ((parity >> 4) & 0x01))>>6);
    P1 = ((((parity >> 1) & 0x01) ^ \
           ((parity >> 3) & 0x01) ^ \
           ((parity >> 4) & 0x01) ^ \
           ((parity >> 5) & 0x01))>>7);
    
    parity |= (P0 | P1);    
    
    return(parity);
}

/**
 *******************************************************************************
 * @brief	   
 * @details 
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
uint8_t Sample_URT_LINSlave_CalcChecksum( uint8_t LIN_Frame_ID, uint8_t *LIN_Data, uint8_t LIN_Data_Length)
{
    uint8_t  LIN_CalcChecksum_Tmp;
    uint16_t LIN_CalcChecksum = 0;
    
    if( LIN_Frame_ID != 0x3C && LIN_Frame_ID!= 0x7D)
    {
        LIN_CalcChecksum = LIN_Frame_ID;
    }
    
    for( LIN_CalcChecksum_Tmp = 0; LIN_CalcChecksum_Tmp < LIN_Data_Length; LIN_CalcChecksum_Tmp++)
    {
        LIN_CalcChecksum += *(LIN_Data++);
        
        if( LIN_CalcChecksum > 0xFF)
        {
            LIN_CalcChecksum = LIN_CalcChecksum - 0xFF;
        }
        
    }
    
    return((uint8_t)(~LIN_CalcChecksum));
    
}

/**
 *******************************************************************************
 * @brief	   
 * @details 
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
void Sample_URT_LINSlave_main(void)
{
    uint8_t URT_LINSlave_mainTmp;
    
    if( LIN_Frame.State != LIN_SLAVE_INIT_DATA)
    {
        LIN_Frame.CheckSum = Sample_URT_LINSlave_CalcChecksum( LIN_Frame.PID, LIN_Frame.Data, LIN_Frame.DataLength);
        
        for( URT_LINSlave_mainTmp = 0; URT_LINSlave_mainTmp < LIN_Frame.DataLength; URT_LINSlave_mainTmp++)
        {
            URT_SetTXData( LIN_URTX, 1, LIN_Frame.Data[URT_LINSlave_mainTmp]);
            while((URT_GetITAllFlagStatus( LIN_URTX) & URT_IT_TX)==0);
        }
        URT_SetTXData( LIN_URTX, 1, LIN_Frame.CheckSum);
        while((URT_GetITAllFlagStatus( LIN_URTX) & URT_IT_TC)==0);
        
        LIN_RX_BreakDetectflag = 0;
        LIN_Frame.State = LIN_IDLE_BREAK;
    }
}

/**
 *******************************************************************************
 * @brief	   
 * @details 
 * @return     
 * @exception   
 * @note           
 *******************************************************************************
 */
__WEAK void LIN_ErrorCallback(void)
{
    //=========================================================
    //NOTE : This function should not be modifyed, when the 
    //       callback is needed, the APX_IRQHandler_Callback 
    //       can be implemented in the user file.      
}








