/**
  ******************************************************************************
 *
 * @file        Sample_TM20_OnePulse.c
 *
 * @brief       Output 2-channels PWM of TM20 dependent on TRGI event.
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2021/05/25
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
* @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_TM_DRV.h"


/* Wizard menu ---------------------------------------------------------------*/
//*** <<< Use Configuration Wizard in Context Menu >>> ***
//  <h> TM20 counter period
//      <o0> prescaler counter range (1~65536)  <1-65536>
//      <o1> main counter range (1~65536) <1-65536>
//  </h>
#define TM20_PrescalerCounter_Range 1
#define TM20_MainCounter_Range      3000

//  <h> Duty cycle setting 
//      <o0> Channel0 match point (1~65536) <1-65536>
//      <o1> Channel1 match point (1~65536) <1-65536>
//  </h>
#define TM20_Channel0_MatchPoint    100
#define TM20_Channel1_MatchPoint    1000

//  <o0> One pulse output - Repetition count (1~256) <1-256>
//      <i> Input range is 1~256. 1 = Only one PWM cycle.
#define RepCOUNT                    10

//*** <<< end of configuration section >>>    ***

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
void Sample_TM20_OnePulse(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief	    Output 2-channels PWM of TM20 when TRGI(ETR) event happened.
 *              Then it will to stop PWM pulse when output N times.
 * @details     1.initial & modify TimeBase parameter 
 *    \n        2.config TM20 channel0/1 function 
 *    \n        3.Enable TM20 channel 0 Output
 *    \n        4.set match point (Duty cycle %) for PWM channel0
 *    \n        5.Enable TM20 channel 1 Output
 *    \n        6.set match point (Duty cycle %) for PWM channel1
 *    \n        7.select Edge Align
 *    \n        8.clear all flag
 *    \n        9.TRGI configuration  
 *    \n        10.One pulse output configuration
 *    \n        11.Enable TM20
 * @return      None
 * @note        It will always output PWM cycle when TRGI happen.
 *******************************************************************************
 */
void Sample_TM20_OnePulse(void)
{  
    TM_TimeBaseInitTypeDef TM_TimeBase_InitStruct;
     

    // make sure :
	
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //Select CK_MAIN Source = CK_HS
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->TM20 = Enable
    //Configure Peripheral On Mode Clock->Port A = Enable
    //Configure Peripheral On Mode Clock->Port B = Enable
	
    //==Set GPIO init 
    //TM20_OC00 pin config:
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOA->PA4
    //GPIO port initial is 0xFFFF
    //PA4 mode is PPO
    //PA4 function TM20_OC00
	
    //==Set GPIO init 
    //TM20_OC10 pin config:
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOA->PA5
    //GPIO port initial is 0xFFFF
    //PA5 mode is PPO
    //PA5 function TM20_OC10

    //==Set GPIO init 
    //TM20_ETR pin config:
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->PB6
    //GPIO port initial is 0xFFFF
    //PB6 mode is DIN with pull-up
    //PB6 function TM20_ETR

	 
    TM_DeInit(TM20);
    
    // ----------------------------------------------------
    // 1.TimeBase structure initial
    TM_TimeBaseStruct_Init(&TM_TimeBase_InitStruct);
    
    // modify parameter
    TM_TimeBase_InitStruct.TM_MainClockDirection    = TM_UpCount;
    TM_TimeBase_InitStruct.TM_Period                = TM20_MainCounter_Range - 1; 
    TM_TimeBase_InitStruct.TM_Prescaler             = TM20_PrescalerCounter_Range - 1;
    TM_TimeBase_InitStruct.TM_CounterMode           = Cascade;
    
    TM_TimeBase_Init(TM20, &TM_TimeBase_InitStruct);
   
    // ----------------------------------------------------
    // 2.config TM20 channel0/1/2 function 
    TM_CH0Function_Config(TM20, TM_16bitPWM);
    TM_CH1Function_Config(TM20, TM_16bitPWM);
    TM_CH2Function_Config(TM20, TM_16bitPWM);
    
    // ----------------------------------------------------
    // 3.Enable TM20 channel 0 Output (just output TM20_OC00)
    TM_OC00Output_Cmd(TM20,ENABLE);    
    TM_InverseOC0z_Cmd(TM20, DISABLE);
    TM_OC0zOutputState_Init(TM20, CLR);
    
    TM_OC0NOutput_Cmd(TM20, DISABLE);
    TM_InverseOC0N_Cmd(TM20, DISABLE);
    // ----------------------------------------------------
    // 4.set match point (Duty cycle %) for PWM channel0
    TM_SetCC0A(TM20,TM20_Channel0_MatchPoint);        
    TM_SetCC0B(TM20,TM20_Channel0_MatchPoint);		// reload value when overflow

    // ----------------------------------------------------
    // 5.TM20 channel 1 Output config (just output TM20_OC10)
    TM_OC10Output_Cmd(TM20,ENABLE);    
    TM_InverseOC1z_Cmd(TM20, DISABLE);
    TM_OC1zOutputState_Init(TM20, CLR);
    
    TM_OC1NOutput_Cmd(TM20, DISABLE);
    TM_InverseOC1N_Cmd(TM20, DISABLE);
    // ----------------------------------------------------
    // 6.set match point (Duty cycle %) for PWM channel1
    TM_SetCC1A(TM20,TM20_Channel1_MatchPoint);        
    TM_SetCC1B(TM20,TM20_Channel1_MatchPoint);		// reload value when overflow

    // ----------------------------------------------------
    // 7.select Edge Align
    TM_AlignmentMode_Select(TM20, TM_EdgeAlign);
    
    // ----------------------------------------------------
    // 8.clear flag
    TM_ClearFlag(TM20, TMx_CF0A | TMx_CF1A | TMx_CF2A);
    
    // ----------------------------------------------------
    // 9.TRGI configuration
    TM_TriggerSource_Select(TM20, TRGI_ETR);
    
    TM_TRGICounter_Select(TM20, TRGI_StartupRising);
    TM_TRGIPrescaler_Select(TM20, TRGI_StartupRising);
    
    // ----------------------------------------------------
    // 10.One pulse output configuration
    TM_RepetitionCounter_Cmd(TM20, ENABLE);         // Enable RCNT
    TM_ControlPWM_Cmd(TM20, ENABLE);                // Stop OC/PWM output when RCNT underflow.
    TM_RepetitionCounter_Config(TM20, RepCOUNT - 1, RepCOUNT - 1);
    TM_RepetitionCounterClock_Select(TM20, TM_RCNT_CKMAIN);     // RCNT clock source from Main counter overflow.
    
    // ----------------------------------------------------
    // 11.Timer enable
    TM_Timer_Cmd(TM20,ENABLE);
    
    return;
}




