
/**
 ******************************************************************************
 *
 * @file        Sample_ADC_ImproveAccuracy.C
 * @brief       MG32 demo main c Code. 
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2022/04/20
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define VBUF14_Addr     (0x1FF20050)

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
uint16_t Sample_ADC_ImproveAccuracy(uint16_t VBUF14_ADC_Value, uint16_t AIN_ADC_Value);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief       Improve ADC accuracy
 * @details     Use the VBUF14 proportional relationship calculate the I/O pin
 *              voltage.
 * @param[in]   VBUF14_ADC_Value: ADC sample VBUF14 ADC data (unit : mV).
 * @param[in]   AIN_ADC_Value: ADC sample external channel ADC data.
 * @return 	    Calculate the I/O pin voltage.
 * @note 
 * @par         Example
 * @code
 
    uint16_t VrefVoltage;                   // Save ADC reference voltage
    uint16_t VBUF14_ADC_Value;              // Convert VBUF14 voltage
    uint16_t AIN_ADC_Value;                 // Calculate AINx voltage
 
    // ------------------------------------------------------------------------
    // 1. Enable VBUF & Stable ...
    // 2. Initial ADC (Trigger source is ADC_START)
    // ------------------------------------------------------------------------
    // to do ...

    // ------------------------------------------------------------------------
    // ADC sample VBUF14 voltage to calibration
    // ------------------------------------------------------------------------
    ADC_InternalChannel_Select(ADC0, ADC_INT_VBUF);         // Select ADC internal channel (VBUF)
    ADC_ClearFlag(ADC0, ADC_E1CNVF);                        // Clear ADC E1CNVF flag
    ADC_SoftwareConversion_Cmd(ADC0, ENABLE);               // Trigger ADC to conversion 
    while(ADC_GetSingleFlagStatus(ADC0, ADC_E1CNVF) == DRV_UnHappened); // Wait for complete
    ADC_ClearFlag(ADC0, ADC_E1CNVF);                        // Clear ADC E1CNVF flag
    
    // ------------------------------------------------------------------------
    // Get ADC Vref Voltage
    // ------------------------------------------------------------------------
    VrefVoltage = (1400 * 4096) / ADC_GetDAT0Data(ADC0);
    
    // ------------------------------------------------------------------------
    // Get VBUF14 Voltage 
    // ------------------------------------------------------------------------
    VBUF14_ADC_Value = ADC_GetDAT0Data(ADC0) / VrefVoltage * 4096;
    
    // ------------------------------------------------------------------------
    // ADC sample external channel-x 
    // ------------------------------------------------------------------------
    ADC_ExternalChannel_Select(ADC0, ADC_ExtAIN2);          // Select ADC external channel (PA2)
    ADC_ClearFlag(ADC0, ADC_E1CNVF);                        // Clear ADC E1CNVF flag
    ADC_SoftwareConversion_Cmd(ADC0, ENABLE);               // Trigger ADC to conversion
    while(ADC_GetSingleFlagStatus(ADC0, ADC_E1CNVF) == DRV_UnHappened); // Wait for complete
    ADC_ClearFlag(ADC0, ADC_E1CNVF);                        // Clear ADC E1CNVF flag
 
    // ------------------------------------------------------------------------
    // Use Sample_ADC_ImproveAccuracy to improve accuracy
    // ------------------------------------------------------------------------
    AIN_ADC_Value = Sample_ADC_ImproveAccuracy(VBUF14_ADC_Value, ADC_GetDAT0Data(ADC0));
    
    // to do ...
    
 * @endcode
 *******************************************************************************
 */
uint16_t Sample_ADC_ImproveAccuracy(uint16_t VBUF14_ADC_Value, uint16_t AIN_ADC_Value)
{
    uint16_t VBUF14_Prestored_Value;
    uint16_t *ptr16;
    
    // ------------------------------------------------------------------------
    // Inernal reference VBIF14 voltage under VDD=3.3V.
    // Use the ADC to read its ADC value to store in flash ROM as the Prestored value.
    // ------------------------------------------------------------------------
    ptr16 = (uint16_t *)VBUF14_Addr;
    VBUF14_Prestored_Value = (*(ptr16)) * 3300 / 4096;     // unit : mV
    
    // ------------------------------------------------------------------------
    // To use the proportional relationship calculate the I/O pin voltage
    // formula = (VBUF14_Prestored_Value * AIN_ADC_Value) / VBUF14_ADC_Value
    // ------------------------------------------------------------------------
    return (VBUF14_Prestored_Value * AIN_ADC_Value / VBUF14_ADC_Value);

}

