/**
 ******************************************************************************
 *
 * @file        Sample_SPI_Flash.h
 * @brief       Use SPI of MG32 to control MX25R512F read / write. 
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2021/05/25
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2020 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */



#ifndef _TST_FLASH_H
#define _TST_FLASH_H


/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_GPIO_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
#define FLASH_SECTOR_ERASE          0x20    /*!< 1 sector = 4K bytes                                    */
#define FLASH_BLOCK32K_ERASE        0x52    /*!< Block erase 32K bytes                                  */
#define FLASH_BLOCK64K_ERASE        0xD8    /*!< Block erase 62K bytes                                  */
#define FLASH_CHIP_ERASE            0x60    /*!< Chip erase                                             */

/**
 * @name    ============== Flash control parameter ==============
 *  
 */ 
///@{
/*! @enum   Flash_ERASE_TypeDef
    @brief  The Flash Erase type.
*/
typedef enum
{
    FLASH_ERASE_SECTOR    = FLASH_SECTOR_ERASE,   /*!< Erase type = Sector(4K byte) erase.              */
    FLASH_ERASE_BLOCK32K  = FLASH_BLOCK32K_ERASE, /*!< Erase type = 32K byte block erase.               */
    FLASH_ERASE_BLOCK64K  = FLASH_BLOCK64K_ERASE, /*!< Erase type = 64K byte block erase.               */
    FLASH_ERASE_CHIP      = FLASH_CHIP_ERASE,     /*!< Erase type = Chip erase.                         */
}Flash_ERASE_TypeDef;
///@}


/*! @struct FlashCTR_TypeDef
    @brief  Flash data control.
*/ 
///@{
typedef struct
{
    uint8_t               *pBuffer;         /*!< Pointer of program data buffer.                        */
    uint8_t               *rBuffer;         /*!< Pointer of recevie buffer.                             */
    ctype                 Address;          /*!< Control Flash address.                                 */
    uint32_t              Total_Length;     /*!< To control Flash read and write total address.         */
    uint32_t              BLen;             /*!< To control Flash write bytes in a time.                */
    uint32_t              DataCnt;          /*!< Use for Flash read and write operation parameter.      */
}FlashCTR_TypeDef;
///@}


/* Private define ------------------------------------------------------------*/
/**
 * @name    ========MG32 Peripheral Define ==============
 *
 */ 
///@{
#define Flash_nCS                   PB0     /*!< Define software nCS Pin.                               */
#define SPIX                        SPI0    /*!< Use the SPI = to control SPI flash .                   */
///@}

/**
 * @name    ============== MX25R512F Command ==============
 *
 */ 
///@{
#define FLASH_NORMAL_READ           0x03    /*!< Read data bytes                                        */
#define FLASH_FAST_READ             0x0B    /*!< Read data bytes at higher speed                        */

#define FLASH_2IO_READ              0xBB    /*!< 2 x I/O read mode                                      */
#define FLASH_1I_2O_READ            0x3B    /*!< Dual read mode                                         */
#define FLASH_4IO_READ              0xEB    /*!< 4 x I/O read mode                                      */
#define FLASH_1I_4O_Read            0x6B    /*!< Quad read mode                                         */

#define FLASH_PAGE_PROGRAM          0x02    /*!< Page program                                           */
#define FLASH_QUADPAGE_PROGRAM      0x38    /*!< 4 x I/O page program                                   */

#define FLASH_SFDP_READ             0x5A    /*!< Read SFDP mode                                         */
#define FLASH_WRITE_ENABLE          0x06    /*!< Write enable                                           */
#define FLASH_WRITE_DISABLE         0x04    /*!< Write disable                                          */
#define FLASH_READ_STATUS           0x05    /*!< Read Status Register                                   */
#define FLASH_READ_CONFIGURATION    0x15    /*!< Read configuration register                            */
#define FLASH_WRITE_STATUS          0x01    /*!< Write status register                                  */
#define FLASH_SUSPEND_PROGRAM       0x75    /*!< Suspends erase                                         */
#define FLASH_SUSPEND_ERASE         0xB0    /*!< Suapends erase                                         */
#define FLASH_RESUMES_PROGRAM       0x7A    /*!< Resume program                                         */
#define FLASH_RESUMES_REASE         0x30    /*!< Resume erase                                           */
#define FLASH_DEEP_POWER_DOWN       0xB9    /*!< Deep power-down                                        */
#define FLASH_SET_BURST_LENGTH      0xC0    /*!< Burst read                                             */
#define FLASH_READ_IDENTIFICATION   0x9F    /*!< Read manufacturer ID + memory type + memory density.   */
#define FLASH_READ_ELECTRONIC_ID    0xAB    /*!< Read electronic ID.                                    */
#define FLASH_READ_REMS             0x90    /*!< Read manufacturer & device ID.                         */
#define FLASH_ENTER_SECURED_OTP     0xB1    /*!< Enter secured OTP                                      */
#define FLASH_EXIT_SECURED_OTP      0xC1    /*!< Exit secured OTP                                       */
#define FLASH_READ_SECURITY_REG     0x2B    /*!< Read security register                                 */
#define FLASH_WRITE_SECURITY_REG    0x2F    /*!< Write security register                                */
#define FLASH_NOP                   0x00    /*!< Nop                                                    */
#define FLASH_RESET_ENABLE          0x66    /*!< Reset enable                                           */
#define FLAHS_RESET_MEMORY          0x99    /*!< Reset memory                                           */
///@}

#define FLASH_nCS_ACTIVE            0       /*!< Define polarity (nCS = Active).                        */
#define FLASH_nCS_NoACTIVE          1       /*!< Define polarity (nCS = No Active).                     */ 
                
#define FLASH_MAXSIZE               64*1024 /*!< Flash max size.                                        */
#define FLASH_PROGRAM_MAX           256     /*!< Define Flash program max bytes  in one write enable.   */ 
        
#define Flash_Dummy_Data            0xFFFFFFFF  /*!< Define DUMMY Data.                                 */
        
#define FLASH_RDID_CMP              0x1028C2    /*!< Manufacturer ID  + Memory type + Memory density.   */
#define FLASH_RES_CMP               0x10    /*!< Electronic ID.                                         */
#define FLASH_REMS_CMP              0xC210  /*!< Manufacturer ID + Device ID.                           */


/* Private macro -------------------------------------------------------------*/
#define Sample_Flash_ReadData()                   Sample_SPI_ReadData()
#define Sample_Flash_SetData(DATALEN, FLAHSDATA)  Sample_SPI_SetData((DATALEN -1),FLAHSDATA)

/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/**
 * @name    ============== Flash sample function ==============
 *  
 */ 
///@{
void Sample_SPI_SetData(uint8_t DataLen , uint32_t FlashData);  /*!< SPI send data.                     */
uint8_t Sample_SPI_ReadData(void);                              /*!< SPI read data.                     */
void Sample_SPI_WriteFlash_Cmd(FunctionalState WREN);           /*!< Flash write control.               */
void Sample_SPI_CheckFlashBusy(void);                           /*!< Check flash busy.                  */
            
void Sample_SPI_SPIMode_Init (void);                            /*!< SPI inital.                        */
uint32_t Sample_SPI_ReadFlashRDID (void);                       /*!< Read flash ID.                     */
uint16_t Sample_SPI_ReadFlashRDCR (void);                       /*!< Control flash register.            */
void Sample_SPI_EraseFlash (Flash_ERASE_TypeDef  , uint32_t);   /*!< Control flash to erase.            */   
            
void Sample_SPI_ReadFlash (FlashCTR_TypeDef*);                  /*!< Read flash data.                   */
void Sample_SPI_ReadFlash_DMA (FlashCTR_TypeDef*);              /*!< Use DMA read flash data.           */
void Sample_QPI_ReadFlash (FlashCTR_TypeDef*);                  /*!< QPI read flash data.               */
#if defined(MG32_2ND) || defined(MG32_3RD)      
void Sample_QPI_ReadFlash_DMA (FlashCTR_TypeDef*);              /*!< QPI use DMA read flash data.       */
#endif      
            
void Sample_SPI_ProgramFlash (FlashCTR_TypeDef*);               /*!< Program data to flash.             */
void Sample_SPI_ProgramFlash_DMA (FlashCTR_TypeDef*);           /*!< Use DMA program data to flash.     */
void Sample_QPI_ProgramFlash (FlashCTR_TypeDef*);               /*!< QPI program data to flash.         */
void Sample_QPI_ProgramFlash_DMA (FlashCTR_TypeDef*);           /*!< QPI program data to flash.         */
void Sample_SPI_MX25R512FFlash (void);                          /*!< Flash sample code.                 */
///@}

/* External vairables --------------------------------------------------------*/
extern FlashCTR_TypeDef Flash0;

#endif



