/**
 ******************************************************************************
 *
 * @file        MG32_LCD_DRV.c
 * @brief       MG32 demo MG32_LCD_DRV c Code. 
 *
 * @par         Project
 *              MG32
 * @version     V1.11
 * @date        2025/06/16
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2024 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Wizard menu ---------------------------------------------------------------*/

/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#if defined(ModuleExist_LCD)
#include "MG32_LCD_DRV.h"

/** @name MG32_DRV
  * @{
  */

/** @name LCD LCD
  * @{
  */

/* External vairables --------------------------------------------------------*/
/** @name LCD_Private_Vairables LCD Private vairables
  * @{
  */


/**
  * @}
  */

/* Private typedef -----------------------------------------------------------*/
/** @name LCD_Private_Typedef LCD Private typedef
  * @{
  */


/**
  * @}
  */

/* Private define ------------------------------------------------------------*/
/** @name LCD_Private_Define LCD Private define
  * @{
  */


/**
  * @}
  */

/* Private macro -------------------------------------------------------------*/
/** @name LCD_Private_Macro LCD Private macro
  * @{
  */


/**
  * @}
  */

/* Private variables ---------------------------------------------------------*/
/** @name LCD_Private_Variables LCD Private variables
  * @{
  */

static uint32_t gBackup_CR0_VS_SEL = 0;
static uint32_t gBackup_CR1_DRV_MDS = 0;
static uint32_t gBackup_CR1_OVD_DIS = 0;
static uint32_t gBackup_CR2_CP_CNT = 0;

/**
  * @}
  */

/* Private function prototypes -----------------------------------------------*/
/** @name LCD_Private_Function_Prototypes LCD Private function prototypes
  * @{
  */


/**
  * @}
  */

/* Exported variables --------------------------------------------------------*/
/** @name LCD_Exported_Variables LCD Exported variables
  * @{
  */


/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/
/** @name LCD_Exported_Functions LCD Exported functions
  * @{
  */

/** @name LCD_Exported_Functions_Clock
  * @{
  */

/**
 *******************************************************************************
 * @brief       LCD input clock CK_LCD_PR divider.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   PreDivider:
 *  @arg\b        PreDivider_8: divided by 8
 *  @arg\b        PreDivider_16 : divided by 16
 *  @arg\b        PreDivider_32 : divided by 32
 *  @arg\b        PreDivider_64 : divided by 64
 *  @arg\b        PreDivider_128 : divided by 128
 *  @arg\b        PreDivider_256 : divided by 256
 *  @arg\b        PreDivider_512 : divided by 512
 *  @arg\b        PreDivider_1024 : divided by 1024
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_SetClockPreDivider(LCD, divided by 8);
                LCD_SetClockPreDivider(LCD, divided by 16);
                LCD_SetClockPreDivider(LCD, divided by 32);
                LCD_SetClockPreDivider(LCD, divided by 64);
                LCD_SetClockPreDivider(LCD, divided by 128);
                LCD_SetClockPreDivider(LCD, divided by 256);
                LCD_SetClockPreDivider(LCD, divided by 512);
                LCD_SetClockPreDivider(LCD, divided by 1024);
 * @endcode
 * @par         Modify
 *              void LCD_SetClockPreDivider(LCD_Struct* LCDx, LCD_ClcokPreDivider_TypeDef PreDivider)
 *******************************************************************************
 */
void LCD_SetClockPreDivider(LCD_Struct* LCDx, LCD_ClcokPreDivider_TypeDef PreDivider)
{
    LCDx->CLK.W = ((LCDx->CLK.W & LCD_CLK_CK_PDIV_mask_w) | PreDivider);
}


/**
 *******************************************************************************
 * @brief       LCD input clock CK_LCD source select.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   SourceSelect:
 *  @arg\b        ClcokSource_CK_LS: CK_LS
 *  @arg\b        ClcokSource_NCO_P0 : NCO_P0
 *  @arg\b        ClcokSource_PROC : PROC : CK_LCD_PR process clock from CSC
 *  @arg\b        ClcokSource_TM01_TRGO : TM01_TRGO
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_SetClockSource(LCD, ClcokSource_CK_LS);
                LCD_SetClockSource(LCD, ClcokSource_NCO_P0);
                LCD_SetClockSource(LCD, ClcokSource_PROC);
                LCD_SetClockSource(LCD, ClcokSource_TM01_TRGO);
 * @endcode
 * @par         Modify
 *              void LCD_SetClockSource(LCD_Struct* LCDx, LCD_ClcokSourceSelect_TypeDef SourceSelect)
 *******************************************************************************
 */
void LCD_SetClockSource(LCD_Struct* LCDx, LCD_ClcokSource_TypeDef SourceSelect)
{
    LCDx->CLK.W = ((LCDx->CLK.W & (~LCD_CLK_CK_SEL_mask_w)) | SourceSelect);
}


/**
 *******************************************************************************
 * @brief       LCD internal clock CK_LCD_INT input divider.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   Divider:
 *  @arg\b        Divider_1 : divided by 1
 *  @arg\b        Divider_2 : divided by 2
 *  @arg\b        Divider_4 : divided by 4
 *  @arg\b        Divider_8 : divided by 8
 *  @arg\b        Divider_16 : divided by 16
 *  @arg\b        Divider_32 : divided by 32
 *  @arg\b        Divider_64 : divided by 64
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_SetClockDivider(LCD, Divider_1);
                LCD_SetClockDivider(LCD, Divider_2);
                LCD_SetClockDivider(LCD, Divider_4);
                LCD_SetClockDivider(LCD, Divider_8);
                LCD_SetClockDivider(LCD, Divider_16);
                LCD_SetClockDivider(LCD, Divider_32);
                LCD_SetClockDivider(LCD, Divider_64);
 * @endcode
 * @par         Modify
 *              void LCD_SetClockDivider(LCD_Struct* LCDx, LCD_ClcokDivider_TypeDef Divider)
 *******************************************************************************
 */
void LCD_SetClockDivider(LCD_Struct* LCDx, LCD_ClcokDivider_TypeDef Divider)
{
    LCDx->CLK.W = ((LCDx->CLK.W & (~LCD_CLK_CK_DIV_mask_w)) | Divider);
}


/**
 *******************************************************************************
 * @brief       LCD internal clock CK_LCD_PHS prescaler.
 * @details     The value range 1~31 is indicated divider 2~32. 
 *              The value 0 is invalid.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   Prescaler:
 *  @arg\b        1 to 31 : indicated divider 2~32.
 *  @arg\b        0 : invalid.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_SetClockPrescaler(LCD, 1); // divider 2
                LCD_SetClockPrescaler(LCD, 2); // divider 3
                LCD_SetClockPrescaler(LCD, 15); // divider 16
                LCD_SetClockPrescaler(LCD, 16); // divider 17
                LCD_SetClockPrescaler(LCD, 30); // divider 31
                LCD_SetClockPrescaler(LCD, 31); // divider 32
 * @endcode
 * @par         Modify
 *              void LCD_SetClockPrescaler(LCD_Struct* LCDx, uint32_t Prescaler)
 *******************************************************************************
 */
void LCD_SetClockPrescaler(LCD_Struct* LCDx, uint32_t Prescaler)
{
    LCDx->CLK.W = ((LCDx->CLK.W & (~LCD_CLK_CK_PSC_mask_w)) | (LCD_CLK_CK_PSC_mask_w & (Prescaler << LCD_CLK_CK_PSC_shift_w)));
}


/**
  * @}
  */


//=================================================================================================


/** @name LCD_Exported_Functions_Power
  * @{
  */


/**
 *******************************************************************************
 * @brief       LCD over voltage detection disable or enable.
 * @details     It must set to enable if LCD_VS_SEL is set 'CP' or LCD_VS_SEL - 
 *               is set 'EXT' under LCD_VT > VDD voltage.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   OVDState:
 *  @arg\b        DISABLE : Over voltage detection disable.
 *  @arg\b        ENABLE : Over voltage detection enable.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_OverVoltageDetection(LCD, DISABLE);
                LCD_OverVoltageDetection(LCD, ENABLE);
 * @endcode
 * @par         Modify
 *              void LCD_OverVoltageDetection(LCD_Struct* LCDx, FunctionalState OVDState)
 *******************************************************************************
 */
void LCD_OverVoltageDetection(LCD_Struct* LCDx, FunctionalState OVDState)
{
    if(OVDState == ENABLE)
        gBackup_CR1_OVD_DIS &= (~LCD_CR1_OVD_DIS_mask_w);
    else
        gBackup_CR1_OVD_DIS |= LCD_CR1_OVD_DIS_mask_w;

    LCDx->CR1.W = (LCDx->CR1.W & (~LCD_CR1_OVD_DIS_mask_w)) | gBackup_CR1_OVD_DIS;
}


/**
 *******************************************************************************
 * @brief       LCD voltage rail LCD_VT / LCD_V1 / LCD_V2 / LCD_V3 pin connection control.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   VxPin:
 *  @arg\b        VTPin_NC : LCD_VT No pin connection.
 *  @arg\b        VTPin_Pin1 : LCD_VT Connect to pin.
 *  @arg\b        V1Pin_NC : LCD_V1 No pin connection.
 *  @arg\b        V1Pin_Pin1 : LCD_V1 Connect to pin.
 *  @arg\b        V2Pin_NC : LCD_V2 No pin connection.
 *  @arg\b        V2Pin_Pin1 : LCD_V2 Connect to pin.
 *  @arg\b        V3Pin_NC : LCD_V3 No pin connection.
 *  @arg\b        V3Pin_Pin1 : LCD_V3 Connect to pin.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_VxPinControl(LCD, VTPin_Pin1);
                LCD_VxPinControl(LCD, (VTPin_Pin1 | V1Pin_Pin1));
                LCD_VxPinControl(LCD, (VTPin_Pin1 | V1Pin_Pin1 | V2Pin_Pin1));
                LCD_VxPinControl(LCD, (VTPin_Pin1 | V1Pin_Pin1 | V2Pin_Pin1 | V3Pin_Pin1));
 * @endcode
 * @par         Modify
 *              void LCD_VxPinControl(LCD_Struct* LCDx, uint32_t VxPin)
 *******************************************************************************
 */
void LCD_VxPinControl(LCD_Struct* LCDx, uint32_t VxPin)
{
    LCDx->CR1.W = ((LCDx->CR1.W & (~(LCD_CR1_VT_CTL_mask_w | LCD_CR1_V1_CTL_mask_w | LCD_CR1_V2_CTL_mask_w | LCD_CR1_V3_CTL_mask_w))) | VxPin);
}


/**
 *******************************************************************************
 * @brief       LCD bias power source selection.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   VTBPSource:
 *  @arg\b        PowerSource_OFF : No power.
 *  @arg\b        PowerSource_AVDD : Internal voltage source (from VDD).
 *  @arg\b        PowerSource_EXT : External voltage source (from LCD_VT).
 *  @arg\b        PowerSource_CP : Internal charge pump.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_VxPinControl(LCD, PowerSource_OFF);
                LCD_VxPinControl(LCD, PowerSource_AVDD);
                LCD_VxPinControl(LCD, PowerSource_EXT);
                LCD_VxPinControl(LCD, PowerSource_CP);
 * @endcode
 * @par         Modify
 *              void LCD_VTBiasPowerSource(LCD_Struct* LCDx, LCD_VTPowerSource_TypeDef VTBPSource)
 *******************************************************************************
 */
void LCD_VTBiasPowerSource(LCD_Struct* LCDx, LCD_VTPowerSource_TypeDef VTBPSource)
{
    gBackup_CR0_VS_SEL = VTBPSource;
    LCDx->CR0.W = ((LCDx->CR0.W & (~LCD_CR0_VS_SEL_mask_w)) | gBackup_CR0_VS_SEL);
}


/**
 *******************************************************************************
 * @brief       LCD Charge Pump Constrast Level function .
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   CPContranst:
 *  @arg\b        CONTRAST_LEVEL0: Charge-pump Ouput Level 2.36 Volt(minimun voltage).
 *  @arg\b        CONTRAST_LEVEL1: Charge-pump Ouput Level 2.44 Volt.
 *  @arg\b        CONTRAST_LEVEL2: Charge-pump Ouput Level 2.52 Volt.
 *  @arg\b        CONTRAST_LEVEL3: Charge-pump Ouput Level 2.60 Volt.
 *  @arg\b        CONTRAST_LEVEL4: Charge-pump Ouput Level 2.68 Volt.
 *  @arg\b        CONTRAST_LEVEL5: Charge-pump Ouput Level 2.76 Volt.
 *  @arg\b        CONTRAST_LEVEL6: Charge-pump Ouput Level 2.84 Volt.
 *  @arg\b        CONTRAST_LEVEL7: Charge-pump Ouput Level 2.92 Volt.
 *  @arg\b        CONTRAST_LEVEL8: Charge-pump Ouput Level 3.00 Volt.
 *  @arg\b        CONTRAST_LEVEL9: Charge-pump Ouput Level 3.08 Volt.
 *  @arg\b        CONTRAST_LEVEL10: Charge-pump Ouput Level 3.16 Volt.
 *  @arg\b        CONTRAST_LEVEL11: Charge-pump Ouput Level 3.24 Volt.
 *  @arg\b        CONTRAST_LEVEL12: Charge-pump Ouput Level 3.32 Volt.
 *  @arg\b        CONTRAST_LEVEL13: Charge-pump Ouput Level 3.40 Volt.
 *  @arg\b        CONTRAST_LEVEL14: Charge-pump Ouput Level 3.48 Volt(maximum voltage).
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_ChargePumpConstrast_Config(LCD, CONTRAST_LEVEL0);
                LCD_ChargePumpConstrast_Config(LCD, CONTRAST_LEVEL4);
                LCD_ChargePumpConstrast_Config(LCD, CONTRAST_LEVEL8);
                LCD_ChargePumpConstrast_Config(LCD, CONTRAST_LEVEL12);
                LCD_ChargePumpConstrast_Config(LCD, CONTRAST_LEVEL14);
 * @endcode
 * @par         Modify
 *              void LCD_ChargePumpConstrast_Config(LCD_Struct* LCDx, LCD_ChargePumpContrast_TypeDef CPContranst)
 *******************************************************************************
 */
void LCD_ChargePumpConstrast_Config(LCD_Struct* LCDx, LCD_ChargePumpContrast_TypeDef CPContranst)
{
    LCDx->CR1.W = ((LCDx->CR1.W & (~LCD_CR1_VT_SEL_mask_w)) | CPContranst);
}


/**
 *******************************************************************************
 * @brief       LCD charge pump clock selection.
 * @details     The value range 0~3 is indicated divider 1, 2, 3, 4  from CK_LS.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   CPClockDivx:
 *  @arg\b        CPClock_Div1: divided by 1
 *  @arg\b        CPClock_Div2: divided by 2
 *  @arg\b        CPClock_Div3: divided by 3
 *  @arg\b        CPClock_Div4: divided by 4
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_SetChargePumpDivider(LCD, CPClock_Div1);
                LCD_SetChargePumpDivider(LCD, CPClock_Div2);
                LCD_SetChargePumpDivider(LCD, CPClock_Div3);
                LCD_SetChargePumpDivider(LCD, CPClock_Div4);
 * @endcode
 * @par         Modify
 *              void LCD_SetChargePumpDivider(LCD_Struct* LCDx, LCD_ChargePumpClcokDivider_TypeDef CPClockDivx)
 *******************************************************************************
 */
void LCD_SetChargePumpDivider(LCD_Struct* LCDx, LCD_ChargePumpClcokDivider_TypeDef CPClockDivx)
{
    LCDx->CLK.W = ((LCDx->CLK.W & (~LCD_CLK_CK_CDIV_mask_w)) | CPClockDivx);
}


/**
 *******************************************************************************
 * @brief       LCD charge pump charging counter value.
 * @details     This counter is counting by CK_LS clock input. 
 *              When the LCD power source selects built-in charge pump in 
 *              LCD_VS_SEL, the charge pump power will be hold and disable the 
 *              charge pump voltage detector after the charging time is reached t
 *              his counter setting. The value range 1 ~ 127 is indicated the 
 *              charge pump duration is 2 ~ 128 CK_LS clock time. The value 0 
 *              is indicated always charging. The value only can change from 
 *              default 0 to other value when LCD_EN has set 'Enable'.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   CPCount:
 *  @arg\b        1 to 127 : charge pump duration is 2 ~ 128 CK_LS clock time.
 *  @arg\b        0 : indicated always charging.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_SetChargePumpCount(LCD, ENABLE);
                LCD_SetChargePumpCount(LCD, DISABLE);
 * @endcode
 * @par         Modify
 *              void LCD_SetChargePumpCount(LCD_Struct* LCDx, uint32_t CPCount)
 *******************************************************************************
 */
void LCD_SetChargePumpCount(LCD_Struct* LCDx, uint32_t CPCount)
{
    gBackup_CR2_CP_CNT = LCD_CR2_CP_CNT_mask_w & (CPCount << LCD_CR2_CP_CNT_shift_w);
    LCDx->CR2.W = ((LCDx->CR2.W & (~LCD_CR2_CP_CNT_mask_w)) | gBackup_CR2_CP_CNT);
}


/**
 *******************************************************************************
 * @brief       LCD charge pump counter synchronization disable or enable.
 * @details     When this bit is set 'Enable', the charge pump counter counting 
 *              start is syncchronized to COM/SEG control timing.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   CPSyncState:
 *  @arg\b        DISABLE: Disable Charge Pump.
 *  @arg\b        ENABLE : Enable Charge Pump.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_ChargePumpSync_Cmd(LCD, ENABLE);
                LCD_ChargePumpSync_Cmd(LCD, DISABLE);
 * @endcode
 * @par         Modify
 *              void LCD_ChargePumpSync_Cmd(LCD_Struct* LCDx, FunctionalState CPSyncState)
 *******************************************************************************
 */
void LCD_ChargePumpSync_Cmd(LCD_Struct* LCDx, FunctionalState CPSyncState)
{
    if(CPSyncState == ENABLE)
        LCDx->CR1.W |= LCD_CR1_CP_SYNC_mask_w;
    else
        LCDx->CR1.W &= ~LCD_CR1_CP_SYNC_mask_w;
}


/**
 *******************************************************************************
 * @brief       LCD built-in charge pump disable or enable.
 * @details     When enables, the GPIO PD6 and PD7 will be forced to switch to 
 *              do as LCD_C1 and LCD_C2 connections. The LCD_BIAS must be 
 *              initialed before this bit is enabled.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   CPState:
 *  @arg\b        DISABLE: Disable Charge Pump.
 *  @arg\b        ENABLE : Enable Charge Pump.
 * @return      DRV_Return
 *  @retval       DRV_False : False
 *  @retval       DRV_True : True
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_ChargePump_Cmd(LCD, ENABLE);
                LCD_ChargePump_Cmd(LCD, DISABLE);
 * @endcode
 * @par         Modify
 *              void LCD_ChargePump_Cmd(LCD_Struct* LCDx, FunctionalState CPState)
 *******************************************************************************
 */
DRV_Return LCD_ChargePump_Cmd(LCD_Struct* LCDx, FunctionalState CPState)
{
    uint32_t lCount32 = 100;

    if(CPState == DISABLE)
        LCDx->CR1.W &= ~LCD_CR1_CP_EN_mask_w;
    else
    {
        LCDx->STA.W = LCD_STA_CPRF_mask_w;
        LCDx->CR1.W |= LCD_CR1_CP_EN_mask_w;

        do{
            lCount32 -- ;
        }while(((LCDx->STA.W & LCD_STA_CPRF_mask_w) == 0) & (lCount32 != 0));       // wait Charge Pump Ready

        //if((lCount32 == 0) & ((LCDx->STA.W & LCD_STA_CPRF_mask_w) == 0))
        //{
        //    LCDx->CR2.W = (LCDx->CR2.W & (~LCD_CR1_CP_EN_mask_w));
        //    return DRV_False;
        //}
    }

    return DRV_True;
}


/**
 *******************************************************************************
 * @brief       LCD off state LCD pin control.
 *              LCD off state LCD pin control.
 * @details     When this bit selects 'Hiz', the LCD_Pn outputs are set to tri-state.
 *              When this bit selects 'Low', the LCD_Pn outputs are set to low level.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   OffState:
 *  @arg\b        OffState_Hiz : Tri-State.
 *  @arg\b        OffState_Low : Output Low.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_SetPinOffState(LCD, OffState_Hiz);  // defaule
                LCD_SetPinOffState(LCD, OffState_Low);
 * @endcode
 * @par         Modify
 *              void LCD_SetPinOffState(LCD_Struct* LCDx, LCD_PinOffState_TypeDef OffState)
 *******************************************************************************
 */
void LCD_SetPinOffState(LCD_Struct* LCDx, LCD_PinOffState_TypeDef OffState)
{
    LCDx->CR0.W = ((LCDx->CR1.W & (~LCD_CR1_OFF_CTL_mask_w)) | OffState);
}


/**
 *******************************************************************************
 * @brief       LCD reference voltage resister ladder selection.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   RLSelect:
 *  @arg\b        Resister_Int : select internal resister ladder.
 *  @arg\b        Resister_Ext : select external resister ladder
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_ReferenceResister(LCD, Resister_Int);
                LCD_ReferenceResister(LCD, Resister_Ext);
 * @endcode
 * @par         Modify
 *              void LCD_ReferenceResister(LCD_Struct* LCDx, LCD_ResisterLadder_TypeDef RLSelect)
 *******************************************************************************
 */
void LCD_ReferenceResister(LCD_Struct* LCDx, LCD_ResisterLadder_TypeDef RLSelect)
{
    LCDx->CR0.W = ((LCDx->CR0.W & (~LCD_CR0_RL_SEL_mask_w)) | RLSelect);
}


/**
 *******************************************************************************
 * @brief       LCD bias voltage select.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   BiasSelect:
 *  @arg\b        BIAS_Static : Static
 *  @arg\b        BIAS_B12 : Bias 1/2
 *  @arg\b        BIAS_B13 : Bias 1/3
 *  @arg\b        BIAS_B14 : Bias 1/4
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_BiasVoltege(LCD, BIAS_B13);
                LCD_BiasVoltege(LCD, BIAS_B14);
                LCD_BiasVoltege(LCD, BIAS_B12);
                LCD_BiasVoltege(LCD, BIAS_Static);
 * @endcode
 * @par         Modify
 *              void LCD_BiasVoltege(LCD_Struct* LCDx, uint32_t BiasSelect)
 *******************************************************************************
 */
void LCD_BiasVoltege(LCD_Struct* LCDx, uint32_t BiasSelect)
{
    LCDx->CR0.W = ((LCDx->CR0.W & (~LCD_CR0_BIAS_mask_w)) | BiasSelect);
}


/**
  * @}
  */


//=================================================================================================


/** @name LCD_Exported_Functions_Mode
  * @{
  */


/**
 *******************************************************************************
 * @brief       LCD frame control mode selection.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   TypeSelect:
 *  @arg\b        FrameTypeA : Type-A frame control mode
 *  @arg\b        FrameTypeB : Type-B frame control mode
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_FrameMode(LCD, FrameTypeA);
                LCD_FrameMode(LCD, FrameTypeB);
 * @endcode
 * @par         Modify
 *              void LCD_FrameMode(LCD_Struct* LCDx, LCD_FrameMode_TypeDef TypeSelect)
 *******************************************************************************
 */
void LCD_FrameMode(LCD_Struct* LCDx, LCD_FrameMode_TypeDef TypeSelect)
{
    LCDx->CR0.W = ((LCDx->CR0.W & (~LCD_CR0_FRM_MDS_mask_w)) | TypeSelect);
}


/**
 *******************************************************************************
 * @brief       LCD duty cycle select.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   DCSelect:
 *  @arg\b        DCStatic : Static duty
 *  @arg\b        DC12Duty : 1/2 duty.
 *  @arg\b        DC13Duty : 1/3 duty.
 *  @arg\b        DC14Duty : 1/4 duty.
 *  @arg\b        DC15Duty : 1/5 duty.
 *  @arg\b        DC16Duty : 1/6 duty.
 *  @arg\b        DC17Duty : 1/7 duty.
 *  @arg\b        DC18Duty : 1/8 duty.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_DutyCycle(LCD, DCStatic);
                LCD_DutyCycle(LCD, DC12Duty);
                LCD_DutyCycle(LCD, DC14Duty);
                LCD_DutyCycle(LCD, DC18Duty);
 * @endcode
 * @par         Modify
 *              void LCD_DutyCycle(LCD_Struct* LCDx, LCD_DutyCycle_TypeDef DCSelect)
 *******************************************************************************
 */
void LCD_DutyCycle(LCD_Struct* LCDx, LCD_DutyCycle_TypeDef DCSelect)
{
    LCDx->CR0.W = ((LCDx->CR0.W & (~LCD_CR0_DUTY_mask_w)) | DCSelect);
}


/**
 *******************************************************************************
 * @brief       LCD COM/SEG waveform inverse.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   InvSelect:
 *  @arg\b        0 to 15 : dead time duration is 0 ~ 15 unit
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_WaveformInverse(LCD, WaveformNormal);   // Default
                LCD_WaveformInverse(LCD, WaveformInverse);
 * @endcode
 * @par         Modify
 *              void LCD_WaveformInverse(LCD_Struct* LCDx, uint32_t InvSelect)
 *******************************************************************************
 */
void LCD_WaveformInverse(LCD_Struct* LCDx, uint32_t InvSelect)
{
    LCDx->CR0.W = ((LCDx->CR0.W & (~LCD_CR0_CS_INV_mask_w)) | InvSelect);
}


/**
 *******************************************************************************
 * @brief       LCD output dead time mode selection.
 * @details     When LCD_FRM_MDS selects 'TYPEB', this bit is only able to set 'Frame'.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   DTMode:
 *  @arg\b        DTFrame : insert dead time between two Frame cycle.
 *  @arg\b        DTDuty : insert dead time between two Duty cycle
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_FrameMode(LCD, DTFrame);
                LCD_FrameMode(LCD, DTDuty);
 * @endcode
 * @par         Modify
 *              void LCD_DeadTimeMode(LCD_Struct* LCDx, LCD_DeadTime_TypeDef DTMode)
 *******************************************************************************
 */
void LCD_DeadTimeMode(LCD_Struct* LCDx, LCD_DeadTime_TypeDef DTMode)
{
    LCDx->CR0.W = ((LCDx->CR0.W & (~LCD_CR0_DT_MDS_mask_w)) | DTMode);
}


/**
 *******************************************************************************
 * @brief       LCD output dead time period width.
 * @details     During the dead time, all COM and SEG pins are driven to ground.
 *              The value range 0 ~ 15 is indicated the dead time duration is 
 *              0 ~ 15 unit. 
 *              The unit is one CK_LCD_PHS (LCD phase period) if LCD_DT_MDS 
 *              selects 'Frame' or one CK_LCD_INT period if LCD_DT_MDS selects 'Duty'.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   DTPeriodWidth:
 *  @arg\b        0 to 15 : dead time duration is 0 ~ 15 unit
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_DeadTimePeriodWidth(LCD, 0);
                LCD_DeadTimePeriodWidth(LCD, 1);
                LCD_DeadTimePeriodWidth(LCD, 7);
                LCD_DeadTimePeriodWidth(LCD, 8);
                LCD_DeadTimePeriodWidth(LCD, 15);
 * @endcode
 * @par         Modify
 *              void LCD_DeadTimePeriodWidth(LCD_Struct* LCDx, uint32_t DTPeriodWidth)
 *******************************************************************************
 */
void LCD_DeadTimePeriodWidth(LCD_Struct* LCDx, uint32_t DTPeriodWidth)
{
    LCD->CR0.W = ((LCDx->CR0.W & (~LCD_CR0_DT_PW_mask_w)) | (LCD_CR0_DT_PW_mask_w & (DTPeriodWidth << LCD_CR0_DT_PW_shift_w)));
}


/**
 *******************************************************************************
 * @brief       LCD drive strength mode selection.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   DSMode:
 *  @arg\b        DSNormal : enable a high resistance(default).
 *  @arg\b        DSHigh : enable a low resistance.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_DriveStrengthMode(LCD, DSNormal);   // Default
                LCD_DriveStrengthMode(LCD, DSHigh);
 * @endcode
 * @par         Modify
 *              void LCD_DriveStrengthMode(LCD_Struct* LCDx, LCD_DriveStrengthMode_TypeDef DSMode)
 *******************************************************************************
 */
void LCD_DriveStrengthMode(LCD_Struct* LCDx, LCD_DriveStrengthMode_TypeDef DSMode)
{
    gBackup_CR1_DRV_MDS = DSMode;
    LCDx->CR1.W = ((LCDx->CR1.W & (~LCD_CR1_DRV_MDS_mask_w)) | gBackup_CR1_DRV_MDS);
}


/**
 *******************************************************************************
 * @brief       LCD drive high strength pulse width.
 * @details     When LCD_DRV_MDS is set 'Normal', this register is written to 
 *              define the drive high pulse duration in terms of CK_LCD_INT 
 *              pulses. The value range 0 ~ 7 is indicated the drive high pulse 
 *              duration is 0 ~ 7.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   DriveHighPulseWidth:
 *  @arg\b        0 to 7 :  drive high pulse duration is 0 ~ 7.
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_DriveStrengthMode(LCD, 0);   // Default
                LCD_DriveStrengthMode(LCD, 1);
                LCD_DriveStrengthMode(LCD, 3);
                LCD_DriveStrengthMode(LCD, 4);
                LCD_DriveStrengthMode(LCD, 6);
                LCD_DriveStrengthMode(LCD, 7);
 * @endcode
 * @par         Modify
 *              void LCD_HighDrivePulseWidth(LCD_Struct* LCDx, uint32_t DriveHighPulseWidth)
 *******************************************************************************
 */
void LCD_HighDrivePulseWidth(LCD_Struct* LCDx, uint32_t DriveHighPulseWidth)
{
    LCDx->CR1.W = ((LCDx->CR1.W & (~LCD_CR1_DRV_PW_mask_w)) | (LCD_CR1_DRV_PW_mask_w & (DriveHighPulseWidth << LCD_CR1_DRV_PW_shift_w)));
}


/**
  * @}
  */


//=================================================================================================


/** @name LCD_Exported_Functions_Interrupt and Flag
  * @{
  */


/**
 *******************************************************************************
 * @brief       LCD Interrupt Config
 * @details     LCD Interrupt Config
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   LCD_ITSrc:
 *  @arg\b        LCD_IT_SOF : Start of frame interrupt.
 *  @arg\b        LCD_IT_UDC : Update display data completed interrupt.
 *  @arg\b        LCD_IT_CPR : Charge pump power ready interrupt.
 *  @arg\b        LCD_IT_BLKON : Blinking on interrupt.
 *  @arg\b        LCD_IT_BLKOFF : Blinking off interrupt.
 *  @arg\b        LCD_IT_IEA : Interrupt all .
 * @param[in]   State:
 *   @arg\b       ENABLE          : Slave Address Detect Enable
 *   @arg\b       DISABLE         : Slave Address Detect Disable. (Default)
 * @return      
 * @exception   No
 * @note 
 * @par         Example
 * @code
                LCD_DriveStrengthMode(LCD, (LCD_IT_CPR | LCD_IT_IEA), ENABLE);
                LCD_DriveStrengthMode(LCD, LCD_IT_CPR, DISABLE);
                LCD_DriveStrengthMode(LCD, (LCD_IT_UDC | LCD_IT_IEA), ENABLE);
                LCD_DriveStrengthMode(LCD, LCD_IT_BLKON, ENABLE);
                LCD_DriveStrengthMode(LCD, LCD_IT_BLKON, DISABLE);
                LCD_DriveStrengthMode(LCD, (LCD_IT_SOF | LCD_IT_IEA), ENABLE);
                LCD_DriveStrengthMode(LCD, (LCD_IT_BLKOFF | LCD_IT_IEA), ENABLE);
                LCD_DriveStrengthMode(LCD, (LCD_IT_BLKOFF | LCD_IT_IEA), DISABLE);
 * @endcode
 * @par         Modify
 *              void LCD_IT_Config(LCD_Struct* LCDx, uint32_t LCD_ITSrc, FunctionalState State)
 *******************************************************************************
 */
void LCD_IT_Config(LCD_Struct* LCDx, uint32_t LCD_ITSrc, FunctionalState State)
{
    if(State == ENABLE)
        LCDx->INT.W |= LCD_ITSrc;
    else
        LCDx->INT.W &= (~LCD_ITSrc);
}


/**
 *******************************************************************************
 * @brief       LCD Interrupt Enable or Disable
 *
 *              LCD Interrupt Enable or Disable
 * @details     LCD Interrupt Enable or Disable
 * @param[in]   LCDx : LCD relationship control.
 * @param[in]   State : 
 *   @arg\b     ENABLE : Slave Address Detect Enable
 *   @arg\b     DISABLE : Slave Address Detect Disable. (Default)
 * @return      No
 * @exception   None
 * @note        None
 * @par         Example
 * @code
                LCD_ITEA_Cmd(LCD, ENABLE);
                LCD_ITEA_Cmd(LCD, DISABLE);
 * @endcode
 * @par         Modify
 *              void LCD_ITEA_Cmd(LCD_Struct* LCDx, FunctionalState State)
 *******************************************************************************
 */
void LCD_ITEA_Cmd(LCD_Struct* LCDx, FunctionalState State)
{
    if(State == ENABLE)
        LCDx->INT.W |= LCD_INT_IEA_mask_w;
    else
        LCDx->INT.W &= ~LCD_INT_IEA_mask_w;
}


/**
 *******************************************************************************
 * @brief       LCD Get IT Source
 *
 *              LCD Get Interrupt Source
 * @details     LCD Get Interrupt Source
 * @param[in]   LCDx : LCD relationship control.
 * @return      uint32_t : 
 *  @see          LCD_IT_IEA = Interrupt all.
 *  @see          LCD_IT_SOF = Start of frame interrupt .
 *  @see          LCD_IT_UDC = Update display data completed interrupt.
 *  @see          LCD_IT_CPR = Charge pump power ready interrupt .
 *  @see          LCD_IT_BLKON = Blinking on interrupt.
 *  @see          LCD_IT_BLKOFF = Blinking off interrupt.
 * @exception   None
 * @note        None
 * @par         Example
 * @code
                if((LCD_GetITSource(LCD) & LCD_IT_IEA) != 0)
                if((LCD_GetITSource(LCD) & LCD_IT_IEA) == 0)
                if((LCD_GetITSource(LCD) & LCD_IT_SOF) != 0)
                if((LCD_GetITSource(LCD) & LCD_IT_SOF) == 0)
                if((LCD_GetITSource(LCD) & LCD_IT_UDC) != 0)
                if((LCD_GetITSource(LCD) & LCD_IT_UDC) == 0)
                if((LCD_GetITSource(LCD) & LCD_IT_CPR) != 0)
                if((LCD_GetITSource(LCD) & LCD_IT_CPR) == 0)
                if((LCD_GetITSource(LCD) & LCD_IT_BLKON) != 0)
                if((LCD_GetITSource(LCD) & LCD_IT_BLKON) == 0)
                if((LCD_GetITSource(LCD) & LCD_IT_BLKOFF) != 0)
                if((LCD_GetITSource(LCD) & LCD_IT_BLKOFF) == 0)
 * @endcode
 * @par         Modify
 *              uint32_t LCD_GetITSource(LCD_Struct* LCDx);
 *******************************************************************************
 */
uint32_t LCD_GetITSource(LCD_Struct* LCDx)
{
    return(LCDx->INT.W & (LCD_IT_IEA | LCD_IT_SOF | LCD_IT_UDC | LCD_IT_CPR | LCD_IT_BLKON | LCD_IT_BLKOFF));
}


/**
 *******************************************************************************
 * @brief       LCD Get All Flag Status
 * @details     LCD status event Flag. 
 *              (set by hardware , clear by software setting 1)
 *              using Byte Mode.
 *              
 * @param[in]   LCDx : LCD relationship control.
 * @return      uint32_t : LCD FLAG Status.
 *  @see          LCD_FLAG_BLKOFF : Blinking off interrupt flag.
 *  @see          LCD_FLAG_BLKONF : Blinking on interrupt flag.
 *  @see          LCD_FLAG_CPRF : Charge pump power ready interrupt flag.
 *  @see          LCD_FLAG_UDCF : Update display data completed interrupt flag.
 *  @see          LCD_FLAG_SOF : Start of frame interrupt flag.
 *  @see          LCD_FLAG_PRDYF : Internal power source ready statue.
 *  @see          LCD_FLAG_DEF : Display enabled status.
 * @exception   None
 * @note        
 * @par         Example
 * @code        
                while((LCD_GetAllFlagStatus(LCD) & LCD_FLAG_SOF) == LCD_FLAG_SOF);
                while((LCD_GetAllFlagStatus(LCD) & LCD_FLAG_UDCF) == LCD_FLAG_UDCF);
                if((LCD_GetAllFlagStatus(LCD) & LCD_FLAG_DEF) == LCD_FLAG_DEF)
                if((LCD_GetAllFlagStatus(LCD) & LCD_FLAG_CPRF) == LCD_FLAG_CPRF)
                if((LCD_GetAllFlagStatus(LCD) & LCD_FLAG_BLKONF) == LCD_FLAG_BLKONF)
                if((LCD_GetAllFlagStatus(LCD) & LCD_FLAG_BLKONF) == LCD_FLAG_BLKONF)
 * @endcode     
 * @par         Modify
 *              uint32_t LCD_GetAllFlagStatus(LCD_Struct* LCDx)
 *******************************************************************************
 */
uint32_t LCD_GetAllFlagStatus(LCD_Struct* LCDx)
{
    return(LCDx->STA.W);
}


/**
 *******************************************************************************
 * @brief       LCD Get Flag
 *              LCD Get Flag
 * @details     LCD Get Flag
 * @param[in]   LCDx : LCD relationship control.
 * @param[in]   LCD_FLAG : LCD flag
 *  @arg\b        LCD_FLAG_BLKOFF : Blinking off interrupt flag.
 *  @arg\b        LCD_FLAG_BLKONF : Blinking on interrupt flag.
 *  @arg\b        LCD_FLAG_CPRF : Charge pump power ready interrupt flag.
 *  @arg\b        LCD_FLAG_UDCF : Update display data completed interrupt flag.
 *  @arg\b        LCD_FLAG_SOF : Start of frame interrupt flag.
 *  @arg\b        LCD_FLAG_PRDYF : Internal power source ready statue.
 *  @arg\b        LCD_FLAG_DEF : Display enabled status.
 * @par         None
 *              None 
 *      \n      global_var1     : global var1 
 * @return      DRV_Return
 * @see           DRV_UnHappened : False
 * @see           DRV_Happened : True
 * @exception   None
 * @note        None
 * @par         Example
 * @code
                while(LCD_GetFlagStatus(LCD, uint32_t LCD_FLAG_CPRF) != DRV_UnHappened)
                if(LCD_GetFlagStatus(LCD, uint32_t LCD_FLAG_UDCF) != DRV_UnHappened)
                if(LCD_GetFlagStatus(LCD, uint32_t LCD_FLAG_BLKONF) != DRV_UnHappened)
                if(LCD_GetFlagStatus(LCD, uint32_t LCD_FLAG_BLKOFF) != DRV_UnHappened)
                while(LCD_GetFlagStatus(LCD, uint32_t LCD_FLAG_CPRF) != DRV_UnHappened)
                while(LCD_GetFlagStatus(LCD, uint32_t LCD_FLAG_CPRF) != DRV_UnHappened)
 * @endcode
 * @par         Modify
 *              DRV_Return LCD_GetFlagStatus(LCD_Struct* LCDx, uint32_t LCD_FLAG)
 *******************************************************************************
 */
DRV_Return LCD_GetFlagStatus(LCD_Struct* LCDx, uint32_t LCD_FLAG)
{
    if((LCDx->STA.W & LCD_FLAG) == LCD_FLAG)
        return (DRV_Happened);
    else
        return (DRV_UnHappened);
}


/**
 *******************************************************************************
 * @brief       LCD Clear Flag
 *              LCD Clear Flag
 * @details     LCD Clear Flag.
 * @param[in]   LCDx : LCD relationship control.
 * @param[in]   LCD_FLAG : LCD flag
 *  @arg\b        LCD_FLAG_BLKOFF : Blinking off interrupt flag.
 *  @arg\b        LCD_FLAG_BLKONF : Blinking on interrupt flag.
 *  @arg\b        LCD_FLAG_CPRF : Charge pump power ready interrupt flag.
 *  @arg\b        LCD_FLAG_UDCF : Update display data completed interrupt flag.
 *  @arg\b        LCD_FLAG_SOF : Start of frame interrupt flag.
 *  @arg\b        LCD_FLAG_PRDYF : Internal power source ready statue.
 *  @arg\b        LCD_FLAG_DEF : Display enabled status.
 * @return      No
 * @exception   None
 * @note        
 * @par         Example
 * @code        
                LCD_ClearFlag(LCD, LCD_FLAG_BLKOFF)
                LCD_ClearFlag(LCD, LCD_FLAG_BLKONF)
                LCD_ClearFlag(LCD, LCD_FLAG_CPRF)
                LCD_ClearFlag(LCD, LCD_FLAG_UDCF)
                LCD_ClearFlag(LCD, LCD_FLAG_SOF)
                LCD_ClearFlag(LCD, LCD_FLAG_PRDYF)
                LCD_ClearFlag(LCD, LCD_FLAG_DEF)
 * @endcode     
 * @par         Modify
 *              void LCD_ClearFlag(LCD_Struct* LCDx, uint32_t LCD_FLAG)
 *******************************************************************************
 */
void LCD_ClearFlag(LCD_Struct* LCDx, uint32_t LCD_FLAG)
{
    LCDx->STA.W = LCD_FLAG;
}


/**
  * @}
  */


//=================================================================================================


/** @name LCD_Exported_Functions_Start_STOP
  * @{
  */


/**
 *******************************************************************************
 * @brief       LCD Start.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @return      DRV_Return
 *  @retval       DRV_False : False
 *  @retval       DRV_True : True
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_Start(LCD);
 * @endcode     
 * @par         Modify
 *              DRV_Return LCD_Start(LCD_Struct* LCDx)
 *******************************************************************************
 */
DRV_Return LCD_Start(LCD_Struct* LCDx)
{
    uint32_t lCount32 = 100;

    LCDx->CR1.W = (LCDx->CR1.W | gBackup_CR1_OVD_DIS);                              // Recover OVD

    LCDx->CR0.W = (LCDx->CR0.W | gBackup_CR0_VS_SEL);                               // Recover VS_SEL

    if((LCDx->CR0.W & LCD_CR0_RL_SEL_mask_w) == LCD_CR0_RL_SEL_int_w)               // Set Internal Resister.
        LCDx->CR1.W = ((LCDx->CR1.W & LCD_CR1_DRV_MDS_mask_w) | LCD_CR1_DRV_MDS_high_w);

    if((LCDx->CR0.W & LCD_CR0_VS_SEL_mask_w) == LCD_CR0_VS_SEL_cp_w)                // Check VT Source Charge Pump.
    {
        LCDx->STA.W = LCD_STA_CPRF_mask_w;                                          // Clear Charge Pump Flag.

        LCDx->CR2.W = LCDx->CR2.W & (~LCD_CR2_CP_CNT_mask_w);                       // Clear CP_CNT.

        LCDx->CR2.W = (LCDx->CR2.W | LCD_CR1_CP_EN_mask_w);                         // Enable Charge Pump.

        do{
            lCount32 -- ;
        }while(((LCDx->STA.W & LCD_STA_CPRF_mask_w) == 0) & (lCount32 != 0));       // wait Charge Pump Ready

        //if((lCount32 == 0) & ((LCDx->STA.W & LCD_STA_CPRF_mask_w) == 0))
        //{
        //    LCDx->CR2.W = (LCDx->CR2.W & (~LCD_CR1_CP_EN_mask_w));
        //    return DRV_False;
        //}
    }

    LCDx->CR0.W = (LCDx->CR0.W | LCD_CR0_EN_enable_w);                              // Enable LCD Module.

    if((LCDx->CR0.W & LCD_CR0_RL_SEL_mask_w) == LCD_CR0_RL_SEL_int_w)
        LCDx->CR1.W = ((LCDx->CR1.W & LCD_CR1_DRV_MDS_mask_w) | gBackup_CR1_DRV_MDS);  // Recover Internal Resister Configuration.

    if((LCDx->CR0.W & LCD_CR0_VS_SEL_mask_w) == LCD_CR0_VS_SEL_cp_w)                // Check VT Source Charge Pump.
        LCDx->CR2.W = (LCDx->CR2.W | gBackup_CR2_CP_CNT);                           // Recover CP_CNT Value.

    return DRV_True;
}


/**
 *******************************************************************************
 * @brief       LCD Stop.
 * @details     
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_Stop(LCD);
 * @endcode     
 * @par         Modify
 *              void LCD_Stop(LCD_Struct* LCDx)
 *******************************************************************************
 */
void LCD_Stop(LCD_Struct* LCDx)
{
    LCDx->CR0.W = (LCDx->CR0.W | LCD_CR0_EN_enable_w); __ISB();                     // Enable LCD Module.
    while((LCDx->CR0.W & LCD_CR0_EN_mask_w) != 0);

    LCDx->CR1.W &= ~LCD_CR1_CP_EN_mask_w;
    LCDx->CR2.W &= ~LCD_CR2_CP_CNT_mask_w;
    LCDx->CR0.W &= ~LCD_CR0_VS_SEL_mask_w;
    LCDx->CR1.W |= LCD_CR1_OVD_DIS_disable_w;

    LCDx->STA.W = 0xFFFFFFFFUL;
}


/**
 *******************************************************************************
 * @brief       LCD module disable or enable
 * @details     LCD module enable bit. When this bit is set 'Disable', all the 
 *              LCD_Pn outputs are disabled.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   LCDState:
 *  @arg\b        DISABLE: Disable LCD function.
 *  @arg\b        ENABLE : Enable LCD function.
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_Stop(LCD);
 * @endcode     
 * @par         Modify
 *              void LCD_Cmd(LCD_Struct* LCDx, FunctionalState LCDState)
 *******************************************************************************
 */
void LCD_Cmd(LCD_Struct* LCDx, FunctionalState LCDState)
{
    if(LCDState == DISABLE)
    {
        LCDx->CR0.W = LCDx->CR0.W & (~LCD_CR0_EN_mask_w);
        while((LCDx->CR0.W & LCD_CR0_EN_mask_w) != 0);
    }
    else
        LCDx->CR0.W = LCDx->CR0.W | LCD_CR0_EN_mask_w;
}


/**
  * @}
  */

//=================================================================================================


/** @name LCD_Exported_Functions_Blink
  * @{
  */

/**
 *******************************************************************************
 * @brief       LCD blinking clock CK_LCD_BLK divider.
 *              LCD blinking clock CK_LCD_BLK divider.
 * @details     LCD blinking clock CK_LCD_BLK divider.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   BDivider:
 *  @arg\b        BDivider_4 :  divided by 4.
 *  @arg\b        BDivider_8 :  divided by 8.
 *  @arg\b        BDivider_16 :  divided by 16.
 *  @arg\b        BDivider_32 :  divided by 32.
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_SetBlinkDivider(LCD, BDivider_4);   // Default
                LCD_SetBlinkDivider(LCD, BDivider_8);
                LCD_SetBlinkDivider(LCD, BDivider_16);
                LCD_SetBlinkDivider(LCD, BDivider_32);
 * @endcode     
 * @par         Modify
 *              void LCD_BlinkDivider(LCD_Struct* LCDx, LCD_BlinkClcokDivider_TypeDef BDivider)
 *******************************************************************************
 */
void LCD_SetBlinkDivider(LCD_Struct* LCDx, LCD_BlinkClcokDivider_TypeDef BDivider)
{
    LCDx->CLK.W = ((LCDx->CLK.W & (~LCD_CLK_CK_BDIV_mask_w)) | BDivider);
}


/**
 *******************************************************************************
 * @brief       LCD blinking maximum frame counter value.
 *              LCD blinking maximum frame counter value
 * @details     The frame counter will start at 0 and be automatically increased 
 *              by 1 at each frame end. When the frame counter reachs this 
 *              register value, it will resets and restarts from 0. The LCD 
 *              display will toggle the blinking display if the LCD_BLK_MDS is 
 *              not setting 'Disable'.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   BCount:
 *  @arg\b        0 to 31 : frame counter duration is 0 ~ 15 unit
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_SetBlinkCount(LCD, 0);   // Default
                LCD_SetBlinkCount(LCD, 1);
                LCD_SetBlinkCount(LCD, 15);
                LCD_SetBlinkCount(LCD, 16);
                LCD_SetBlinkCount(LCD, 30);
                LCD_SetBlinkCount(LCD, 31);
 * @endcode     
 * @par         Modify
 *              void LCD_SetBlinkCount(LCD_Struct* LCDx, uint32_t BCount)
 *******************************************************************************
 */
void LCD_SetBlinkCount(LCD_Struct* LCDx, uint32_t BCount)
{
    LCDx->CR2.W = ((LCDx->CR2.W & (~LCD_CR2_BCNT_mask_w)) | (LCD_CR2_BCNT_mask_w & (BCount << LCD_CR2_BCNT_shift_w)));
}


/**
 *******************************************************************************
 * @brief       LCD blinking mode selection.
 *              LCD blinking mode selection.
 * @details     LCD blinking mode selection.
 * @param[in]   LCDx:
 *  @arg\b        LCD: LCD relationship control.
 * @param[in]   BMode:
 *  @arg\b        BlinkDisable : Blinking disable.
 *  @arg\b        BlinkAll : Blinking all dots on all SEGs for all COMs.
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_BlinkMode_Config(LCD, BlinkDisable);   // Default
                LCD_BlinkMode_Config(LCD, BlinkAll);
 * @endcode     
 * @par         Modify
 *              void LCD_BlinkMode_Config(LCD_Struct* LCDx, LCD_BlinkMode_TypeDef BMode)
 *******************************************************************************
 */
void LCD_BlinkMode_Config(LCD_Struct* LCDx, LCD_BlinkMode_TypeDef BMode)
{
    LCDx->CR2.W = ((LCDx->CR2.W & (~LCD_CR2_BLK_MDS_mask_w)) | BMode);
    while((LCDx->CR2.W & LCD_CR2_BLK_MDS_mask_w) != BMode);
}


/**
 *******************************************************************************
 * @brief       LCD Get Blinking Mode state.
 *              LCD Get Blinking Mode state.
 * @details     LCD Get Blinking Mode state.
 * @param[in]   LCDx:
 *  @arg\b        LCD : LCD relationship control.
 * @return      DRV_Return
 *   @see         DRV_UnHappened : Blinking disable.
 *   @see         DRV_Happened : Blinking all dots on all SEGs for all COMs.
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                while(LCD_GetBlinkMode(LCD) != BlinkDisable);
 * @endcode     
 * @par         Modify
 *              DRV_Return LCD_GetBlinkMode(LCD_Struct* LCDx)
 *******************************************************************************
 */
DRV_Return LCD_GetBlinkMode(LCD_Struct* LCDx)
{
    if((LCDx->CR2.W & LCD_CR2_BLK_MDS_mask_w) == 0)
        return (DRV_UnHappened);
    else
        return (DRV_Happened);
}


/**
  * @}
  */


//=================================================================================================


/** @name LCD_Exported_Functions_WriteData
  * @{
  */


/**
 *******************************************************************************
 * @brief       LCD COM SEG select.
 *              LCD COM SEG select.
 * @details     LCD COM SEG select.
 * @param[in]   LCDx:
 *  @arg\b        LCD : LCD relationship control.
 * @param[in]   MUXn:
 *   @arg\b       0 = SEG : Segment line.
 *   @arg\b       1 = COM : Common line.
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_IOConfigCOMSEG(LCD, &MUXn);
 * @endcode     
 * @par         Modify
 *              void LCD_IOConfigCOMSEG(LCD_Struct* LCDx, uint32_t* MUXn)
 *******************************************************************************
 */
void LCD_IOConfigCOMSEG(LCD_Struct* LCDx, uint32_t* MUXn)
{
    LCDx->MUX0.W = MUXn[0];
    LCDx->MUX1.W = MUXn[1];
}


/**
 *******************************************************************************
 * @brief       LCD display memory data write.
 *              LCD display memory data write.
 * @details     LCD display memory data write.
 * @param[in]   LCDx :
 *  @arg\b        LCD : LCD relationship control.
 * @param[in]   WData :
 *  @arg\b        LCD_Pn pin display data bits or COM selection bits. 
 *                Refer the register description of LCD_Mn for detail information.
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_WriteData(LCD, &WData[0])
 * @endcode     
 * @par         Modify
 *              void LCD_WriteData(LCD_Struct* LCDx, uint32_t* WData)
 *******************************************************************************
 */
void LCD_WriteData(LCD_Struct* LCDx, uint32_t* WData)
{
    LCDx->MD0.W = WData[0];
    LCDx->MD1.W = WData[1];
    LCDx->MD2.W = WData[2];
    LCDx->MD3.W = WData[3];
    LCDx->MD4.W = WData[4];
    LCDx->MD5.W = WData[5];
    LCDx->MD6.W = WData[6];
    LCDx->MD7.W = WData[7];
    LCDx->MD8.W = WData[8];
    LCDx->MD9.W = WData[9];
    LCDx->MD10.W = WData[10];
}


/**
 *******************************************************************************
 * @brief       LCD display memory clear enable.
 *              LCD display memory clear enable.
 * @details     When enables, the chip will clear all LCD display memory registers LCD_MDn. 
 *              The bit is automatically reset when the LCD memory is cleared. 
 *              (set by software and clear by hardware)
 * @param[in]   LCDx :
 *  @arg\b        LCD : LCD relationship control.
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_ClearMemory(LCD)
 * @endcode     
 * @par         Modify
 *              void LCD_ClearMemory(LCD_Struct* LCDx)
 *******************************************************************************
 */
void LCD_ClearMemory(LCD_Struct* LCDx)
{
    LCDx->CR2.W |= LCD_CR2_MEM_CLR_mask_w;
}


/**
 *******************************************************************************
 * @brief       LCD segments all off enable or disable.
 *              LCD segments all off enable or disable.
 * @details     When selects 'Off', the chip will turn off all segment lines. 
 *              When selects 'On', all LCD segment lines are turning on or off 
 *              according to their corresponding memory data.
 * @param[in]   LCDx :
 *  @arg\b        LCD : LCD relationship control.
 * @param[in]   SegmentCTL :
 *  @arg\b        Segment_ON : All LCD segment lines are turning on.
 *  @arg\b        Segment_OFF : Turn off all segment lines.
 * @return      
 * @exception   No
 * @note        
 * @par         Example
 * @code        
                LCD_DisplayCTL(LCD, Segment_ON);
                LCD_DisplayCTL(LCD, Segment_OFF);
 * @endcode     
 * @par         Modify
 *              void LCD_DisplayCTL(LCD_Struct* LCDx, LCD_SegmentOff_TypeDef SegmentCTL)
 *******************************************************************************
 */
void LCD_DisplayCTL(LCD_Struct* LCDx, LCD_SegmentOff_TypeDef SegmentCTL)
{
    LCDx->CR2.W = ((LCDx->CR2.W & (~LCD_CR2_SEG_OFF_mask_w)) | SegmentCTL);
}

/**
  * @}
  */


//=================================================================================================


/**
  * @}
  */

/* Private function ----------------------------------------------------------*/
/** @name LCD_Private_Function LCD Private Function
  * @{
  */


/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
  #endif

