

/**
 ******************************************************************************
 *
 * @file        MG32_CAN_MID.h
 * @brief       The CAN module middleware H file.
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2024/07/01
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

#ifndef _MG32_CAN_MID_H
#define _MG32_CAN_MID_H

/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_RST.h"
#include "MG32_CAN.h"
#include "MG32_Common_MID.h"

/* Typedef -------------------------------------------------------------------*/

/*! @enum  MID_CAN_StateTypeDef
    @brief CAN state 
*/
typedef enum
{
    MID_CAN_STATE_RESET         = 0x00000000U,     /*!< CAN not yet initialized or disabled*/
    MID_CAN_STATE_READY         = 0x00000001U,     /*!< CAN initialized and ready for use*/
    MID_CAN_STATE_LISTENING     = 0x00000002U,     /*!< CAN receive process is ongoing*/
    MID_CAN_STATE_SLEEP_PENDING = 0x00000003U,     /*!< CAN sleep request is pending*/
    MID_CAN_STATE_SLEEP_ACTIVE  = 0x00000004U,     /*!< CAN sleep mode is active*/
    MID_CAN_STATE_ERROR         = 0x00000005U,     /*!< CAN error state*/ 
    
}MID_CAN_StateTypeDef;


/*! @struct  CAN_InitTypeDef
    @brief   CAN initial structure definition.
*/
typedef struct 
{
    uint32_t             Mode;                    /*!< Specifies the CAN operating mode.
                                                       This parameter can be a value of ref CAN_Operating_Mode*/
    uint32_t             SyncJumpWidth;           /*!< Specifies the maximum number of time quanta the CAN hardware
                                                       is allowed to lengthen or shorten a bit to perform resynchronization.
                                                       This parameter can be a value of ref CAN_Synchronisation_Jump_Width*/
    uint32_t             TimeSeg1;                /*!< Specifies number of time quanta in bit segment 1
                                                       This parameter can be a value of ref CAN_TimeQuantum_In_Bit_Segment1*/
    uint32_t             TimeSeg2;                /*!< Specifies number of time quanta in bit segment 2
                                                       This parameter can be a value of ref CAN_TimeQuantum_In_Bit_Segment2*/
    uint16_t             Prescaler;               /*!< Specifies the length of time quantum.
                                                       This parameter must be a number between Min Data = 2 and Max Data = 1024*/
    FunctionalState      AutoWakeup;              /*!< Enable or disable the automatic wake-up mode.
                                                       This parameter can be set to ENABLE or DISABLE*/
    FunctionalState      AutoRetransmission;      /*!< Enable or disable the automatic retransmission mode.
                                                       This parameter can be set to ENABLE or DISABLE*/
    FunctionalState      ReceiveFifoLocked;       /*!< Enable or disable the receive FIFO lock mode.
                                                       This parameter can be set to ENABLE or DISABLE*/
    FunctionalState      TransmitFifoPriority;    /*!< Enable or disable the transmit FIFO priority.
                                                       This parameter can be set to ENABLE or DISABLE
                                                       - DISABLE : Priority driven by the identifier of the message
                                                       - ENABLE  : Priority driven by the request order(chronologically)*/
    FunctionalState      FDTolerant;              /*!< Enable or disable the FD tolerant mode.
                                                       This parameter can be set to ENABLE or DISABLE*/
    FunctionalState      PinSwap;                 /*!< Enable or disable the CAN_TX and CAN_RX pin swap
                                                       This parameter can be set to ENABLE or DISABLE*/
    
}CAN_InitTypeDef;


/*! @struct CAN_RXBuf_TypeDef
    @brief  Acceptance filter bank struct.
*/
typedef struct
{
    union
    {
        uint32_t W[2];
        struct
        {
            struct
            {
                uint32_t            :1;   
                uint32_t  RTR       :1;   /*!< Compare ID's RTR (Remote transmission request bit).*/
                uint32_t  IDE       :1;   /*!< Compare ID's IDE (Identifier extension bit).*/
                uint32_t  EID       :18;  /*!< Compare ID's EID (Identifier extension ID-17 to ID-10).*/
                uint32_t  SID       :11;  /*!< Compare ID's SID (Base identifier ID-28 to ID-18).*/  
            }ID;         
            struct
            {
                uint32_t            :1;
                uint32_t  RTR       :1;   /*!< Mask ID's RTR (Remote transmission request bit)
                                               0: Don't care 
                                               1: Must match.*/
                uint32_t  IDE       :1;   /*!< Mask ID's IDE (Identifier extension bit)
                                               0: Don't care 
                                               1: Must match.*/
                uint32_t  EID       :18;  /*!< Mask ID's EID (Identifier extension ID-17 to ID-0)
                                               0: Don't care 
                                               1: Must match.*/
                uint32_t  SID       :11;  /*!< Mask ID's SID (Base identifier ID-28 to ID-18)
                                               0: Don't care 
                                               1: Must match.*/    
            }Mask;                        /*!< Acceptance filter mask configure struct.*/
        }Mask32;
        struct
        {
            struct
            {
                uint32_t            :1;
                uint32_t  RTR       :1;   /*!< List0 ID's RTR (Remote transmission request bit).*/
                uint32_t  IDE       :1;   /*!< List0 ID's IDE (Identifier extension bit).*/
                uint32_t  EID       :18;  /*!< List0 ID's EID (Identifier extension ID-17 to ID-10).*/
                uint32_t  SID       :11;  /*!< List0 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List0;       
            struct                  
            {                       
                uint32_t            :1;
                uint32_t  RTR       :1;   /*!< List1 ID's RTR (Remote transmission request bit).*/
                uint32_t  IDE       :1;   /*!< List1 ID's IDE (Identifier extension bit).*/
                uint32_t  EID       :18;  /*!< List1 ID's EID (Identifier extension ID-17 to ID-10).*/
                uint32_t  SID       :11;  /*!< List1 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List1;                       
        }List32;
        struct
        {
            struct
            {
                uint16_t  EID_17_15 :3;   /*!< Compare0 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t  RTR       :1;   /*!< Compare0 ID's RTR (Remote transmission request bit).*/
                uint16_t  IDE       :1;   /*!< Compare0 ID's EID (Identifier extension bit).*/
                uint16_t  SID       :11;  /*!< Compare0 ID's SID (Base identifier ID-28 to ID-18).*/    
            }ID0;                         
            struct
            {
                uint16_t  EID_17_15 :3;   /*!< Mask0 ID's EID (Identifier extension ID-17 to ID-15)
                                               0: Don't care 
                                               1: Must match.*/             
                uint16_t  RTR       :1;   /*!< Mask0 ID's RTR (Remote transmission request bit)
                                               0: Don't care 
                                               1: Must match.*/              
                uint16_t  IDE       :1;   /*!< Mask0 ID's IDE (Identifier extension bit)
                                               0: Don't care 
                                               1: Must match.*/               
                uint16_t  SID       :11;  /*!< Mask ID's SID (Base identifier ID-28 to ID-18)
                                               0: Don't care 
                                               1: Must match.*/      
            }Mask0;
            struct
            {
                uint16_t  EID_17_15 :3;   /*!< Compare1 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t  RTR       :1;   /*!< Compare1 ID's RTR (Remote transmission request bit).*/
                uint16_t  IDE       :1;   /*!< Compare1 ID's EID (Identifier extension bit).*/
                uint16_t  SID       :11;  /*!< Compare1 ID's SID (Base identifier ID-28 to ID-18).*/       
            }ID1;
            struct
            {
                uint16_t  EID_17_15 :3;   /*!< Mask1 ID's EID (Identifier extension ID-17 to ID-15)
                                               0: Don't care 
                                               1: Must match.*/             
                uint16_t  RTR       :1;   /*!< Mask1 ID's RTR (Remote transmission request bit)
                                               0: Don't care 
                                               1: Must match.*/              
                uint16_t  IDE       :1;   /*!< Mask1 ID's IDE (Identifier extension bit)
                                               0: Don't care 
                                               1: Must match.*/               
                uint16_t  SID       :11;  /*!< Mask1 ID's SID (Base identifier ID-28 to ID-18)
                                               0: Don't care 
                                               1: Must match.*/  
            }Mask1;
        }Mask16;     
        struct
        {
            struct
            {
                uint16_t EID_17_15 : 3;   /*!< List0 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t RTR       : 1;   /*!< List0 ID's RTR (Remote transmission request bit).*/
                uint16_t IDE       : 1;   /*!< List0 ID's IDE (Identifier extension bit).*/
                uint16_t SID       :11;   /*!< List0 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List0;
            struct
            {
                uint16_t EID_17_15 : 3;   /*!< List1 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t RTR       : 1;   /*!< List1 ID's RTR (Remote transmission request bit).*/
                uint16_t IDE       : 1;   /*!< List1 ID's IDE (Identifier extension bit).*/
                uint16_t SID       :11;   /*!< List1 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List1;
            struct
            {
                uint16_t EID_17_15 : 3;   /*!< List2 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t RTR       : 1;   /*!< List2 ID's RTR (Remote transmission request bit).*/
                uint16_t IDE       : 1;   /*!< List2 ID's IDE (Identifier extension bit).*/
                uint16_t SID       :11;   /*!< List2 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List2;           
            struct
            {
                uint16_t EID_17_15 : 3;   /*!< List3 ID's EID (Identifier extension ID-17 to ID-15).*/
                uint16_t RTR       : 1;   /*!< List3 ID's RTR (Remote transmission request bit).*/
                uint16_t IDE       : 1;   /*!< List3 ID's IDE (Identifier extension bit).*/
                uint16_t SID       :11;   /*!< List3 ID's SID (Base identifier ID-28 to ID-18).*/  
            }List3;
        }List16;
    }AFnR;
}CAN_FilterIDMaskTypeDef;

/*! @struct CAN_FilterTypeDef
    @brief  CAN filter configuration structure definition 
*/
typedef struct
{
    CAN_FilterIDMaskTypeDef  FilterIdMask;          /*!< Specifies the filter identification number or mask number*/
    uint32_t                 FilterBank;            /*!< Specifies the filter bank which will be initialized.
                                                         The parameter must be a number between Min_Data = 0 and Max_Data = 5*/
    uint32_t                 FilterMode;            /*!< Specifies the filter mode to initialized.
                                                         This parameter can be a value of ref CAN_Filter_Mode*/
    uint32_t                 FilterScale;           /*!< Specifies the filter scale.
                                                         This parameter can be a value of ref CAN_Filter_Scale*/
    uint32_t                 FilterActivation;      /*!< Enable or disable the filter.
                                                         This parameter can be a value of ref CAN_Filter_Activation*/
    uint32_t                 FilterFIFOAssignment;  /*!< Specifies the FIFO which will be assigned to the filter.
                                                          This parameter can be a value of ref CAN_Filter_FIFO*/
}CAN_FilterTypeDef;

/*! @struct CAN_FilterBankTypeDef
    @brief  CAN filter bank register 
*/
typedef struct
{
    __IO uint32_t AFnR0;    /*!< Filter bank register 0*/
    __IO uint32_t AFnR1;    /*!< Filter bank register 1*/
}CAN_FilterBankTypeDef;


/*! @struct  CAN_TxHeaderTypeDef
    @brief   CAN TX message header structure definition
*/
typedef struct
{    
    union
    {
        uint32_t  W[2];              /*!< Word 32-bit type*/
        struct
        {
            uint32_t Ext_ID : 18;    /*!< Specifies the identifier extension*/
            uint32_t BaseID : 11;    /*!< Specifies the base identifier*/
            uint32_t RTR    :  1;    /*!< Specifies the type of frame for the message that was received*/
            uint32_t IDE    :  1;    /*!< Specifies the type of identifier for the message that was received*/
            uint32_t        :  1;    /*!< Reserved bit*/
            uint32_t DLC    :  4;    /*!< Specifies the length of the frame that be received*/
            uint32_t        : 28;    /*!< Reserved bit*/
        }Field;
    }Format;
}CAN_TxHeaderTypeDef;

/*! @struct CAN_TxMailBox_TypeDef
    @brief  CAN transmit buffer data register.
*/
typedef struct
{
  __IO uint32_t TDATx1;    /*!< IDE,RTR,SID,EID*/  
  __IO uint32_t TDATx2;    /*!< DLC*/
  __IO ctype    TDATx3;    /*!< DATA3 ~ DATA0*/
  __IO ctype    TDATx4;    /*!< DATA7 ~ DATA4*/
} CAN_TxMailBox_TypeDef;


/*! @struct  CAN_RxHeaderTypeDef
    @brief   CAN RX message header structure definition
*/
typedef struct
{    
    union
    {
        uint32_t  W[2];               /*!< Word 32-bit type*/
        struct
        {
            uint32_t Ext_ID : 18;     /*!< Specifies the identifier extension*/
            uint32_t BaseID : 11;     /*!< Specifies the base identifier*/
            uint32_t RTR    :  1;     /*!< Specifies the type of frame for the message that was received*/
            uint32_t IDE    :  1;     /*!< Specifies the type of identifier for the message that was received*/
            uint32_t        :  1;     /*!< Reserved bit*/
            uint32_t DLC    :  4;     /*!< Specifies the length of the frame that be received*/
            uint32_t        : 28;     /*!< Reserved bit*/
        }Field;
    }Format;
}CAN_RxHeaderTypeDef;

/*! @struct CAN_RxFIFO_TypeDef
    @brief  CAN receive FIFO data register 
*/
typedef struct
{
  __IO uint32_t RDATx1;    /*!< IDE,RTR,SID,EID*/
  __IO uint32_t RDATx2;    /*!< DLC*/
  __IO ctype    RDATx3;    /*!< DATA3 ~ DATA0*/
  __IO ctype    RDATx4;    /*!< DATA7 ~ DATA4*/
} CAN_RxFIFO_TypeDef;

/*! @struct  CAN_HandleTypeDef
    @brief   CAN handle Structure definition.
*/
typedef struct
{
    CAN_Struct                 *Instance;          /*!< Register base address*/
                       
    CAN_InitTypeDef            Init;               /*!< CAN required parameter*/
    __IO uint32_t              ErrorCode;          /*!< CAN error code. This parameter can be a value of ref CAN_Error_Code*/
    __IO uint16_t              BusErrorCode;       /*!< CAN bus error code. This parameter bit define can be to ref CAN_BusErrorCode,
                                                        CAN_BusErrorCode_Direction and CAN_BusErrorCode_Segment*/
    __IO uint8_t               ArbitrationLostBit; /*!< CAN arbiration lost at which bit*/
    __IO MID_CAN_StateTypeDef  State;              /*!< CAN communication state*/

}CAN_HandleTypeDef;


/* Define --------------------------------------------------------------------*/


///@{
/**
 * @name CAN_Error_Code : CAN error code
 *   		
 */ 
#define MID_CAN_ERROR_NONE            (0x00000000U)     /*!< No error*/
                                                        
#define MID_CAN_ERROR_EWG             (0x00000001U)     /*!< Error warning*/
#define MID_CAN_ERROR_EPV             (0x00000002U)     /*!< Error passive*/
#define MID_CAN_ERROR_BOF             (0x00000004U)     /*!< Bus-off */
#define MID_CAN_ERROR_STF             (0x00000008U)     /*!< Stuff error*/
#define MID_CAN_ERROR_FOR             (0x00000010U)     /*!< Form error*/
#define MID_CAN_ERROR_BIT             (0x00000020U)     /*!< Bit error*/
#define MID_CAN_ERROR_CRC             (0x00000040U)     /*!< CRC error*/
#define MID_CAN_ERROR_OTHER           (0x00000080U)     /*!< the other error*/
                                                        
#define MID_CAN_ERROR_RX_FOV0         (0x00000100U)     /*!< Rx FIFO 0 overrun error*/
#define MID_CAN_ERROR_RX_FOV1         (0x00000200U)     /*!< Rx FIFO 1 overrun error*/
                                                        
#define MID_CAN_ERROR_TX_TERR         (0x00001000U)     /*!< CAN transmission happen error */
#define MID_CAN_ERROR_TX_ALST         (0x00002000U)     /*!< TxMailbox transmit failure due to arbitration lost*/
#define MID_CAN_ERROR_RX_RERR         (0x00004000U)     /*!< CAN reception happen error */
                                                        
#define MID_CAN_ERROR_TIMEOUT         (0x00020000U)     /*!< Timeout error*/
#define MID_CAN_ERROR_NOT_INITIALIZED (0x00040000U)     /*!< Peripheral not initialized*/
#define MID_CAN_ERROR_NOT_READY       (0x00080000U)     /*!< Peripheral not ready*/
#define MID_CAN_ERROR_NOT_STARTED     (0x00100000U)     /*!< Peripheral not started*/
#define MID_CAN_ERROR_PARAM           (0x00200000U)     /*!< Parameter error*/
///@}


///@{
/**
 * @name CAN_Operating_Mode: CAN operating mode
 *   		
 */ 
#define CAN_MODE_NORMAL              (0x00000000U)    /*!< Normal mode*/
#define CAN_MODE_LOOPBACK            (0x00000001U)    /*!< Loopback mode*/ 
#define CAN_MODE_SILENT              (0x00000002U)    /*!< Silent mode*/
#define CAN_MODE_SILENT_LOOPBACK     (0x00000003U)    /*!< Loopback combined with silent mode*/

/**
 * @name CAN_Synchronisation_Jump_Width : CAN synchronization jump width
 *
 */
#define CAN_SJW_1TQ                  (0x00000000U)    /*!< 1 time quantum*/    
#define CAN_SJW_2TQ                  (0x00010000U)    /*!< 2 time quantum*/
#define CAN_SJW_3TQ                  (0x00020000U)    /*!< 3 time quantum*/
#define CAN_SJW_4TQ                  (0x00030000U)    /*!< 4 time quantum*/

/**
 * @name CAN_TimeQuantum_In_Bit_Segment1 : CAN time quantum in bit segment 1
 *
 */
#define CAN_BS1_1TQ                  (0x00000000U)    /*!< 1 time quantum*/
#define CAN_BS1_2TQ                  (0x00000001U)    /*!< 2 time quantum*/               
#define CAN_BS1_3TQ                  (0x00000002U)    /*!< 3 time quantum*/
#define CAN_BS1_4TQ                  (0x00000003U)    /*!< 4 time quantum*/
#define CAN_BS1_5TQ                  (0x00000004U)    /*!< 5 time quantum*/
#define CAN_BS1_6TQ                  (0x00000005U)    /*!< 6 time quantum*/
#define CAN_BS1_7TQ                  (0x00000006U)    /*!< 7 time quantum*/
#define CAN_BS1_8TQ                  (0x00000007U)    /*!< 8 time quantum*/
#define CAN_BS1_9TQ                  (0x00000008U)    /*!< 9 time quantum*/
#define CAN_BS1_10TQ                 (0x00000009U)    /*!< 10 time quantum*/
#define CAN_BS1_11TQ                 (0x0000000AU)    /*!< 11 time quantum*/
#define CAN_BS1_12TQ                 (0x0000000BU)    /*!< 12 time quantum*/
#define CAN_BS1_13TQ                 (0x0000000CU)    /*!< 13 time quantum*/
#define CAN_BS1_14TQ                 (0x0000000DU)    /*!< 14 time quantum*/
#define CAN_BS1_15TQ                 (0x0000000EU)    /*!< 15 time quantum*/
#define CAN_BS1_16TQ                 (0x0000000FU)    /*!< 16 time quantum*/

/**
 * @name CAN_TimeQuantum_In_Bit_Segment2 : CAN time quantum in bit segment 2
 *
 */
#define CAN_BS2_2TQ                  (0x00000100U)    /*!< 2 time quantum*/               
#define CAN_BS2_3TQ                  (0x00000200U)    /*!< 3 time quantum*/
#define CAN_BS2_4TQ                  (0x00000300U)    /*!< 4 time quantum*/
#define CAN_BS2_5TQ                  (0x00000400U)    /*!< 5 time quantum*/
#define CAN_BS2_6TQ                  (0x00000500U)    /*!< 6 time quantum*/
#define CAN_BS2_7TQ                  (0x00000600U)    /*!< 7 time quantum*/
#define CAN_BS2_8TQ                  (0x00000700U)    /*!< 8 time quantum*/


///@}


///@{
/**
 * @name Filter_Banck_Number_Max : filter bank max number
 *   		
 */
#define CAN_FILTERBANK_MAX           (6 - 1)     /*!< Filter banck max number*/

/**
 * @name Filter_Reigster_Shift : Filter register shift related
 *   		
 */
#define CAN_FILTERBANK_SHIFT         0x30       /*!< CAN filter related register shift in CAN register*/   
#define CAN_FILTERBANKX_SHIFT        0x08       /*!< Each CAN filter group register shift*/


/**
 * @name CAN_Filter_Mode : CAN filter mode
 *   		
 */
#define CAN_FILTERMODE_IDMASK        (CAN_AFC1_AF0_MDS_maskmode_w)  /*!< Identifier mask mode*/
#define CAN_FILTERMODE_IDLIST        (CAN_AFC1_AF0_MDS_listmode_w)  /*!< Identifier list mode*/

/**
 * @name CAN_Filter_Scale : CAN filter scale
 *   		
 */
#define CAN_FILTERSCALE_16BIT        (0x00000000U)   /*!< Two 16-bit filters */
#define CAN_FILTERSCALE_32BIT        (0x00000001U)   /*!< One 32-bit filter  */

/**
 * @name CAN_Filter_Activation : CAN filter Activation
 *   		
 */
#define CAN_FILTER_DISABLE           (0x00000000U)  /*!< Disable filter.*/
#define CAN_FILTER_ENABLE            (0x00000001U)  /*!< Enable filter.*/


/**
 * @name CAN_Filter_FIFO : CAN acceptance filter FIFO select
 *   		
 */
#define CAN_FILTER_FIFO0             (CAN_AFC0_AF0_FSEL_fifo0_w) /*!< Filter FIFO 0 assignment for filter*/  
#define CAN_FILTER_FIFO1             (CAN_AFC0_AF0_FSEL_fifo1_w) /*!< Filter FIFO 1 assignment for filter*/

///@}


///@{
/**
 * @name CAN_Receive_FIFO_Number : CAN receive FIFO number
 *   		
 */
#define CAN_RX_FIFO0                 (0x00000000U)  /*!< CAN receive FIFO 0*/
#define CAN_RX_FIFO1                 (0x00000001U)  /*!< CAN receive FIFO 1*/
///@}

///@{
/**
 * @name CAN_TX_Mailboxes : CAN TX mailboxes
 *   		
 */ 
#define CAN_TX_MAILBOX0               (0x00000001U)  /*!< TX Mailbox 0*/
#define CAN_TX_MAILBOX1               (0x00000002U)  /*!< TX Mailbox 1*/
#define CAN_TX_MAILBOX2               (0x00000004U)  /*!< TX Mailbox 2*/

///@}

///@{
/**
 * @name CAN_TDATn_Shift : TDATn  register shift related.
 *   		
 */ 
#define CAN_TXBUF_SHIFT               0xC0   /*!< CAN transmit buffer register address shift in CAN module register.*/
#define CAN_TXBUFX_SHIFT              0x10   /*!< Each CAN transmit buffer group register shift*/
///@}

///@{
/**
 * @name CAN_RDATn_Shift : RDATn register shift related.
 *   		
 */ 
#define CAN_RXBUF_SHIFT               0xA0   /*!< CAN receive FIFO register address shift in CAN module register*/
#define CAN_RXBUFX_SHIFT              0x10   /*!< Each CAN receive FIFO group register shift*/
///@}


///@{
/**
 * @name CAN_BusErrorCode_Mask : CAN bus error code related mask
 *   		
 */ 
#define CAN_ERROR_CODE_MASK           0x0700    /*!< CAN Bus error code mask*/
#define CAN_ERROR_DIR_MASK            0x0080    /*!< CAN Bus error code direct mask*/
#define CAN_ERROR_SCODE_MASK          0x001F    /*!< CAN Bus error segment code mask.*/

/**
 * @name CAN_BusErrorCode : Bus error code (Bit 10..8)
 *   		
 */ 
#define CAN_ERR_CODE_BIT              0x0100    /*!< Bit error.*/
#define CAN_ERR_CODE_FORM             0x0200    /*!< Form error.*/
#define CAN_ERR_CODE_STUFF            0x0300    /*!< Stuff error.*/
#define CAN_ERR_CODE_CRC              0x0400    /*!< CRC error.*/
#define CAN_ERR_CODE_OTHER            0x0700    /*!< The other error.*/

/**
 * @name CAN_BusErrorCode_Direction : Record direction when bus error occure (Bit 7)
 *   		
 */                                     
#define CAN_ERR_DIR_TX                0x0000    /*!< Error occur in transmit.*/
#define CAN_ERR_DIR_RX                0x0080    /*!< Error occur in receive.*/

/**
 * @name CAN_BusErrorCode_Segment : CAN bus error segment code (Bit 4..0)
 *   		
 */
#define CAN_ERR_SCODE_ID28_ID21       0x0002    /*!< Error happen in ID28 ~ ID21.*/
#define CAN_ERR_SCODE_SOF             0x0003    /*!< Error happen in SOF.*/
#define CAN_ERR_SCODE_SRTR            0x0004    /*!< Error happen in RTR of stardard format or SRR of extended format.*/
#define CAN_ERR_SCODE_IDE             0x0005    /*!< Error happen in IDE.*/
#define CAN_ERR_SCODE_ID20_ID18       0x0006    /*!< Error happen in ID20 ~ ID18.*/
#define CAN_ERR_SCODE_ID17_ID13       0x0007    /*!< Error happen in ID17 ~ ID13.*/
#define CAN_ERR_SCODE_CRC             0x0008    /*!< Error happen in CRC.*/
#define CAN_ERR_SCODE_RESBIT0         0x0009    /*!< Error happen in r0 (reserved bit 0).*/
#define CAN_ERR_SCODE_DATAFIELD       0x000A    /*!< Error happen in data field.*/
#define CAN_ERR_SCODE_DLC             0x000B    /*!< Error happen in DLC.*/
#define CAN_ERR_SCODE_RTR             0x000C    /*!< Error happen in RTR of extended format.*/
#define CAN_ERR_SCODE_RESBIT1         0x000D    /*!< Error happen in r1 (reserved bit 1).*/
#define CAN_ERR_SCODE_ID4_ID0         0x000E    /*!< Error happen in ID4 ~ ID0.*/
#define CAN_ERR_SCODE_ID12_ID5        0x000F    /*!< Error happen in ID12 ~ ID5.*/
#define CAN_ERR_SCODE_AERROR          0x0011    /*!< Error happen in active error flag.*/
#define CAN_ERR_SCODE_INTERMISSION    0x0012    /*!< Error happen in intermission.*/
#define CAN_ERR_SCODE_TOLERATE        0x0013    /*!< Error happen in tolerate dominant bits.*/
#define CAN_ERR_SCODE_PERROR          0x0016    /*!< Error happen in Passive error flag.*/
#define CAN_ERR_SCODE_ERRORDELIMITER  0x0017    /*!< Error happen in error delimiter.*/
#define CAN_ERR_SCODE_CRCDELIMITER    0x0018    /*!< Error happen in CRC delimiter.*/
#define CAN_ERR_SCODE_ACK             0x0019    /*!< Error happen in ACK.*/
#define CAN_ERR_SCODE_EOF             0x001A    /*!< Error happen in EOF.*/
#define CAN_ERR_SCODE_ACKDELIMITER    0x001B    /*!< Error happen in ACK delimiter.*/
#define CAN_ERR_SCODE_OVERLOADFRAME   0x001C    /*!< Error happen in overload frame.*/
///@}


///@{
/**
 * @name CAN_flags : CAN flag related define
 *   		
 */ 
/*Transmit Interrupt Flag*/
#define CAN_ITFLAG_TX0      CAN_INT_TX0_IE_mask_w     /*!< Transmit mailbox 0 empty interrupt flag*/
#define CAN_ITFLAG_TX1      CAN_INT_TX1_IE_mask_w     /*!< Transmit mailbox 1 empty interrupt flag*/
#define CAN_ITFLAG_TX2      CAN_INT_TX2_IE_mask_w     /*!< Transmit mailbox 2 empty interrupt flag*/
#define CAN_ITFLAG_ALST     CAN_INT_ALOS_IE_mask_w    /*!< Arbitration lost interrupt flag*/

/*Transmit Event Flag*/
#define CAN_FLAG_TC0        CAN_STA2_TC0_STA_mask_w   /*!<Transmit successful mailbox 0 flag*/
#define CAN_FALG_TB0        CAN_STA2_TB0_STA_mask_w   /*!<Mailbox 0 buffer release flag*/
#define CAN_FLAG_TC1        CAN_STA2_TC1_STA_mask_w   /*!<Transmit successful mailbox 0 flag*/
#define CAN_FALG_TB1        CAN_STA2_TB1_STA_mask_w   /*!<Mailbox 0 buffer release flag*/
#define CAN_FLAG_TC2        CAN_STA2_TC2_STA_mask_w   /*!<Transmit successful mailbox 0 flag*/
#define CAN_FALG_TB2        CAN_STA2_TB2_STA_mask_w   /*!<Mailbox 0 buffer release flag*/

/*Receive Interrupt Flag*/
#define CAN_ITFLAG_ROVR0    CAN_INT_ROVR0_IE_mask_w   /*!< FIFO 0 overrun interrupt flag*/
#define CAN_ITFLAG_RFUL0    CAN_INT_RFUL0_IE_mask_w   /*!< FIFO 0 full interrupt flag*/
#define CAN_ITFLAG_RPEND0   CAN_INT_RPEND0_IE_mask_w  /*!< FIFO 0 message pending interrupt flag*/
#define CAN_ITFLAG_RX0      CAN_INT_RX0_IE_mask_w     /*!< FIFO 0 receive interrupt flag*/ 

#define CAN_ITFLAG_ROVR1    CAN_INT_ROVR1_IE_mask_w   /*!< FIFO 1 overrun interrupt flag*/
#define CAN_ITFLAG_RFUL1    CAN_INT_RFUL1_IE_mask_w   /*!< FIFO 1 full interrupt flag*/
#define CAN_ITFLAG_RPEND1   CAN_INT_RPEND1_IE_mask_w  /*!< FIFO 1 message pending interrupt flag*/
#define CAN_ITFLAG_RX1      CAN_INT_RX1_IE_mask_w     /*!< FIFO 1 receive interrupt flag*/ 

/*Operating Mode Interrupt Flag*/
#define CAN_ITFLAG_WKU      CAN_INT_WUP_IE_mask_w     /*!< Wake-up interrupt flag*/


/*Operating Mode Flag*/
#define CAN_FLAG_LP         CAN_STA_LP_STA_mask_w     /*!< CAN module low power status*/
#define CAN_FLAG_INIT       CAN_STA_INIT_STA_mask_w   /*!< CAN initial mode status*/
#define CAN_FLAG_BOF        CAN_STA_BUS_STA_mask_w    /*!< CAN bus-off status*/
#define CAN_FLAG_EWG        CAN_STA_EW_STA_mask_w     /*!< CAN error warning limit status*/
#define CAN_FLAG_EPV        CAN_STA_EP_STA_mask_w     /*!< CAN error passive status*/

/*Error Interrupt Flag*/
#define CAN_ITFLAG_EWG      CAN_INT_EW_IE_mask_w      /*!< Error warning interrupt flag*/
#define CAN_ITFLAG_EPV      CAN_INT_EP_IE_mask_w      /*!< Error passive interrupt flag*/
#define CAN_ITFLAG_BOF      CAN_INT_BUS_IE_mask_w     /*!< Bus-off interrupt flag*/
#define CAN_ITFLAG_BERR     CAN_INT_BERR_IE_mask_w    /*!< Bus error interrupt flag*/

///@}


///@{
/**
 * @name CAN_Interrupts : CAN Interrupts
 *   		
 */ 
/*Transmit Interrupt*/
#define CAN_IT_TX_MAILBOX0_EMPTY  CAN_INT_TX0_IE_mask_w     /*!< Transmit mailbox 0 empty interrupt*/
#define CAN_IT_TX_MAILBOX1_EMPTY  CAN_INT_TX1_IE_mask_w     /*!< Transmit mailbox 1 empty interrupt*/
#define CAN_IT_TX_MAILBOX2_EMPTY  CAN_INT_TX2_IE_mask_w     /*!< Transmit mailbox 2 empty interrupt*/
#define CAN_IT_ALST               CAN_INT_ALOS_IE_mask_w    /*!< Arbitration lost interrupt*/

/*Receive Interrupt*/
#define CAN_IT_RX_FIFO0_OVERRUN      CAN_INT_ROVR0_IE_mask_w   /*!< FIFO 0 overrun interrupt*/
#define CAN_IT_RX_FIFO0_FULL         CAN_INT_RFUL0_IE_mask_w   /*!< FIFO 0 full interrupt*/
#define CAN_IT_RX_FIFO0_PENDING      CAN_INT_RPEND0_IE_mask_w  /*!< FIFO 0 message pending interrupt*/
#define CAN_IT_RX_FIFO0_MSG_PENDING  CAN_INT_RX0_IE_mask_w     /*!< FIFO 0 receive interrupt*/ 

#define CAN_IT_RX_FIFO1_OVERRUN      CAN_INT_ROVR1_IE_mask_w   /*!< FIFO 0 overrun interrupt*/
#define CAN_IT_RX_FIFO1_FULL         CAN_INT_RFUL1_IE_mask_w   /*!< FIFO 0 full interrupt*/
#define CAN_IT_RX_FIFO1_PENDING      CAN_INT_RPEND1_IE_mask_w  /*!< FIFO 0 message pending interrupt*/
#define CAN_IT_RX_FIFO1_MSG_PENDING  CAN_INT_RX1_IE_mask_w     /*!< FIFO 0 receive interrupt*/ 


/*Operating Mode Interrupt*/
#define CAN_IT_WAKEUP       CAN_INT_WUP_IE_mask_w     /*!< Wake-up interrupt*/

/*Error Interrupt*/
#define CAN_IT_ERROR_WARNING CAN_INT_EW_IE_mask_w      /*!< Error warning interrupt*/
#define CAN_IT_ERROR_PASSIVE CAN_INT_EP_IE_mask_w      /*!< Error passive interrupt*/
#define CAN_IT_BUSOFF        CAN_INT_BUS_IE_mask_w     /*!< Bus-off interrupt*/ 
#define CAN_IT_BUS_ERROR     CAN_INT_BERR_IE_mask_w    /*!< Bus error interrupt*/


/*CAN Interrupt All Enable*/
#define CAN_IT_IEA          CAN_INT_IEA_mask_w        /*!< CAN interrupt all enable (CAN module interrupt main enable)*/

///@}


/* Exported Macro ------------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief  Reset CAN handle state
 * @param[out]  "__HANDLE__": Specifies CAN handle.
 * @return 
 *******************************************************************************
 */
#define __DRV_CAN_RESET_HANDLE_STATE(__HANDLE__)  ((__HANDLE__)->State = MID_CAN_STATE_RESET)

/**
 *******************************************************************************
 * @brief  Enable the specified CAN interrupts.
 * @param[in] "__HANDLE__": Specifies CAN handle
 * @param[in] "__INTERRUPT__": CAN Interrupt sources to enable
 *                           This parameter can be any combination of the following values:
 *   @arg CAN_IT_TX_MAILBOX0_EMPTY    : Transmit mailbox 0 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX1_EMPTY    : Transmit mailbox 1 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX2_EMPTY    : Transmit mailbox 2 empty interrupt.
 *   @arg CAN_IT_RX_FIFO0_OVERRUN     : FIFO 0 overrun interrupt
 *   @arg CAN_IT_RX_FIFO0_FULL        : FIFO 0 full interrupt
 *   @arg CAN_IT_RX_FIFO0_PENDING     : FIFO 0 message pending interrupt
 *   @arg CAN_IT_RX_FIFO0_MSG_PENDING : FIFO 0 receive interrupt
 *   @arg CAN_IT_RX_FIFO1_OVERRUN     : FIFO 1 overrun interrupt
 *   @arg CAN_IT_RX_FIFO1_FULL        : FIFO 1 full interrupt
 *   @arg CAN_IT_RX_FIFO1_PENDING     : FIFO 1 message pending interrupt
 *   @arg CAN_IT_RX_FIFO1_MSG_PENDING : FIFO 1 receive interrupt
 *   @arg CAN_IT_WAKEUP               : Wake-up interrupt
 *   @arg CAN_IT_ERROR_WARNING        : Error warning interrupt
 *   @arg CAN_IT_ERROR_PASSIVE        : Error passive interrupt
 *   @arg CAN_IT_BUSOFF               : Bus-off interrupt
 *   @arg CAN_IT_BUS_ERROR            : Bus error interrupt
 *   @arg CAN_IT_ALST                 : Arbitration lost interrupt
 *   @arg CAN_IT_IEA                  : CAN interrupt all enable (CAN module interrupt main enable)
 * @return 
 *******************************************************************************
 */
#define __DRV_CAN_ENABLE_IT(__HANDLE__, __INTERRUPT__) (((__HANDLE__)->Instance->INT.W) |= (__INTERRUPT__))

/**
 *******************************************************************************
 * @brief  Disable the specified CAN interrupts.
 * @param[in] "__HANDLE__": Specifies CAN handle
 * @param[in] "__INTERRUPT__": CAN Interrupt sources to disable
 *                           This parameter can be any combination of the following values:
 *   @arg CAN_IT_TX_MAILBOX0_EMPTY    : Transmit mailbox 0 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX1_EMPTY    : Transmit mailbox 1 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX2_EMPTY    : Transmit mailbox 2 empty interrupt.
 *   @arg CAN_IT_RX_FIFO0_OVERRUN     : FIFO 0 overrun interrupt
 *   @arg CAN_IT_RX_FIFO0_FULL        : FIFO 0 full interrupt
 *   @arg CAN_IT_RX_FIFO0_PENDING     : FIFO 0 message pending interrupt
 *   @arg CAN_IT_RX_FIFO0_MSG_PENDING : FIFO 0 receive interrupt
 *   @arg CAN_IT_RX_FIFO1_OVERRUN     : FIFO 1 overrun interrupt
 *   @arg CAN_IT_RX_FIFO1_FULL        : FIFO 1 full interrupt
 *   @arg CAN_IT_RX_FIFO1_PENDING     : FIFO 1 message pending interrupt
 *   @arg CAN_IT_RX_FIFO1_MSG_PENDING : FIFO 1 receive interrupt
 *   @arg CAN_IT_WAKEUP               : Wake-up interrupt
 *   @arg CAN_IT_ERROR_WARNING        : Error warning interrupt
 *   @arg CAN_IT_ERROR_PASSIVE        : Error passive interrupt
 *   @arg CAN_IT_BUSOFF               : Bus-off interrupt
 *   @arg CAN_IT_BUS_ERROR            : Bus error interrupt
 *   @arg CAN_IT_ALST                 : Arbitration lost interrupt
 *   @arg CAN_IT_IEA                  : CAN interrupt all enable (CAN module interrupt main enable)
 * @return 
 *******************************************************************************
 */
#define __DRV_CAN_DISABLE_IT(__HANDLE__, __INTERRUPT__) (((__HANDLE__)->Instance->INT.W) &= ~(__INTERRUPT__))


/**
 *******************************************************************************
 * @brief  Check if the specified CAN interrupt source is enable or disabled.
 * @param[in] "__HANDLE__": Specifies CAN handle.
 * @param[in] "__INTERRUPT__": Specifies the CAN interrupt source to check.
 *                           This parameter can be a value of the following values:
 *   @arg CAN_IT_TX_MAILBOX0_EMPTY    : Transmit mailbox 0 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX1_EMPTY    : Transmit mailbox 1 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX2_EMPTY    : Transmit mailbox 2 empty interrupt.
 *   @arg CAN_IT_RX_FIFO0_OVERRUN     : FIFO 0 overrun interrupt
 *   @arg CAN_IT_RX_FIFO0_FULL        : FIFO 0 full interrupt
 *   @arg CAN_IT_RX_FIFO0_PENDING     : FIFO 0 message pending interrupt
 *   @arg CAN_IT_RX_FIFO0_MSG_PENDING : FIFO 0 receive interrupt
 *   @arg CAN_IT_RX_FIFO1_OVERRUN     : FIFO 1 overrun interrupt
 *   @arg CAN_IT_RX_FIFO1_FULL        : FIFO 1 full interrupt
 *   @arg CAN_IT_RX_FIFO1_PENDING     : FIFO 1 message pending interrupt
 *   @arg CAN_IT_RX_FIFO1_MSG_PENDING : FIFO 1 receive interrupt
 *   @arg CAN_IT_WAKEUP               : Wake-up interrupt
 *   @arg CAN_IT_ERROR_WARNING        : Error warning interrupt
 *   @arg CAN_IT_ERROR_PASSIVE        : Error passive interrupt
 *   @arg CAN_IT_BUSOFF               : Bus-off interrupt
 *   @arg CAN_IT_BUS_ERROR            : Bus error interrupt
 *   @arg CAN_IT_ALST                 : Arbitration lost interrupt
 *   @arg CAN_IT_IEA                  : CAN interrupt all enable (CAN module interrupt main enable)
 * @return The state of __INTERRUPT__ (TRUE or FALSE)
 *******************************************************************************
 */
#define __DRV_CAN_GET_IT_SOURCE(__HANDLE__, __INTERRUPT__) (((__HANDLE__)->Instance->INT.W) & (__INTERRUPT__))


/**
 *******************************************************************************
 * @brief  Clear the specified CAN pending flag
 * @param[in] "__HANDLE__": Specifies CAN handle.
 * @param[in] "__FLAG__": Specifies the flag to check 
 *                      This parameter can be a value of the following values:
 *   @arg CAN_ITFLAG_TX0    : Transmit mailbox 0 empty interrupt.
 *   @arg CAN_ITFLAG_TX1    : Transmit mailbox 1 empty interrupt.
 *   @arg CAN_ITFLAG_TX2    : Transmit mailbox 2 empty interrupt.
 *   @arg CAN_ITFLAG_ROVR0  : FIFO 0 overrun interrupt
 *   @arg CAN_ITFLAG_RFUL0  : FIFO 0 full interrupt
 *   @arg CAN_ITFLAG_RPEND0 : FIFO 0 message pending interrupt
 *   @arg CAN_ITFLAG_RX0    : FIFO 0 receive interrupt
 *   @arg CAN_ITFLAG_ROVR1  : FIFO 1 overrun interrupt
 *   @arg CAN_ITFLAG_RFUL1  : FIFO 1 full interrupt
 *   @arg CAN_ITFLAG_RPEND1 : FIFO 1 message pending interrupt
 *   @arg CAN_ITFLAG_RX1    : FIFO 1 receive interrupt
 *   @arg CAN_ITFLAG_WKU    : Wake-up interrupt
 *   @arg CAN_ITFLAG_EWG    : Error warning interrupt
 *   @arg CAN_ITFLAG_EPV    : Error passive interrupt
 *   @arg CAN_ITFLAG_BOF    : Bus-off interrupt
 *   @arg CAN_ITFLAG_BERR   : Bus error interrupt
 *   @arg CAN_ITFLAG_ALST   : Arbitration lost interrupt
 *   @arg CAN_ITFLAG_IEA    : CAN interrupt all enable 
 * @return 
 *******************************************************************************
 */
#define __DRV_CAN_CLEAR_FLAG(__HANDLE__, __FLAG__) ((__HANDLE__)->Instance->STA.W) = (__FLAG__)


/**
 *******************************************************************************
 * @brief  Get Error Code
 * @param[in] "__HANDLE__": Specifies CAN handle.
 * @return Last error code related information 
 * @note      The error code define can refer to CAN_ErrorCode.
 *   \n Bit 10..8 : Error Code:
 *   \n             - CAN_ERR_CODE_BIT   : Bit error
 *   \n             - CAN_ERR_CODE_FORM  : Form error
 *   \n             - CAN_ERR_CODE_STUFF : Stuff error
 *   \n             - CAN_ERR_CODE_CRC   : CRC error
 *   \n             - CAN_ERR_CODE_OTHER : the other error
 *   \n Bit 7     : Error Direction:
 *   \n             - CAN_ERR_DIR_TX     : The error occurred during reception.
 *   \n             - CAN_ERR_DIR_RX     : The error occurred during transmission.
 *   \n Bit 4..0  : Error Segment Code:
 *   \n             - CAN_ERR_SCODE_ID28_ID21     : ID28 to ID21
 *   \n             - CAN_ERR_SCODE_SOF           : Start of frame
 *   \n             - CAN_ERR_SCODE_SRTR          : SRTR bit
 *   \n             - CAN_ERR_SCODE_IDE           : IDE bit
 *   \n             - CAN_ERR_SCODE_ID20_ID18     : ID20 to ID18
 *   \n             - CAN_ERR_SCODE_ID17_ID13     : ID17 to ID13
 *   \n             - CAN_ERR_SCODE_CRC           : CRC sequence
 *   \n             - CAN_ERR_SCODE_RESBIT0       : Reserved bit 0
 *   \n             - CAN_ERR_SCODE_DATAFIELD     : Data Field
 *   \n             - CAN_ERR_SCODE_DLC           : Data Length Code
 *   \n             - CAN_ERR_SCODE_RTR           : RTR bit
 *   \n             - CAN_ERR_SCODE_RESBIT1       : Reserved bit 1
 *   \n             - CAN_ERR_SCODE_ID4_ID0       : ID4 to ID0
 *   \n             - CAN_ERR_SCODE_ID12_ID5      : ID12 to ID5
 *   \n             - CAN_ERR_SCODE_AERROR        : Active error flag
 *   \n             - CAN_ERR_SCODE_INTERMISSION  : Intermission
 *   \n             - CAN_ERR_SCODE_TOLERATE      : Tolerate dominant bits
 *   \n             - CAN_ERR_SCODE_PERROR        : Passive error flag
 *   \n             - CAN_ERR_SCODE_ERRORDELIMITER: Error delimiter
 *   \n             - CAN_ERR_SCODE_CRCDELIMITER  : CRC delimiter
 *   \n             - CAN_ERR_SCODE_ACK           : Acknowledge
 *   \n             - CAN_ERR_SCODE_EOF           : End of frame
 *   \n             - CAN_ERR_SCODE_ACKDELIMITER  : Acknowledge delimiter
 *   \n             - CAN_ERR_SCODE_OVERLOADFRAME : Overload frame
 *******************************************************************************
 */
#define __DRV_CAN_GET_ERRORCODE(__HANDLE__)  ((__HANDLE__)->Instance->STA3.H[0]







/* Exported Function  --------------------------------------------------------*/


///@{
/**
 * @name Initialization and de-initialization functions
 *   		
 */ 
MID_StatusTypeDef MID_CAN_DeInit( CAN_HandleTypeDef *mCAN);
MID_StatusTypeDef MID_CAN_Init( CAN_HandleTypeDef *mCAN);
void MID_CAN_MspInit( CAN_HandleTypeDef *mCAN);
void MID_CAN_MspDeInit( CAN_HandleTypeDef *mCAN);

MID_StatusTypeDef MID_CAN_ConfigFilter(CAN_HandleTypeDef *mCAN, CAN_FilterTypeDef *FilterConfig);
///@}

///@{
/**
 * @name Control function
 *   		
 */ 

MID_StatusTypeDef MID_CAN_Start( CAN_HandleTypeDef *mCAN);
MID_StatusTypeDef MID_CAN_Stop( CAN_HandleTypeDef *mCAN);
MID_StatusTypeDef MID_CAN_RequestSleep( CAN_HandleTypeDef *mCAN);
MID_StatusTypeDef MID_CAN_WakeUp( CAN_HandleTypeDef *mCAN);
uint32_t MID_CAN_IsSleepActive( CAN_HandleTypeDef *mCAN);

MID_StatusTypeDef MID_CAN_AddTxMessage( CAN_HandleTypeDef *mCAN, CAN_TxHeaderTypeDef *pHeader, uint8_t aData[], uint32_t *pTxMailbox);
MID_StatusTypeDef MID_CAN_AbortTxRequest(CAN_HandleTypeDef *mCAN, uint32_t TxMailbox);
uint32_t MID_CAN_GetTxMailboxesFreeLevel( CAN_HandleTypeDef *mCAN);
uint32_t MID_CAN_IsTxMessagePending( CAN_HandleTypeDef *mCAN, uint32_t TxMailbox);

MID_StatusTypeDef MID_CAN_GetRxMessage(CAN_HandleTypeDef* mCAN, uint32_t RxFIFO, CAN_RxHeaderTypeDef *pHeader, uint8_t aData[]);
uint32_t MID_CAN_GetRxFifoFillLevel( CAN_HandleTypeDef *mCAN, uint32_t RxFIFO);

///@}


///@{
/**
 * @name Interrupt management
 *   		
 */ 
MID_StatusTypeDef MID_CAN_ActivateNotification( CAN_HandleTypeDef *mCAN, uint32_t ActiveITs);
MID_StatusTypeDef MID_CAN_DeactivateNotification( CAN_HandleTypeDef *mCAN, uint32_t InactiveITs);

void MID_CAN_IRQHandler( CAN_HandleTypeDef *mCAN);
///@}

///@{
/**
 * @name Call back function
 *   		
 */ 
void MID_CAN_TxMailbox0CompleteCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_TxMailbox0FailCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_TxMailbox1CompleteCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_TxMailbox1FailCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_TxMailbox2CompleteCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_TxMailbox2FailCallback(CAN_HandleTypeDef *mCAN);

void MID_CAN_RxFifo0FullCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_RxFifo0MsgPendingCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_RxFifo1FullCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_RxFifo1MsgPendingCallback(CAN_HandleTypeDef *mCAN);

void MID_CAN_WakeupFromRxMsgCallback(CAN_HandleTypeDef *mCAN);

void MID_CAN_ReBelowWarnLimitCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_ReEnterErrorActiveCallback(CAN_HandleTypeDef *mCAN);
void MID_CAN_ErrorCallback(CAN_HandleTypeDef *mCAN);

///@}

///@{
/**
 * @name Peripheral state and error functions
 *   		
 */ 
MID_CAN_StateTypeDef MID_CAN_GetState(CAN_HandleTypeDef *mCAN);
uint32_t MID_CAN_GetError( CAN_HandleTypeDef *mCAN);
MID_StatusTypeDef MID_CAN_ResetError( CAN_HandleTypeDef *mCAN);

///@}

/* Exported variables --------------------------------------------------------*/


/* Private Macro -----------------------------------------------*/
#define IS_CAN_ALL_INSTANCE(INSTANCE) (((INSTANCE) == CAN0))

#define IS_CAN_MODE(MODE) (((MODE) == CAN_MODE_NORMAL) || \
                           ((MODE) == CAN_MODE_LOOPBACK)|| \
                           ((MODE) == CAN_MODE_SILENT) || \
                           ((MODE) == CAN_MODE_SILENT_LOOPBACK))


#define IS_CAN_SJW(SJW) (((SJW) == CAN_SJW_1TQ) || ((SJW) == CAN_SJW_2TQ) || \
                         ((SJW) == CAN_SJW_3TQ) || ((SJW) == CAN_SJW_4TQ))
                         
#define IS_CAN_BS1(BS1) (((BS1) == CAN_BS1_1TQ) || ((BS1) == CAN_BS1_2TQ) || \
                         ((BS1) == CAN_BS1_3TQ) || ((BS1) == CAN_BS1_4TQ) || \
                         ((BS1) == CAN_BS1_5TQ) || ((BS1) == CAN_BS1_6TQ) || \
                         ((BS1) == CAN_BS1_7TQ) || ((BS1) == CAN_BS1_8TQ) || \
                         ((BS1) == CAN_BS1_9TQ) || ((BS1) == CAN_BS1_10TQ)|| \
                         ((BS1) == CAN_BS1_11TQ)|| ((BS1) == CAN_BS1_12TQ)|| \
                         ((BS1) == CAN_BS1_13TQ)|| ((BS1) == CAN_BS1_14TQ)|| \
                         ((BS1) == CAN_BS1_15TQ)|| ((BS1) == CAN_BS1_16TQ))
                         
#define IS_CAN_BS2(BS2) (((BS2) == CAN_BS2_2TQ) || \
                         ((BS2) == CAN_BS2_3TQ) || ((BS2) == CAN_BS2_4TQ) || \
                         ((BS2) == CAN_BS2_5TQ) || ((BS2) == CAN_BS2_6TQ) || \
                         ((BS2) == CAN_BS2_7TQ) || ((BS2) == CAN_BS2_8TQ))
                         
#define IS_CAN_PRESCALER(PRESCALER) (((PRESCALER) >= 2U) && ((PRESCALER) <= 1024U))


#define IS_CAN_FILTER_MODE(MODE) (((MODE) == CAN_FILTERMODE_IDMASK) || \
                                  ((MODE) == CAN_FILTERMODE_IDLIST))
                                  
#define IS_CAN_FILTER_SCALE(SCALE) (((SCALE) == CAN_FILTERSCALE_16BIT) || \
                                    ((SCALE) == CAN_FILTERSCALE_32BIT))
                                    
#define IS_CAN_FILTER_FIFO(FIFO) (((FIFO) == CAN_FILTER_FIFO0) || \
                                  ((FIFO) == CAN_FILTER_FIFO1))
                                  
#define IS_CAN_FILTER_ACTIVATION(ACTIVATION) (((ACTIVATION) == CAN_FILTER_DISABLE) || \
                                              ((ACTIVATION) == CAN_FILTER_ENABLE))   

#define IS_CAN_FILTER_BANK(BANK) ((BANK) <= 5U)

#define IS_CAN_TX_MAILBOX_LIST(TRANSMITMAILBOX)  ((TRANSMITMAILBOX) <= (CAN_TX_MAILBOX0 | CAN_TX_MAILBOX1 | CAN_TX_MAILBOX2))


#define IS_CAN_RX_FIFO(FIFO) (((FIFO)== CAN_RX_FIFO0) || ((FIFO)== CAN_RX_FIFO1))

#define IS_CAN_IT(IT) ((IT) <= (CAN_IT_TX_MAILBOX0_EMPTY    | CAN_IT_TX_MAILBOX1_EMPTY |\
                                CAN_IT_TX_MAILBOX2_EMPTY    | CAN_IT_RX_FIFO0_OVERRUN  |\
                                CAN_IT_RX_FIFO0_FULL        | CAN_IT_RX_FIFO0_PENDING  |\
                                CAN_IT_RX_FIFO0_MSG_PENDING | CAN_IT_RX_FIFO1_OVERRUN  |\
                                CAN_IT_RX_FIFO1_FULL        | CAN_IT_RX_FIFO1_PENDING  |\
                                CAN_IT_RX_FIFO1_MSG_PENDING | CAN_IT_WAKEUP            |\
                                CAN_IT_ERROR_WARNING        | CAN_IT_ERROR_PASSIVE     |\
                                CAN_IT_BUSOFF               | CAN_IT_BUS_ERROR         |\
                                CAN_IT_ALST                 | CAN_IT_IEA))


#endif





