
/**
 ******************************************************************************
 *
 * @file        BSP_I2C_SHT31.c
 * @brief       Read SHT31 humidity and temperature Code. 
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2024/03/14
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/*==============================================================================
            Adafruit SHT31-D Temperature & Humidity Sensor Breakout
            
Power Pins: 
    Vin - this is the power pin. The chip can use 2.5-5VDC for power. To power the
          board, give it the same power as the logic level of your microcontroller.
    GND - common ground for power and logic.

I2C Logic pins:
    SCL - I2C clock pin, connect to your microcontrollers I2C clock line. This pin has
          a 10K pullup resistor to Vin.
    SDA - I2C data pin, connect to your microcontrollers I2C data line. This pin has a
          10K pullup resistor to Vin.
      
Other Pins:
    ADR - This is the I2C address selection pin. This pin has a 10K pull down resistor
          to make the default I2C address 0x44. You can tie this pin to Vin to make the
          address 0x45.
    RST - Hardware reset pint. Has a 10K pullup on it to make the chip active by
          default. Connect to ground to do a hardware reset!
    ALR - Alert/Interrupt output. You can set up the sensor to alert you when an
          event has occured. Check the datasheet for how you can set up the alerts.
      

                                 User NOTES
How To use this function:
-----------------------
   + SHT31_I2C_Init() - Initial I2C peripheral to access SHT31 sensor.
   + SHT31_readTemperature() - Convert enviroment Temperature by SHT31.
   + SHT31_readHumidity() - Get enviroment Humidity by SHT31.
   + SHT31_Read() - Get SHT31 sensor data.


Example:
-----------------------
    SHT31_I2C_Init(0x44);

    // Get SHT31 sensor data.
    SHT31_Read();

    // Get Temperature
    Temperature = SHT31_readTemperature(Celcius);

    // Get Humidity
    Humidity = SHT31_readHumidity(Celcius);

Driver architecture:
--------------------
   + MG32_GPIO_DRV
   + MG32_I2C_DRV

Known Limitations:
------------------
   1- Run in 48MHz. (CSC control).
   2- GPIO PB2/PB3 must be I2C interface pin.
==============================================================================*/

/*==============================================================================
                                 Require parameter
Require module : CSC / GPIO / I2C

CSC Module :
    Clock source                : CK_IHRCO (12M Hz)
    Clock frequency             : CK_MAIN select CK_PLLO (PLL to 48MHz)
    CK_APB frequency            : 48MHz
    CK_AHB frequency            : 48MHz
    Clock enable in 'On Mode'   : GPIOB / TM20

GPIO pin configuration : 
    Pin /   IO mode   /   AFS
    ---   ------------  --------
    PB3 / ODO+Pull up / I2C0_SCL
    PB2 / ODO+Pull up / I2C0_SDA
                                           
I2C Module :                   
    I2C protocol 
    
==============================================================================*/



/* Includes ------------------------------------------------------------------*/
#include "MG32_I2C_DRV.h"
#include "BSP_I2C_SHT31.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

//SHT31_I2CDef SHT31_I2C;
#if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
    #pragma clang diagnostic push
    #pragma clang diagnostic ignored "-Wmissing-variable-declarations"
        SHT31_I2CDef SHT31_I2C;
    #pragma clang diagnostic pop
#else
    SHT31_I2CDef SHT31_I2C;
#endif

/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
//void SHT31_I2C_Init(void);

/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief	    SHT31 I2C bus initial with SHT31's address.
 * @param[in]   Addr : SHT31 address of I2C. 
 *              User can connect ADR pin High or Low to choice 0x45 or 0x44.
 * @return		None.
 *******************************************************************************
 */
boolean SHT31_I2C_Init(uint8_t Addr)
{
    // ------------------------------------------------------------------------
    // I2C initital
    // ------------------------------------------------------------------------
    // clock initial
    I2C_SetClockSource(I2C0, I2C_CLK_SRC_PROC);         //    __I2C_SetClockSource(SI2C->Instance, I2C_CLK_SRC_PROC);
    I2C_SetClockPrescaler(I2C0, I2C_CLK_PSC_8);         //    __I2C_SetClockPrescaler(SI2C->Instance, lI2C_Pre - 1);
    I2C_SetClockDivider(I2C0, I2C_CLK_DIV_16);          //    __I2C_SetClockDivider(SI2C->Instance, (lI2C_DIV << 4));
    I2C_SetSCLLowTime(I2C0, 15);                        //    __I2C_SetSCLLowTime(SI2C->Instance, (lI2C_HT_LT >> 1) - 1);
    I2C_SetSCLHighTime(I2C0, 15);                       //    __I2C_SetSCLHighTime(SI2C->Instance, (lI2C_HT_LT - (lI2C_HT_LT >> 1)));
    
    I2C_SetPreDriveTime(I2C0, I2C_PDRV_0T);             //    __I2C_SetPreDriveTime(SI2C->Instance, I2C_PDRV_0T);

    I2C_GeneralCallAddress_Cmd(I2C0, DISABLE);                                          //    __I2C_GeneralCallAddress_Disable(SI2C->Instance);
    I2C_SlaveAddressDetect_Cmd(I2C0, (I2C_SADR_0 | I2C_SADR_1 | I2C_SADR_2), DISABLE);  //    __I2C_SlaveAddressDetect_Disable(SI2C->Instance, (I2C_SADR_0 | I2C_SADR_1 | I2C_SADR_2));

    I2C_SetSlaveAddress(I2C0, I2C_SADR_1, 0x44);        //    __I2C_SetSlaveAddress1(SI2C->Instance, SI2C->Init.OwnAddress1);
    I2C_SetSlaveAddress(I2C0, I2C_SADR_2, 0x44);        //    __I2C_SetSlaveAddress2(SI2C->Instance, SI2C->Init.OwnAddress2);

    I2C_SetSlaveAddress1Mask(I2C0, 0xFE);               //    __I2C_SetSlaveAddress1Mask(SI2C->Instance, 0xFE); // Compare Address Bit All 

    //===== I2C Timeout Config =====//
    (I2C0->TMOUT.W &= (~I2C_TMOUT_TMO_EN_mask_w));                                      //    __I2C_TMO_Disable(SI2C->Instance);
    I2C0->CLK.W = (((I2C0->CLK.W) & (~I2C_TMO_CKS_MASK)) | (I2C_TMO_CKS_DIV64));        //    __I2C_SetTimeOutClockSource(SI2C->Instance, I2C_TMO_CKS_DIV64);
    I2C0->TMOUT.W = (((I2C0->TMOUT.W) & (~I2C_TMO_MDS_MASK)) | (I2C_TMO_MDS_GENERAL));  //    __I2C_SetTimeOutDetectionMode(SI2C->Instance, I2C_TMO_MDS_GENERAL);
    I2C0->TMOUT.B[1] = (I2C_TMO_MDS_GENERAL);                                           //    __I2C_SetTimeOutCount(SI2C->Instance, I2C_TMO_MDS_GENERAL);

    //===== I2C Enable =====//
    I2C_Cmd(I2C0, ENABLE);

    I2C_ClearFlag(I2C0, (I2C_FLAG_NACKF | I2C_FLAG_ALOSF | I2C_FLAG_BERRF | I2C_FLAG_ROVRF | I2C_FLAG_TOVRF | I2C_FLAG_STOPF | I2C_FLAG_RSTRF));
    
    SHT31_I2C.I2C_CMD_STATE = 0x2C10;
    
    // Check SHT31 I2C address
    if ((Addr == 0x44) || (Addr == 0x45))
    {
        SHT31_I2C.ADDR          = Addr;
        return TRUE;
    }
    else
    {
        return FALSE;
    }

    
}

/**
 *******************************************************************************
 * @brief	    Read temperature
 * @param[in]   S : Scale. Boolean value.
 *  @arg\b	    Fahrenheit : One shot 
 *	@arg\b	    Celcius : scan mode
 * @param[in]   force : true if in force mode.
 * @return		emperature value in selected scale
 *******************************************************************************
 */
float SHT31_readTemperature(boolean S) 
{
    ctype HT_Convert;                           // HT_Convert : Combine Humidity / Temperature
    
    // ----------------------------------------------------------------
    // Temperature conversion formula (result in 'C' & 'F'):
    // ----------------------------------------------------------------
    HT_Convert.H[0] = (uint16_t) (SHT31_I2C.TemperatureVal);
    
    if (S == Fahrenheit) 
        return (float) ((315 * HT_Convert.H[0] / 65535) - 49.0);
    
    return (float) ((175 * HT_Convert.H[0] / 65535.0) - 45.0);
}


/**
 *******************************************************************************
 * @brief	    Read Humidity.
 * @return		float value - humidity in percent
 *******************************************************************************
 */
float SHT31_readHumidity(void) 
{
    ctype HT_Convert;                           // HT_Convert : Combine Humidity / Temperature
    
    // ------------------------------------------------------------------------
    // Relative humidity conversion formula (result in %RH):
    // ------------------------------------------------------------------------    
    HT_Convert.H[0] = (uint16_t) (SHT31_I2C.HumidityVal);

    
    return (float) (100 * HT_Convert.H[0] / 65535.0);
}


/**
 *******************************************************************************
 * @brief	    Readout of Measurement Results
 * @return		float value
 *******************************************************************************
 */
boolean SHT31_Read(void)
{
    uint8_t I2C_RData0, I2C_RData1, I2C_RCRC;
    
    /*
    ---------------------------------------------------------------------------         
    [S]->[I2C Address][W][ACK] -> [Command MSB][ACK] -> [Command LSB][ACK] -> [P]
    
    [S]->[I2C Address][R][ACK] -> ... SCL pulled low ... ->
    [Temperature MSB][ACK] -> [Temperature LSB][ACK] -> [CRC][ACK]->
    [Humidity MSB][ACK] -> [Humidity LSB][ACK] -> [CRC][NAK] -> [P]
    ---------------------------------------------------------------------------             
    */
    
    // ------------------------------------------------------------------------
    // START [S]
    I2C_BusStart();
    
    // SLA+W [I2C Address][W]
    I2C_Transmiter((uint8_t) ((SHT31_I2C.ADDR << 1) | I2C_ADDR_WR));
    
    // COMMAND (MSB+LSB)
    I2C_Transmiter((uint8_t) (SHT31_I2C.I2C_CMD_STATE >> 8));
    I2C_Transmiter((uint8_t) (SHT31_I2C.I2C_CMD_STATE & 0x00FF));
    
    // STOP [P]
    I2C_Master_BusStop();

    // ------------------------------------------------------------------------
    // START [S]
    I2C_BusStart();
    
    // SLA+W [I2C Address][R]
    I2C_Transmiter((uint8_t) ((SHT31_I2C.ADDR << 1) | I2C_ADDR_RD));
    
    // Get Temperature (MSB / LSB / CRC)
    I2C_Receive(&I2C_RData0, I2C_ACK);
    I2C_Receive(&I2C_RData1, I2C_ACK);
    I2C_Receive(&I2C_RCRC, I2C_ACK);
    // preserve 
    SHT31_I2C.TemperatureVal = (uint16_t) ((I2C_RData0 << 8) | I2C_RData1);
    SHT31_I2C.Temperature_CRC = I2C_RCRC;
    
    // Get Humidiry (MSB / LSB / CRC)
    I2C_Receive(&I2C_RData0, I2C_ACK);
    I2C_Receive(&I2C_RData1, I2C_ACK);
    I2C_Receive(&I2C_RCRC, I2C_NACK);
    // preserve 
    SHT31_I2C.HumidityVal = (uint16_t) ((I2C_RData0 << 8) | I2C_RData1);
    SHT31_I2C.Humidity_CRC = I2C_RCRC;
    
    // STOP [P]
    I2C_Master_BusStop();
    
    return TRUE;
    
}

/**
 *******************************************************************************
 * @brief       I2C Output Bus Start
 * @details     Output Bus Start
 * @return      TRUE : SUCCESS.
 *              FALSE : FAILURE
 *******************************************************************************
 */
boolean I2C_BusStart(void)
{
    uint8_t lEventCode;

    // ------------------------------------------------------------------------
    lEventCode = (I2C0->STA2.B[0] & 0xF8);
    if(lEventCode != 0xF8)                      // Bus Idle    
    {
        // Wait I2C bus event
        while((I2C0->STA2.B[1] & I2C_STA2_EVENTF2_mask_b1) == 0);
        lEventCode = (I2C0->STA2.B[0] & 0xF8);  // Update lEventCode
    }

    // ------------------------------------------------------------------------
    switch (lEventCode)
    {
    case 0x08:                                  // A START condition has been transmitted
    case 0x10:                                  // Repeated start condition
        return TRUE;

    default:

    case 0xF8:                                  // Bus Idle    
        Set_STOP_CLR(I2C0);
        Set_START_SET(I2C0);
    
        // Clear I2C_EVENTF 
        I2C0->STA2.B[1] = I2C_STA2_EVENTF2_mask_b1;
        while((I2C0->STA2.B[1] & 0x01) == 0);   // Wait complete   
    
        return TRUE;
    }
}

/**
 *******************************************************************************
 * @brief       I2C Master Output Bus Stop
 * @details     Master Output Bus Stop
 * @return      TRUE : SUCCESS.
 *              FALSE : FAILURE
 *******************************************************************************
 */
boolean I2C_Master_BusStop(void)
{
    uint8_t lEventCode;

    // wait I2C bus event
    while((I2C0->STA2.B[1] & I2C_STA2_EVENTF2_mask_b1) == 0);
    lEventCode = (I2C0->STA2.B[0] & 0xF8);      // Update lEventCode

    switch (lEventCode)
    {
    case 0xF8:                                  // Bus Idle    
    case 0xA0:                                  // A STOP or repeated START has been received while will addressed as SLV/REC
        return TRUE;

    default:
        Set_STA_STO_AA_010(I2C0);               // Send 'STOP'
    
        // Clear I2C bus event
        I2C0->STA2.B[1] = I2C_STA2_EVENTF2_mask_b1;
        while(((I2C0)->CR2.B[0] & I2C_CR2_STO_mask_b0) != 0);
        return TRUE;
    }
    
    return FALSE;
}


/**
 *******************************************************************************
 * @brief       I2C Transmiter Data
 * @details     Transmiter Data
 * @param[in]   TxData : Value of Output Data
 * @return      TRUE : SUCCESS.
 *              FALSE : FAILURE
 *******************************************************************************
 */
boolean I2C_Transmiter(uint8_t TxData)
{
    uint8_t lEventCode;

    // wait I2C bus event
    while((I2C0->STA2.B[1] & I2C_STA2_EVENTF2_mask_b1) == 0);
    lEventCode = (I2C0->STA2.B[0] & 0xF8);      // Update lEventCode

    switch (lEventCode)
    {
    case 0x08:                                  // A START condition has been transmitted
    case 0x10:                                  // Repeated start condition
        Set_START_CLR(I2C0);

        #if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
            __attribute__((fallthrough));
        #endif

    case 0x18:                                  // MT SLA+W sent and ACK received
    case 0x20:                                  // MT SLA+W sent NACK received 
    case 0x28:                                  // MT DATA sent and ACK received
    case 0x30:                                  // MT DATA sent NACK received
    case 0xB8:                                  // Data byte in SIDAT has been transmitted ACK has been received
        Set_ASSERT_ACKNOWLEDGE_SET(I2C0);
        I2C_SendSBUF(I2C0, TxData);

        #if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
            __attribute__((fallthrough));
        #endif

    case 0xA8:                                  // Own SLA+R has bee Received ACK has been returned
    case 0xB0:                                  // Arbitration lost in SLA+R/W as master,
                                                // Own SLA+R has been Received ACK has bee returned
        I2C0->STA2.B[1] = I2C_STA2_EVENTF2_mask_b1;
        return TRUE;

    case 0xC0:                                  // Data byte or Last data byte in SIDAT has been transmitted Not ACK has been received
    default:
        return FALSE;
    }
}

/**
 *******************************************************************************
 * @brief       I2C Receive Data
 * @details     Receive Data
 * @param[out]  RxData : Value of Receive Data.
 * @param[in]   Acknowledge : Freeback ACK / NACK to Transmiter.
 *          \n  0 : I2C_NACK
 *          \n  1 : I2C_ACK
 * @return      TRUE : SUCCESS.
 *              FALSE : FAILURE
 *******************************************************************************
 */
boolean I2C_Receive(uint8_t *RxData, uint8_t Acknowledge)
{
    uint8_t lEventCode;

    // wait I2C bus event
    while((I2C0->STA2.B[1] & I2C_STA2_EVENTF2_mask_b1) == 0);
    lEventCode = (I2C0->STA2.B[0] & 0xF8);      // Update lEventCode
    
    switch (lEventCode)
    {
    case 0x40:                                  // SLA+R sent and ACK received
    case 0x50:                                  // Data Received and ACK sent
    case 0x80:                                  // Data byte has been received ACK has been return
    case 0x90:                                  // Previously address with General Call address
                                                // Data byte has been received ACK has been return
        if(Acknowledge != 0)
            Set_ASSERT_ACKNOWLEDGE_SET(I2C0);
        else
            Set_ASSERT_ACKNOWLEDGE_CLR(I2C0);
        
        #if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
            __attribute__((fallthrough));
        #endif

    case 0x58:                                  // Data Received and NACK sent
    case 0x88:                                  // Data byte has been received Not ACK has been return
    case 0x98:                                  // Previously address with General Call address
                                                // Data byte has been received Not ACK has been return
        I2C0->STA2.B[1] = I2C_STA2_EVENTF2_mask_b1;
        while((I2C0->STA2.B[1] & I2C_STA2_EVENTF2_mask_b1) == 0);
        *RxData = I2C0->SBUF.B[0];
        return TRUE;

    case 0x60:                                  // Own SLA+W has bee Received ACK has been returned
    case 0x68:                                  // Arbitration lost in SLA+R/W as master,
                                                // Own SLA+W has been Received ACK has bee returned
    case 0x70:                                  // General Call address has been received ACK has been returned
    case 0x78:                                  // Arbitration lost in SLA+R/W as master,
                                                // General Call address has been received ACK has been returned
        I2C0->STA2.B[1] = I2C_STA2_EVENTF2_mask_b1;
        return TRUE;

    case 0xA0:                                  // A STOP or repeated START has been received while will addressed as SLV/REC
        I2C0->STA2.B[1] = I2C_STA2_EVENTF2_mask_b1;
        return FALSE;
    default:
        return FALSE;
    }
    
}



