/*********************************************************************
*	 Project: MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=24MHz, SysCLK=24MHz
*	 Description:
*			TWSI0 (SCL/P40,	SDA/P41) Master	interrupt
*			TWSI1 (SCL/P10,	SDA/P11) Master	interrupt
*	 Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		24000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)


#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	IO_TWI0_SCL		P40
#define	IO_TWI0_SDA		P41

#define	IO_TWI1_SCL		P10
#define	IO_TWI1_SDA		P11

#define	I2C_SLAVE_ADDR		0xA0

bit	bES0;
u8 LedTime;
u8 Cnt100;

u8 TWI0TestTime;
u8 TWI1TestTime;

u8 TWI0OvTime;
u8 TWI1OvTime;



enum{
	EE_IDLE=0,
	EE_SEND_START,
	EE_SEND_SLA,
	EE_WRITE_ADDR_H,
	EE_WRITE_ADDR_L,
	EE_SEND_RESTART,
	EE_WRITE_DATA,
	EE_READ_DATA,
	EE_SEND_STOP
}EE_STATUS;

#define	EE_SUCCESS		0x00

#define	DATA_LENGTH			 16
WordTypeDef	TWI0TestAddr;
WordTypeDef	TWI1TestAddr;

xdata u8 TWI0WriteBuf[DATA_LENGTH];
xdata u8 TWI0ReadBuf[DATA_LENGTH];
xdata u8 TWI1WriteBuf[DATA_LENGTH];
xdata u8 TWI1ReadBuf[DATA_LENGTH];

#define	TWI0_TEST_INTERVAL_TIME	(400/100)			// *100ms	TWI0 test interval time
#define	TWI1_TEST_INTERVAL_TIME	(700/100)			// *100ms	TWI1 test interval time

#define	IIC_OVTIME_MAX		20						// *1ms	IIC	overtime


#define	EE_MODE_WRITE		0
#define	EE_MODE_READ		1

typedef	struct
{
	u8 EEMode;
	u8 EEStatus;
	u8 EEDeviceAddr;
	u16	EEDataAddr;
	u8 *EEDataBuf;
	u8 EEFlag;
	u8 EEDataInx;
	u8 EEDataLength;
}EEPROM_PROC_STRUCT;

idata EEPROM_PROC_STRUCT EE_TWI0,EE_TWI1;


u8 rand;

/***********************************************************************************
*Function:		void INT_T0(void)
*Description:	T0 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_T0(void) interrupt	INT_VECTOR_T0
{
	TH0=TIMER_12T_1ms_TH;
	TL0=TIMER_12T_1ms_TL;
	if(LedTime!=0) LedTime--;
	if(TWI0OvTime!=0) TWI0OvTime--;
	if(TWI1OvTime!=0) TWI1OvTime--;
	Cnt100++;
	if(Cnt100>=100)
	{
		Cnt100=0;
		if(TWI0TestTime!=0)	TWI0TestTime--;
		if(TWI1TestTime!=0)	TWI1TestTime--;
	}
}

/***********************************************************************************
*Function:		void INT_TWI(void)
*Description:	TWI0 Interrupt handler

*Input:
*Output:
*************************************************************************************/
void INT_TWI0()	interrupt INT_VECTOR_TWI0
{
	_push_(SFRPI);
	SFR_SetPage(0);

	if(SISTA==0x08)
	{ // Tx:Start
		SICON=SICON&(~STA);
		SIDAT=EE_TWI0.EEDeviceAddr&0xFE;
		EE_TWI0.EEStatus=EE_SEND_SLA;
	}
	else if(SISTA==0x18)
	{ // Tx:SLA+W Rx:ACK
		// To send RegAddr high
		SICON=SICON|(AA);
		SIDAT=(u8)(EE_TWI0.EEDataAddr>>8);
		EE_TWI0.EEStatus=EE_WRITE_ADDR_H;
	}
	else if(SISTA==0x20)
	{ // Tx:SLA+W Rx:NACK
		EE_TWI0.EEFlag=EE_TWI0.EEStatus+0x80;
		EE_TWI0.EEStatus=EE_SEND_STOP;
		SICON=SICON|(STO);
	}
	else if(SISTA==0x28)
	{ // Tx:DAT	Rx:ACK
		if(EE_TWI0.EEStatus==EE_WRITE_ADDR_H)
		{
			// To send RegAddr LOW
			SICON=SICON|(AA);
			SIDAT=(u8)(EE_TWI0.EEDataAddr);
			EE_TWI0.EEStatus=EE_WRITE_ADDR_L;
		}
		else if(EE_TWI0.EEStatus==EE_WRITE_ADDR_L)
		{
			// Send	RegAddr	Done
			if(EE_TWI0.EEMode==EE_MODE_READ)
			{
				// Read	mode: To Tx	ReStart;
				SICON=SICON|(STA);
				EE_TWI0.EEStatus=EE_SEND_RESTART;
			}
			else
			{
				// Write mode: To Tx Data;
				SICON=SICON|(AA);
				EE_TWI0.EEStatus=EE_WRITE_DATA;
				EE_TWI0.EEDataInx=0;
				SIDAT=EE_TWI0.EEDataBuf[EE_TWI0.EEDataInx];
				EE_TWI0.EEDataInx++;
			}
		}
		else if(EE_TWI0.EEStatus==EE_WRITE_DATA)
		{
			if(EE_TWI0.EEDataInx>=EE_TWI0.EEDataLength)
			{
				SICON=SICON|(STO);
				EE_TWI0.EEFlag=	EE_SUCCESS;
				EE_TWI0.EEStatus=EE_SEND_STOP;
			}
			else
			{
				SICON=SICON|(AA);
				SIDAT=EE_TWI0.EEDataBuf[EE_TWI0.EEDataInx];
				EE_TWI0.EEDataInx++;
			}
		}
		else
		{
			SICON=SICON|(STO);
			EE_TWI0.EEFlag=EE_TWI0.EEStatus+0x80;
			EE_TWI0.EEStatus=EE_SEND_STOP;
		}
	}
	else if(SISTA==0x30)
	{ // Tx:DAT	Rx:NACK
		EE_TWI0.EEFlag=	EE_TWI0.EEStatus+0x80;
		if(EE_TWI0.EEStatus==EE_WRITE_DATA)
		{
			if(EE_TWI0.EEDataInx==EE_TWI0.EEDataLength)
			{
				EE_TWI0.EEFlag=	EE_SUCCESS;
			}
		}
		SICON=SICON|(STO);
		EE_TWI0.EEStatus=EE_SEND_STOP;
	}
	else if(SISTA==0x10)
	{ // Tx:ReStart
		SICON=SICON&(~STA);
		SIDAT=EE_TWI0.EEDeviceAddr|0x01;
		EE_TWI0.EEStatus=EE_SEND_SLA;
	}
	else if(SISTA==0x40)
	{ // Tx:SLA+R Rx:ACK
		SICON=SICON|(AA);
		EE_TWI0.EEDataInx=0;
		EE_TWI0.EEStatus=EE_READ_DATA;
	}
	else if(SISTA==0x48)
	{ // Tx:SLA+R Rx:NACK
		EE_TWI0.EEFlag=EE_TWI0.EEStatus+0x80;
		EE_TWI0.EEStatus=EE_SEND_STOP;
		SICON=SICON|(STO);
	}
	else if(SISTA==0x50)
	{ // Rx:DAT	Tx:ACK
		if(EE_TWI0.EEStatus==EE_READ_DATA)
		{
			if(EE_TWI0.EEDataInx>=EE_TWI0.EEDataLength)
			{
				SICON=SICON|(STO);
				EE_TWI0.EEFlag=	EE_SUCCESS;
				EE_TWI0.EEStatus=EE_SEND_STOP;
			}
			else
			{
				EE_TWI0.EEDataBuf[EE_TWI0.EEDataInx]=SIDAT;
				EE_TWI0.EEDataInx++;
				if(EE_TWI0.EEDataInx>=EE_TWI0.EEDataLength-1)
				{
					SICON=SICON&(~AA);
				}
				else
				{
					SICON=SICON|(AA);
				}
			}
		}
		else
		{
			EE_TWI0.EEFlag=EE_TWI0.EEStatus+0x80;
			EE_TWI0.EEStatus=EE_SEND_STOP;
			SICON=SICON|(STO);
		}
	}
	else if(SISTA==0x58)
	{ // Rx:DAT	Tx:NACK
		EE_TWI0.EEFlag=	EE_TWI0.EEStatus+0x80;
		if(EE_TWI0.EEStatus==EE_READ_DATA)
		{
			EE_TWI0.EEDataBuf[EE_TWI0.EEDataInx]=SIDAT;
			EE_TWI0.EEDataInx++;
			if(EE_TWI0.EEDataInx==EE_TWI0.EEDataLength)
			{
				EE_TWI0.EEFlag=	EE_SUCCESS;
			}
		}
		SICON=SICON|(STO);
		EE_TWI0.EEStatus=EE_SEND_STOP;
	}
	TWI0OvTime=IIC_OVTIME_MAX;
	SICON=SICON&(~SI);
	_pop_(SFRPI);
}


/***********************************************************************************
*Function:		void INT_TWI1(void)
*Description:	TWI1 Interrupt handler

*Input:
*Output:
*************************************************************************************/
void INT_TWI1()	interrupt INT_VECTOR_TWI1
{
	_push_(SFRPI);
	SFR_SetPage(1);

	if(SI1STA==0x08)
	{ // Tx:Start
		SI1CON=SI1CON&(~STA1);
		SI1DAT=EE_TWI1.EEDeviceAddr&0xFE;
		EE_TWI1.EEStatus=EE_SEND_SLA;
	}
	else if(SI1STA==0x18)
	{ // Tx:SLA+W Rx:ACK
		// To send RegAddr high
		SI1CON=SI1CON|(AA1);
		SI1DAT=(u8)(EE_TWI1.EEDataAddr>>8);
		EE_TWI1.EEStatus=EE_WRITE_ADDR_H;
	}
	else if(SI1STA==0x20)
	{ // Tx:SLA+W Rx:NACK
		EE_TWI1.EEFlag=EE_TWI1.EEStatus+0x80;
		EE_TWI1.EEStatus=EE_SEND_STOP;
		SI1CON=SI1CON|(STO1);
	}
	else if(SI1STA==0x28)
	{ // Tx:DAT	Rx:ACK
		if(EE_TWI1.EEStatus==EE_WRITE_ADDR_H)
		{
			// To send RegAddr LOW
			SI1CON=SI1CON|(AA1);
			SI1DAT=(u8)(EE_TWI1.EEDataAddr);
			EE_TWI1.EEStatus=EE_WRITE_ADDR_L;
		}
		else if(EE_TWI1.EEStatus==EE_WRITE_ADDR_L)
		{
			// Send	RegAddr	Done
			if(EE_TWI1.EEMode==EE_MODE_READ)
			{
				// Read	mode: To Tx	ReStart;
				SI1CON=SI1CON|(STA1);
				EE_TWI1.EEStatus=EE_SEND_RESTART;
			}
			else
			{
				// Write mode: To Tx Data;
				SI1CON=SI1CON|(AA1);
				EE_TWI1.EEStatus=EE_WRITE_DATA;
				EE_TWI1.EEDataInx=0;
				SI1DAT=EE_TWI1.EEDataBuf[EE_TWI1.EEDataInx];
				EE_TWI1.EEDataInx++;
			}
		}
		else if(EE_TWI1.EEStatus==EE_WRITE_DATA)
		{
			if(EE_TWI1.EEDataInx>=EE_TWI1.EEDataLength)
			{
				SI1CON=SI1CON|(STO1);
				EE_TWI1.EEFlag=	EE_SUCCESS;
				EE_TWI1.EEStatus=EE_SEND_STOP;
			}
			else
			{
				SI1CON=SI1CON|(AA1);
				SI1DAT=EE_TWI1.EEDataBuf[EE_TWI1.EEDataInx];
				EE_TWI1.EEDataInx++;
			}
		}
		else
		{
			SI1CON=SI1CON|(STO1);
			EE_TWI1.EEFlag=EE_TWI1.EEStatus+0x80;
			EE_TWI1.EEStatus=EE_SEND_STOP;
		}
	}
	else if(SI1STA==0x30)
	{ // Tx:DAT	Rx:NACK
		EE_TWI1.EEFlag=	EE_TWI1.EEStatus+0x80;
		if(EE_TWI1.EEStatus==EE_WRITE_DATA)
		{
			if(EE_TWI1.EEDataInx==EE_TWI1.EEDataLength)
			{
				EE_TWI1.EEFlag=	EE_SUCCESS;
			}
		}
		SI1CON=SI1CON|(STO1);
		EE_TWI1.EEStatus=EE_SEND_STOP;
	}
	else if(SI1STA==0x10)
	{ // Tx:ReStart
		if(EE_TWI1.EEMode==EE_MODE_READ)
		{
			SI1CON=SI1CON&(~STA1);
			SI1DAT=EE_TWI1.EEDeviceAddr|0x01;
			EE_TWI1.EEStatus=EE_SEND_SLA;
		}
		else
		{
			EE_TWI1.EEFlag=EE_TWI1.EEStatus+0x80;
			EE_TWI1.EEStatus=EE_SEND_STOP;
			SI1CON=SI1CON|(STO1);
		}
	}
	else if(SI1STA==0x40)
	{ // Tx:SLA+R Rx:ACK
		SI1CON=SI1CON|(AA1);
		EE_TWI1.EEDataInx=0;
		EE_TWI1.EEStatus=EE_READ_DATA;
	}
	else if(SI1STA==0x48)
	{ // Tx:SLA+R Rx:NACK
		EE_TWI1.EEFlag=EE_TWI1.EEStatus+0x80;
		EE_TWI1.EEStatus=EE_SEND_STOP;
		SI1CON=SI1CON|(STO1);
	}
	else if(SI1STA==0x50)
	{ // Rx:DAT	Tx:ACK
		if(EE_TWI1.EEStatus==EE_READ_DATA)
		{
			if(EE_TWI1.EEDataInx>=EE_TWI1.EEDataLength)
			{
				SI1CON=SI1CON|(STO1);
				EE_TWI1.EEFlag=	EE_SUCCESS;
				EE_TWI1.EEStatus=EE_SEND_STOP;
			}
			else
			{
				EE_TWI1.EEDataBuf[EE_TWI1.EEDataInx]=SI1DAT;
				EE_TWI1.EEDataInx++;
				if(EE_TWI1.EEDataInx>=EE_TWI1.EEDataLength-1)
				{
					SI1CON=SI1CON&(~AA1);
				}
				else
				{
					SI1CON=SI1CON|(AA1);
				}
			}
		}
		else
		{
			EE_TWI1.EEFlag=EE_TWI1.EEStatus+0x80;
			EE_TWI1.EEStatus=EE_SEND_STOP;
			SI1CON=SI1CON|(STO1);
		}
	}
	else if(SI1STA==0x58)
	{ // Rx:DAT	Tx:NACK
		EE_TWI1.EEFlag=	EE_TWI1.EEStatus+0x80;
		if(EE_TWI1.EEStatus==EE_READ_DATA)
		{
			EE_TWI1.EEDataBuf[EE_TWI1.EEDataInx]=SI1DAT;
			EE_TWI1.EEDataInx++;
			if(EE_TWI1.EEDataInx==EE_TWI1.EEDataLength)
			{
				EE_TWI1.EEFlag=	EE_SUCCESS;
			}
		}
		SI1CON=SI1CON|(STO1);
		EE_TWI1.EEStatus=EE_SEND_STOP;
	}
	TWI1OvTime=IIC_OVTIME_MAX;
	SI1CON=SI1CON&(~SI1);
	_pop_(SFRPI);
}

/***********************************************************************************
*Function:		void SendByte(u8 tByte)
*Description:	send byte
*Input:			u8 tByte: the data to be send
*Output:
*************************************************************************************/
void SendByte(u8 tByte)
{
	bES0=ES0;
	ES0=0;
	S0BUF=tByte;
	while(TI0==0);
	TI0=0;
	ES0=bES0;
}

/***********************************************************************************
*Function:		void SendStr(BYTE* PStr)
*Description:	send string
*Input:			u8*	PStr:the string	to be send
*Output:
*************************************************************************************/
void SendStr(u8* PStr)
{
	while(*PStr	!= 0)
	{
		SendByte(*PStr);
		PStr ++;
	}
}

/***********************************************************************************
*Function:		void SendHex(u8	ToSend)
*Description:	send data for HEX
*Input:			u8 ToSend: the data	to be send
*Output:
*************************************************************************************/
void SendHex(u8	ToSend)
{
	u8 x;
	x =	ToSend >> 4;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
	x= ToSend&0x0F;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
}

/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		void InitUart0_S0BRG()
*Description:	Initialize Uart0, The source of	baud rate was S0BRG
*Input:
*Output:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R.	source:	S0BRG
	UART0_SetS0BRGBaudRateX2();			// S0BRG x2
	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source: SYSCLK
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_24000000_1T);

	UART0_EnS0BRG();					// Enable S0BRG
}

/***********************************************************************************
*Function:		void InitTimer0(void)
*Description:	Initialize Timer0  Sysclk/12
*Input:
*Output:
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// Set T0 mode:16bit timer
	TM_SetT0Clock_SYSCLKDiv12();			// Set T0 clock:SYSCLK/12
	TM_SetT0Gate_Disable();

	TM_SetT0LowByte(TIMER_12T_1ms_TL);					// Set T0 low byte
	TM_SetT0HighByte(TIMER_12T_1ms_TH);					// Set T0 high byte

	TM_EnableT0();							// Enable T0
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// Set P22,P24,P26 as push-pull(LED)
	PORT_SetP3QuasiBi(BIT0|BIT1);					// Set P30,P31 as Quasi-Bi
	PORT_SetP1OpenDrainPu(BIT0|BIT1);				// Set P10,P11 as open-drain with pull-high(TWI1:SCL/SDA)
	PORT_SetP4OpenDrainPu(BIT0|BIT1);				// Set P40,P41 as open-drain with pull-high(TWI0:SCL/SDA)
}

/***********************************************************************************
*Function:	 void InitTWI0(void)
*Description:Initialize	TWI0
*Input:
*Output:
*************************************************************************************/
void InitTWI0()
{
	TWI0_Clear();
	IO_TWI0_SCL=0;
	IO_TWI0_SDA=1;
	TWI0_SetClock(TWI0_CLK_SYSCLK_DIV_60);				// SYSCLK/60=24MHz/60=400KHz
	TWI0_SetTWI0SCLTWI0SDA_P40P41();
	IO_TWI0_SCL=1;
	TWI0_Enable();
}

/***********************************************************************************
*Function:	 void InitTWI1(void)
*Description:Initialize	TWI1
*Input:
*Output:
*************************************************************************************/
void InitTWI1()
{
	TWI1_Clear();
	IO_TWI1_SCL=0;
	IO_TWI1_SDA=1;
	TWI1_SetClock((TWI0_CLK_SYSCLK_DIV_60));				// SYSCLK/60=24MHz/60=400KHz
	TWI1_SetTWI1SCLTWI1SDA_P10P11();
	IO_TWI1_SCL=1;
	TWI1_Enable();
}

/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:		void InitInterrupt()
*Description:	Initialize Interrupt
*Input:
*Output:
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER0();						// Enable Timer0 interrupt
	INT_EnTWI0();						// Enable TWI0 interrupt
	INT_EnTWI1();						// Enable TWI1 interrupt
	INT_EnAll();						// Enable global interrupt
}

/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitTWI0();
	InitTWI1();
	InitTimer0();
	InitUart0_S0BRG();
	InitInterrupt();
}

/***********************************************************************************
*Function:		 void TWI0_StartWriteBuf(u8	DevAddr,u16	RegStartAddr,u8	*pByte)
*Description:	 Send data to slave	through	TWI0
*Input:
*				u8 DevAddr:	Slave Device address
*				u16	RegStartAddr: Slave	data address
*				u8 *pBuf: Address of the buffer	to be send
*				u8 Len:	Data length
*Output:
*************************************************************************************/
void TWI0_StartWriteBuf(u8 DevAddr,u16 RegStartAddr,u8 *pBuf,u8	Len)
{

	EE_TWI0.EEMode=EE_MODE_WRITE;
	EE_TWI0.EEDataBuf=pBuf;
	EE_TWI0.EEDeviceAddr=DevAddr;
	EE_TWI0.EEDataAddr=RegStartAddr;
	EE_TWI0.EEDataLength=Len;
	EE_TWI0.EEStatus=EE_SEND_START;
	// start TWI0
	TWI0OvTime=IIC_OVTIME_MAX;
	TWI0_SendSTART();
}

/***********************************************************************************
*Function:		 void TWI0_StartReadBuf(u8 DevAddr,u16 RegStartAddr,u8 *pByte)
*Description:	 Read data from	slave through TWI0
*Input:
*				u8 DevAddr:	Slave Device address
*				u16	RegStartAddr: Slave	data address
*				u8 *pBuf: Address of the buffer	to be save
*				u8 Len:	Data length
*Output:
*************************************************************************************/
void TWI0_StartReadBuf(u8 DevAddr,u16 RegStartAddr,u8 *pBuf,u8 Len)
{
	EE_TWI0.EEMode=EE_MODE_READ;
	EE_TWI0.EEDataBuf=pBuf;
	EE_TWI0.EEDeviceAddr=DevAddr;
	EE_TWI0.EEDataAddr=RegStartAddr;
	EE_TWI0.EEDataLength=Len;
	EE_TWI0.EEStatus=EE_SEND_START;
	// start TWI0
	TWI0OvTime=IIC_OVTIME_MAX;
	TWI0_SendSTART();
}

/***********************************************************************************
*Function:		 void TWI1_StartWriteBuf(u8	DevAddr,u16	RegStartAddr,u8	*pByte)
*Description:	 Send data to slave	through	TWI1
*Input:
*				u8 DevAddr:	Slave Device address
*				u16	RegStartAddr: Slave	data address
*				u8 *pBuf: Address of the buffer	to be send
*				u8 Len:	Data length
*Output:
*************************************************************************************/
void TWI1_StartWriteBuf(u8 DevAddr,u16 RegStartAddr,u8 *pBuf,u8	Len)
{

	EE_TWI1.EEMode=EE_MODE_WRITE;
	EE_TWI1.EEDataBuf=pBuf;
	EE_TWI1.EEDeviceAddr=DevAddr;
	EE_TWI1.EEDataAddr=RegStartAddr;
	EE_TWI1.EEDataLength=Len;
	EE_TWI1.EEStatus=EE_SEND_START;
	// start TWI1
	TWI1OvTime=IIC_OVTIME_MAX;
	TWI1_SendSTART();
}

/***********************************************************************************
*Function:		 void TWI0_StartReadBuf(u8 DevAddr,u16 RegStartAddr,u8 *pByte)
*Description:	 Read data from	slave through TWI1
*Input:
*				u8 DevAddr:	Slave Device address
*				u16	RegStartAddr: Slave	data address
*				u8 *pBuf: Address of the buffer	to be save
*				u8 Len:	Data length
*Output:
*************************************************************************************/
void TWI1_StartReadBuf(u8 DevAddr,u16 RegStartAddr,u8 *pBuf,u8 Len)
{
	EE_TWI1.EEMode=EE_MODE_READ;
	EE_TWI1.EEDataBuf=pBuf;
	EE_TWI1.EEDeviceAddr=DevAddr;
	EE_TWI1.EEDataAddr=RegStartAddr;
	EE_TWI1.EEDataLength=Len;
	EE_TWI1.EEStatus=EE_SEND_START;
	// start TWI1
	TWI1OvTime=IIC_OVTIME_MAX;
	TWI1_SendSTART();
}


void TWI0_TestProc(void)
{
	u8 i;

	if(EE_TWI0.EEStatus==EE_IDLE)
	{
		if(TWI0TestTime==0)
		{
			if(EE_TWI0.EEMode!=EE_MODE_WRITE)
			{ // Writing test start
				SendStr("\nTWI0	Write EEPROM Start...");
				SendHex(TWI0TestAddr.B.BHigh);
				SendHex(TWI0TestAddr.B.BLow);
				i=0;
				for(i=0;i<DATA_LENGTH;i++)
				{
					TWI0WriteBuf[i]=i+rand;
					i++;
				}
				TWI0_StartWriteBuf(I2C_SLAVE_ADDR, TWI0TestAddr.W, &TWI0WriteBuf[0],DATA_LENGTH);
			}
			else
			{
				SendStr("\nTWI0	Read EEPROM	Start...");
				SendHex(TWI0TestAddr.B.BHigh);
				SendHex(TWI0TestAddr.B.BLow);
				TWI0_StartReadBuf(I2C_SLAVE_ADDR, TWI0TestAddr.W, &TWI0ReadBuf[0],DATA_LENGTH);
				TWI0TestAddr.W=TWI0TestAddr.W+DATA_LENGTH;
				if(TWI0TestAddr.W>=0x1000)
				{
					TWI0TestAddr.W=0;
				}
			}
		}
	}
	else if(EE_TWI0.EEStatus==EE_SEND_STOP)
	{
		TWI0TestTime=TWI0_TEST_INTERVAL_TIME;
		if(EE_TWI0.EEFlag != EE_SUCCESS)
		{
			SendStr("\nTWI0	Err: 0x");
			SendHex(EE_TWI0.EEFlag);
		}
		else
		{
			if(EE_TWI0.EEMode==EE_MODE_READ)
			{
				for(i=0;i<DATA_LENGTH;i++)
				{
					if(TWI0WriteBuf[i]!=TWI0ReadBuf[i])	break;
				}
				if(i>=DATA_LENGTH)
				{//	Check OK
					IO_LED_R=0;
					SendStr("\nTWI0	WR Success!!");
					DelayXms(50);
					IO_LED_R=1;
				}
				else
				{//	Check failed
					SendStr("\nTWI0	Verify Fail!! 0x");
					SendHex(i);
				}
			}
			else
			{
				SendStr("\nTWI0	Wrtie Done!");
				DelayXms(5);
				TWI0TestTime=0;
			}
		}
		EE_TWI0.EEStatus=EE_IDLE;
	}
	else
	{
		if(TWI0OvTime==0)
		{ //
			SendStr("\nTWI0	Fail-->TWI0_Reset..");
			InitTWI0();
			EE_TWI0.EEStatus=EE_IDLE;
			EE_TWI0.EEMode=EE_MODE_READ;
			TWI0TestTime=TWI0_TEST_INTERVAL_TIME;
		}
	}
}

void TWI1_TestProc(void)
{
	u8 i;

	if(EE_TWI1.EEStatus==EE_IDLE)
	{
		if(TWI1TestTime==0)
		{
			if(EE_TWI1.EEMode!=EE_MODE_WRITE)
			{ // Writing test start
				SendStr("\nTWI1	Write EEPROM Start...");
				SendHex(TWI1TestAddr.B.BHigh);
				SendHex(TWI1TestAddr.B.BLow);
				i=0;
				for(i=0;i<DATA_LENGTH;i++)
				{
					TWI1WriteBuf[i]=i+rand;
					i++;
				}
				TWI1_StartWriteBuf(I2C_SLAVE_ADDR, TWI1TestAddr.W, &TWI1WriteBuf[0],DATA_LENGTH);
			}
			else
			{
				SendStr("\nTWI1	Read EEPROM	Start...");
				SendHex(TWI1TestAddr.B.BHigh);
				SendHex(TWI1TestAddr.B.BLow);
				TWI1_StartReadBuf(I2C_SLAVE_ADDR, TWI1TestAddr.W, &TWI1ReadBuf[0],DATA_LENGTH);
				TWI1TestAddr.W=TWI1TestAddr.W+DATA_LENGTH;
				if(TWI1TestAddr.W>=0x1000)
				{
					TWI1TestAddr.W=0;
				}
			}
		}
	}
	else if(EE_TWI1.EEStatus==EE_SEND_STOP)
	{
		TWI1TestTime=TWI1_TEST_INTERVAL_TIME;
		if(EE_TWI1.EEFlag != EE_SUCCESS)
		{
			SendStr("\nTWI1	Err: 0x");
			SendHex(EE_TWI1.EEFlag);
		}
		else
		{
			if(EE_TWI1.EEMode==EE_MODE_READ)
			{
				for(i=0;i<DATA_LENGTH;i++)
				{
					if(TWI1WriteBuf[i]!=TWI1ReadBuf[i])	break;
				}
				if(i>=DATA_LENGTH)
				{//	Check OK
					IO_LED_Y=0;
					SendStr("\nTWI1	WR Success!!");
					DelayXms(50);
					IO_LED_Y=1;
				}
				else
				{//	Check failed
					SendStr("\nTWI1	Verify Fail!! 0x");
					SendHex(i);
				}
			}
			else
			{
				SendStr("\nTWI1	Wrtie Done!");
				DelayXms(5);
				TWI1TestTime=0;
			}
		}
		EE_TWI1.EEStatus=EE_IDLE;
	}
	else
	{
		if(TWI1OvTime==0)
		{ //
			SendStr("\nTWI1	Fail-->TWI1_Reset..");
			InitTWI1();
			EE_TWI1.EEStatus=EE_IDLE;
			EE_TWI1.EEMode=EE_MODE_READ;
			TWI1TestTime=TWI1_TEST_INTERVAL_TIME;
		}
	}
}

void main()
{
	InitSystem();

	SendStr("\nStart");
	IO_LED_G=0;IO_LED_Y=0;IO_LED_R=0;
	DelayXms(1000);
	IO_LED_G=1;IO_LED_Y=1;IO_LED_R=1;

	rand = 0x80;

	TWI0TestAddr.W=0x0000;
	EE_TWI0.EEStatus=EE_IDLE;
	EE_TWI0.EEMode=EE_MODE_READ;
	TWI0TestTime=TWI0_TEST_INTERVAL_TIME;

	TWI1TestAddr.W=0x0000;
	EE_TWI1.EEStatus=EE_IDLE;
	EE_TWI1.EEMode=EE_MODE_READ;
	TWI1TestTime=TWI1_TEST_INTERVAL_TIME;

	while(1)
	{
		rand++;
		if(LedTime==0)
		{
			IO_LED_G = ~IO_LED_G;
			LedTime=200;
		}
		TWI0_TestProc();
		TWI1_TestProc();

	}
}


