/*********************************************************************
*    Project: MGEQ1C064-DEMO
*			MGEQ1C064AD48 LQFP48 EV_V10 (TH249A)
*			CpuCLK=32MHz, SysCLK=32MHz
*    Description:
*			Split T2 mode: capture with auto-zero
*			Split T3 mode: capture with auto-zero 
*			Split T4 mode: capture with auto-zero 
*    Note:
*			
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MGEQ1C064.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MGEQ1C064.H"
#include ".\include\API_Uart_BRGRL_MGEQ1C064.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection: 
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	33170000,36000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		32000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define IO_TEST_0		P52
#define IO_TEST_1		P53
#define IO_TEST_2		P54
#define IO_TEST_3		P55
#define IO_TEST_4		P56
#define IO_TEST_5		P57

#define LED_G_0			P55
#define LED_R			P56
#define LED_G_1			P57

#define CAPTURE_BUF_SIZE	8
bit bT2Capture;
xdata u8 T2CaptureData[CAPTURE_BUF_SIZE];
u8 T2CaptureCnt;

bit bT3Capture;
xdata u8 T3CaptureData[CAPTURE_BUF_SIZE];
u8 T3CaptureCnt;

bit bT4Capture;
xdata u8 T4CaptureData[CAPTURE_BUF_SIZE];
u8 T4CaptureCnt;

/***********************************************************************************
*Function:   	void INT_T2(void)
*Description:	T2 Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_T2(void) interrupt INT_VECTOR_T2
{
	_push_(SFRPI);
	SFR_SetPage(0);			// TF2,EXF2 only Page-0
	if(TF2)
	{
		IO_TEST_0=!IO_TEST_0;
		TF2=0;
	}
	if(EXF2)
	{
		EXF2=0;
		if(bT2Capture==0)
		{
			T2CaptureData[T2CaptureCnt]=RCAP2H;
			T2CaptureCnt++;
			if(T2CaptureCnt>=CAPTURE_BUF_SIZE)
			{
				bT2Capture=TRUE;
			}
		}
	}
	if(RCLK_TF2L)
	{
		IO_TEST_3=!IO_TEST_3;
		RCLK_TF2L=0;
	}
	_pop_(SFRPI);
}

/***********************************************************************************
*Function:   	void INT_T3(void)
*Description:	T3 Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_T3(void) interrupt INT_VECTOR_T3
{
	_push_(SFRPI);
	SFR_SetPage(1);				// TF3,EXF3 only Page-1
	if(TF3)
	{
		TF3=0;
		SFR_SetPage(0);			// P5 only Page-0
		IO_TEST_1=!IO_TEST_1;
	}
	SFR_SetPage(1);				// TF3,EXF3 only Page-1
	if(EXF3)
	{
		EXF3=0;
		if(bT3Capture==0)
		{
			T3CaptureData[T3CaptureCnt]=RCAP3H;
			T3CaptureCnt++;
			if(T3CaptureCnt>=CAPTURE_BUF_SIZE)
			{
				bT3Capture=TRUE;
			}
		}
	}
	SFR_SetPage(1);				// TF3L only Page-1
	if(TF3L)
	{
		TF3L=0;
		SFR_SetPage(0); 		// P5 only Page-0
		IO_TEST_4=!IO_TEST_4;
	}
	_pop_(SFRPI);
}

/***********************************************************************************
*Function:   	void INT_T4(void)
*Description:	T4 Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_T4(void) interrupt INT_VECTOR_T4
{
	_push_(SFRPI);
	SFR_SetPage(2);				// TF4,EXF4 only Page-2
	if(TF4)
	{
		TF4=0;
		SFR_SetPage(0);			// P5 only Page-0
		IO_TEST_2=!IO_TEST_2;
	}
	SFR_SetPage(2);				// TF4,EXF4 only Page-2
	if(EXF4)
	{
		EXF4=0;
		if(bT4Capture==0)
		{
			T4CaptureData[T4CaptureCnt]=RCAP4H;
			T4CaptureCnt++;
			if(T4CaptureCnt>=CAPTURE_BUF_SIZE)
			{
				bT4Capture=TRUE;
			}
		}
	}
	SFR_SetPage(2);				// TF4L only Page-2
	if(TF4L)
	{
		TF4L=0;
		SFR_SetPage(0); 		// P5 only Page-0
		IO_TEST_5=!IO_TEST_5;
	}
	_pop_(SFRPI);
}

/***********************************************************************************

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:		void SendByte(u8 tByte)
*Description:	send byte
*Input:			u8 tByte: the data to be send
*Output:     
*************************************************************************************/
void SendByte(u8 tByte)
{
	bit bES0;
	bES0=ES0;
	ES0=0;
	S0BUF=tByte;
	while(TI0==0);
	TI0=0;
	ES0=bES0;
}

/***********************************************************************************
*Function:		void SendStr(BYTE* PStr)
*Description:	send string
*Input: 			u8* PStr:the string to be send
*Output:     
*************************************************************************************/
void SendStr(u8* PStr)
{
	while(*PStr != 0)
	{
		SendByte(*PStr);
		PStr ++;
	}
}

/***********************************************************************************
*Function:   	void SendHex(u8 ToSend)
*Description:	send data for HEX
*Input: 			u8 ToSend: the data to be send
*Output:     
*************************************************************************************/
void SendHex(u8 ToSend)
{
	u8 x;
//	SendByte(' ');
//	SendByte('0');
//	SendByte('x');
	x = ToSend >> 4;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
	x= ToSend&0x0F;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
}

/***********************************************************************************
*Function:   	void InitTimer2(void)
*Description:	Initialize Timer2
*Input:   
*Output:     
*************************************************************************************/
void InitTimer2(void)
{
	TM_SetT2Mode_CapturewithAutoZero();		// T2 mode: capture with Auto-Zero
	TM_SetT2Clock_SYSCLKDiv12();			// T2 clock source: SYSCLK/12
	TM_SetTL2Clock_SYSCLKDiv12();			// T2L clock source: SYSCLK/12
	TM_EnableAccessTF2L();					// Enable access TF2L and TF2L interrupt
	TM_DisableTR2LC();						// Disable TR2LC
	TM_EnT2Split();							// Enable split T2
	
	TM_SetT2LowByte(0);						// T2 low byte				
	TM_SetT2RLLowByte(0x40);				// T2 low byte reload  ---T2L overflow: SYSCLK/12/(256-0x40)

	TM_SetT2HighByte(0);					// T2 high byte
	TM_SetT2RLHighByte(0);					// T2 high byte reload ---T2H overflow: SYSCLK/12/(256-0x00)

	TM_SetT2Capture_T2EXPin();				// T2 EXI: T2EX pin
	TM_SetT2CKOT2EX_P10P11();				// T2EX pin: P11
	TM_EnableT2EX_DetectFalling();			// T2EX detect: Falling

	TM_EnableT2();							// Enable T2
	TM_EnableTL2();							// Enable TL2
}

/***********************************************************************************
*Function:   	void InitTimer3(void)
*Description:	Initialize Timer3
*Input:   
*Output:     
*************************************************************************************/
void InitTimer3(void)
{
	TM_SetT3Mode_CapturewithAutoZero();		// T3 mode: capture with Auto-Zero
	TM_SetT3Clock_SYSCLKDiv12();			// T3 clock source: SYSCLK/12
	TM_SetTL3Clock_SYSCLKDiv12();			// T3L clock source: SYSCLK/12
	
	TM_SetT3LowByte(0);						// T3 low byte				
	TM_SetT3RLLowByte(0x80);				// T3 low byte reload  ---T3L overflow:  SYSCLK/12/(256-0x80)

	TM_SetT3HighByte(0);					// T3 high byte
	TM_SetT3RLHighByte(0);					// T3 high byte reload

	TM_SetT3Capture_T3EXPin();				// T3 EXI: T3EX pin
	TM_SetT3CKOT3EX_P33P32();				// T3/T3CKO:P33,T3EX:P32
	TM_EnableT3EX_DetectFalling();			// T3EX detect: Falling
	TM_EnT3Split();							// Enable split T3
	
	TM_DisableTR3LC();						// Disable TR3LC

	TM_EnableT3();							// Enable T3
	TM_EnableTL3();							// Enable TL3

}

/***********************************************************************************
*Function:   	void InitTimer4(void)
*Description:	Initialize Timer4
*Input:   
*Output:     
*************************************************************************************/
void InitTimer4(void)
{
	TM_SetT4Mode_CapturewithAutoZero();		// T4 mode: capture with Auto-Zero
	TM_SetT4Clock_SYSCLKDiv12();			// T4 clock source: SYSCLK/12
	TM_SetTL4Clock_SYSCLKDiv12();			// T4L clock source: SYSCLK/12
	
	
	TM_SetT4LowByte(0);						// T4 low byte				
	TM_SetT4RLLowByte(0x80);				// T4 low byte reload  ---T4L overflow:  SYSCLK/12/(256-0x80)

	TM_SetT4HighByte(0);					// T4 high byte
	TM_SetT4RLHighByte(0);					// T4 high byte reload

	TM_SetT4Capture_T4EXPin();				// T4 EXI: T4EX pin
	TM_SetT4CKOT4EX_P36P37();				// T4/T4CKO:P36,T4EX:P37
	TM_EnableT4EX_DetectFalling();			// T4EX detect: Falling
	TM_EnT4Split();							// Enable split T4
	
	TM_DisableTR4LC();						// Disable TR4LC

	TM_EnableT4();							// Enable T4
	TM_EnableTL4();							// Enable TL4

}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER2();
	INT_EnTIMER2L();
	INT_EnTIMER3();
	INT_EnTIMER3L();
	INT_EnTIMER4();
	INT_EnTIMER4L();
}

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP5PushPull(BIT2|BIT3|BIT4|BIT5|BIT6|BIT7);					// Set P52,P53,P54,P55,P56,P57 as Push Pull
	PORT_SetP1OpenDrainPu(BIT1);										// Set P11 as open-drain with pull-high
	PORT_SetP3QuasiBi(BIT2|BIT3|BIT6|BIT7);								// Set P32,P33,P36,P37 as Quasi-Bidirectional
}

/***********************************************************************************
*Function:   	void InitClock()
*Description:	Initialize clock
*Input:   
*Output:   		
*************************************************************************************/
void InitClock(void)
{
#if (MCU_SYSCLK==11059200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==12000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=12MHz CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==22118400)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==24000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==29491200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==32000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==36000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// CKMIx6,x8,x12
	CLK_SetCKM_x6x8x12();	
	// SysClk=36MHz CpuClk=18MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx6, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4_X6|OSCIn_IHRCO);
#else
	// CKMIx6,x8,x12
	CLK_SetCKM_x6x8x12();	
	// SysClk=36MHz CpuClk=18MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx6, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4_X6|OSCIn_IHRCO);
#endif
#endif


#if (MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if (MCU_SYSCLK==48000000)
	// SysClk=48MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:   	void InitUart0_S0BRG()
*Description:	Initialize Uart0, The source of baud rate was S0BRG
*Input:   
*Output:   		
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	
	UART0_SetAccess_S0CR1();			// Enable access S0CR1 
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R. source: S0BRG
	UART0_SetS0BRGBaudRateX2();			// S0BRG x2
	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source: SYSCLK

	// Sets B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_57600_2X_32000000_1T);
	
	UART0_EnS0BRG();					// Enable S0BRG
}

/***********************************************************************************
*Function:       void InitSystem(void)
*Description:    Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	
	InitPort();
	InitClock();
	InitUart0_S0BRG();
	
	InitTimer2();			
	InitTimer3();			
	InitTimer4();

	InitInterrupt();
	
	INT_EnAll();			// Enable global interrupt

}



void main()
{
	
	u8 i;
	
    InitSystem();
	
 	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
 	LED_G_0=1;LED_G_1=1;LED_R=1;

					
    while(1)
    {
    	DelayXms(100);
    	if(bT2Capture)
    	{
			SendStr("\nT2 Caputre:");
			for(i=0;i<CAPTURE_BUF_SIZE;i++)
			{
				SendByte('0');
				SendByte('x');
				SendHex(T2CaptureData[i]);
				SendByte(' ');
			}
			T2CaptureCnt=0;
			bT2Capture=0;
    	}
		if(bT3Capture)
		{
			SendStr("\nT3 Caputre:");
			for(i=0;i<CAPTURE_BUF_SIZE;i++)
			{
				SendByte('0');
				SendByte('x');
				SendHex(T3CaptureData[i]);
				SendByte(' ');
			}
			T3CaptureCnt=0;
			bT3Capture=0;
		}
		if(bT4Capture)
		{
			SendStr("\nT4 Caputre:");
			for(i=0;i<CAPTURE_BUF_SIZE;i++)
			{
				SendByte('0');
				SendByte('x');
				SendHex(T4CaptureData[i]);
				SendByte(' ');
			}
			T4CaptureCnt=0;
			bT4Capture=0;
		}
    }
}


