/**
 * @file    mg32_syscfg.c
 * @author  MegawinTech Application Team
 * @version V1.0.0
 * @date    01-October-2024
 * @brief   This file contains all the SYSCFG firmware functions 
 */

/* Define to prevent recursive inclusion -----------------------------------------------------------------------------*/
#define _MG32_SYSCFG_C_

/* Files includes ----------------------------------------------------------------------------------------------------*/
#include "mg32_syscfg_lib.h"

/** @addtogroup MG32_StdPeriph_Driver
  * @{
  */

/** @addtogroup SYSCFG
  * @{
  */

/** @defgroup SYSCFG_Private_Defines
  * @{
  */

/**
  * @}
  */

/** @defgroup SYSCFG_Private_Functions
  * @{
  */


/**
  * @brief  Deinitializes the SYSCFG registers to their default reset values.
  * @param  None
  * @retval None
  */
void SYSCFG_DeInit(void)
{
    /* Set SYSCFG_CFGR1 register to reset value without affecting MEM_MODE bits */
    SYSCFG->CFGR &= SYSCFG_CFGR_MEM_MODE_Msk;
    /* Set EXTICRx registers to reset value */
    SYSCFG->EXTICR[0] = 0;
    SYSCFG->EXTICR[1] = 0;
    SYSCFG->EXTICR[2] = 0;
    SYSCFG->EXTICR[3] = 0;
}

/**
* @brief  Configures the memory mapping at address 0x00000000.
* @param  SYSCFG_MemoryRemap: selects the memory remapping.
*          This parameter can be one of the following values:
*            @arg SYSCFG_MemoryRemap_Flash: Main Flash memory mapped at 0x00000000
*            @arg SYSCFG_MemoryRemap_SystemMemory: System Flash memory mapped at 0x00000000
*            @arg SYSCFG_MemoryRemap_SRAM: Embedded SRAM mapped at 0x00000000
* @retval None
*/
void SYSCFG_MemoryRemapConfig(uint32_t SYSCFG_MemoryRemap)
{
    MODIFY_REG(SYSCFG->CFGR, SYSCFG_CFGR_MEM_MODE_Msk, SYSCFG_MemoryRemap);
}

/**
  * @brief  Selects the GPIO pin used as EXTI Line.
  * @param  port_source_gpio: EXTI_PortSourceGPIOx .
  * @param  pin_source: EXTI_PinSourcex.
  * @retval None.
  */
void SYSCFG_EXTILineConfig(uint8_t port_source_gpio, uint8_t pin_source)
{
    uint32_t tmp = 0x00;

    tmp = ((uint32_t)0x0F) << (0x04 * (pin_source & (uint8_t)0x03));
    SYSCFG->EXTICR[pin_source >> 0x02] &= ~tmp;
    SYSCFG->EXTICR[pin_source >> 0x02] |= (((uint32_t)port_source_gpio) << (0x04 * (pin_source & (uint8_t)0x03)));
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
