/**
  ******************************************************************************
 *
 * @file        Sample_MID_TM36_8bitx2PWM_DTG_CentralAlign_DMA.c
 *
 * @brief       TM36 8bitx2 PWM with central align, Deadtime control and DMA access.
 *
 * @par         Project
 *              MG32
 * @version     V1.14
 * @date        2021/05/27
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
* @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_TM_MID.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static TM_HandleTypeDef    mTM36;
static DMA_HandleTypeDef   PWMChannel0, PWMChannel1, PWMChannel2;

static uint16_t            DChaneel0[] = {0x1010, 0x2020, 0x3030};
static uint16_t            DChaneel1[] = {0x4040, 0x5050, 0x6060};
static uint16_t            DChaneel2[] = {0x7070, 0x8080, 0x9090};

/* Private function prototypes -----------------------------------------------*/
void Sample_MID_TM36_8bitx2PWM_DTG_CentralAlign_DMA(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief	    TM36 8bitx2 PWM with central align, Deadtime control and DMA access.
 * @details     1.Initial TM36 mode.
 *    \n        2.Clock initial.
 *    \n        3.8bitx2 PWM Output channel configuration.
 *    \n        4.Initial DMA.
 *    \n        5.Set Break in cycle-by-cycle mode.
 *    \n        6.Config TM36 Central-Align.
 *    \n        7.Config OC/PWM update timing.
 *    \n        8.Start PWM output with DMA.
 * @return      None
 *******************************************************************************
 */
void Sample_MID_TM36_8bitx2PWM_DTG_CentralAlign_DMA(void)
{      
    TM_OC_InitTypeDef               sConfig;    
    TM_ClockConfigTypeDef           CKConfig;
    TM_BreakDeadTimeConfigTypeDef   BKConfig;
    
    
    // make sure :
    
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //  Select CK_PLLI Divider = CK_HS/2
    //  Select CK_PLL Multiplication factor = CK_PLLIx16
    //  Select CK_PLLO Divider = CK_PLL/4
    //Select CK_MAIN Source = CK_PLLO
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->TM36 = Enable
    //Configure Peripheral On Mode Clock->DMA  = Enable
    //Configure Peripheral On Mode Clock->TM10 = Enable
	
    //==Set GPIO init 
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOC->PC0,PC2,PD0
    //GPIO port initial is 0xFFFF
    //PC0,PC2,PD0 mode is PPO
    //PC0,PC2,PD0 function TM36_OC00,TM36_OC10,TM36_OC2
    
    // ------------------------------------------------------------------------
    // 1.Initial TM36 mode.
    // ------------------------------------------------------------------------
    mTM36.Instance                  = TM36;
    mTM36.Init.TM_CounterMode       = TM_CASCADE_UP;
    mTM36.Init.TM_Period            = 0xFF;
    mTM36.Init.TM_Prescaler         = 10;
    MID_TM_8Bx2PWM_DTG_Init(&mTM36);
    
    // ------------------------------------------------------------------------
    // 2.Clock initial.
    // ------------------------------------------------------------------------
    CKConfig.TM_ClockSource         = TM_INTERNAL_CLOCK;
    CKConfig.TM_ExternalClockSource = TM_EXTERNAL_ITR0;             // Setup ITR0 from TM10_TRGO
    CKConfig.TM_InternalClockSource = TM_INTERNALCLOCK_PROC;
    CKConfig.TM_INTClockDivision    = TM_INTERNALCLOCK_DIVDER_DIV1;
    
    MID_TM_ConfigClockSource(&mTM36, &CKConfig);  
    
    // ------------------------------------------------------------------------
    // 3.8bitx2 PWM Output channel configuration.
    // ------------------------------------------------------------------------
    MID_TM_OC_Struct_Init(&sConfig);                                // default initial (output state)
    sConfig.OCMode                  = TM_CH_8bitx2_PWM_COMPLEMENT;
    sConfig.Pulse                   = 0;                            // Duty cycle
    MID_TM_PWM_ConfigChannel(&mTM36, &sConfig, MID_TM_Channel0);
    MID_TM_PWM_ConfigChannel(&mTM36, &sConfig, MID_TM_Channel1);    
    MID_TM_PWM_ConfigChannel(&mTM36, &sConfig, MID_TM_Channel2);    

    // ------------------------------------------------------------------------
    // 4.Initial DMA. (M2P)
    // ------------------------------------------------------------------------
    // DMAChannel0
    PWMChannel0.Instance            = DMA;
	PWMChannel0.DMAChannelIndex     = DMAChannel0;                  // DMA channel0 can support M2P mode  
	PWMChannel0.Init.SrcPeri        = MID_DMA_MEMORY_READ;          // Source symbol is Memory
	PWMChannel0.Init.DesPeri        = MID_DMA_TM36_CC0B_WRITE;      // Destination symbol is TM36_OC0
	PWMChannel0.Init.BSize          = DMA_BSIZE_2BYTE;              // Burst size is 2 byte
	PWMChannel0.Init.MEMMode        = MID_DMA_MEMORY_NORMAL;        // Normal memory access mode    
    PWMChannel0.Init.LoopMode       = DMA_LOOP_DISABLE;             // DISABLE Loop mode
    PWMChannel0.Parent              = &mTM36;
	
    MID_DMA_Init(&PWMChannel0);

    // DMAChannel1
    PWMChannel1.Instance            = DMA;
	PWMChannel1.DMAChannelIndex     = DMAChannel1;                  // DMA channel1 can support M2P mode  
	PWMChannel1.Init.SrcPeri        = MID_DMA_MEMORY_READ;          // Source symbol is Memory
	PWMChannel1.Init.DesPeri        = MID_DMA_TM36_CC1B_WRITE;      // Destination symbol is TM36_OC1
	PWMChannel1.Init.BSize          = DMA_BSIZE_2BYTE;              // Burst size is 2 byte
	PWMChannel1.Init.MEMMode        = MID_DMA_MEMORY_NORMAL;        // Normal memory access mode    
    PWMChannel1.Init.LoopMode       = DMA_LOOP_DISABLE;             // DISABLE Loop mode
    PWMChannel1.Parent              = &mTM36;
    
	MID_DMA_Init(&PWMChannel1);

    // DMAChannel2
    PWMChannel2.Instance            = DMA;
	PWMChannel2.DMAChannelIndex     = DMAChannel2;                  // DMA channel2 can support M2P mode  
	PWMChannel2.Init.SrcPeri        = MID_DMA_MEMORY_READ;          // Source symbol is Memory
	PWMChannel2.Init.DesPeri        = MID_DMA_TM36_CC2B_WRITE;      // Destination symbol is TM36_OC2
	PWMChannel2.Init.BSize          = DMA_BSIZE_2BYTE;              // Burst size is 2 byte
	PWMChannel2.Init.MEMMode        = MID_DMA_MEMORY_NORMAL;        // Normal memory access mode    
    PWMChannel2.Init.LoopMode       = DMA_LOOP_DISABLE;             // DISABLE Loop mode
    PWMChannel2.Parent              = &mTM36;
    
	MID_DMA_Init(&PWMChannel2);
    
        
    // ------------------------------------------------------------------------
    // 5.Set Break in cycle-by-cycle mode.
    // ------------------------------------------------------------------------
    BKConfig.BreakMode              = TM_BK_CYCLE_BY_CYCLE;
    BKConfig.BreakCHxOutputMode     = MID_BREAK_STOP_STATE;
    BKConfig.BreakSourceSel         = MID_BK_ExtPin | MID_BK_ClockFailure | MID_BK_CPULOCKUP | MID_BK_BOD1;
    BKConfig.DeatTimeClockDivision  = MID_TM_CKDTG_DIV8;
    BKConfig.DeadTime               = 26;
    MID_TM_ConfigBreakDeadTime(&mTM36, &BKConfig);
    
    // ------------------------------------------------------------------------
    // 6.Config TM36 Central-Align.
    // ------------------------------------------------------------------------
    __DRV_TM_ENABLE_PWM_CENTRAL(&mTM36);

    // ------------------------------------------------------------------------
    // 7.Config OC/PWM update timing.
    // ------------------------------------------------------------------------
//    __DRV_TM_SET_DMA_UPLOAD_TIMING(&mTM36, TM_ITR_EVENT_TIMING);        // Default TM36 internal trigger signals is TM10_TRGO       
    __DRV_TM_SET_DMA_UPLOAD_TIMING(&mTM36, TM_UPDATEG_EVENT_TIMING);    // Default update mode 
    
    // ------------------------------------------------------------------------
    // 8.Start PWM output with DMA.
    // ------------------------------------------------------------------------
    // User maybe modify DMA parameters after DMA complete. 
    // So, user can modify 'MG32_IRQHandler.c'.
    // Like this.
    // void DMA_IRQHandler(void)
    // {
    //     MID_DMA_IRQHandler(&PWMChannel0);
    //     MID_DMA_IRQHandler(&PWMChannel1);
    //     MID_DMA_IRQHandler(&PWMChannel2);
    // }
    
    mTM36.hDMA[0] = &PWMChannel0;
    mTM36.hDMA[1] = &PWMChannel1;
    mTM36.hDMA[2] = &PWMChannel2;
    
    // for [-Wcast-align] issue
#if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
    #pragma clang diagnostic push
    #pragma clang diagnostic ignored "-Wcast-align"
        MID_TM_8Bx2PWM_DTG_Start_DMA(&mTM36, MID_TM_Channel0, (uint32_t *) &DChaneel0, 6);
        MID_TM_8Bx2PWM_DTG_Start_DMA(&mTM36, MID_TM_Channel1, (uint32_t *) &DChaneel1, 6);
        MID_TM_8Bx2PWM_DTG_Start_DMA(&mTM36, MID_TM_Channel2, (uint32_t *) &DChaneel2, 6);
    #pragma clang diagnostic pop
#else
    MID_TM_8Bx2PWM_DTG_Start_DMA(&mTM36, MID_TM_Channel0, (uint32_t *) &DChaneel0, 6);
    MID_TM_8Bx2PWM_DTG_Start_DMA(&mTM36, MID_TM_Channel1, (uint32_t *) &DChaneel1, 6);
    MID_TM_8Bx2PWM_DTG_Start_DMA(&mTM36, MID_TM_Channel2, (uint32_t *) &DChaneel2, 6);
#endif
    
    return;

}

