
/**
  ******************************************************************************
 *
 * @file        Sample_MID_CMP_WakeupFromSLEEPMode.c
 *
 * @brief       Initial COMP recipe for SLEEP mode (Middleware). 
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2023/03/16
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
 
/* Includes ------------------------------------------------------------------*/
#include "MG32_MID.h"
#include "RTE_Components.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

#if defined(IRQHandler_Middleware_Level_)       
    #if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
        #pragma clang diagnostic push
        #pragma clang diagnostic ignored "-Wmissing-variable-declarations"
        CMP_HandleTypeDef   mCMP;
        #pragma clang diagnostic pop
    // ------------------------------------------------------------------------
    #else
        CMP_HandleTypeDef   mCMP;
    #endif
#endif

/* Private function prototypes -----------------------------------------------*/
void Sample_MID_CMP_WakeupFromSLEEPMode(void);
void COMP_IRQHandler(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/
 
 
/**
 *******************************************************************************
 * @brief       Initial COMP recipe for SLEEP mode. 
 * @details     1. Systick initial & setting priority
 *      \n      2. COMP & GPIOA Clock initial
 *      \n      3. GPIO initial
 *      \n      4. ACx(0~3) initial
 *      \n      5. Interrupt and NVIC configuration
 *      \n      6. COMP sleep mode continuous run configure.
 * @return	    None	
 * @note        
 * @par         Example
 * @code
    Sample_MID_CMP_WakeupFromSLEEPMode();
    // To do ...
    {
    
    }
    MID_SYSTICK_DisableIRQ();
    SLEEP_WFI();
    MID_SYSTICK_EnableIRQ();
 * @endcode
 *******************************************************************************
*/
void Sample_MID_CMP_WakeupFromSLEEPMode(void)
{
    GPIO_InitTypeDef    GPIOX;

    // ===CSC Initialize Set Description====
    // MG32_CSC_Init.h(Configuration Wizard)
    // Select CK_HS source = CK_IHRCO
    // Select IHRCO = 12Mz
    // Select CK_MAIN Source = CK_HS
    // Select CK_LS Source = CK_ILRCO
    // Configure Select APB Prescaler = CK_MAIN/1
    // Configure Peripheral On Mode Clock->RTC = Enable
    
    // (GPIO Config)
    // AC0 needs PA8/PA9 or PB0/PB1 pin.
    // AC1 needs PA10/PA11 or PB0/PB1 pin.
    // AC2 needs PA12/PA13 or PB0/PB1 pin. (for MG32F02A132/A072)
    // AC3 needs PA14/PA15 or PB0/PB1 pin. (for MG32F02A132/A072)
    // MG32_GPIO_Init.H
    // Select GPIO Initial -> GPIOA configuration -> PA8/PA9/PA10/PA11/PA12/PA13/PA14/PA15 configuration
    // Select GPIO Initial -> GPIOB configuration -> PB0/PB1 configuration
    // Select Pin mode = AIO
    // Select Pin alternate function = GPA8/GPA9/GPA10/GPA11/GPA12/GPA13/GPA14/GPA15
    // Other selection keep default value


    // ------------------------------------------------------------------------
    // 1. Systick initial & setting priority
    // ------------------------------------------------------------------------
    MID_Init();
    
    // ------------------------------------------------------------------------
    // 2. COMP & GPIOA Clock initial (User can config in MG32_CSC_Init.h.)
    // ------------------------------------------------------------------------
    MID_UnProtectModuleReg(CSCmodule);
    __DRV_CSC_GPIOA_CLK_ENABLE();                               // Enable PortA clock    
    
    __DRV_CSC_CMP_CLK_ENABLE();                                 // Enable CMP clock
    __DRV_CSC_CMP_CKS_CK_APB();                                 // CMP select AHB
    
    __DRV_CSC_SLP_CMP_CLK_ENABLE();                             // *** Enable CMP sleep clock ***
    MID_ProtectModuleReg(CSCmodule);
    
    // ------------------------------------------------------------------------
    // 3. GPIO initial (User can config in MG32_GPIO_LQFP80_Init.h.)
    // ------------------------------------------------------------------------
    // for AC0
    GPIOX.Pin           = ( GPIO_Pin_8 | GPIO_Pin_9);
    GPIOX.Mode          = GPIO_MODE_ANALOG_IO;
    GPIOX.Pull          = GPIO_NOPULLUP;
    GPIOX.Speed         = GPIO_SPEED_LOW;
    GPIOX.Inverse       = GPIO_INVERSE_DISABLE;
    GPIOX.OUTDrive      = GPIO_OUTDRIVE_LEVEL0;
    GPIOX.FilterDivider = GPIO_FILTERDIVIDER_BYPASS;
    GPIOX.Alternate     = 0;                          // 0 = GPIO mode.
    MID_GPIO_Init (IOMA ,&GPIOX);  

    
    // for AC1
    GPIOX.Pin           = ( GPIO_Pin_10 | GPIO_Pin_11);
    MID_GPIO_Init (IOMA ,&GPIOX);  
    
    #if defined(MG32_1ST)
    // for AC2
    GPIOX.Pin           = ( GPIO_Pin_12 | GPIO_Pin_13);
    MID_GPIO_Init (IOMA ,&GPIOX);  
    
    // for AC3
    GPIOX.Pin           = ( GPIO_Pin_14 | GPIO_Pin_15);
    MID_GPIO_Init (IOMA ,&GPIOX);  
    #endif
    
    // ------------------------------------------------------------------------
    // 4. AC0 initial (User can config in MG32_CMP_Init.h.)
    // ------------------------------------------------------------------------
    // CMPAC0 config
    
    mCMP.Instance               = CMPAC0;
    mCMP.Init.InvertingInput    = CMP_INVERTINGINPUT_IO0;   // Inverting PIN Input
    mCMP.Init.NonInvertingInput = CMP_NONINVERTINGINPUT_IO1;// Non-Inverting PIN Input
    mCMP.Init.ComparatorPol     = CMP_POSITIVE_POLARITY;    // CMP Possitive polarity
    mCMP.Init.OutputPol         = CMP_OUTPUTPOL_NONINVERTED;// CMP non-invert output
    mCMP.Init.Hysteresis        = CMP_HYSTERESIS_NONE;      // Hysteresis window config
    mCMP.Init.Mode              = CMP_MODE_HIGHSPEED;       // High speed for CMP
    mCMP.Init.TriggerMode       = CMP_TRIGGERMODE_IT_RISING_FALLING;    // External Interrupt Mode with Rising/Falling edge trigger detection
    mCMP.Init.OutputSync        = CMP_SYNCHRONIZE_BYPASS;   // CMP_OUT without filter function
    mCMP.Init.SyncFilterDIV     = CMP_SYNCHRONIZE_DIV1;     // no used when select BYPASS option
    mCMP.IntVref.IntVREF0_Level = 32;                       // Internal VREF0 selection
    mCMP.IntVref.IntVREF1_Level = 32;                       // Internal VREF1 selection
    
    MID_CMP_Init(&mCMP);
    MID_CMP_Start_IT(&mCMP);
    
    // ------------------------------------------------------------------------
    // AC1 initial 
    // ------------------------------------------------------------------------
    // CMPAC1 config
    mCMP.Instance                   = CMPAC1;
    MID_CMP_Init(&mCMP);
    MID_CMP_Start_IT(&mCMP);
   
    #if defined(MG32_1ST)
    // ------------------------------------------------------------------------
    // AC2 initial 
    // ------------------------------------------------------------------------
    // CMPAC2 config
    mCMP.Instance                   = CMPAC2;
    MID_CMP_Init(&mCMP);
    MID_CMP_Start_IT(&mCMP);
    
    // ------------------------------------------------------------------------
    // AC2 initial 
    // ------------------------------------------------------------------------
    // CMPAC3 config
    mCMP.Instance                   = CMPAC3;
    MID_CMP_Init(&mCMP);
    MID_CMP_Start_IT(&mCMP);
    #endif
    
    // ------------------------------------------------------------------------
    // 5. Interrupt and NVIC configuration
    // ------------------------------------------------------------------------
    __DRV_CMP_CLEAR_FLAG (CMP_CMP0_FLAG_ALL);
    __DRV_CMP_CLEAR_FLAG (CMP_CMP1_FLAG_ALL);
        
    #if defined(MG32_1ST)
    __DRV_CMP_CLEAR_FLAG (CMP_CMP2_FLAG_ALL);
    __DRV_CMP_CLEAR_FLAG (CMP_CMP3_FLAG_ALL);
    #endif
    
    NVIC_ClearPendingIRQ(COMP_IRQn);    
    NVIC_EnableIRQ(COMP_IRQn);
    
    // ------------------------------------------------------------------------
    // 6. COMP sleep mode continuous run configure.
    // ------------------------------------------------------------------------
    MID_UnProtectModuleReg(PWmodule);
    __DRV_PW_CMP0_SLEEP_MODE_POWER_ON_ENABLE();             // Enable CMP in SLEEP
    __DRV_PW_CMP1_SLEEP_MODE_POWER_ON_ENABLE();             // Enable CMP in SLEEP
    #if defined(MG32_1ST)           
    __DRV_PW_CMP2_SLEEP_MODE_POWER_ON_ENABLE();             // Enable CMP in SLEEP
    __DRV_PW_CMP3_SLEEP_MODE_POWER_ON_ENABLE();             // Enable CMP in SLEEP
    #endif
    MID_ProtectModuleReg(PWmodule);

    // ------------------------------------------------------------------------
    // SLEEP test
    // ------------------------------------------------------------------------
//    MID_SYSTICK_DisableIRQ();
//    SLEEP_WFI();
//    MID_SYSTICK_EnableIRQ();
    
    
}


/**
 *******************************************************************************
 * @brief       COMP: An interrupt, or IRQ, is an exception signalled by a 
 *              peripheral, or generated by a software request. 
 * @details     COMP handles port CMP0~3 interrupt.
 * @return      No
 *******************************************************************************
 */
#ifndef MG32_COMP_IRQ_
void COMP_IRQHandler(void)
{
    // Middleware level
    MID_CMP_IRQHandler(&mCMP);
}
#endif


