/**
 ******************************************************************************
 *
 * @file        BSP_I2C_SHT31.h
 *
 * @brief       This file contains all the functions prototypes for the SHT31 
 *              firmware library.
 *
 * @par         Project
 *              MG32
 * @version     V1.17
 * @date        2022/01/07
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2016 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par 		Disclaimer 
 *		The Demo software is provided "AS IS"  without any warranty, either 
 *		expressed or implied, including, but not limited to, the implied warranties 
 *		of merchantability and fitness for a particular purpose.  The author will 
 *		not be liable for any special, incidental, consequential or indirect 
 *		damages due to loss of data or any other reason. 
 *		These statements agree with the world wide and local dictated laws about 
 *		authorship and violence against these laws. 
 ******************************************************************************
 ******************************************************************************
 */ 
/*==============================================================================
            Adafruit SHT31-D Temperature & Humidity Sensor Breakout
            
Power Pins: 
    Vin - this is the power pin. The chip can use 2.5-5VDC for power. To power the
          board, give it the same power as the logic level of your microcontroller.
    GND - common ground for power and logic.

I2C Logic pins:
    SCL - I2C clock pin, connect to your microcontrollers I2C clock line. This pin has
          a 10K pullup resistor to Vin.
    SDA - I2C data pin, connect to your microcontrollers I2C data line. This pin has a
          10K pullup resistor to Vin.
      
Other Pins:
    ADR - This is the I2C address selection pin. This pin has a 10K pull down resistor
          to make the default I2C address 0x44. You can tie this pin to Vin to make the
          address 0x45.
    RST - Hardware reset pint. Has a 10K pullup on it to make the chip active by
          default. Connect to ground to do a hardware reset!
    ALR - Alert/Interrupt output. You can set up the sensor to alert you when an
          event has occured. Check the datasheet for how you can set up the alerts.
      

                                 User NOTES
How To use this function:
-----------------------
   + SHT31_I2C_Init() - Initial SHT31 sensor.
   + SHT31_readTemperature() - Convert enviroment Temperature by SHT31.
   + SHT31_readHumidity() - Get enviroment Humidity by SHT31.
   + SHT31_Read() - Get SHT31 sensor data.


Example:
-----------------------
    SHT31_I2C_Init(0x44);
    
    // Start AM2302 peripheral
    SHT31_Read();
    
    // Get Humidity
    Humidity = SHT31_readHumidity();
    
    // Get Temperature
    Temperature = SHT31_readTemperature(Celcius);


Driver architecture:
--------------------
   + MG32_GPIO_DRV
   + MG32_I2C_DRV

Known Limitations:
------------------
   1- Run in 48MHz. (CSC control).
   2- GPIO PA8 must be GPIO mode.
==============================================================================*/

/*==============================================================================
                                 Require parameter
Require module : CSC / GPIO / TM20

CSC Module :
    Clock source                : CK_IHRCO (12M Hz)
    Clock frequency             : CK_MAIN select CK_PLLO (PLL to 48MHz)
    CK_APB frequency            : 48MHz
    CK_AHB frequency            : 48MHz
    Clock enable in 'On Mode'   : GPIOB / TM20

GPIO pin configuration : 
    Pin /   IO mode   /   AFS
    ---   ------------  --------
    PB3 / ODO+Pull up / I2C0_SCL
    PB2 / ODO+Pull up / I2C0_SDA
                                           
I2C Module :                   
    I2C protocol 
    
==============================================================================*/


#ifndef _MG32_SHT31_I2C_H

/* Includes ------------------------------------------------------------------*/
#include "MG32_I2C_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/**
 * @name	SHT31_I2C_address
 *   		SHT31 I2C default address
 */ 
//!@{
#define SHT_DEFAULT_ADDRESS     0x44
//!@}
    
/**
 * @name	SLA_Read_Write_Control
 *   		I2C address with Write / Read control
 */ 
//!@{
#define I2C_ADDR_WR             0x00
#define I2C_ADDR_RD             0x01
//!@}
    
/**
 * @name	I2C_ACK_NACK_Control 
 *   		Response ACK or NACK in I2C receive state.
 */ 
//!@{
#define I2C_ACK                 1
#define I2C_NACK                0
//!@}


/**
 * @name	Temperature unit
 *   		SHT31 Temperature unit
 */ 
//!@{
#define Fahrenheit                  1
#define Celcius                     0
//!@}



/* Private typedef -----------------------------------------------------------*/
typedef struct  __attribute__((packed))
{

    // ------------------------------------------------------------------------
    // Like anduino code
    boolean (*begin)(uint8_t Addr);
    boolean (*read)(void);
    float (*readTemperature)(boolean S);
    float (*readHumidity)(void);
    
    // ------------------------------------------------------------------------
    uint16_t I2C_CMD_STATE;                     // Write command
    
    uint16_t TemperatureVal;                    // Temperature data
    uint16_t HumidityVal;                       // Humidity data
    
    uint8_t Temperature_CRC;                    // Temperature CRC8
    uint8_t Humidity_CRC;                       // Humidity CRC8
    
    uint8_t ADDR;                               // SHT31 I2C address
    
} SHT31_I2CDef;

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/

/**
 * @name    Function announce
 * @brief   SHT31 base function group
 */ 
///@{
boolean SHT31_I2C_Init(uint8_t Addr);
boolean SHT31_Read(void);
float SHT31_readTemperature(boolean S);
float SHT31_readHumidity(void);
///@}

/* Exported variables --------------------------------------------------------*/

/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 * @name    Function announce
 * @brief   I2C base function group (poll mode)
 */ 
///@{
boolean I2C_BusStart(void);
boolean I2C_Master_BusStop(void);
boolean I2C_Transmiter(uint8_t TxData);
boolean I2C_Receive(uint8_t *RxData, uint8_t Acknowledge);
void I2C_ByteMode_Handle(void);
///@}


#endif
