/**
 ******************************************************************************
 *
 * @file        BSP_BLE.c
 * @brief       MG32 BLE board support c code. 
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2022/04/18
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2022 megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/*==============================================================================
                                 User NOTES
How To use this function:
-----------------------
   + BSP_BLE_Init();
   
   + BSP_BLE_SetInitRunningMode();                      // Set BLE init running mode
   + BSP_BLE_RadioInit(TXPWR_0DBM, &ble_mac_addr);      // Radio init BLE
   + BSP_BLE_RunIniterruptStart(160*2);                 // 320*0.625=200 ms
   
   + IrqMcuGotoSleepAndWakeup();                        // IRQ MCU goto sleep and wakeup
   
Driver architecture:
--------------------  
   + void BSP_BLE_WriteTxBuffer (u8* SrcIndex, u8 len); // Update BLE_TX_BUF
   + void BSP_BLE_ClearTxBuffer (void);                 // Clear BLE_TX_BUFF index and flag
   + u8 BSP_BLE_ReadRxBuffer (u8* DestIndex);           // Read BLE_RX_BUFF
   + void BSP_BLE_ClearRxBuffer (void);                 // Clear BLE_RX_BUFF


Known Limitations:
------------------
   1- Run in 24MHz. (CSC control).
   2- 
   3- 
   4- 
   
Require parameter:
------------------   
    Require module : CSC / SPI / EXIC / IWDT
                       
    CSC Module :
        Clock source                : CK_IHRCO (12M Hz)
        CK_MAIN frequency           : CK_PLLO
        CK_APB frequency            : 24MHz
        CK_AHB frequency            : 24MHz
        Clock enable in 'On Mode'   : GPIOB / GPIOC / GPIOE

    SPI Module :
        Clock source                : CK_APB
        Clock frequency             : 24MHz

    EXIC Module : 
        Clock source                : CK_APB 
        Clock frequency             : 24MHz

    IWDT Module :
        Clock source                : CK_APB
        Clock frequency             : 32KHz
    
    GPIO pin configuration : 
        Pin / IO mode  / AFS        / BLE Pin
        ---- ---------- ------------ --------------------
        PB0 / PPO      / GPIO(NSS)  / CEN
        PB1 / DIN      / SPI0_MISO  / MI
        PB2 / PPO      / SPI0_CLK   / SCK
        PB3 / PPO      / SPI0_MOSI  / MO
        PB10/ PPO      / PPO        / IRQ
        PE13/ PPO      / GPIO       /  
           
Example codes:
------------------   
// Includes ------------------------------------------------------------------// 
#include "BSP_BLE.h"
#include "BSP_BLE_API.h"

// Wizard menu ---------------------------------------------------------------//
// Private typedef -----------------------------------------------------------//
// Private define ------------------------------------------------------------//
// Private macro -------------------------------------------------------------//
// Private variables ---------------------------------------------------------//
static unsigned char *ble_mac_addr;

// Private function prototypes -----------------------------------------------//
// Exported variables --------------------------------------------------------//
// Exported functions --------------------------------------------------------//
void ChipInit(void);
void BSP_BLE_WriteTxBuffer (u8* Index, u8 len);
void BSP_BLE_ClearBleTxBuffer (void);
u8 BSP_BLE_ReadRxBuffer (u8* DestIndex);
void BSP_BLE_ClearBleRxBuffer (void);

// External variables --------------------------------------------------------//
extern u8 BLE_RXF;
extern u8 BLE_TCF;



void BLE_Sample (void)
{
    u8 Text[30] = {0};                                      // String array
    u8 TX_CNT;                                              // String length
    

    
    ChipInit();                                             // Initial CSC modle
    BSP_Init();                                             // Initial SPI0, URT0, EXIC, IWDT, LED, GPIO and Tick
    BSP_BLE_SetInitRunningMode();                           // Set BLE init running mode
    BSP_BLE_RadioInit(TXPWR_0DBM, &ble_mac_addr);           // Set Radio init BLE
    BSP_BLE_RunIniterruptStart(160*2);                      // Set interrupt time 320*0.625 = 200 ms

    while(1)
    {
        BSP_BLE_IrqMcuGotoSleepAndWakeup();                         // IRQ MCU go to sleep and wakeup

        if(BLE_RXF != 0)                                    // Get BLE recevie flag
        {
            Text[0] = 'G';                                  // Set text buffer "Get "
            Text[1] = 'e';
            Text[2] = 't';
            Text[3] = ' ';
            TX_CNT = 4;                                     // The text buffer count 4

            TX_CNT += BSP_BLE_ReadRxBuffer(&Text[4]);       // Get BLE receive date to "Text" and update TX_CNT.  
            BSP_BLE_ClearBleRxBuffer();                     // Clear RXF and 
            BSP_BLE_WriteTxBuffer (Text, TX_CNT);           // Update BLE TX buffer data
        }
    }
}      
==============================================================================*/



/* Includes ------------------------------------------------------------------*/
#include "BSP_BLE.h"
#include "BSP_BLE_API.h"
#include "MG32_CSC_Init.h"
#include "MG32__Common_DRV.h"
#include "MG32_Common_MID.h"
#include "MG32_SPI_MID.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define MAX_SIZE 200            // rxBuf/txBuf data size

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static u16 BLE_TX_INDEX = 0;    // BLE TX index.

/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
static u8 BLE_RX_BUF[MAX_SIZE];         // BLE RX buffer. // u8 txBuf[MAX_SIZE];    // BLE RX buffer
static u8 BLE_TX_BUF[MAX_SIZE];         // BLE TX buffer. // u8 rxBuf[MAX_SIZE];    // BLE TX buffer
static u8 BLE_RX_STR = 0;               // BLE RX start
static u8 BLE_RX_LEN = 0;               // BLE RX length
volatile u8 BLE_RXF = 0;                // BLE RX update compete flag.
static u8 BLE_TCF = 1;                // BLE TX transmited complet flag.
static unsigned char Temp8;
static unsigned short Temp16;
static unsigned long Temp24;
static unsigned long Temp32;

//static volatile uint8_t SPI_RDAT;                       /* Add volatile */
static volatile uint8_t CPU_CLK_FAST_ENABLE = FALSE;    /* Add volatile */

/* Exported functions --------------------------------------------------------*/
extern char GetConnectedStatus(void);
void BLE_EXIC_Init(void);
void EXINT1_IRQHandler(void);
void IWDT_Init(void);
void BLE_SPI_Init (MID_SPI_Struct* BLE_MD);
void BLE_LED_Init(void);
char IsIrqEnabled (void);
unsigned int GetSysTickCount(void);
void SysTick_Handler(void);
void BSP_BLE_WriteRxBuffer (u8* DestIndex, u8 len);
void BSP_BLE_SendData (void);
void UsrProcCallback(void);
unsigned char SPI_WriteBuf(unsigned char reg, unsigned char const *pBuf, unsigned char len);
unsigned char SPI_ReadBuf(unsigned char reg, unsigned char *pBuf, unsigned char len);
void SysClk8to48(void);
void SysClk48to8(void);
void DisableEnvINT(void);
void EnableEnvINT(void);
void UpdateLEDValueAll(void);
void McuGotoSleepAndWakeup(void);
unsigned char aes_encrypt_HW(unsigned char *_data, unsigned char *_key);

/* External variables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief       EXIC module initial for BLE
 * @details
 * @return      None
 * @note        Used pin
 *                  PB10 AFS GPB10 (EXIC Interrupt)
 * @par         Example
 * @code
    EXIC_Init();
 * @endcode
 *******************************************************************************
 */
void BLE_EXIC_Init(void)
{
    PIN_InitTypeDef  PINX_InitStruct;
    EXIC_TRGSTypeDef EXIC_TRGS;
    
    UnProtectModuleReg(CSCprotect);                                         // Unprotect CSC module register
    CSC_PeriphOnModeClock_Config(CSC_ON_PortB,ENABLE);                      // Emable port B moudle clock
    ProtectModuleReg(CSCprotect);                                           // Protect CSC module register
    
    //====GPIO Initial==== 
//    PINX_InitStruct.PINX_Pin                = PX_Pin;                       // Select Pin of the port to initial (PX_Pin_All is all pin of the port.)
    PINX_InitStruct.PINX_Mode               = PINX_Mode_Digital_I;          // Select GPIO mode 
                                                                            //     1.QB: Quasi-Bidirection mode only for PC ) 
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;      // Select wether enable internal pull-up R or not.
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;               // Select wehter enable high speed mode
                                                                            //     1.(PINX_Speed_High mode only for PC0~3 , PD0~3 )    
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;         // Select output drive strength 
                                                                            //     1.(Level 0 & level 3 only for PE0~PE3)
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;    // Select input filter divider.
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;         // Select input signal whether inverse or not.
                                                                            //     1.PINX_Inverse_Disable = L level or falling edge.
                                                                            //     2.PINX_Inverse_Enable  = H level or Rising edge.
    PINX_InitStruct.PINX_Alternate_Function = 0;                            // Select GPIO mode
//    GPIO_PortMode_Config(IOMB, &PINX_InitStruct);                            // (Pin0 & Pin1) of PorA configuration // Initial PortA
    GPIO_PinMode_Config(EXIC_CFG_PIN,&PINX_InitStruct);
    
    //====EXIC Initial====                                   
    EXIC_TRGS.EXIC_Pin = EXIC_TRGS_PINX;    //1. You can select any pin of the port.
    EXIC_TRGS.EXIC_TRGS_Mode = Edge;                                        // External interrupt pin edge/level trigger event select.
    EXIC_PxTriggerMode_Select(EXIC_PB,&EXIC_TRGS);                          // EXIC trigger mode configuration.
    
    EXIC_PxTriggerAndMask_Select(EXIC_PB , EXIC_PX_AND_MASK_PINX);          // EXIC_PA  AF trigger event select.
                                                                            //     1. Select PA_AF trigger pin(PA0, PA1).
                                                                            //     2. PA_AF is set if trigger event of select pin have to happen at same time(PA0 & PA1).  
    EXIC_PxTriggerOrMask_Select(EXIC_PB  , EXIC_PX_OR_MASK_PINX);           // EXIC_PA  OF trigger event select.
                                                                            //     1. Select PA_OF trigger pin(PA2, PA3).
                                                                            //     2. PA_OF is set if anyone trigger event of select pin happen(PA2 | PA3).   
    EXIC_ClearPxTriggerEventFlag(EXIC_PB, EXIC_PX_PINX );                   // Clear All pin of the port event flag.    
    
    EXIC_PxTriggerITEA_Cmd(EXIC_PB_IT , ENABLE);                            // Enable EXIC interrupt
    NVIC_EnableIRQ(EXINT1_IRQn);                                            // Enable EXIC interrupt of NVIC
}


/**
 *******************************************************************************
 * @brief       EXINT1 IRQ Handler
 * @details
 * @return      None
 * @note        ble has used, modify code must be careful
 * @par         Example
 * @code

 * @endcode
 *******************************************************************************
 */
void EXINT1_IRQHandler(void)
{
   EXIC_ClearPxTriggerEventFlag(EXIC_PB, EXIC_PX_PINX );            // Clear 
   EXIC_ClearPxTriggerITFlag(EXIC_PB_ITF,EXIC_PX_OF | EXIC_PX_AF);
   ble_run(0);
}


/**
 *******************************************************************************
 * @brief       IWDT module initial
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    IWDT_Init();
 * @endcode
 *******************************************************************************
 */
void IWDT_Init(void)
{
    /*=== 1. Configure IWDT clock ===*/
    UnProtectModuleReg(IWDTprotect);                                // Unprotect IWDT module
    IWDT_Divider_Select(IWDT_DIV_128);                              // DIV output = CK_IWDT /128

    /*=== 2. Configure RST module ===*/                         
    UnProtectModuleReg(RSTprotect);                                 // Unprotect RST moduel register
    RST_WRstSource_Config(RST_IWDT_WE, ENABLE);                     // Enable IWDT event trigger warm reset
    ProtectModuleReg(RSTprotect);                                   // Protect RST moduel register

    IWDT_IT_Config(IWDT_INT_TF, ENABLE);                            // Enable IWDT timeout interupt 
    NVIC_EnableIRQ(SYS_IRQn);
    SYS_ITEA_Cmd(ENABLE);                                           // Enable SYS interupt in NVIC
    IWDT_ClearFlag(IWDT_TF);                                        // Clear IWDT timeout flag

    /*=== 3. Enable IWDT module ===*/
    IWDT_Cmd(ENABLE);                                               // Enable IWDT module
    ProtectModuleReg(IWDTprotect);                                  // Protect IWDT module
    
    /*=== 4. Check flag action ===*/
    while(IWDT_GetSingleFlagStatus(IWDT_EW1F) == DRV_UnHappened);   // Wait IWDT early wakeup-1 happened
    IWDT_ClearFlag(IWDT_EW1F);                                      // Clear EW1F flag 
    
    while(IWDT_GetSingleFlagStatus(IWDT_EW0F) == DRV_UnHappened);   // Wait IWDT early wakeup-0 happened
    IWDT_ClearFlag(IWDT_EW0F);                                      // Clear EW0F flag

    IWDT_RefreshCounter();                                          // Clear IWDT timer
}


/**
 *******************************************************************************
 * @brief       SPI0 module initial for BLE
 * @details
 * @return      None
 * @note        Used pins
 *                  When select SPI0
 *                  PB0 AFS GPB0 (Software NSS)
 *                  PB1 AFS SPI0_MISO
 *                  PB2 AFS SPI0_CLK
 *                  PB3 AFS SPI0_MOSI
 *
 *                  When select URT1 and deviece is MG32F02A132
 *                  PD6 AFS GPB0 (Software NSS)
 *                  PD5 AFS SPI0_MISO
 *                  PD2 AFS SPI0_CLK
 *                  PD4 AFS SPI0_MOSI
 *
 *                  When select URT1 and deviece is MG32F02A032
 *                  PC13 AFS GPB0 (Software NSS)
 *                  PC9  AFS SPI0_MISO
 *                  PC3  AFS SPI0_CLK
 *                  PC8  AFS SPI0_MOSI
 *
 *                  When select URT1 and deviece is MG32F02U128/MG32F02U064/MG32F02A128/MG32F02A064
 *                  PC7 AFS GPB0 (Software NSS)
 *                  PC9 AFS SPI0_MISO
 *                  PC3 AFS SPI0_CLK
 *                  PC8 AFS SPI0_MOSI
 *
 *                  When select URT1 and deviece is MG32F02V032
 *                  PC13 AFS GPB0 (Software NSS)
 *                  PC9 AFS SPI0_MISO
 *                  PC1 AFS SPI0_CLK
 *                  PC8 AFS SPI0_MOSI
 *
 * @par         Example
 * @code
    BLE_SPI_Init();
 * @endcode
 *******************************************************************************
 */
void BLE_SPI_Init (MID_SPI_Struct* BLE_MD)
{
    PIN_InitTypeDef PINX_InitStruct;

    
    
    /*=== 1. Config SPI0 IO ===*/
    if(BLE_MD->STA.W == SPI0->STA.W)
    {
        // For SPI0 pins setting
        PINX_InitStruct.PINX_Mode               = PINX_Mode_PushPull_O;     // Pin select pusu pull mode
        PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;  // Enable pull up resistor
        PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;           // Pin output low speed mode
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;     // Pin output driver full strength.
        PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;// Pin input deglitch filter clock divider bypass
        PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;     // Pin input data not inverse
        PINX_InitStruct.PINX_Alternate_Function = 2;                        // Pin AFS = 2
        GPIO_PinMode_Config(PINB(2),&PINX_InitStruct);                      // CLK setup at PB2
        
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level2;     // Pin output drive strength 1/4
        GPIO_PinMode_Config(PINB(3),&PINX_InitStruct);                      // MOSI setup at PB3
        
        PINX_InitStruct.PINX_Alternate_Function = 0;                        // Pin AFS = 0
        GPIO_PinMode_Config(PINB(0),&PINX_InitStruct);                      // NSS setup at PB0
        
        PINX_InitStruct.PINX_Mode               = PINX_Mode_Digital_I;      // Setting digital  mode
        PINX_InitStruct.PINX_Alternate_Function = 2;                        // Pin AFS = 2
        GPIO_PinMode_Config(PINB(1),&PINX_InitStruct);  
    }
    else
    {
#if defined(MG32_1ST)
        // For URT1 pins setting
        PINX_InitStruct.PINX_Mode               = PINX_Mode_PushPull_O;     // Pin select pusu pull mode
        PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;  // Enable pull up resistor
        PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;           // Pin output low speed mode
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;     // Pin output driver full strength.
        PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;// Pin input deglitch filter clock divider bypass
        PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;     // Pin input data not inverse
        PINX_InitStruct.PINX_Alternate_Function = 3;                        // Pin AFS = 9
        GPIO_PinMode_Config(PIND(2),&PINX_InitStruct);                      // CLK setup at PD2
        
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level2;     // Pin output drive strength 1/4
        GPIO_PinMode_Config(PIND(4),&PINX_InitStruct);                      // MOSI setup at PD4
        
        PINX_InitStruct.PINX_Alternate_Function = 0;                        // Pin AFS = 0
        GPIO_PinMode_Config(PIND(6),&PINX_InitStruct);                      // NSS setup at PD6
        
        PINX_InitStruct.PINX_Mode               = PINX_Mode_Digital_I;      // Setting digital  mode
        PINX_InitStruct.PINX_Alternate_Function = 3;                        // Pin AFS = 2
        GPIO_PinMode_Config(PIND(5),&PINX_InitStruct);                      // MISO setup at D5
#endif

#if defined(MG32_2ND)
        // For URT1 pins setting
        PINX_InitStruct.PINX_Mode               = PINX_Mode_PushPull_O;     // Pin select pusu pull mode
        PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;  // Enable pull up resistor
        PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;           // Pin output low speed mode
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;     // Pin output driver full strength.
        PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;// Pin input deglitch filter clock divider bypass
        PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;     // Pin input data not inverse
        PINX_InitStruct.PINX_Alternate_Function = 4;                        // Pin AFS = 9
        GPIO_PinMode_Config(PINC(3),&PINX_InitStruct);                      // CLK setup at PC3
        
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level2;     // Pin output drive strength 1/4
        GPIO_PinMode_Config(PINC(8),&PINX_InitStruct);                      // MOSI setup at PC8
        
        PINX_InitStruct.PINX_Alternate_Function = 0;                        // Pin AFS = 0
        GPIO_PinMode_Config(PINC(13),&PINX_InitStruct);                      // NSS setup at PC13
        
        PINX_InitStruct.PINX_Mode               = PINX_Mode_Digital_I;      // Setting digital  mode
        PINX_InitStruct.PINX_Alternate_Function = 4;                        // Pin AFS = 2
        GPIO_PinMode_Config(PINC(9),&PINX_InitStruct);                      // MISO setup at PC9
#endif

#if defined(MG32_3RD)
        // For URT1 pins setting
        PINX_InitStruct.PINX_Mode               = PINX_Mode_PushPull_O;     // Pin select pusu pull mode
        PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;  // Enable pull up resistor
        PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;           // Pin output low speed mode
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;     // Pin output driver full strength.
        PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;// Pin input deglitch filter clock divider bypass
        PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;     // Pin input data not inverse

        PINX_InitStruct.PINX_Alternate_Function = 4;                        // Pin AFS = 4
        GPIO_PinMode_Config(PINC(3),&PINX_InitStruct);                      // CLK at PC3
        
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level2;     // Pin output drive strength 1/4
        GPIO_PinMode_Config(PINC(8),&PINX_InitStruct);                      // MOSI at PC8
        
        PINX_InitStruct.PINX_Alternate_Function = 0;                        // Pin AFS = 0
        GPIO_PinMode_Config(PINC(7),&PINX_InitStruct);                      // NSS at PC7
        
        PINX_InitStruct.PINX_Mode               = PINX_Mode_Digital_I;      // Setting digital  mode
        PINX_InitStruct.PINX_Alternate_Function = 4;                        // Pin AFS = 2
        GPIO_PinMode_Config(PINC(9),&PINX_InitStruct);                      // MISO at PC9
#endif

#if defined(MG32_4TH)
        // For URT1 pins setting
        PINX_InitStruct.PINX_Mode               = PINX_Mode_PushPull_O;     // Pin select pusu pull mode
        PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;  // Enable pull up resistor
        PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;           // Pin output low speed mode
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;     // Pin output driver full strength.
        PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;// Pin input deglitch filter clock divider bypass
        PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;     // Pin input data not inverse
        PINX_InitStruct.PINX_Alternate_Function = 4;                        // Pin AFS = 4
        GPIO_PinMode_Config(PINC(1),&PINX_InitStruct);                      // CLK setup at PC1
        
        PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level2;     // Pin output drive strength 1/4
        GPIO_PinMode_Config(PINC(8),&PINX_InitStruct);                      // MOSI setup at PC8
        
        PINX_InitStruct.PINX_Alternate_Function = 0;                        // Pin AFS = 0
        GPIO_PinMode_Config(PINC(13),&PINX_InitStruct);                      // NSS setup at PC13
        
        PINX_InitStruct.PINX_Mode               = PINX_Mode_Digital_I;      // Setting digital  mode
        PINX_InitStruct.PINX_Alternate_Function = 4;                        // Pin AFS = 2
        GPIO_PinMode_Config(PINC(9),&PINX_InitStruct);                      // MISO setup at PC9
#endif
    }
    /*=== 2. Configure SPI ===*/
    SPIx_Init(BLE_MD, CONF_CK_APB_FREQ, 0);
}


/**
 *******************************************************************************
 * @brief       LED module initial for BLE
 * @details
 * @return      None
 * @note        Used pins
 *                  PD8 for MG32F02A032
 *                  PE13 for MG32F02A132/072 MA862
 * @par         Example
 * @code
    BLE_LED_Init();
 * @endcode
 *******************************************************************************
 */
void BLE_LED_Init(void)
{
    PIN_InitTypeDef        PINX_InitStruct;
    
    PINX_InitStruct.PINX_Mode               = PINX_Mode_PushPull_O;     // Pin select pusu pull mode
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;  // Enable pull up resistor
    PINX_InitStruct.PINX_Speed              = PINX_Speed_Low;           // Pin output low speed mode
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;     // Pin output driver full strength.
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;// Pin input deglitch filter clock divider bypass
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;     // Pin input data not inverse
    PINX_InitStruct.PINX_Alternate_Function = 0;                        // Pin AFS = 0 (GPIO)

    GPIO_PinMode_Config(LED1_PINSET,&PINX_InitStruct);
    LED1_PIN = LED_OFF; 
}


/**
 *******************************************************************************
 * @brief       LED module initial
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    LED_Flash();
 * @endcode
 *******************************************************************************
 */
void LED_Flash(void)
{
    LED1_PIN ^= 0x1;
}


/**
 *******************************************************************************
 * @brief       BSP initial
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    BSP_Init();
 * @endcode
 *******************************************************************************
 */
void BSP_Init (void)
{
    BLE_SPI_Init (SPI_SPI0);
    
    /*=== Try ===*/
    BLE_EXIC_Init();
    BLE_LED_Init();
    InitTick(12000000,0);
    IWDT_Init();
}


/**
 *******************************************************************************
 * @brief       IS IRQ Enable
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    IsIrqEnabled();
 * @endcode
 *******************************************************************************
 */
char IsIrqEnabled (void) //porting api
{
    return !(EXIC_PIN);
}


/**
 *******************************************************************************
 * @brief       IRQ MCU go to sleep and wakeup
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    IrqMcuGotoSleepAndWakeup();
 * @endcode
 *******************************************************************************
 */
void IrqMcuGotoSleepAndWakeup(void)
{
    if(ble_run_interrupt_McuCanSleep())
    {
        //mcu lowpower
    }
}


static volatile unsigned int SysTick_Count = 0;


/**
 *******************************************************************************
 * @brief       Get sys tick count
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    GetSysTickCount();
 * @endcode
 *******************************************************************************
 */
unsigned int GetSysTickCount(void) //porting api
{
    return SysTick_Count;
}


/**
 *******************************************************************************
 * @brief       Sys Tick Handler
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    SysTick_Handler();
 * @endcode
 *******************************************************************************
 */
void SysTick_Handler(void)
{
    SysTick_Count ++;
    ble_nMsRoutine();
}


/**
 * @name    BLE release 
 *
 */ 
///@{
/**
 *******************************************************************************
 * @brief       Write BLE_TX_BUF buffer
 * @details
 * @param[in]   SrcIndex: Source address.
 * @param[in]   len: Transfer length bytes number. 1 ~ 200 bytes.
 * @return      None
 * @note
 * @par         Example
 * @code
    BSP_BLE_WriteTxBuffer(TXBUF, 10);
    BSP_BLE_WriteTxBuffer(&TXBUF[0], 10);
 * @endcode
 *******************************************************************************
 */
void BSP_BLE_WriteTxBuffer (u8* SrcIndex, u8 len)
{
    u8  i;                                  // 

    for(i=len; i>0; i--)                        // for loop 
    {
        BLE_TX_BUF[BLE_TX_INDEX] = *SrcIndex;   // Move data to BLE_TX_BUF.
        SrcIndex++;                             // Update source index address.
        BLE_TX_INDEX++;                         // Update BLE TX BUF index.
        if(BLE_TX_INDEX >= MAX_SIZE)            // When BLE TX BUF index >= MAX_SIZE (200).
        {
            BLE_TX_INDEX = 0;                   // BLE TX BUF index = 0.
        }
    }
}


/**
 *******************************************************************************
 * @brief       Move BLE_RX_BUF data to DestIndex 
 * @details
 * @param[in]   DestIndex: Destination address.
 * @return      Received data byte number. 1 ~ 200 bytes.
 * @note
 * @par         Example
 * @code
    Length = BSP_BLE_ReadRxBuffer(RXBUF);
 * @endcode
 *******************************************************************************
 */
u8 BSP_BLE_ReadRxBuffer (u8* DestIndex)
{
    u8  i;                                          // Loop parameter
    u8  Len;                                        // Get received length

    Len = BLE_RX_LEN - BLE_RX_STR;                  // Get length
    for(i=Len; i>0; i--)                            // Move loop
    {
        *DestIndex++ = BLE_RX_BUF[BLE_RX_STR++];    // Move data to DestIndex
    }
    return Len;                                     // Return receive length
}


/**
 *******************************************************************************
 * @brief       Clear BLE_TX_BUF index and set BLE_TCF = 1
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    BSP_BLE_ClearBleTxBuffer();
 * @endcode
 *******************************************************************************
 */
void BSP_BLE_ClearBleTxBuffer (void)
{
    BLE_TX_INDEX = 0;
    BLE_TCF = 1;
    (void)BLE_TCF;
}


/**
 *******************************************************************************
 * @brief       Clear CLE_RX_BUF index and BLE_RXF = 0
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    BSP_BLE_ClearBleRxBuffer();
 * @endcode
 *******************************************************************************
 */
void BSP_BLE_ClearBleRxBuffer (void)
{
    BLE_RX_STR = 0;
    BLE_RX_LEN = 0;
    BLE_RXF = 0;
}
///@} 


/**
 *******************************************************************************
 * @brief       Write BLE_RX_BUF buffer
 * @details
 * @param[in]   DestIndex: Destination address.
 * @param[in]   len: Transfer length bytes number. 1 ~ 200 bytes.
 * @return      None
 * @note
 * @par         Example
 * @code
    BSP_BLE_WriteRxBuffer(RXBUF, 10);
    BSP_BLE_WriteRxBuffer(&RXBUF[0], 10);
 * @endcode
 *******************************************************************************
 */
void BSP_BLE_WriteRxBuffer (u8* DestIndex, u8 len)          //api, BLE received data to txBuf   
{
    unsigned char i;

    if ((BLE_RX_LEN+len)<MAX_SIZE)                          // When txBuf waiting transmit index + updata length < MAX_SIZE (200)
    {
        for (i=0;i<len;i++)                                 // Move BLE get data to BLE_RX_BUF[]
        {
            BLE_RX_BUF[BLE_RX_LEN+i] = *(DestIndex+i);
        }
        BLE_RX_LEN += len;                                  // txBuf waiting transmit index + update length
    }
}


#define BLE_TxBufIndex  BLE_TX_INDEX        // BLE transmit buffer index
static volatile u16 BLE_TxdBufIndex = 0;           // BLE transmitted buffer index


/**
 *******************************************************************************
 * @brief       Send BLE_TX_BUF data
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    BSP_BLE_SendData();
 * @endcode
 *******************************************************************************
 */
void BSP_BLE_SendData (void) 
{
    u16 send;

    if(BLE_TxBufIndex != BLE_TxdBufIndex)           // When BLE transmit buffer index != BLE transmitted buffer index
    {
        
        if(!GetConnectedStatus())                   // When not Connected
        {
            BLE_TxdBufIndex = BLE_TxBufIndex;       // Update BLE transmitted buffer index, BLE transmit buffer index.
        }
        else                                        // When Connected
        {
            if(BLE_TxBufIndex > BLE_TxdBufIndex)    // When BLE transmit buffer index > BLE transmitted buffer index
            {
                send = sconn_notifydata(BLE_TX_BUF+BLE_TxdBufIndex, (unsigned char)(BLE_TxBufIndex - BLE_TxdBufIndex));
                                                    // BLE transmit data and return transmit length.
                BLE_TxdBufIndex += send;            // Updata BLE transmitted index
            }
            else                                    // When BLE transmit buffer index =< BLE transmitted buffer index
            {
                send = sconn_notifydata(BLE_TX_BUF+BLE_TxdBufIndex, (unsigned char)(MAX_SIZE - BLE_TxdBufIndex));
                                                    // BLE transmit data and return transmit length.
                BLE_TxdBufIndex += send;            // Updata BLE transmitted buffer index
                BLE_TxdBufIndex %= MAX_SIZE;        // Calculate BLE transmitted index.
                
                /*=== Transmit test ===*/
            }
        }
        
        if(BLE_TxBufIndex == BLE_TxdBufIndex)       // When BLE_TX_BUF transmit complete.
        {
            BLE_TCF = 1;                            // BLE TX transmited complet flag.
        }
    }
}


/**
 *******************************************************************************
 * @brief       User Proc Callback
 * @details
 * @return      None
 * @note
 * @par         Example
 * @code
    UsrProcCallback();
 * @endcode
 *******************************************************************************
 */
void UsrProcCallback(void)                  // Porting api
{    
    IWDT_RefreshCounter();                  // Refresh IWDT counter
    
    LED_Flash();                            // Switch LED outpput state (Flashing)
    
    BSP_BLE_SendData();                     // Move rxBuf data to BLE
    
    if ((BLE_RX_LEN) && (0 == BLE_RX_STR))  // When BLE receive start = 0 and BLE recevied length
    {
        BLE_RXF = 1;                        /* Trigger BLE RX Flag */
    }
}


/**
 *******************************************************************************
 * @brief       SPI0 module initial
 * @details
 * @param[in]   ProcessClockRate
 *  @arg\b          12000000 ~ 48000000
 * @param[in]   BLEClockRate
 *  @arg\b          0, is auto setting BLE clock rate 6000000 ~ 10000000.
 *  @arg\b          6000000 ~ 10000000, setting BLE clock rate.
 * @return      None
 * @note
 * @par         Example
 * @code
    SPIx_Init(12000000, 0);
 * @endcode
 *******************************************************************************
 */
//DRV_Return SPIx_Init (uint32_t ProcessClockRate, uint32_t BLEClockRate)
DRV_Return SPIx_Init (MID_SPI_Struct* BLE_MD, uint32_t ProcessClockRate, uint32_t BLEClockRate)
{
//    volatile ctype    URT_RecData;              /* volatile */
    volatile uint8_t  tmp;                      /* volatile */
    volatile uint8_t  CNT;                      /* volatile */
    volatile float    SPI_Clock;                /* volatile */
    volatile uint32_t DIV;                      /* volatile */



    (void)BLEClockRate;                                                 // Solve the parameter warning 
    // When CK_AHB > 12MHz
    if(ProcessClockRate > 12000000)
    {
        CPU_CLK_FAST_ENABLE = TRUE;
    }
    else
    {
        CPU_CLK_FAST_ENABLE = FALSE;
    }

    // When connect is SPI0
    if(BLE_MD->STA.W == SPI0->STA.W)
    {
    //    /*=== 1. Default Initial SPI ===*/
    //    SPI_DeInit(SPI0);

    //    /*=== 2. Configure clock divider ===*/                      // SPI clock = 1MHz
    //    SPI_Clock_Select(SPI0, SPI_CK_SPIx_PR);                     // CK_SPIx = CK_SPIx_PR
    //    SPI_PreDivider_Select(SPI0, SPI_PDIV_1);                    // PDIV outpu = CK_SPIx /2
    //    SPI_Prescaler_Select(SPI0, SPI_PSC_1);                      // Prescaler outpu = PDIV outpu /3
    //    SPI_Divider_Select(SPI0, SPI_DIV_2);                        // DIV outpu = PDIV outpu /2

    //    /*=== 3. Configure SPI data line, mode and data size... ===*/
    //    SPI_DataLine_Select(SPI0, SPI_Standard);                    // SPI Standard mode 
    //    SPI_ModeAndNss_Select(SPI0, SPI_Master);                    // Master
    //    SPI_ClockPhase_Select(SPI0, SPI_LeadingEdge);               // CPHA = 0
    //    SPI_ClockPolarity_Select(SPI0, SPI_Low);                    // CPOL = 0
    //    SPI_FirstBit_Select(SPI0, SPI_MSB);                         // MSB first
    //    SPI_DataSize_Select(SPI0, SPI_8bits);                       // Data size 8bits

    //    /*=== 4. Enable SPI ===*/
    //    SPI_Cmd(SPI0, ENABLE);                                      // Enable SPI0 module
        
        /* Get Prescaler setting */
        DIV = 0xFF;
        CNT = 0;
        do
        {
            CNT++;
            SPI_Clock = (float) ProcessClockRate;
            SPI_Clock = (SPI_Clock /CNT) /2;
            if((SPI_Clock >= 6000000) && (SPI_Clock < 10000000))        // When SPI clock range in 6M ~ 10M
            {
                DIV = CNT;                                              // Store DIV 
            }
        }while(CNT<8);                                                  // When 
        
        DIV--;
        if(DIV > 8)                                                     // When not get DIV
            return DRV_Failure;
        
        /*=== 1. Default Initial SPI ===*/
        BLE_MD->CR0.W = 0x00000000;
        BLE_MD->STA.W = 0x00000FD8;
        BLE_MD->INT.W = 0x00000000;
        BLE_MD->CLK.W = 0x00000000;
        BLE_MD->CR1.W = 0x00000000;
        BLE_MD->CR2.W = 0x03000100;

        /*=== 2. Configure clock divider ===*/                          // SPI clock = 1MHz
        BLE_MD->CLK.W = SPI_CLK_CK_SEL_proc_w |                          // CK_SPIx = CK_SPIx_PR 
                       SPI_CLK_CK_DIV_div2_w |                          // DIV outpu = PDIV outpu /1
                       ((uint32_t)DIV << 8) |                           // Prescaler outpu = PDIV outpu /1
                       SPI_CLK_CK_PDIV_div1_w;                          // PDIV outpu = CK_SPIx /2
     
        /*=== 3. Configure SPI data line, mode and data size... ===*/
        BLE_MD->CR2.W |= SPI_CR2_DAT_LINE_spi_w |                        // SPI Standard mode 
                        (SPI_CR2_DSIZE_mask_w & 0x00080000);            // Data size 8bits

        BLE_MD->CR0.W |= SPI_CR0_MDS_master_w |
                        SPI_CR0_CPHA_leading_edge_w |
                        SPI_CR0_CPOL_low_w |
                        SPI_CR0_LSB_EN_disable_w;

        /*=== 4. Enable SPI ===*/
        BLE_MD->CR0.W |= SPI_CR0_EN_enable_w;                            // Enable SPI0 module
    }
    else
    {
        //==Set CSC init
        //MG32_CSC_Init.h(Configuration Wizard)
        //Select CK_HS source = CK_IHRCO
        //Select IHRCO = 11.0592M
        //Select CK_MAIN Source = CK_HS
        //Configure PLL->Select APB Prescaler = CK_MAIN/1
        //Configure Peripheral On Mode Clock->Port B/URT0 = Enable
        //Configure Peripheral On Mode Clock->URT0->Select URT0_PR Source = CK_APB(11.0592)
        
        //==Set GPIO init
        //1. MOSI Pin
        //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin8
        //    (2).GPIO port initial is 0xFFFF
        //    (3).Pin8 mode is PPO (Push pull output)
        //    (4).Pin8 pull-up resister Enable
        //    (5).Pin8 function URT0_TX
        //2. MISO Pin
        //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin9
        //    (2).GPIO port initial is 0xFFFF
        //    (3).Pin8 mode is ODO (Open drain)
        //    (4).Pin8 pull-up resister Enable
        //    (5).Pin9 function URT0_RX
        //3. SPICLK Pin
        //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOC->Pin0
        //    (2).GPIO port initial is 0xFFFF
        //    (3).Pin8 mode is PPO (Push pull output)
        //    (4).Pin8 pull-up resister Enable
        //    (5).Pin3 function URT0_CLK
        //4. NSS Pin
        //    (1).MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOB->Pin10
        //    (2).GPIO port initial is 0xFFFF
        //    (3).Pin8 mode is PPO (Push pull output)
        //    (4).Pin8 pull-up resister Enable
        //    (5).Pin3 function GPIO (the NSS is software control)
        
        /* Get Prescaler setting */
        DIV = 0xFF;
        CNT = 0;
        do
        {
            CNT++;
            SPI_Clock = (float) ProcessClockRate;
            SPI_Clock = (SPI_Clock /(CNT+1)) ;
            if((SPI_Clock >= 6000000) && (SPI_Clock < 10000000))        // When SPI clock range in 6M ~ 10M
            {
                DIV = CNT;                                              // Store DIV 
            }
        }while(CNT<8);                                                  // When 
        
        if(DIV > 8)                                                     // When not get DIV
            return DRV_Failure;
        
        //=====Set Clock=====//
        //---Set BaudRate---//
        BLE_MD->CLK.W = (BLE_MODULE->CLK.W & ~(URT_CLK_BR_MDS_mask_w | URT_CLK_CK_SEL_mask_w)) |
                       URT_CLK_CK_SEL_proc_w | URT_CLK_BR_MDS_separated_w;
        BLE_MD->RLR.B[0] = 0;
        BLE_MD->RLR.B[1] = 0;
        BLE_MD->CLK.W = ((BLE_MODULE->CLK.W & (~URT_CLK_BR_EN_mask_w)) |
                       URT_CLK_BR_EN_mask_w);

        //---TX/RX Clock---//
        BLE_MD->CLK.W = ((BLE_MODULE->CLK.W & ~(URT_CLK_TX_CKS_mask_w | URT_CLK_RX_CKS_mask_w)) | 
                       URT_CLK_TX_CKS_internal_w | URT_CLK_RX_CKS_internal_w);
        BLE_MD->CR1.B[3] = (uint8_t)DIV;
        BLE_MD->CR1.B[1] = (uint8_t)DIV;
        BLE_MD->CR0.W = ((BLE_MODULE->CR0.W & ~(URT_CR0_OS_MDS_mask_w)) | 
                       URT_CR0_OS_MDS_three_w);
        BLE_MD->CR2.W = ((BLE_MODULE->CR2.W & ~(URT_CR2_TX_EN_mask_w | URT_CR2_RX_EN_mask_w)) | 
                       URT_CR2_TX_EN_enable_w | URT_CR2_RX_EN_enable_w);

        BLE_MD->CR1.B[2] = URT_CR1_TXDSIZE_8bit_b2 | URT_CR1_TXMSB_EN_enable_b2 | URT_CR1_TXPAR_EN_disable_b2 | URT_CR1_TXSTP_LEN_1bit_b2;
        BLE_MD->CR1.B[0] = URT_CR1_RXDSIZE_8bit_b0 | URT_CR1_RXMSB_EN_enable_b0 | URT_CR1_RXPAR_EN_disable_b0 | URT_CR1_RXSTP_LEN_1bit_b0;

        tmp = BLE_MODULE->CR4.B[0] & (~(URT_CR4_RDAT_INV_mask_b0|URT_CR4_TDAT_INV_mask_b0));
        tmp = tmp | URT_CR4_RDAT_INV_disable_b0 | URT_CR4_TDAT_INV_disable_b0;
        BLE_MD->CR4.B[0] = tmp;

        // Set Mode Select
        BLE_MD->CR0.W = (BLE_MODULE->CR0.W & (~URT_CR0_MDS_mask_w) ) | URT_CR0_MDS_sync_w; 
//        BLE_MODULE->MUTE.W = BLE_MODULE->MUTE.W & (~URT_MUTE_MUTE_EN_mask_w);
        // Set DataLine Select
        BLE_MD->CR0.W = ((BLE_MODULE->CR0.W & (~URT_CR0_DAT_LINE_mask_w)) | URT_CR0_DAT_LINE_2_w);
        BLE_MD->CR3.W = ((BLE_MODULE->CR3.W & (~URT_CR3_CPHA_mask_w)) | URT_CR3_CPHA_leading_edge_w);
        BLE_MD->CR3.W = ((BLE_MODULE->CR3.W & (~URT_CR3_CPOL_mask_w)) | URT_CR3_CPOL_low_w);

        // Flash_nCS = FLASH_nCS_NoACTIVE;
        SPI_NSS = 1;
        // Set Data Control
        BLE_MD->CR3.B[2] = 0;
        // Enable CLK(SYSCLK)
        BLE_MD->CLK.W = (( BLE_MODULE->CLK.W & (~URT_CLK_CLK_EN_mask_w)) | ( URT_CLK_CLK_EN_mask_w * 1));
        // Enable URT
        BLE_MD->CR0.W = ((BLE_MODULE->CR0.W & (~URT_CR0_EN_mask_w)) | (URT_CR0_EN_mask_w * 1));
    }
    
    return DRV_Success;
}


/**
 *******************************************************************************
 * @brief       SPI write buffer
 * @details  
 * @param[in]   reg: Command
 * @param[in]   *pBuf: Write buffer index
 * @param[in]   len: Data length
 *  @arg\b          0~255
 * @return      DRV_Return
 *  @arg\b          0
 * @note
 * @par         Example
 * @code
    SPI_WriteBuf(CMD, &TXBuf, 128);
 * @endcode
 *******************************************************************************
 */
unsigned char SPI_WriteBuf(unsigned char reg, unsigned char const *pBuf, unsigned char len)
{
    volatile signed char i=0,j=0;               /* volatile */
    
    
    
    SPI_NSS = 0;                                                    // NSS = 0
    
    BLE_MODULE->TDAT.B[0] = reg;                                    // Write 1 byte
    
    for (i=(signed char)len;i>3;i-=4)
    {                                                               // Write 4 bytes
        Temp32 = *(pBuf+j) | 
                 (unsigned long) (*(pBuf+j+1)<<8) | 
                 (unsigned long) (*(pBuf+j+2)<<16) | 
                 (unsigned long) (*(pBuf+j+3)<<24);
        j+=4;
        BLE_MODULE->TDAT.W = Temp32;
        while((BLE_MODULE->STA.B[0] & SPI_STA_TXF_mask_b0) == 0);   // Wait TDAT data move to TX shadow buffer
    }
    
    for (;i>2;i-=3)
    {                                                               // Write 3 bytes
        Temp24 = *(pBuf+j) | 
                 (unsigned long) (*(pBuf+j+1)<<8) | 
                 (unsigned long) (*(pBuf+j+2)<<16);
        j+=3;
        BLE_MODULE->TDAT3.W = Temp24;
        while((BLE_MODULE->STA.B[0] & SPI_STA_TXF_mask_b0) == 0);   // Wait TDAT data move to TX shadow buffer
    }
    
    for (;i>1;i-=2)
    {                                                               // Write 2 bytes
        Temp16 = *(pBuf+j) | 
                 (unsigned short)(*(pBuf+j+1)<<8);
        j+=2;                                                       // Index parameter + 2
        BLE_MODULE->TDAT.H[0] = Temp16;                             // Write 2 bytes to TX
        while((BLE_MODULE->STA.B[0] & SPI_STA_TXF_mask_b0) == 0);   // Wait TDAT data move to TX shadow buffer
    }
    
    for (;i>0;i--)
    {
        Temp8 = *(pBuf+j);                                          // Write 1 byte
        j++;
        BLE_MODULE->TDAT.B[0] = Temp8;
        while((BLE_MODULE->STA.B[0] & SPI_STA_TXF_mask_b0) == 0);   // Wait TDAT data move to TX shadow buffer
    }    
    
    
    if(&BLE_MODULE->STA.W == &SPI0->STA.W)                          // When BLE module used SPI0
    {
        while((BLE_MODULE->STA.B[0] & SPI_STA_TCF_mask_b0) == 0);   // Wait TX data transmit complete
        BLE_MODULE->CR1.B[0] = SPI_CR1_RDAT_CLR_mask_b0;            // Clear RX buffe data
        BLE_MODULE->STA.H[0] = SPI_ALLF;                            // Clear SPI all flag
    }
    else                                                            // When BLE moudle used URTx
    {
        while((BLE_MODULE->STA.B[0] & URT_STA_TCF_mask_b0) == 0);   // Wait TX data transmit complete
        BLE_MODULE->CR4.B[0] = URT_CR4_RDAT_CLR_mask_b0;            // Clear RX buffe data
        BLE_MODULE->STA.W = URT_IT_Mask;                            // Clear URTx all flag
    }

    SPI_NSS = 1;                        // NSS = 1
    return 0;
}


/**
 *******************************************************************************
 * @brief       SPI Read buffer
 * @details  
 * @param[in]   reg: Command
 * @param[in]   *pBuf: Read buffer index
 * @param[in]   len: Data length
 *  @arg\b          0~255
 * @return      DRV_Return
 *  @arg\b          0
 * @note
 * @par         Example
 * @code
    SPI_ReadBuf(CMD, &RXBuf, 128);
 * @endcode
 *******************************************************************************
 */
unsigned char SPI_ReadBuf(unsigned char reg, unsigned char *pBuf, unsigned char len)
{
    volatile unsigned char i=0;                 /* volatile */



    SPI_NSS = 0;                                                    // NSS = 0
    
    BLE_MODULE->TDAT.B[0] = reg;                                    // Write 1 byte
    if(CPU_CLK_FAST_ENABLE == FALSE)                                // When CK_AHB <= 12MHz
    {
        BLE_MODULE->TDAT.B[0] = 0xff;                                   // Write 1 byt 
    }
    while((BLE_MODULE->STA.B[0] & SPI_STA_RXF_mask_b0) == 0);       // Wait RDAT not empty
    BLE_MODULE->RDAT.B[0];                                          // Clear RDAT
    
    for (i=0;i<len;i++)                                             // Waiting  
    {
        BLE_MODULE->TDAT.B[0] = 0xff;                               // Write 1 byt
        while((BLE_MODULE->STA.B[0] & SPI_STA_RXF_mask_b0) == 0);   // Wait RDAT not empty
        *(pBuf+i) = BLE_MODULE->RDAT.B[0];                          // Read data
    }
       /*wait tCF transm complete*/
    SPI_NSS = 1;                                                    // NSS = 1
    
    if(&BLE_MODULE->STA.W == &SPI0->STA.W)                          // When BLE module used SPI0
    {
        BLE_MODULE->CR1.B[0] = SPI_CR1_RDAT_CLR_mask_b0;            // Clear RX buffe data
    }
    else
    {
        BLE_MODULE->CR4.B[0] = URT_CR4_RDAT_CLR_mask_b0;            // Clear RX buffe data
    }
    return 0;
}


//////DO NOT REMOVE, used in ble lib///////
void SysClk8to48(void)
{
}
void SysClk48to8(void)
{
}
void DisableEnvINT(void)
{
}
void EnableEnvINT(void)
{
}
void UpdateLEDValueAll(void)
{
}
void McuGotoSleepAndWakeup(void)
{
}

unsigned char aes_encrypt_HW(unsigned char *_data, unsigned char *_key)
{
    (void)_data;    // Solve the parameter warning      
    (void)_key;     // Solve the parameter warning  
    return 0;       //Hardware AES not supported
}


