/**
  ******************************************************************************
 *
 * @file        Sample_MID_TM26_OnePulse.c
 *
 * @brief       Output 2-channels PWM of TM26 dependent on TRGI event.
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2021/05/27
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
* @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_TM_MID.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
//*** <<< Use Configuration Wizard in Context Menu >>> ***
//  <h> TM26 counter period
//      <o0> prescaler counter range (1~65536)  <1-65536>
//      <o1> main counter range (1~65536) <1-65536>
//  </h>
#define TM26_PrescalerCounter_Range 1
#define TM26_MainCounter_Range      1024

//  <h> Duty cycle setting 
//      <o0> Channel0 match point (1~65536) <1-65536>
//      <o1> Channel1 match point (1~65536) <1-65536>
//  </h>
#define TM26_Channel0_MatchPoint    100
#define TM26_Channel1_MatchPoint    200

//  <o0> One pulse output - Repetition count (1~256) <1-256>
//      <i> Input range is 1~256. 1 = Only one PWM cycle.
#define RepCOUNT                    10

//*** <<< end of configuration section >>>    ***

/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static TM_HandleTypeDef    mTM26;

/* Private function prototypes -----------------------------------------------*/
void Sample_MID_TM26_OnePulse(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	    Output 2-channels PWM of TM26 when TRGI(ETR) event happened.
 *              Then it will to stop PWM pulse when output N times.
 * @details     1.Initial TM26 32 bit counter range.
 *    \n        2.Initial TM26 clock mode.
 *    \n        3.TM26 PWM output channel configuration.
 *    \n        4.Config TM26 with slave mode.
 *    \n        5.Config One Pulse mode and start it.
 *    \n        6.Start PWM output.
 *    \n        7.Update duty cycle.
 * @return      None
 * @note        It will always output PWM cycle when TRGI happen.
 *******************************************************************************
 */
void Sample_MID_TM26_OnePulse(void)
{      
    TM_OC_InitTypeDef       sConfig;    
    TM_ClockConfigTypeDef   CKConfig;
    TM_SlaveConfigTypeDef   sSlave;
    TM_RCNTConfigTypeDef    TMRCNT;
    

    // make sure :
    
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //  Select CK_PLLI Divider = CK_HS/2
    //  Select CK_PLL Multiplication factor = CK_PLLIx16
    //  Select CK_PLLO Divider = CK_PLL/4
    //Select CK_MAIN Source = CK_PLLO
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->TM26 = Enable
	
    //==Set GPIO init 
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIAC->PA12, PA13
    //GPIO port initial is 0xFFFF
    //PA12, PA13 mode is PPO
    //PA12, PA13 function TM26_OC00, TM26_OC10
    
    //==Set GPIO init 
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIAB->PB5
    //GPIO port initial is 0xFFFF
    //PB5 mode is DIN with pull-up
    //PB5 function TM26_ETR
    
    // ------------------------------------------------------------------------
    // 1.Initial TM26 for 10-bit PWM.
    // ------------------------------------------------------------------------
    mTM26.Instance                  = TM26;
    mTM26.Init.TM_CounterMode       = TM_SEPARATE_UP;
    mTM26.Init.TM_Period            = TM26_MainCounter_Range - 1;
    mTM26.Init.TM_Prescaler         = TM26_PrescalerCounter_Range - 1;
    MID_TM_PWM_Init(&mTM26);
    
    // ------------------------------------------------------------------------
    // 2.Initial TM26 clock mode.
    // ------------------------------------------------------------------------
    CKConfig.TM_ClockSource         = TM_INTERNAL_CLOCK;
    CKConfig.TM_ExternalClockSource = 0;
    CKConfig.TM_INTClockDivision    = TM_INTERNALCLOCK_DIVDER_DIV1;
    CKConfig.TM_InternalClockSource = TM_INTERNALCLOCK_PROC;
    MID_TM_ConfigClockSource(&mTM26, &CKConfig);  
    
    // ------------------------------------------------------------------------
    // 3.TM26 PWM output channel configuration.
    // ------------------------------------------------------------------------
    MID_TM_OC_Struct_Init(&sConfig);            // default initial (output state)
    
    sConfig.OCMode                  = TM_CH_16bit_PWM;
    sConfig.Pulse                   = TM26_Channel0_MatchPoint; // Modify Duty cycle
    MID_TM_PWM_ConfigChannel(&mTM26, &sConfig, MID_TM_Channel0);
    
    sConfig.Pulse                   = TM26_Channel1_MatchPoint; // Modify Duty cycle
    MID_TM_PWM_ConfigChannel(&mTM26, &sConfig, MID_TM_Channel1);    
    
    // ------------------------------------------------------------------------
    // 4.Config TM26 with slave mode.
    // ------------------------------------------------------------------------
    sSlave.SlaveMode                = TM_SLAVEMODE_TRIGGER_RISINGEDGE;
    sSlave.InputTrigger             = TM_TS_ETR;
    MID_TM_SlaveConfigSynchronization(&mTM26, &sSlave);
    
    // ------------------------------------------------------------------------
    // 5.Config One Pulse mode and start it.
    // ------------------------------------------------------------------------
    TMRCNT.RCNTClkSrc               = TM_RCNT_MAIN_EVENT;
    TMRCNT.RCNTCounter              = RepCOUNT - 1; // first ouput times    
    TMRCNT.RCNTReload               = RepCOUNT - 1; // reload times
    MID_TM_OnePulse_Init(&mTM26, &TMRCNT);
    MID_TM_OnePulse_Start(&mTM26);
//    MID_TM_OnePulse_Start_IT(&mTM26);
    
    // ------------------------------------------------------------------------
    // 6.Start PWM output.
    // ------------------------------------------------------------------------
    MID_TM_PWM_Start(&mTM26, MID_TM_Channel0);      // Channel0 be 16bit PWM output (Blocking mode: Polling) 
    MID_TM_PWM_Start(&mTM26, MID_TM_Channel1);      // Channel1 be 16bit PWM output (Blocking mode: Polling)

	// ------------------------------------------------------------------------
	// 7.Update duty cycle.
	// ------------------------------------------------------------------------
	__DRV_TM_SET_COMPARE_B(&mTM26, MID_TM_Channel0, 511);
	__DRV_TM_SET_COMPARE_B(&mTM26, MID_TM_Channel1, 255);

    
    return;
    
}

//void MID_TM_OnePulseCallback(TM_HandleTypeDef* mTM)
//{
//    PB6 = !PB6;
//}
