/**
  ******************************************************************************
 *
 * @file        Sample_RTC_GeneralTimer.c
 *
 * @brief       RTC general timer sample code,  every overflow into interrupt.
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2022/12/19
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 Megawin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

 

/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"
#include "MG32.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define RTC_Rload_Max 0xFFFFFFFE
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
#if defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) /* ARM Compiler V6 */
    #pragma clang diagnostic push
    #pragma clang diagnostic ignored "-Wmissing-variable-declarations"
        uint32_t RTC_CNT;									  /* Get RTC Counter Value */
    #pragma clang diagnostic pop
#else
    uint32_t RTC_CNT;									      /* Get RTC Counter Value */
#endif

/* Private function prototypes -----------------------------------------------*/
uint32_t Sample_RTC_GeneralTimer (void);
void SYS_IRQHandler(void);
void RTC_IRQ(void);
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief  	    Sample RTC init
 * @details     1. Enable CSC to RTC clock
 *      \n      2. Configure RTC clock
 *      \n      3. Set RTC timer value
 *      \n      4. Set RTC Interrupt Set
 *      \n      5. Enable RTC module
 * @return	    None	
 * @note        
 * @par         Example
 * @code
    Sample_RTC_Init();
 * @endcode
 *******************************************************************************
 */
uint32_t Sample_RTC_GeneralTimer (void)
{
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12Mz
    //Select CK_MAIN Source = CK_HS
    //Select CK_LS Source = CK_ILRCO
    //Configure Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->RTC = Enable

    /*=== 1. Enable CSC to RTC clock ===*/
    UnProtectModuleReg(CSCprotect);                                 // Unprotect CSC module
    CSC_PeriphOnModeClock_Config(CSC_ON_RTC, ENABLE);               // Enable RTC module clock
    ProtectModuleReg(CSCprotect);                                   // protect CSC module
      
    /*=== 2. Configure RTC clock ===*/
    UnProtectModuleReg(RTCprotect);                                 // Unprotect RTC module
    RTC_CLK_Select(RTC_CK_LS);                                      // RTC clock source = CK_LS
    RTC_PreDivider_Select(RTC_PDIV_4096);                           // PDIV output = RTC clock / 4096
    RTC_Divider_Select(RTC_DIV_4);                                  // DIV output = (RTC clock / 4096) / 8
    RTC_OutputSignal_Select(RTC_PC);                                // RTC output = DIV otuput frequency
    
    /*=== 3. Set RTC timer value ===*/
    RTC_RCR_Mode_Select(RTC_RCR_MOD_ForceReload);                   // RTC switch to reload mode
    RTC_SetReloadReg(RTC_Rload_Max);                                // Set reload data
    RTC_TriggerStamp_SW();                                          // Trigger reload data update to RTC timer
    while(RTC_GetSingleFlagStatus(RTC_RCRF) == DRV_UnHappened);     // Waiting reload complete
    RTC_ClearFlag(RTC_ALLF);                                        // Clear flag
    
    /*=== 4. Set RTC Interrupt Set ===*/
    RTC_IT_Config(RTC_INT_TO, ENABLE);                              // RTC Over Float Interrupt Enable
    RTC_ITEA_Cmd(ENABLE);                                           // RTC Interrupt Switch Enable.
    SYS_ITEA_Cmd(ENABLE);                                           // SYS Interrupt Switch Enable.
    NVIC_EnableIRQ(SYS_IRQn);                                       
    
    /*=== 5. Enable RTC module ===*/
    RTC_Cmd (ENABLE);                                               // Enable RTC module
    ProtectModuleReg(RTCprotect);                                   // Protect RTC module
    
    return DRV_Success;
}

/**
 *******************************************************************************
 * @brief       SYS: An interrupt, or IRQ, is an exception signalled by a
 *              peripheral, or generated by a software request. 
 * @details     SYS_IRQ hava IWDT, PW, RTC, CSC, APB,MEM or EMB interrupt source.
 * @return      No
 *******************************************************************************
 */
void SYS_IRQHandler(void)
{
    RTC_IRQ();                          // RTC Interrupt Service Routine
}

/**
 *******************************************************************************
 * @brief  	    RTC module reset event service
 * @details     Clear RTC interrupt flag.
 * @return	    
 * @note
 *******************************************************************************
 */
void RTC_IRQ(void)
{
    RTC_ClearFlag(RTC_TOF);             // Clear RTC Over Float Flag
}



