
/**
 ******************************************************************************
 *
 * @file        MG32_Simple_URT.C
 * @brief       Simple URT C code. 
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2021/06/01
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32__Common_DRV.h"
#include "MG32_URT_DRV.h"
#include "MG32_GPIO_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
// <<< Use Configuration Wizard in Context Menu >>>
//<o0> URT BaudRateGenerator clock source frequence is(2000000-48000000) <2000000-48000000>
//<o1> Baudrate is(600-256000) <600-6000000>
#define URT_BDCLK_SOURCE     36000000
#define URT_BR_VALUE         115200
// <<< end of configuration section >>>

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/



/**
 *****************************************************************************************
 * @fn      UART_TXRXBaudRate_Configuration
 * @brief   URT interrupts all enable or disable
 * @details  
 * @param   URTx          : where x can be (0 to 3) to select the UART peripheral
 * @param   URT_Frequence : PROC frequence value (ex 12MHz =  12000000)
 * @param   URTBaudRate   : TX and RX baudrate value (ex 9600)
 * @return        
 *****************************************************************************************
 */
DRV_Return URT_TXRXBaudRate_Configuration(URT_Struct* URTX , uint32_t URT_Frequence , uint32_t URTBaudRate )
{
	int8_t   i,j,k;
	uint32_t tmp;
	uint8_t  TXOverSamplingSampleNumber;
    uint8_t  RXOverSamplingSampleNumber;
	URT_BRG_TypeDef URT_BRGStruct;
    
	
	
	
	
	tmp = URT_Frequence / URTBaudRate;											
	//================TX and RX oversamplig value===================
	j = 0;
	for(i=8;i<32;i++)
	{
		k = tmp % i;
		if(k==0)
		{
			TXOverSamplingSampleNumber = (i - 1);
			RXOverSamplingSampleNumber = (i - 1);
			break;
		}
		else
		{
			if((i-k)>j)
			{
				j = i - k;
				TXOverSamplingSampleNumber = (i - 1);
				RXOverSamplingSampleNumber = (i - 1);
			}
		}
	}
	
	//==========prescaler and baudrate counter reload value============
	tmp = tmp / (TXOverSamplingSampleNumber  + 1);
	j = 0;
	for(i=2;i<16;i++)															
	{
		k = tmp % i;
		if(k==0)
		{
			URT_BRGStruct.URT_PrescalerCounterReload = (i - 1);
			break;
		}
		else
		{
			if((i-k)>j)
			{
				j = (i - k);
				URT_BRGStruct.URT_PrescalerCounterReload = (i - 1);
			}
		}
	}
	tmp = tmp / (URT_BRGStruct.URT_PrescalerCounterReload + 1);
	if(tmp==0)
	{
		return(DRV_Failure);												    //TX and RX baudrate can't setting.
	}
	URT_BRGStruct.URT_BaudRateCounterReload = (tmp - 1);
	
	
	    
    //----------------Baud Rate Generator Configuration----------------
    URT_BRGStruct.URT_InternalClockSource = URT_BDClock_PROC;
    URT_BRGStruct.URT_BaudRateMode = URT_BDMode_Separated;
    URT_BaudRateGenerator_Config(URTX,&URT_BRGStruct);
    
    //----------------RX / TX Baud Rate Configuration------------------
    URT_TXClockSource_Select(URTX , URT_TXClock_Internal);
    URT_RXClockSource_Select(URTX , URT_RXClock_Internal);
    URT_RXOverSamplingMode_Select( URTX, URT_RXSMP_3TIME);
    URT_TXOverSamplingSampleNumber_Select( URTX, TXOverSamplingSampleNumber);
	URT_RXOverSamplingSampleNumber_Select( URTX, RXOverSamplingSampleNumber);
	URT_BaudRateGenerator_Cmd(URTX , ENABLE);
	

	return(DRV_Success);
}

/**
 *****************************************************************************************
 * @fn      UART_Inital
 * @brief   UART inital
 * @details  
 * @param   URTx          : where x can be (0 to 3) to select the UART peripheral
 * @param   
 * @param   
 * @return        
 *****************************************************************************************
 */
DRV_Return URT_Init(URT_Struct* URTX)
{
    DRV_Return URT_Init_Result;
    
	URT_Data_TypeDef URT_Datastruct;

    
    //---------Data character configuration-----------
    URT_Datastruct.URT_TX_DataLength = URT_DataLength_8;
    URT_Datastruct.URT_RX_DataLength = URT_DataLength_8;
    URT_Datastruct.URT_TX_DataOrder  = URT_DataTyped_LSB;
    URT_Datastruct.URT_RX_DataOrder  = URT_DataTyped_LSB;
    URT_Datastruct.URT_TX_Parity     = URT_Parity_No;
    URT_Datastruct.URT_RX_Parity     = URT_Parity_No;
    URT_Datastruct.URT_TX_StopBits   = URT_StopBits_1_0;
    URT_Datastruct.URT_RX_StopBits   = URT_StopBits_1_0;
    URT_Datastruct.URT_TX_DataInverse= DISABLE;
    URT_Datastruct.URT_RX_DataInverse= DISABLE;
    URT_DataCharacter_Config(URTX,&URT_Datastruct);   
    
    
    //---------------Mode setting----------------------
    URT_Mode_Select( URTX , URT_URT_mode);                                                   // Normal URT mode
    URT_DataLine_Select(URTX,URT_DataLine_2);                                                // Data bus is 2 line ( RX and TX)
	
    //----------------Data control---------------------
    URT_RXShadowBufferThreshold_Select(URTX,URT_RXTH_1BYTE );                                // RX shadowBuffer level is 1 byte.
    
    //------------Baudrate configuration--------------
    URT_Init_Result = URT_TXRXBaudRate_Configuration(URTX,URT_BDCLK_SOURCE,URT_BR_VALUE);

    if(URT_Init_Result==DRV_Success)
    {
        
       URT_TX_Cmd(URTX,ENABLE);
       URT_RX_Cmd(URTX,ENABLE);
	   URT_Cmd(URTX,ENABLE);
    }
    return(URT_Init_Result);
}


































