/*********************************************************************
*    Project: MG82F5B32-DEMO
*			MG82F5B32 LQFP32_V10 EV Board (TH156A)
*			CpuCLK=12MHz, SysCLK=12MHz
*    Description:
*			Use XTAL as SysClk,CpuClk
*    Note:
* 
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F5Bxx.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F5B32.H"
#include ".\include\API_Uart_BRGRL_MG82F5B32.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		48000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.32MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
//#define MCU_CPUCLK		(MCU_SYSCLK)
#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P25
#define LED_R		P26
#define LED_G_1		P27

#define _ENABLE_CKM_		// Enable clock multiply

u8 LedTime;

/***********************************************************************************
*Function:   void INT_T0(void)
*Description:T0 Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_T0(void) interrupt INT_VECTOR_T0
{
	TF0 = 0;
	TR0 = 0;
	TH0=TIMER_12T_1ms_TH;
	TL0=TIMER_12T_1ms_TL;
	TR0 = 1;
	
	if(LedTime!=0) LedTime--;

}

/*************************************************
*Function: 		char putchar (char c)   
*Description: 	send for printf
*Input:    		char c
*Output:     
*************************************************/
char putchar (char c)   
{      
	bit bES;
	bES=ES0;
    ES0=0;        
    S0BUF = c;        
    while(TI0==0);        
    TI0=0;        
    ES0=bES;        
    return 0;
}

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

#ifndef _ENABLE_CKM_
/***********************************************************************************
*Function:   	void InitClock_Xtal(void)
*Description:	Initialize clock,use XTAL
*Input:   
*Output:   		
*************************************************************************************/
void InitClock_Xtal(void)
{
	// Enable XTAL,Enable IHRCO, MCK=OSCin, OSCin=IHRCO
	CLK_SetCKCON2(GAIN_FOR_2M_25M|ENABLE_XTAL|ENABLE_IHRCO|MCK_OSCin|OSCIn_IHRCO);
	DelayXms(2);
	// Wait for XTAL stanby
	while((CKCON1&XTOR)==0);
	DelayXms(2);
	PCON1=MCDF;		// Clear MCDF
	// Enable XTAL,Enable IHRCO, MCK=OSCin, OSCin=XTAL
	CLK_SetCKCON2(GAIN_FOR_2M_25M|ENABLE_XTAL|ENABLE_IHRCO|MCK_OSCin|OSCIn_XTAL);
	DelayXms(2);
	// Enable XTAL,Disable IHRCO, MCK=OSCin, OSCin=XTAL
	CLK_SetCKCON2(GAIN_FOR_2M_25M|ENABLE_XTAL|DISABLE_IHRCO|MCK_OSCin|OSCIn_XTAL);

}

#else
/***********************************************************************************
*Function:   	void InitClock_Xtal_CKM(void)
*Description:	Initialize clock,use XTAL,enable clock multiply
*Input:   
*Output:   		
*************************************************************************************/
void InitClock_Xtal_CKM(void)
{
	// CpuCLK=SysCLK/2,  SysCLK=MCKDO, Enable CKM, CKMI=OSCin/2	
	CLK_SetCKCON0(CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);

	// Enable XTAL,Enable IHRCO, MCK=OSCin, OSCin=IHRCO
	CLK_SetCKCON2(GAIN_FOR_2M_25M|ENABLE_XTAL|ENABLE_IHRCO|MCK_OSCin|OSCIn_IHRCO);
	DelayXms(2);
	// Wait for XTAL stanby
	while((CKCON1&XTOR)==0);
	DelayXms(2);
	PCON1=MCDF;		// Clear MCDF
	// Enable XTAL,Enable IHRCO, MCK=OSCin, OSCin=XTAL
	CLK_SetCKCON2(GAIN_FOR_2M_25M|ENABLE_XTAL|ENABLE_IHRCO|MCK_OSCin|OSCIn_XTAL);
	DelayXms(2);
	// Enable XTAL,Disable IHRCO, MCK=CKMIx8, OSCin=XTAL
	CLK_SetCKCON2(GAIN_FOR_2M_25M|ENABLE_XTAL|DISABLE_IHRCO|MCK_CKMI_X8|OSCIn_XTAL);
}

#endif

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2OpenDrain(BIT4|BIT5|BIT6|BIT7);			// Set P24,P25,P26,P27 as open drain
	PORT_EnP24P25P26P27PullHigh();						// Enable P24,P25,P26,P27 pull-high

	PORT_SetP2PushPull(BIT2);							// Set P22 as push pull
}

/***********************************************************************************
*Function:   	void InitTimer0(void)
*Description:	Initialize Timer0
*Input:   
*Output:   		
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// TIMER0 Mode: 16-bit
	TM_SetT0Clock_SYSCLKDiv12();			// TIMER0 Clock source: SYSCLK/12
	TM_SetT0Gate_Disable();

	TM_SetT0LowByte(TIMER_12T_1ms_TL);		// Set TL0 value
	TM_SetT0HighByte(TIMER_12T_1ms_TH);		// Set TH0 value

	TM_EnableT0();							// Enable TIMER0
}

/***********************************************************************************
*Function:   	void InitUart0_S1BRG()
*Description:	Initialize Uart0, The source of baud rate was S1BRG
*Input:   
*Output:   		
*************************************************************************************/
void InitUart0_S1BRG(void)
{	
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS1BRG();			// B.R. source: S1BRG
	UART0_SetS1BRGBaudRateX1();			// S1BRG x2
	UART1_SetS1BRGSelSYSCLK();			// S1BRG clock source: SYSCLK

    // Sets B.R. value
	UART1_SetS1BRGValue(S1BRG_BRGRL_9600_1X_48000000_1T);
	
	UART1_EnS1BRG();					// Enable S1BRG
}

/***********************************************************************************
*Function:   	void InitPCA_PWM(void)
*Description:	Initialize PCA for PWM
*Input:   
*Output:   		
*************************************************************************************/
void InitPCA_PWM(void)
{
	PCA_SetCLOCK_SYSCLK();							// PCA clock: SysClk
	
	PCA_CH0_SetMode_PWM();

	PCA_CH0_SetPWM_8Bit();

	PCA_SetPWM_EdgeAligned();						// PWM Edge Aligned

	// Set PWM duty
	PCA_CH0_SetValue_L(0x80); 				
	PCA_CH0_SetValue_9thBit_L_0();					// Bit 9
	PCA_CH0_SetValue_H(0x80);						// Reload value
	PCA_CH0_SetValue_9thBit_H_0();					// Bit 9


	// Set Counter
	PCA_SetCounter(0);
	PCA_SetCounterReload(0);

	PCA_EnPCACounter();								// Enable PCA
	
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	IE=0;EIE1=0;								// Clear interrupt
	INT_EnTIMER0();								// Enable T0 interrupt

	SFIE=0x00;									// Clear SFIE
	INT_EnSF_MCD();								// Enable MCD interrupt
	
}

/***********************************************************************************
*Function:       void InitSystem(void)
*Description:    Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
#ifndef _ENABLE_CKM_
	InitClock_Xtal();
#else
	InitClock_Xtal_CKM();
#endif
	InitUart0_S1BRG();
	InitPCA_PWM();
	InitTimer0();
	InitInterrupt();
}

void main()
{

    InitSystem();
	
	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
	LED_G_0=1;LED_G_1=1;LED_R=1;
	printf("\nHello...Clock Xtal...");

	LedTime=200;
	
	INT_EnAll();

    while(1)
    {
		if(LedTime==0)
		{
			LedTime=200;
			LED_G_0=!LED_G_0;
			printf("\nSysTik 200ms");
		}
		
		if((PCON1&MCDF)!=0)
		{
			if((CKCON1&XTOR)!=0)
			{
				#ifndef _ENABLE_CKM_
					InitClock_Xtal();
				#else
					InitClock_Xtal_CKM();
				#endif
				
				PWMCR=PWMCR&(~PFCF);	// Clear PFCF, start PWM output
			}
		}
    }
}
