/*********************************************************************
*Project:MG82F6B08_001_104-DEMO
*			MG82F6B08_001_104 SOP16_V11 EV Board (TH221A)
*			CpuCLK=8MHz, SysCLK=8MHz
*Description:
*			UART0, LIN Slave
*Note:
*
*Creat time:
*Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F6B08_001_104.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6B08_001_104.H"
#include ".\include\API_Uart_BRGRL_MG82F6B08_001_104.H"


/*************************************************
*Set SysClk (MAX.24MHz)
*Selection: 
*	8000000,16000000
*	11059200,22118400
*************************************************/
#define MCU_SYSCLK		8000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.16MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_R		P32
#define LED_G_0		P33
#define LED_G_1		P46
#define LIN_TXD			P31
#define LIN_RXD			P30

#define LIN_TIME_MAX				10

#define LIN_SYNC_BYTE				0x55

#define LIN_AUTO_BAUDRATE_FLAG		0		// 0->Disable auto Baud-Rate 1->Enable auto Baud-Rate			

#define LIN_PID_MASTER_TX_MSG0		0x10		
#define LIN_PID_MASTER_TX_MSG1		0x11	

#define LIN_PID_MASTER_RX_MSG0		0x21
#define LIN_PID_MASTER_RX_MSG1		0x31

// define LIN status
typedef enum
{
  LIN_IDLE_BREAK=0,
  LIN_SYNC,
  LIN_PID,
  LIN_MASTER_RX_DATA,
  LIN_MASTER_RX_CHECKSUM,
  LIN_MASTER_RX_DONE,
  LIN_MASTER_TX_DATA,
  LIN_MASTER_TX_CHECKSUM,
  LIN_MASTER_TX_DONE,
  LIN_ERR
}LIN_State;

// define LIN mode
typedef enum
{
  	LIN_MODE_MASTER_TX=0,
	LIN_MODE_MASTER_RX,
	LIN_MODE_SLAVE_TX,
	LIN_MODE_SLAVE_RX
}LIN_Mode;

// define frame struct
typedef struct
{
	LIN_Mode Mode;
  	LIN_State State;
  	u8 PID ;
  	u8 Data[8];
  	u8 DataLength;
  	u8 DataInx;
  	u8 Check;
} LIN_Frame ;

#define BIT(A,B)      ((A>>B)&0x01)   // A Variable

idata LIN_Frame LINFrame;

// define slave PID
typedef struct
{
  	u8 SPIDMTx0_P;		// Slave PID 0, for Master send/Slave recive
  	u8 SPIDMTx1_P;		// Slave PID 1, for Master send/Slave recive
  	u8 SPIDMRx0_P;		// Slave PID 0, for Master recive/Slave send
  	u8 SPIDMRx1_P;		// Slave PID 1, for Master recive/Slave send
} LIN_SlavePID_Def ;
idata LIN_SlavePID_Def LINSlavePID;

u8 LedTime;
u8 LINOverTime;

void LIN_Slave_StartFrame(void);
u8 LIN_CalcParity(u8 id);
u8 LIN_CalcChecksum(u8 id,u8 *lindata,u8 length);
u8 LIN_CalcChecksum_INT(u8 id,u8 *lindata,u8 length);

/***********************************************************************************
*Function:	u8 LIN_CalcChecksum_INT(uint8 id,uint8 *data,uint8 length)
*Description: LIN calculate checksum 
*Input:	
*Output:	  
*************************************************************************************/
u8 LIN_CalcChecksum_INT(u8 id,u8 *lindata,u8 length)
{
	u8 i;
	u16 check_sum = 0;
	//0x3C 0x3D  standard checksum
	if(id != 0x3c && id != 0x7d)  //Enhanced checksum
	{
	 	check_sum  = id ;
	} 
	else // standard checksum
	{
	 	check_sum = 0 ;
	}
	for (i = 0; i < length; i++) 
	{    
	  	check_sum += *(lindata++);
	  
	  	if (check_sum > 0xFF)      // carry
	  	{
	  		check_sum -= 0xFF;
	  	}
	}
	return (~check_sum);  //not
}

/***********************************************************************************
*Function:   void INT_UART0(void)
*Description:UART0 Interrupt handler
		 
*Input:   
*Output:     
*************************************************************************************/
void INT_UART0(void) interrupt INT_VECTOR_UART0
{
	u8 x;
	if(RI0==1)
	{
		// LIN status
		if(LINFrame.State==LIN_IDLE_BREAK)
		{
			if((S0CFG1&SBF0)!=0)
			{ // Rx Break done
				UART0_LIN_ClrSBF0();
				#if (LIN_AUTO_BAUDRATE_FLAG==TRUE)
				{
					UART0_LIN_AutoBR();
					while(LIN_TXD==0);
					LINFrame.State=LIN_PID;				// if Auto Baud Rate , SYNC(0x55) for ABR, so the next data is PID
				}
				#else
				{
					UART0_LIN_DisAutoBR();		
					LINFrame.State=LIN_SYNC;			// if no ABR, the next data is SYNC(0x55)
				}
				#endif
				LINOverTime=LIN_TIME_MAX;
			}
		}
		/**/
		else if(LINFrame.State==LIN_SYNC)
		{ // Rx SYNC done
			x=S0BUF;
			if(x == LIN_SYNC_BYTE)
			{
				LINFrame.State++;
				LINOverTime=LIN_TIME_MAX;
			}
			else
			{
				LINFrame.State=LIN_ERR;
			}
		}
		/**/
		else if(LINFrame.State==LIN_PID)
		{ // Rx PID done
			LINOverTime=LIN_TIME_MAX;
			LINFrame.PID = S0BUF;
			if((LINFrame.PID == LINSlavePID.SPIDMRx0_P)||(LINFrame.PID == LINSlavePID.SPIDMRx1_P))
			{ // PID for Master get msg 
					// Lin salve data init
					for(x = 0 ;x<8;x++)
					{
						LINFrame.Data[x] = 0x58+x ;
					}
					LINFrame.DataLength=8;
					LINFrame.Check = LIN_CalcChecksum_INT(LINFrame.PID, &LINFrame.Data[0], LINFrame.DataLength);
					LINFrame.DataInx=1;
					// send first data
					UART0_LIN_SetTX();
					S0BUF=LINFrame.Data[0];
					LINFrame.State=LIN_MASTER_RX_DATA;
			}
			else if((LINFrame.PID == LINSlavePID.SPIDMTx0_P)||(LINFrame.PID == LINSlavePID.SPIDMTx1_P))
			{// PID for Master send msg 
				LINFrame.State=LIN_MASTER_TX_DATA;
				LINFrame.DataInx=0;
			}
			else 
			{// PID error
				LINFrame.State=LIN_ERR;
			}

		}
		else if(LINFrame.State==LIN_MASTER_TX_DATA)
		{
			LINOverTime=LIN_TIME_MAX;
			LINFrame.Data[LINFrame.DataInx] = S0BUF;
			LINFrame.DataInx++;
			if(LINFrame.DataInx>=LINFrame.DataLength)
			{
				LINFrame.State++;
			}

		}
		else if(LINFrame.State==LIN_MASTER_TX_CHECKSUM)
		{
			LINOverTime=LIN_TIME_MAX;
			LINFrame.Check=S0BUF;
			LINFrame.State++;
		}
		else if(LINFrame.State==LIN_MASTER_TX_DONE)
		{
			
		}
		else
		{
			LINFrame.State=LIN_ERR;
		}
		RI0 = 0;
	}
	else
	{
		TI0 = 0;
		if(LINFrame.State==LIN_MASTER_RX_DATA)
		{
			LINOverTime=LIN_TIME_MAX;
			S0BUF=LINFrame.Data[LINFrame.DataInx];
			LINFrame.DataInx++;
			if(LINFrame.DataInx>=LINFrame.DataLength)
			{
				LINFrame.State++;
			}
		}
		else if(LINFrame.State==LIN_MASTER_RX_CHECKSUM)
		{
			LINOverTime=LIN_TIME_MAX;
			S0BUF=LINFrame.Check;
			LINFrame.State++;
			
		}
		else if(LINFrame.State==LIN_MASTER_RX_DONE)
		{
			
		}
		else
		{
			LINFrame.State=LIN_ERR;
		}
	}
}

/***********************************************************************************
*Function:   void INT_T0(void)
*Description:T0 Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_T0(void) interrupt INT_VECTOR_T0
{
	TH0=TIMER_12T_1ms_TH;
	TL0=TIMER_12T_1ms_TL;

	if(LedTime!=0) LedTime--;
	if(LINOverTime!=0) LINOverTime--;
}

/*************************************************
*Function:     void DelayXus(u8 xUs)
*Description:  delay,unit:us
*Input:     u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     void DelayXms(u16 xMs)
*Description:delay,unit:ms
*Input:     u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:   void InitUart0_LIN(void)
*Description:   Initialize Uart0 for LIN mode
*Input:   
*Output:     
*************************************************************************************/
void InitUart0_LIN(void)
{
	
	UART0_SetAccess_S0CR1();			// Set SFR 0xB9 address to access S0CR1
	UART0_SetModeLIN();					// UART0 Mode:LIN 		
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R. source: S0BRG
	UART0_SetS0BRGBaudRateX2();			// S0BRG x2
	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source:SYSCLK

	// Set S0BRG Value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_8000000_1T);
	
	UART0_EnS0BRG();					// Enable S0BRG
	
	
}

/***********************************************************************************
*Function:   void InitTimer0(void)
*Description:Initialize Timer0
*Input:   
*Output:     
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// TIMER0 Mode: 16-bit
	TM_SetT0Clock_SYSCLKDiv12();					// TIMER0 Clock source: SYSCLK/12
	TM_SetT0Gate_Disable();

	TM_SetT0LowByte(TIMER_12T_1ms_TL);		// Set TL0 value
	TM_SetT0HighByte(TIMER_12T_1ms_TH);		// Set TH0 value

	TM_EnableT0();							// Enable T0
}

/***********************************************************************************
*Function:   void InitInterrupt(void)
*Description:   Initialize Interrupt 
*Input:   
*Output:     
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnUART0();				// Enable UART0 interrupt
	INT_EnTIMER0();				// Enable T0 interrupt
}	

/***********************************************************************************
*Function:   void InitPort(void)
*Description:   Initialize IO Port
*Input:   
*Output:     
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP3QuasiBi(BIT0|BIT1|BIT2|BIT3);		// Set P30,P31,P32,P33 as Quasi-Bidirectional,for RXD/TXD,LED
	PORT_SetP4OpenDrainPu(BIT6);				// Set P46 as Open-Drain with pull-up resistor,for LED
}

/***********************************************************************************
*Function:   void InitClock(void)
*Description:   Initialize clock	
*Input:   
*Output:     
*************************************************************************************/
void InitClock(void)
{
#if (MCU_SYSCLK==8000000)
	// Default: IHRCO=16MHz, OSCin=IHRCO,MCK=OSCin,MCKDO=MCK/2
#endif

#if (MCU_SYSCLK==11059200)
	// IHRCO=22.12MHz, OSCin=IHRCO,MCK=OSCin,MCKDO=MCK/2
	CLK_SelIHRCO_22118400Hz();
#endif
	
#if (MCU_SYSCLK==16000000)
	// IHRCO=16MHz, OSCin=IHRCO,MCK=OSCin,MCKDO=MCK
	CLK_SetCKCON3(WDTCS_ILRCO|FWKP_NORMAL_120uS|MCKDO_MCK_DIV_1);
#endif

#if (MCU_SYSCLK==22118400)
	// IHRCO=22.12MHz, OSCin=IHRCO,MCK=OSCin,MCKDO=MCK
	CLK_SelIHRCO_22118400Hz();
	CLK_SetCKCON3(WDTCS_ILRCO|FWKP_NORMAL_120uS|MCKDO_MCK_DIV_1);
#endif
	

#if (MCU_CPUCLK==MCU_SYSCLK)
	CLK_SetCpuclk_Sysclk();
#else
	CLK_SetCpuclk_SysclkDiv2();
#endif

	// P45 ouput MCK/4
	//CLK_P45OC_MCKDiv4();
}

/***********************************************************************************
*Function:   void InitSystem(void)
*Description:   Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitTimer0();
	InitUart0_LIN();
	InitInterrupt();
}

/*
*************************************************************************************
*  Function
*
*************************************************************************************
*/
/***********************************************************************************
*Function:   void LIN_Slave_StartFrame(void)
*Description:   LIN  start receive
*Input:   
*Output:     
*************************************************************************************/
void LIN_Slave_StartFrame(void)
{
	// set S0BRG Value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_8000000_1T);
	UART0_LIN_SetRX();
	UART0_LIN_WaitSYNC();

	LINFrame.State=LIN_IDLE_BREAK;
}

/***********************************************************************************
*Function:	uint8 LIN_CalcParity(uint8 id)
*Description: LIN calculate parity
*Input:	
*Output:	  
*************************************************************************************/
u8 LIN_CalcParity(u8 id)
{
	u8 parity, p0,p1;
	parity=id; 
	p0=(BIT(parity,0)^BIT(parity,1)^BIT(parity,2)^BIT(parity,4))<<6;     //Even
	p1=(!(BIT(parity,1)^BIT(parity,3)^BIT(parity,4)^BIT(parity,5)))<<7;  //Odd
	parity|=(p0|p1);
	return parity;
}

/***********************************************************************************
*Function:	u8 LIN_CalcChecksum(uint8 id,uint8 *data,uint8 length)
*Description: LIN calculate checksum
*Input:	
*Output:	  
*************************************************************************************/
u8 LIN_CalcChecksum(u8 id,u8 *lindata,u8 length)
{
	u8 i;
	u16 check_sum = 0;
	//0x3C 0x3D use standard checksum
 	if(id != 0x3c && id != 0x7d)  // use enhanced checksum
 	{
	 	check_sum  = id ;
	} 
	else //use standard checksum
	{
	 	check_sum = 0 ;
	}
	for (i = 0; i < length; i++) 
	{    
	 	 check_sum += *(lindata++);
	  
	  	if (check_sum > 0xFF)      //carry
	  	{
	  		check_sum -= 0xFF;
	  	}
	}
	return (~check_sum);  // not
}

/*
*************************************************************************************
*/ 

void main()
{

	InitSystem();
	
	LED_G_0 =0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
	LED_G_0 =1;LED_G_1=1;LED_R=1;	

	LINSlavePID.SPIDMTx0_P=LIN_CalcParity(LIN_PID_MASTER_TX_MSG0);
	LINSlavePID.SPIDMTx1_P=LIN_CalcParity(LIN_PID_MASTER_TX_MSG1);
	LINSlavePID.SPIDMRx0_P=LIN_CalcParity(LIN_PID_MASTER_RX_MSG0);
	LINSlavePID.SPIDMRx1_P=LIN_CalcParity(LIN_PID_MASTER_RX_MSG1);

	INT_EnAll();
	LIN_Slave_StartFrame();

	while(1)
    {
		if(LedTime==0)
		{
			LedTime=100;
			LED_G_0 = !LED_G_0;	
		}
		if(LINFrame.State==LIN_MASTER_RX_DONE)
		{
			LED_G_1=0;
			DelayXms(2);
			LED_G_1=1;
			DelayXms(2);
			LIN_Slave_StartFrame();
			
		}
		if(LINFrame.State==LIN_MASTER_TX_DONE)
		{
			// Check data
			if(LINFrame.Check == LIN_CalcChecksum(LINFrame.PID, &LINFrame.Data[0], LINFrame.DataLength))
			{ // Check success
				LED_R=0;
				DelayXms(2);
				LED_R=1;
				DelayXms(2);
			}
			LIN_Slave_StartFrame();
			
		}

		if(LINFrame.State==LIN_ERR)
		{
			LIN_Slave_StartFrame();
			
		}

		if((LINFrame.State != LIN_IDLE_BREAK)&&(LINOverTime ==0))
		{
			LINFrame.State = LIN_ERR;
		}
    }

}


