/*********************************************************************
*    Project: MG82F5B32-DEMO
*			MG82F5B32 LQFP32_V10 EV Board (TH156A)
*			CpuCLK=12MHz, SysCLK=12MHz
*    Description:
*			ADC 10bit, P10 for input
*			Through the IVREF 2.4v, calculate the P10 input voltage
*    Note:
* 
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F5Bxx.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F5B32.H"
#include ".\include\API_Uart_BRGRL_MG82F5B32.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.32MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P25
#define LED_R		P26
#define LED_G_1		P27


#define TEST_ADC_DATA_CNT		64
xdata WordTypeDef TestBuf[TEST_ADC_DATA_CNT];
u8 TestBufLen;

#define _RESOLUTION_10BIT_	0
#define _RESOLUTION_12BIT_	1

#define _DEFAULT_ADC_RESOLUTION_	_RESOLUTION_12BIT_

#if (_DEFAULT_ADC_RESOLUTION_==_RESOLUTION_10BIT_)
#define VCC_REF					(3300L)
#define ADC_VALUE_MAX			(1024L)
#define IVR_ADC_MIN				(714L)		// 2.3v@3.3V 
#define IVR_ADC_MAX				(776L)		// 2.5v@3.3V
#define IVR_ADC_DEFAULT			(745L)		// 2.4v@3.3V
#endif

#if (_DEFAULT_ADC_RESOLUTION_==_RESOLUTION_12BIT_)
#define VCC_REF					(3300L)
#define ADC_VALUE_MAX			(4096L)
#define IVR_ADC_MIN				(2855L)		// 2.3v@3.3V 
#define IVR_ADC_MAX				(3103L)		// 2.5v@3.3V
#define IVR_ADC_DEFAULT			(2979L)		// 2.4v@3.3V
#endif

/*************************************************
*Function: 		char putchar (char c)   
*Description: 	send for printf
*Input:    		char c
*Output:     
*************************************************/
char putchar (char c)   
{      
	bit bES;
	bES=ES0;
    ES0=0;        
    S0BUF = c;        
    while(TI0==0);        
    TI0=0;        
    ES0=bES;        
    return 0;
}

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:   	void InitUart0_S1BRG()
*Description:	Initialize Uart0, The source of baud rate was S1BRG
*Input:   
*Output:   		
*************************************************************************************/
void InitUart0_S1BRG(void)
{	
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS1BRG();			// B.R. source: S1BRG
	UART0_SetS1BRGBaudRateX2();			// S1BRG x2
	UART1_SetS1BRGSelSYSCLK();			// S1BRG clock source: SYSCLK

    // Sets B.R. value
	UART1_SetS1BRGValue(S1BRG_BRGRL_9600_2X_12000000_1T);
	
	UART1_EnS1BRG();					// Enable S1BRG
}

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2OpenDrain(BIT4|BIT5|BIT6|BIT7);			// Set P24,P25,P26,P27 as open drain
	PORT_EnP24P25P26P27PullHigh();						// Enable P24,P25,P26,P27 pull-high
	PORT_SetP1AInputOnly(BIT0);  					    // Set P10 as Analog-Input-Only for ADC input
}

/***********************************************************************************
*Function:   	void InitTimer0(void)
*Description:	Initialize Timer0
*Input:   
*Output:   		
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// TIMER0 Mode: 16-bit
	TM_SetT0Clock_SYSCLKDiv12();			// TIMER0 Clock source: SYSCLK/12
	TM_SetT0Gate_Disable();

	TM_SetT0LowByte(0);						// Set TL0 value
	TM_SetT0HighByte(0);					// Set TH0 value

	TM_DisableT0();							// Disable TIMER0
}

/***********************************************************************************
*Function:   void InitADC(void)
*Description:   Initialize ADC
*Input:   
*Output:     
*************************************************************************************/
void InitADC(void)
{
	ADC_Enable();						// Enable ADC
	ADC_SetClock_SYSCLKDiv2();			// ADC Clock = SYSCLK/2       sps= 12M/2/30=200k
	ADC_SetMode_SetADCS();				// ADC Trigger mode: set ADCS
	ADC_SetRightJustified();			// ADC Right-Justified

#if (_DEFAULT_ADC_RESOLUTION_==_RESOLUTION_10BIT_)
	ADC_SetADCData_10Bit();
#endif
	
#if (_DEFAULT_ADC_RESOLUTION_==_RESOLUTION_12BIT_)
	ADC_SetADCData_12Bit();
#endif

	ADC_SetVREF_IVR24();				// Set VREF+ to IVR 2.4V

	ADC_SetChannel_AIN0();				// ADC channel: AIN0(P10)
}

/***********************************************************************************
*Function:   	void InitClock()
*Description:	Initialize clock
*Input:   
*Output:   		
*************************************************************************************/
void InitClock(void)
{
#if (MCU_SYSCLK==11059200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==12000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=12MHz CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==22118400)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==24000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==29491200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==32000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if (MCU_SYSCLK==48000000)
	// SysClk=48MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitUart0_S1BRG();
	InitADC();
	InitTimer0();
}

/***********************************************************************************
*Function:   	u16 GetAdcValue()
*Description:	read ADC result of the current channel  
*Input:   
*Output:   		u16 :ADC Value
*************************************************************************************/
u16 GetAdcValue()
{
	WordTypeDef wAdcValue;
	ADCON0 = ADCON0|ADCS;							// set ADCS,Start ADC
    while((ADCON0&ADCI)==0);           				// wait ADC complete
    wAdcValue.B.BHigh=ADCDH;
    wAdcValue.B.BLow=ADCDL;
 	ADCON0 = ADCON0&(~ADCI);		           		// clear ADCI flag
#if (_DEFAULT_ADC_RESOLUTION_==_RESOLUTION_10BIT_)
	return wAdcValue.W&0x03FF;
#endif
		
#if (_DEFAULT_ADC_RESOLUTION_==_RESOLUTION_12BIT_)
	return wAdcValue.W&0x0FFF;
#endif
}

/***********************************************************************************
*Function:   	unsigned int GetIVRef()
*Description:	read ADC result of the IVRef 
*Input:   
*Output:   		unsigned int :ADC result of the IVRef(12 Bit) when VDD is 3.3V
*************************************************************************************/
unsigned int GetIVRef()
{
unsigned char IVR_CodeHigh,IVR_CodeLow;
	ISPCR = 0x80;
	IFMT = 0x06;
	IFADRH = 0x00;
	IFADRL = 0xC0;
	
	SCMD = 0x46;
	SCMD = 0xB9;
	IVR_CodeHigh= IFD;

	IFADRL ++;
	SCMD = 0x46;
	SCMD = 0xB9;
	IVR_CodeLow= IFD;

	ISPCR = 0x00;
	
	return (IVR_CodeHigh*256+IVR_CodeLow);
}

void main()
{
	u8 i;
	DWordTypeDef dwTemp;
	WordTypeDef wVinValue,wIVRefValue;
	WordTypeDef wIVRAdjAdc;

	u16 wAdcMIN,wAdcMAX,wAdcAVG;
	WordTypeDef wAdcTime;
	
    InitSystem();
	
	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
	LED_G_0=1;LED_G_1=1;LED_R=1;
	
	printf("\nHello...ADC Vref = IVR 2.4V...");

#if (_DEFAULT_ADC_RESOLUTION_==_RESOLUTION_10BIT_)
	wIVRAdjAdc.W=GetIVRef()>>2;				
#endif
#if (_DEFAULT_ADC_RESOLUTION_==_RESOLUTION_12BIT_)
		wIVRAdjAdc.W=GetIVRef(); 			
#endif

	// Uart output
	printf("\nIVR ADC@3.3V:%d",wIVRAdjAdc.W);
	// Detect IVRef calibration value is within the range(2.3~2.5)
	if((wIVRAdjAdc.W <IVR_ADC_MIN)||(wIVRAdjAdc.W>=IVR_ADC_MAX))
	{ // The stored IVRef value is faulty, set to default value 2.4V@3.3V
		// Uart output
		wIVRAdjAdc.W=IVR_ADC_DEFAULT;
		printf(" Err..Set Default:%d",wIVRAdjAdc.W);
	}
	// Calculate the voltage of the IVR
	dwTemp.DW=wIVRAdjAdc.W;
	wIVRefValue.W=dwTemp.DW*VCC_REF/ADC_VALUE_MAX;
	printf(", VRef Voltage:%dmV",wIVRefValue.W);


	TestBufLen=0;

	TH0=0;
	TL0=0;
	TM_EnableT0();
    while(1)
    {
    	TestBuf[TestBufLen].W=GetAdcValue();
		LED_R = !LED_R;
		TestBufLen++;
		if(TestBufLen>=TEST_ADC_DATA_CNT)
		{
			TM_DisableT0();
			wAdcTime.B.BHigh=TH0;
			wAdcTime.B.BLow=TL0;
			
			dwTemp.DW=0;
			wAdcMIN=0xFFFF;
			wAdcMAX=0x0000;
			
			for(i=0;i<(TEST_ADC_DATA_CNT);i++)
			{
				dwTemp.DW=dwTemp.DW+TestBuf[i].W;
				if(wAdcMIN>=TestBuf[i].W)
				{
					wAdcMIN=TestBuf[i].W;
				}
				if(wAdcMAX<TestBuf[i].W)
				{
					wAdcMAX=TestBuf[i].W;
				}
				
			}
			wAdcAVG=dwTemp.DW/TEST_ADC_DATA_CNT;		
			printf("\nADC ");
			printf("Sample cnt:%bd,",i);
			printf("Total time:%d,",wAdcTime.W);
			printf("MIN:%04X,",wAdcMIN);
			printf("MAX:%04X,",wAdcMAX);
			printf("AVG:%04X,",wAdcAVG);

			
			// Calculate the input voltage of AIN0
			dwTemp.DW=wIVRefValue.W;
			wVinValue.W=dwTemp.DW*wAdcAVG/ADC_VALUE_MAX;
			
			// Uart output
			printf("IO Voltage:%dmV",wVinValue.W);
			DelayXms(1000);
			TestBufLen=0;

			TH0=0;
			TL0=0;
			TM_EnableT0();
		}
    }
}



