#include <intrins.h>
#include "REG_MG82F6D64.h"
#include "TYPEDEF.h"
#include "stdio.h"
#include "MG82F6D64_COMMON_DRV.h"
#include "MG82F6D64_GPIO_DRV.h"
#include "MG82F6D64_Timer_DRV.h"
#include "MG82F6D64_UART0_DRV.h"
#include "MG82F6D64_INT_DRV.h"
#include "MG82F6D64_INT_VECTOR.h"

/**
 ******************************************************************************
 * do-while template
 ******************************************************************************
 */
#define MWT( __stuff__ )  do { __stuff__ } while (0)

/****************************************************************
 *  Function : UART0_9600_Init
 *  Use the Timer1(T1OF) as UART0 clock source.
 ****************************************************************/
void UART0_9600_Init(void)
{
    // UART0 Register initial Tx:P31 Rx:P30
    __DRV_URT0_SetSM10(UART0_MODE1_8BIT);
    __DRV_URT0_SerialReception_Cmd(MW_ENABLE);
    TI0 = 1;

    // Use Timer1 Auto reload mode to generate the baud rate 9600
    __DRV_TIMER1_Mode_Select(TIMER1_MODE2_8BIT_AUTORELOAD);
    __DRV_TIMER1_Clock_Source_Select(TIMER1_CLOCK_SOURCE_SYSCLK);
    __DRV_TIMER1_Set8BitIntervalAutoReload(217);
    __DRV_TIMER1_Run_Cmd(MW_ENABLE);
}

/****************************************************************
Variables
Timer2 overflow Interrupt(TH2) => TF2_Flag
Timer2 external Interrupt      => EXF2_Flag
Timer2 overflow Interrupt(TL2) => TF2L_Flag
****************************************************************/
unsigned char TF2_Flag, EXF2_Flag, TF2L_Flag;

void main()
{
    /****************************************************************
    Variables
    Temp_TH2    => Store the TH2 value
    Temp_TL2    => Store the TL2 value
    Temp_RCAP2H => Store the RCAP2H value
    Temp_RCAP2L => Store the RCAP2L value
    Period      => [TH2:TL2]
    Duty        => [RCAP2H:RCAP2L]
    ****************************************************************/
    unsigned char Temp_RCAP2H = 0, Temp_RCAP2L = 0;
    unsigned char Temp_TH2 = 0, Temp_TL2 = 0;
    unsigned int Period = 0, Duty = 0;

    /****************************************************************
     *  UART0 Initialized
     ****************************************************************/
    UART0_9600_Init();

    /****************************************************************
     *  Interrupt Initialized
     *  1. Enable Timer2 interrupt (ET2) = Enable;
     *  2. Global enables all interrupts (EA) = Enable;
     ****************************************************************/
    __DRV_TIMER2_IT_Cmd(MW_ENABLE);
    __DRV_INT_ITEA_Enable();

    /****************************************************************
     *  GPIO Initialized
     *  1. P55 used for LED_D4
     *  2. P11 used for T2EX Pin
     *  3. P34 used for T0CKO
     ****************************************************************/
    __DRV_GPIO_P5ModeSelect(P55_PushPull);
    __DRV_GPIO_P1ModeSelect(P11_OpenDrainPullUp);
    __DRV_GPIO_P3ModeSelect(P34_QuasiMode);

    /****************************************************************
     *  Timer0 Initialized
     *  PWM Duty high = 90.2% (230/255)
     *  PWM Duty low = 9.8% (25/255)
     ****************************************************************/
    __DRV_TIMER0_Mode_Select(TIMER0_MODE0_8BIT_PWM);
    __DRV_TIMER0_Clock_Source_Select(TIMER0_CLOCK_SOURCE_ILRCO);
    __DRV_TIMER0_SetHighByte(25);
    __DRV_TIMER0_T0CKO_Cmd(MW_ENABLE);
    __DRV_TIMER0_Run_Cmd(MW_ENABLE);

    /****************************************************************
     *  Timer2 Initialized
     *  Interrupt time => (1 / (SYSCLK/12)) * 65536 * 15 = 0.983 S
     ****************************************************************/
    __DRV_TIMER2_Mode_Select(TIMER2_MODE6_DUTY_CAPTURE);
    __DRV_TIMER2_Clock_Source_Select(TIMER2_CLOCK_SOURCE_SYSCLK_DIV_12);
    __DRV_TIMER2_Capture_Source_Dectect(Timer2_RISING_EDGE);
    __DRV_TIMER2_Run_Cmd(MW_ENABLE);

    while (1)
    {
        /****************************************************************
         *  Timer2 Duty Capture
         *  T0CKO PWM Signal(P34) output to LED_D4(P55) and T2EX Pin(P11)
         ****************************************************************/
        __DRV_GPIO_WriteP5(P55, P34);
        __DRV_GPIO_WriteP1(P11, P34);
        if (EXF2_Flag)
        {
            Temp_TH2 = DRV_TIMER2_GetTH2();
            Temp_TL2 = DRV_TIMER2_GetTL2();
            Period = Temp_TH2 << 8 | Temp_TL2;
            Temp_RCAP2H = DRV_TIMER2_GetRCAP2H();
            Temp_RCAP2L = DRV_TIMER2_GetRCAP2L();
            Duty = Temp_RCAP2H << 8 | Temp_RCAP2L;
            printf("Duty Time = %.3f mS, Period Time = %.3fmS\n", Duty * 0.001, Period * 0.001);
            printf("Duty = %.3f%% \n\n", ((float)Duty / (float)Period) * 100);
            EXF2_Flag = 0;
        }
    }
}

/****************************************************************
Timer2 interrupt
The Timer2 has TF2, EXF2, TF2L flags to trigger the Timer2 interrupt.
****************************************************************/
void TIMER2_TF2_EXF2_TF2L_ISR(void) interrupt TIMER2_ISR_VECTOR
{
    if (DRV_TIMER2_GetTF2() == 1)
    {
        TF2_Flag = 1;
        DRV_TIMER2_ClearTF2();
    }

    if (DRV_TIMER2_GetTF2L() == 1)
    {
        TF2L_Flag = 1;
        DRV_TIMER2_ClearTF2L();
    }

    if (DRV_TIMER2_GetEXF2() == 1)
    {
        EXF2_Flag = 1;
        DRV_TIMER2_ClearEXF2();
    }
}
