#include "bsp_uart5.h"


/**
  * @name    uart5_init
  * @brief   Init UART5
  * @param   apbclk: UART5 APB Clock frequency
  * @param   baud: Baudrate to configure
  * @retval  None
  */
void uart5_init(uint32_t apbclk, uint32_t baud)
{
  /* Enable UART5 and GPIOC Clock */
  RCC->APB2ENR |= RCC_APB2ENR_IOPCEN | RCC_APB2ENR_IOPDEN;
  RCC->APB1ENR |= RCC_APB1ENR_UART5EN;
  
  /* Reset UART5 */
  RCC->APB1RSTR |= RCC_APB1RSTR_UART5RST;
  RCC->APB1RSTR &= ~RCC_APB1RSTR_UART5RST;
  
  /* PC12(UART5_TX) configure to push pull, PD2(UART5_RX) configure to Input with Pull-up resistance */
  GPIOC->CRH = (GPIOC->CRH & 0xFFF0FFFF) | 0x000B0000;
  GPIOD->CRH = (GPIOD->CRL & 0xFFFFF0FF) | 0x00000800;
  GPIOD->BSRR = GPIO_BSRR_BS2;
  
  /* Configure UART5 */
  UART5->CR2 = 0x0000;
  UART5->CR1 = USART_CR1_TE | USART_CR1_RE;
  UART5->BRR = (apbclk + (baud >> 1)) / baud;  
  
  /* Enable UART5 */
  UART5->CR1 |= USART_CR1_UE;
}


/**
  * @name    uart5_send
  * @brief   Send data through UART5
  * @param   buffer: Buffer to send
  * @param   length: Buffer length
  * @retval  None
  */
void uart5_send(const uint8_t* buffer, uint32_t length)
{
  while(length)
  {
    while(!(UART5->SR & USART_SR_TC));
    UART5->DR = *buffer;
    buffer++;  length--;
  }
  while(!(UART5->SR & USART_SR_TC));
}







