/**
 * @file    mg32f157_can.h
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file contains all the functions prototypes for the CAN firmware
 *          library.
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MG32F157_CAN_H
#define __MG32F157_CAN_H

#ifdef __cplusplus
 extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "mg32f157.h"

/** @addtogroup MG32F157_StdPeriph_Driver
  * @{
  */

/** @addtogroup CAN
  * @{
  */

/* Exported types ------------------------------------------------------------*/

/** 
  * @brief  CAN Init structure definition  
  */
typedef struct
{
  uint8_t CAN_Mode;                               /*!< Specifies the CAN operating mode.
                                                       This parameter can be a value of @ref CAN_Mode */

  uint8_t CAN_Sample;                             /*!< Specifies the Number of bus level samples.
                                                       This parameter can be a value of @ref CAN_Sample_Level. */

  uint8_t AutoRetransmiteCmd;                     /*!< Enable or Disable the auto retransmission.
                                                       This parameter can be set either to ENABLE or DISABLE. */

  uint8_t CAN_Prescaler;                          /*!< Specifies the length of a time quantum.
                                                       It ranges from 1 to 128. */

  uint8_t CAN_SJW;                                /*!< Specifies the maximum number of time quanta
                                                       the CAN hardware is allowed to lengthen or
                                                       shorten a bit to perform resynchronization.
                                                       This parameter can be a value of @ref CAN_Synchronisation_Jump_Width. */

  uint8_t CAN_SEG1;                               /*!< Specifies the number of time quanta in Bit Segment 1.
                                                       This parameter can be a value of @ref CAN_Time_Quantum_In_Bit_Segment_1 */

  uint8_t CAN_SEG2;                               /*!< Specifies the number of time quanta in Bit Segment 2.
                                                       This parameter can be a value of @ref CAN_Time_Quantum_In_Bit_Segment_2 */
} CAN_InitTypeDef;

/** 
  * @brief  CANFD Init structure definition  
  */
typedef struct
{
  uint8_t CAN_Mode;                               /*!< Specifies the CAN operating mode.
                                                       This parameter can be a value of @ref CAN_Mode */

  uint8_t CAN_Sample;                             /*!< Specifies the Number of bus level samples.
                                                       This parameter can be a value of @ref CAN_Sample_Level. */

  uint8_t AutoRetransmiteCmd;                     /*!< Enable or Disable the auto retransmission.
                                                       This parameter can be set either to ENABLE or DISABLE. */

  uint8_t CAN_Prescaler;                          /*!< Specifies the length of a time quantum.
                                                       It ranges from 1 to 128. */

  uint8_t CAN_SJW;                                /*!< Specifies the maximum number of time quanta
                                                       the CAN hardware is allowed to lengthen or
                                                       shorten a bit to perform resynchronization.
                                                       This parameter can be a value of @ref CAN_Synchronisation_Jump_Width. */

  uint8_t CAN_SEG1;                               /*!< Specifies the number of time quanta in Bit Segment 1.
                                                       This parameter can be a value of @ref CAN_Synchronisation_Jump_Width_1 */

  uint8_t CAN_SEG2;                               /*!< Specifies the number of time quanta in Bit Segment 2.
                                                       This parameter can be a value of @ref CAN_Time_Quantum_In_Bit_Segment_2 */

  uint16_t FD_ArbitrationPrescaler;               /*!< Specifies the CAN FD arbitration phase length of time quantum.
                                                       It ranges from 1 to 2048. */

  uint8_t  FD_ArbitrationSJW;                     /*!< Specifies the CAN FD arbitration phase maximum number of time quanta
                                                       the CAN hardware is allowed to lengthen or
                                                       shorten a bit to perform resynchronization.
                                                       This parameter can be a value of @ref CANFD_Arbitrations_Synchronisation_Jump_Width. */

  uint8_t  FD_ArbitrationSEG1;                    /*!< Specifies the CAN FD arbitration phase number of time quanta in Bit Segment 1.
                                                       This parameter can be a value of @ref CANFD_Arbitrations_Time_Quantum_In_Bit_Segment_1 */

  uint8_t  FD_ArbitrationSEG2;                    /*!< Specifies the CAN FD arbitration phase number of time quanta in Bit Segment 2.
                                                       This parameter can be a value of @ref CANFD_Arbitrations_Time_Quantum_In_Bit_Segment_2 */

  uint8_t  FD_DataPrescaler;                      /*!< Specifies the CAN FD Data phase length of time quantum.
                                                       It ranges from 1 to 64. */

  uint8_t  FD_DataSJW;                            /*!< Specifies the CAN FD Data phase maximum number of time quanta
                                                       the CAN hardware is allowed to lengthen or
                                                       shorten a bit to perform resynchronization.
                                                       This parameter can be a value of @ref CANFD_Data_Synchronisation_Jump_Width. */

  uint8_t  FD_DataSEG1;                           /*!< Specifies the CAN FD Data phase number of time quanta in Bit Segment 1.
                                                       This parameter can be a value of @ref CANFD_Data_Time_Quantum_In_Bit_Segment_1 */

  uint8_t  FD_DataSEG2;                           /*!< Specifies the CAN FD Data phase number of time quanta in Bit Segment 2.
                                                       This parameter can be a value of @ref CANFD_Data_Time_Quantum_In_Bit_Segment_2 */

  uint8_t  FD_TransmateDelayCompensationCmd;      /*!< Enable or Disable the transmate delay compensation.
                                                       This parameter can be set either to ENABLE or DISABLE. */

  uint8_t  FD_TransmateDelayCompensationOffset;   /*!< Specifies the CAN FD transmate delay compensation offset.
                                                       This parameter can be a value of @ref CANFD_Transmate_Delay_Compensation_Offset */

  uint8_t  FD_RestrictedOperationCmd;             /*!< Enable or Disable the restricted operation mode.
                                                       This parameter can be set either to ENABLE or DISABLE. */

  uint8_t  FD_ISOFrameFormatSection;              /*!< Specifies ISO CAN FD frame format selection ISO11898-1 or Bosch.
                                                       This parameter can be a value of @ref CANFD_ISO_Frame_Format_Section */

  uint8_t  FD_ArbitrationTimingSection;           /*!< Specifies  Bit Time prescaler in Arbitration phase.
                                                       This parameter can be a value of @ref CANFD_Arbitration_Timing_Section */

  uint8_t FD_DataTimingSection;                   /*!< Specifies  Bit Time prescaler in Data phase.
                                                       This parameter can be a value of @ref CANFD_Data_Timing_Section */

  uint8_t  FD_FrameFormat;                        /*!< Specifies CAN FD frame format.
                                                       This parameter can be a value of @ref CANFD_Frame_Format */
} CANFD_InitTypeDef;

/**
  * @brief  CAN Filter Config structure definition
  */
typedef struct
{
  uint8_t Filter_Mode;                            /*!< Specifies CAN Filter Mode, dual filter or single filter
                                                       This parameter can be a value of @ref CAN_Filter_Mode */

  uint8_t Filter_FrameFormat;                    /*!< Specifies CAN Filter1 Format, Standard Frame or Extended Frame
                                                       This parameter can be a value of @ref CAN_Filter_FrameFormat */

  uint8_t Filter1_Type;                           /*!< Specifies CAN Filter1 Type, remote frame or data frame.
                                                       This parameter can be a value of @ref CAN_Filter_Type */

  uint8_t Filter1_RTRCmd;                         /*!< Specifies whether the filter1 filters RTR bits.
                                                       This parameter can be set either to ENABLE or DISABLE. */

  uint32_t Filter1_Code;                          /*!< Specifies CAN Filter1 Code.
                                                       It ranges from 0x0 to 0xFFFFFFFF. */

  uint32_t Filter1_Mask;                          /*!< Specifies CAN Filter1 Mask.
                                                       It ranges from 0x0 to 0xFFFFFFFF. */

  uint8_t Filter2_Type;                           /*!< Specifies CAN Filter2 Type, remote frame or data frame.
                                                       This parameter can be a value of @ref CAN_Filter_Type */

  uint8_t Filter2_RTRCmd;                         /*!< Specifies whether the filter1 filters RTR bits.
                                                       This parameter can be set either to ENABLE or DISABLE. */

  uint32_t Filter2_Code;                          /*!< Specifies CAN Filter2 Code.
                                                       It ranges from 0x0 to 0xFFFFFFFF. */

  uint32_t Filter2_Mask;                          /*!< Specifies CAN Filter2 Mask.
                                                       It ranges from 0x0 to 0xFFFFFFFF. */

} CAN_FilterInitTypeDef;

/** 
  * @brief  CAN Message structure definition  
  */
typedef struct
{
  uint32_t Identifier;                            /*!< Specifies the identifier.
                                                       This parameter must be a number between:
                                                       - 0 and 0x7FF, if IdType is CAN_IdType_Standard.
                                                       - 0 and 0x1FFFFFFF, if IdType is CAN_IdType_Extended */

  uint8_t IDType;                                 /*!< Specifies the identifier type for the message that will be transmitted.
                                                       etended ID or standard ID.
                                                       This parameter can be a value of @ref CAN_IDType */

  uint8_t FrameType;                              /*!< Specifies the frame type of the message that will be transmitted.
                                                       remote frame or data frame.
                                                       This parameter can be a value of @ref CAN_FrameType */

  uint8_t DataLength;                             /*!< Specifies the length of the frame that will be transmitted.
                                                       This parameter can be a value of @ref CAN_DataLength */

  uint8_t Data[8];                                /*!< Contains the data to be transmitted. It ranges from 0
                                                       to 0xFF. */
} CAN_Msg;

/** 
  * @brief  CANFD Message structure definition  
  */
typedef struct
{
  uint32_t Identifier;                            /*!< Specifies the identifier.
                                                       This parameter must be a number between:
                                                       - 0 and 0x7FF, if IdType is CAN_IdType_Standard
                                                       - 0 and 0x1FFFFFFF, if IdType is CAN_IdType_Extended */

  uint8_t IDType;                                 /*!< Specifies the identifier type for the message that will be transmitted.
                                                       etended ID or standard ID.
                                                       This parameter can be a value of @ref CAN_IDType */

  uint8_t FrameType;                              /*!< Specifies the frame type of the message that will be transmitted.
                                                       remote frame or data frame.
                                                       This parameter can be a value of @ref CAN_Frame_Type */

  uint8_t FrameFormat;                            /*!< Specifies the frame format for the message that will be transmitted.
                                                       class frame or FD frame.
                                                       This parameter can be a value of @ref CAN_Frame_Format */

  uint8_t BitRateSwitch;                          /*!< Specifies whether the Tx frame will be transmitted with or without
                                                       bit rate switching.
                                                       This parameter can be a value of @ref CAN_BitRate_Switch */

  uint8_t ErrorStateIndicator;                    /*!< Specifies the error state indicator.
                                                       This parameter can be a value of @ref CAN_Error_State_Indicator */

  uint8_t DataLength;                             /*!< Specifies the length of the frame that will be transmitted.
                                                       This parameter can be a value between 0 to 64 */
    
  uint8_t Data[64];

} CANFD_Msg;

/* Exported constants --------------------------------------------------------*/

/** @defgroup CAN_Exported_Constants 
  * @{
  */

#define IS_CAN_ALL_PERIPH(PERIPH) ((PERIPH) == CAN1)


/** @defgroup CAN_Mode
  * @{
  */
#define CAN_Mode_Normal                       ((uint8_t)0x00)
#define CAN_Mode_Silent                       ((uint8_t)0x01)
#define CAN_Mode_LoopBack                     ((uint8_t)0x11)
#define CAN_Mode_LoopBack_Silent              ((uint8_t)0x13)
/**
  * @}
  */


/** @defgroup CAN_Sample_Level 
  * @{
  */
#define CAN_Sample_Once                       ((uint8_t)0x00)
#define CAN_Sample_Three                      ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_PRESCALER 
  * @{
  */
#define IS_CAN_PRESCALER(PRESCALER) (((PRESCALER) >= 2) && ((PRESCALER) <= 128) && (!(PRESCALER % 2)))
/**
  * @}
  */


/** @defgroup CAN_Synchronisation_Jump_Width 
  * @{
  */
#define CAN_SJW_1tq                           ((uint8_t)0x00)
#define CAN_SJW_2tq                           ((uint8_t)0x01)
#define CAN_SJW_3tq                           ((uint8_t)0x02)
#define CAN_SJW_4tq                           ((uint8_t)0x03)
/**
  * @}
  */


/** @defgroup CAN_Synchronisation_Jump_Width_1 
  * @{
  */
#define CAN_SEG1_1tq                          ((uint8_t)0x00)
#define CAN_SEG1_2tq                          ((uint8_t)0x01)
#define CAN_SEG1_3tq                          ((uint8_t)0x02)
#define CAN_SEG1_4tq                          ((uint8_t)0x03)
#define CAN_SEG1_5tq                          ((uint8_t)0x04)
#define CAN_SEG1_6tq                          ((uint8_t)0x05)
#define CAN_SEG1_7tq                          ((uint8_t)0x06)
#define CAN_SEG1_8tq                          ((uint8_t)0x07)
#define CAN_SEG1_9tq                          ((uint8_t)0x08)
#define CAN_SEG1_10tq                         ((uint8_t)0x09)
#define CAN_SEG1_11tq                         ((uint8_t)0x0A)
#define CAN_SEG1_12tq                         ((uint8_t)0x0B)
#define CAN_SEG1_13tq                         ((uint8_t)0x0C)
#define CAN_SEG1_14tq                         ((uint8_t)0x0D)
#define CAN_SEG1_15tq                         ((uint8_t)0x0E)
#define CAN_SEG1_16tq                         ((uint8_t)0x0F)
/**
  * @}
  */


/** @defgroup CAN_Time_Quantum_In_Bit_Segment_2 
  * @{
  */
#define CAN_SEG2_1tq                          ((uint8_t)0x00)
#define CAN_SEG2_2tq                          ((uint8_t)0x01)
#define CAN_SEG2_3tq                          ((uint8_t)0x02)
#define CAN_SEG2_4tq                          ((uint8_t)0x03)
#define CAN_SEG2_5tq                          ((uint8_t)0x04)
#define CAN_SEG2_6tq                          ((uint8_t)0x05)
#define CAN_SEG2_7tq                          ((uint8_t)0x06)
#define CAN_SEG2_8tq                          ((uint8_t)0x07)
/**
  * @}
  */


/** @defgroup CANFD_Arbitrations_Pracaler 
  * @{
  */
#define IS_CANFD_ARBITRATION_PRESCALER(PRESCALER) (((PRESCALER) >= 1) && ((PRESCALER) <= 2048) && (!(PRESCALER % 2)))
/**
  * @}
  */


/** @defgroup CANFD_Arbitrations_Synchronisation_Jump_Width 
  * @{
  */
#define CANFD_ArbitrationSJW_1tq              ((uint8_t)0x00)
#define CANFD_ArbitrationSJW_2tq              ((uint8_t)0x01)
#define CANFD_ArbitrationSJW_3tq              ((uint8_t)0x02)
#define CANFD_ArbitrationSJW_4tq              ((uint8_t)0x03)
#define CANFD_ArbitrationSJW_5tq              ((uint8_t)0x04)
#define CANFD_ArbitrationSJW_6tq              ((uint8_t)0x05)
#define CANFD_ArbitrationSJW_7tq              ((uint8_t)0x06)
#define CANFD_ArbitrationSJW_8tq              ((uint8_t)0x07)
#define CANFD_ArbitrationSJW_9tq              ((uint8_t)0x08)
#define CANFD_ArbitrationSJW_10tq             ((uint8_t)0x09)
#define CANFD_ArbitrationSJW_11tq             ((uint8_t)0x0A)
#define CANFD_ArbitrationSJW_12tq             ((uint8_t)0x0B)
#define CANFD_ArbitrationSJW_13tq             ((uint8_t)0x0C)
#define CANFD_ArbitrationSJW_14tq             ((uint8_t)0x0D)
#define CANFD_ArbitrationSJW_15tq             ((uint8_t)0x0E)
#define CANFD_ArbitrationSJW_16tq             ((uint8_t)0x0F)
/**
  * @}
  */


/** @defgroup CANFD_Arbitrations_Time_Quantum_In_Bit_Segment_1 
  * @{
  */
#define CANFD_ArbitrationSEG1_1tq             ((uint8_t)0x00)
#define CANFD_ArbitrationSEG1_2tq             ((uint8_t)0x01)
#define CANFD_ArbitrationSEG1_3tq             ((uint8_t)0x02)
#define CANFD_ArbitrationSEG1_4tq             ((uint8_t)0x03)
#define CANFD_ArbitrationSEG1_5tq             ((uint8_t)0x04)
#define CANFD_ArbitrationSEG1_6tq             ((uint8_t)0x05)
#define CANFD_ArbitrationSEG1_7tq             ((uint8_t)0x06)
#define CANFD_ArbitrationSEG1_8tq             ((uint8_t)0x07)
#define CANFD_ArbitrationSEG1_9tq             ((uint8_t)0x08)
#define CANFD_ArbitrationSEG1_10tq            ((uint8_t)0x09)
#define CANFD_ArbitrationSEG1_11tq            ((uint8_t)0x0A)
#define CANFD_ArbitrationSEG1_12tq            ((uint8_t)0x0B)
#define CANFD_ArbitrationSEG1_13tq            ((uint8_t)0x0C)
#define CANFD_ArbitrationSEG1_14tq            ((uint8_t)0x0D)
#define CANFD_ArbitrationSEG1_15tq            ((uint8_t)0x0E)
#define CANFD_ArbitrationSEG1_16tq            ((uint8_t)0x0F)
#define CANFD_ArbitrationSEG1_17tq            ((uint8_t)0x10)
#define CANFD_ArbitrationSEG1_18tq            ((uint8_t)0x11)
#define CANFD_ArbitrationSEG1_19tq            ((uint8_t)0x12)
#define CANFD_ArbitrationSEG1_20tq            ((uint8_t)0x13)
#define CANFD_ArbitrationSEG1_21tq            ((uint8_t)0x14)
#define CANFD_ArbitrationSEG1_22tq            ((uint8_t)0x15)
#define CANFD_ArbitrationSEG1_23tq            ((uint8_t)0x16)
#define CANFD_ArbitrationSEG1_24tq            ((uint8_t)0x17)
#define CANFD_ArbitrationSEG1_25tq            ((uint8_t)0x18)
#define CANFD_ArbitrationSEG1_26tq            ((uint8_t)0x19)
#define CANFD_ArbitrationSEG1_27tq            ((uint8_t)0x1A)
#define CANFD_ArbitrationSEG1_28tq            ((uint8_t)0x1B)
#define CANFD_ArbitrationSEG1_29tq            ((uint8_t)0x1C)
#define CANFD_ArbitrationSEG1_30tq            ((uint8_t)0x1D)
#define CANFD_ArbitrationSEG1_31tq            ((uint8_t)0x1E)
#define CANFD_ArbitrationSEG1_32tq            ((uint8_t)0x1F)
#define CANFD_ArbitrationSEG1_33tq            ((uint8_t)0x20)
#define CANFD_ArbitrationSEG1_34tq            ((uint8_t)0x21)
#define CANFD_ArbitrationSEG1_35tq            ((uint8_t)0x22)
#define CANFD_ArbitrationSEG1_36tq            ((uint8_t)0x23)
#define CANFD_ArbitrationSEG1_37tq            ((uint8_t)0x24)
#define CANFD_ArbitrationSEG1_38tq            ((uint8_t)0x25)
#define CANFD_ArbitrationSEG1_39tq            ((uint8_t)0x26)
#define CANFD_ArbitrationSEG1_40tq            ((uint8_t)0x27)
#define CANFD_ArbitrationSEG1_41tq            ((uint8_t)0x28)
#define CANFD_ArbitrationSEG1_42tq            ((uint8_t)0x29)
#define CANFD_ArbitrationSEG1_43tq            ((uint8_t)0x2A)
#define CANFD_ArbitrationSEG1_44tq            ((uint8_t)0x2B)
#define CANFD_ArbitrationSEG1_45tq            ((uint8_t)0x2C)
#define CANFD_ArbitrationSEG1_46tq            ((uint8_t)0x2D)
#define CANFD_ArbitrationSEG1_47tq            ((uint8_t)0x2E)
#define CANFD_ArbitrationSEG1_48tq            ((uint8_t)0x2F)
#define CANFD_ArbitrationSEG1_49tq            ((uint8_t)0x30)
#define CANFD_ArbitrationSEG1_50tq            ((uint8_t)0x31)
#define CANFD_ArbitrationSEG1_51tq            ((uint8_t)0x32)
#define CANFD_ArbitrationSEG1_52tq            ((uint8_t)0x33)
#define CANFD_ArbitrationSEG1_53tq            ((uint8_t)0x34)
#define CANFD_ArbitrationSEG1_54tq            ((uint8_t)0x35)
#define CANFD_ArbitrationSEG1_55tq            ((uint8_t)0x36)
#define CANFD_ArbitrationSEG1_56tq            ((uint8_t)0x37)
#define CANFD_ArbitrationSEG1_57tq            ((uint8_t)0x38)
#define CANFD_ArbitrationSEG1_58tq            ((uint8_t)0x39)
#define CANFD_ArbitrationSEG1_59tq            ((uint8_t)0x3A)
#define CANFD_ArbitrationSEG1_60tq            ((uint8_t)0x3B)
#define CANFD_ArbitrationSEG1_61tq            ((uint8_t)0x3C)
#define CANFD_ArbitrationSEG1_62tq            ((uint8_t)0x3D)
#define CANFD_ArbitrationSEG1_63tq            ((uint8_t)0x3E)
#define CANFD_ArbitrationSEG1_64tq            ((uint8_t)0x3F)
/**
  * @}
  */


/** @defgroup CANFD_Arbitrations_Time_Quantum_In_Bit_Segment_2 
  * @{
  */
#define CANFD_ArbitrationSEG2_1tq             ((uint8_t)0x00)
#define CANFD_ArbitrationSEG2_2tq             ((uint8_t)0x01)
#define CANFD_ArbitrationSEG2_3tq             ((uint8_t)0x02)
#define CANFD_ArbitrationSEG2_4tq             ((uint8_t)0x03)
#define CANFD_ArbitrationSEG2_5tq             ((uint8_t)0x04)
#define CANFD_ArbitrationSEG2_6tq             ((uint8_t)0x05)
#define CANFD_ArbitrationSEG2_7tq             ((uint8_t)0x06)
#define CANFD_ArbitrationSEG2_8tq             ((uint8_t)0x07)
#define CANFD_ArbitrationSEG2_9tq             ((uint8_t)0x08)
#define CANFD_ArbitrationSEG2_10tq            ((uint8_t)0x09)
#define CANFD_ArbitrationSEG2_11tq            ((uint8_t)0x0A)
#define CANFD_ArbitrationSEG2_12tq            ((uint8_t)0x0B)
#define CANFD_ArbitrationSEG2_13tq            ((uint8_t)0x0C)
#define CANFD_ArbitrationSEG2_14tq            ((uint8_t)0x0D)
#define CANFD_ArbitrationSEG2_15tq            ((uint8_t)0x0E)
#define CANFD_ArbitrationSEG2_16tq            ((uint8_t)0x0F)
/**
  * @}
  */


/** @defgroup CANFD_Data_Prescaler 
  * @{
  */
#define IS_CANFD_DATA_PRESCALER(PRESCALER) (((PRESCALER) >= 1) && ((PRESCALER) <= 64) && (!(PRESCALER % 2)))
/**
  * @}
  */


/** @defgroup CANFD_Data_Synchronisation_Jump_Width 
  * @{
  */
#define CANFD_DataSJW_1tq                     ((uint8_t)0x00)
#define CANFD_DataSJW_2tq                     ((uint8_t)0x01)
#define CANFD_DataSJW_3tq                     ((uint8_t)0x02)
#define CANFD_DataSJW_4tq                     ((uint8_t)0x03)
#define CANFD_DataSJW_5tq                     ((uint8_t)0x04)
#define CANFD_DataSJW_6tq                     ((uint8_t)0x05)
#define CANFD_DataSJW_7tq                     ((uint8_t)0x06)
#define CANFD_DataSJW_8tq                     ((uint8_t)0x07)
#define CANFD_DataSJW_9tq                     ((uint8_t)0x08)
#define CANFD_DataSJW_10tq                    ((uint8_t)0x09)
#define CANFD_DataSJW_11tq                    ((uint8_t)0x0A)
#define CANFD_DataSJW_12tq                    ((uint8_t)0x0B)
#define CANFD_DataSJW_13tq                    ((uint8_t)0x0C)
#define CANFD_DataSJW_14tq                    ((uint8_t)0x0D)
#define CANFD_DataSJW_15tq                    ((uint8_t)0x0E)
#define CANFD_DataSJW_16tq                    ((uint8_t)0x0F)
/**
  * @}
  */


/** @defgroup CANFD_Data_Time_Quantum_In_Bit_Segment_1 
  * @{
  */
#define CANFD_DataSEG1_1tq                    ((uint8_t)0x00)
#define CANFD_DataSEG1_2tq                    ((uint8_t)0x01)
#define CANFD_DataSEG1_3tq                    ((uint8_t)0x02)
#define CANFD_DataSEG1_4tq                    ((uint8_t)0x03)
#define CANFD_DataSEG1_5tq                    ((uint8_t)0x04)
#define CANFD_DataSEG1_6tq                    ((uint8_t)0x05)
#define CANFD_DataSEG1_7tq                    ((uint8_t)0x06)
#define CANFD_DataSEG1_8tq                    ((uint8_t)0x07)
#define CANFD_DataSEG1_9tq                    ((uint8_t)0x08)
#define CANFD_DataSEG1_10tq                   ((uint8_t)0x09)
#define CANFD_DataSEG1_11tq                   ((uint8_t)0x0A)
#define CANFD_DataSEG1_12tq                   ((uint8_t)0x0B)
#define CANFD_DataSEG1_13tq                   ((uint8_t)0x0C)
#define CANFD_DataSEG1_14tq                   ((uint8_t)0x0D)
#define CANFD_DataSEG1_15tq                   ((uint8_t)0x0E)
#define CANFD_DataSEG1_16tq                   ((uint8_t)0x0F)
#define CANFD_DataSEG1_17tq                   ((uint8_t)0x10)
#define CANFD_DataSEG1_18tq                   ((uint8_t)0x11)
#define CANFD_DataSEG1_19tq                   ((uint8_t)0x12)
#define CANFD_DataSEG1_20tq                   ((uint8_t)0x13)
#define CANFD_DataSEG1_21tq                   ((uint8_t)0x14)
#define CANFD_DataSEG1_22tq                   ((uint8_t)0x15)
#define CANFD_DataSEG1_23tq                   ((uint8_t)0x16)
#define CANFD_DataSEG1_24tq                   ((uint8_t)0x17)
#define CANFD_DataSEG1_25tq                   ((uint8_t)0x18)
#define CANFD_DataSEG1_26tq                   ((uint8_t)0x19)
#define CANFD_DataSEG1_27tq                   ((uint8_t)0x1A)
#define CANFD_DataSEG1_28tq                   ((uint8_t)0x1B)
#define CANFD_DataSEG1_29tq                   ((uint8_t)0x1C)
#define CANFD_DataSEG1_30tq                   ((uint8_t)0x1D)
#define CANFD_DataSEG1_31tq                   ((uint8_t)0x1E)
#define CANFD_DataSEG1_32tq                   ((uint8_t)0x1F)
/**
  * @}
  */


/** @defgroup CANFD_Data_Time_Quantum_In_Bit_Segment_2 
  * @{
  */
#define CANFD_DataSEG2_1tq                    ((uint8_t)0x00)
#define CANFD_DataSEG2_2tq                    ((uint8_t)0x01)
#define CANFD_DataSEG2_3tq                    ((uint8_t)0x02)
#define CANFD_DataSEG2_4tq                    ((uint8_t)0x03)
#define CANFD_DataSEG2_5tq                    ((uint8_t)0x04)
#define CANFD_DataSEG2_6tq                    ((uint8_t)0x05)
#define CANFD_DataSEG2_7tq                    ((uint8_t)0x06)
#define CANFD_DataSEG2_8tq                    ((uint8_t)0x07)
/**
  * @}
  */


/** @defgroup CANFD_Transmate_Delay_Compensation_Offset 
  * @{
  */
#define IS_CANFD_TransmateDelayCompensationOffset(OFFSET) (((OFFSET) >= 1) && ((OFFSET) <= 128))
/**
  * @}
  */


/** @defgroup CANFD_ISO_Frame_Format_Section 
  * @{
  */
#define CANFD_ISOFrameFormatSection_BoschCAN           ((uint8_t)0x00)
#define CANFD_ISOFrameFormatSection_ISO11898           ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CANFD_Arbitration_Timing_Section 
  * @{
  */
#define CANFD_ArbitrationTimingSection_BT             ((uint8_t)0x00)
#define CANFD_ArbitrationTimingSection_NBT            ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CANFD_Data_Timing_Section 
  * @{
  */
#define CANFD_DataTimingSection_HOLD                  ((uint8_t)0x00)
#define CANFD_DataTimingSection_DBT                   ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CANFD_Frame_Format 
  * @{
  */
#define CANFD_FrameFormat_Classic                     ((uint8_t)0x00)
#define CANFD_FrameFormat_FD                          ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_Filter_Mode
  * @{
  */
#define CAN_Filter_Mode_Dual                          ((uint8_t)0x00)
#define CAN_Filter_Mode_Single                        ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_Filter_FrameFormat
  * @{
  */
#define CAN_Filter_FrameFormat_Standard               ((uint8_t)0x00)
#define CAN_Filter_FrameFormat_Extended               ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_Filter_Type 
  * @{
  */
#define CAN_Filter_Type_Data                          ((uint8_t)0x00)
#define CAN_Filter_Type_Remote                        ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_IDType 
  * @{
  */
#define CAN_IDType_Standard                           ((uint8_t)0x00)
#define CAN_IDType_Extended                           ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_Frame_Type 
  * @{
  */
#define CAN_FrameType_Data                            ((uint8_t)0x00)
#define CAN_FrameType_Remote                          ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_Frame_Format 
  * @{
  */
#define CAN_FrameFormat_Class                         ((uint8_t)0x00)
#define CAN_FrameFormat_FD                            ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_BitRate Switch 
  * @{
  */
#define CAN_BitRateSwitch_WithoutSwitch               ((uint8_t)0x00)
#define CAN_BitRateSwitch_Switch                      ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_Error_State_Indicator 
  * @{
  */
#define CAN_ErrorStateIndicator_PassiveErrorNode      ((uint8_t)0x00)
#define CAN_ErrorStateIndicator_ActiveErrorNode       ((uint8_t)0x01)
/**
  * @}
  */


/** @defgroup CAN_Cmd 
  * @{
  */
#define CAN_Cmd_TransmitAbort                         ((uint32_t)0x01U << 9)
#define CAN_Cmd_TransmitStart                         ((uint32_t)0x02U << 9)
#define CAN_Cmd_TransmitSingle                        ((uint32_t)0x03U << 9)
/**
  * @}
  */


/** @defgroup CAN_Transmit_Mode 
  * @{
  */
#define CAN_TransmitMode_Retransmission               ((uint32_t)0x02U << 9)
#define CAN_TransmitMode_Single                       ((uint32_t)0x03U << 9)
/**
  * @}
  */


/** @defgroup CAN_Interrupt_Define 
  * @{
  */
#define CAN_IT_DOI                                    ((uint32_t)0x01U << 0)      /*!< CAN Data Overrun Interrupt Enable*/
#define CAN_IT_BEI                                    ((uint32_t)0x01U << 1)      /*!< CAN Buss Error Interrupt Enable*/
#define CAN_IT_TI                                     ((uint32_t)0x01U << 2)      /*!< CAN Transmission Interrupt Enable*/
#define CAN_IT_RI                                     ((uint32_t)0x01U << 3)      /*!< CAN Receive Interrupt Enable*/
#define CAN_IT_EPI                                    ((uint32_t)0x01U << 4)      /*!< CAN Error Passive Interrupt Enable*/
#define CAN_IT_EWI                                    ((uint32_t)0x01U << 5)      /*!< CAN Error Warning Interrupt Enable*/
#define CAN_IT_ALI                                    ((uint32_t)0x01U << 6)      /*!< CAN Arbitration Lost Interrupt Enable*/
/**
  * @}
  */


/** @defgroup CAN_Interrupt_Flag 
  * @{
  */
#define CAN_Flag_DOI                                  ((uint32_t)0x01U << 24)     /*!< CAN Data Overrun Interrupt Flag Bit*/
#define CAN_Flag_BEI                                  ((uint32_t)0x01U << 25)     /*!< CAN Buss Error Interrupt Flag Bit*/
#define CAN_Flag_TI                                   ((uint32_t)0x01U << 26)     /*!< CAN Transmission Interrupt Flag Bit*/
#define CAN_Flag_RI                                   ((uint32_t)0x01U << 27)     /*!< CAN Receive Interrupt Flag Bit*/
#define CAN_Flag_EPI                                  ((uint32_t)0x01U << 28)     /*!< CAN Error Passive Interrupt Flag Bit*/
#define CAN_Flag_EWI                                  ((uint32_t)0x01U << 29)     /*!< CAN Error Warning Interrupt Flag Bit*/
#define CAN_Flag_ALI                                  ((uint32_t)0x01U << 30)     /*!< CAN Arbitration Lost Interrupt Flag Bit*/
/**
  * @}
  */


/** @defgroup CAN_Status_Define 
  * @{
  */
#define CAN_Status_BS                                 ((uint32_t)0x01U << 16)     /*!< CAN Bus off Status*/
#define CAN_Status_ES                                 ((uint32_t)0x01U << 17)     /*!< CAN Error Status*/
#define CAN_Status_TS                                 ((uint32_t)0x01U << 18)     /*!< CAN Transmit Status*/
#define CAN_Status_RS                                 ((uint32_t)0x01U << 19)     /*!< CAN Receive Status*/
#define CAN_Status_TBS                                ((uint32_t)0x01U << 21)     /*!< CAN Transmit Buffer Status*/
#define CAN_Status_DSO                                ((uint32_t)0x01U << 22)     /*!< CAN Data Overrun Status*/
#define CAN_Status_RBS                                ((uint32_t)0x01U << 23)     /*!< CAN Receive Buffer Status*/
/**
  * @}
  */


/** @defgroup CAN_Error_Code 
  * @{
  */
#define CAN_Error_Code_BER                            ((uint32_t)CAN_ECC_BER)
#define CAN_Error_Code_STFER                          ((uint32_t)CAN_ECC_STFER)
#define CAN_Error_Code_CRCER                          ((uint32_t)CAN_ECC_CRCER)
#define CAN_Error_Code_FRMER                          ((uint32_t)CAN_ECC_FRMER)
#define CAN_Error_Code_ACKER                          ((uint32_t)CAN_ECC_ACKER)
#define CAN_Error_Code_EDIR                           ((uint32_t)CAN_ECC_EDIR)
#define CAN_Error_Code_TXWRN                          ((uint32_t)CAN_ECC_TXWRN)
#define CAN_Error_Code_RXERR                          ((uint32_t)CAN_ECC_RXERR)
/**
  * @}
  */


/** @defgroup CANFD_Status_Define 
  * @{
  */
#define CANFD_Status_Init                             ((uint32_t)0x00U << 14)
#define CANFD_Status_Idle                             ((uint32_t)0x01U << 14)
#define CANFD_Status_Recv                             ((uint32_t)0x02U << 14)
#define CANFD_Status_Send                             ((uint32_t)0x03U << 14)
/**
  * @}
  */


/** @defgroup CANFD_Error 
  * @{
  */
#define CANFD_Error_BITERR                            ((uint32_t)0x01U << 8)
#define CANFD_Error_CRCERR                            ((uint32_t)0x01U << 9)
#define CANFD_Error_FRMERR                            ((uint32_t)0x01U << 10)
#define CANFD_Error_STFERR                            ((uint32_t)0x01U << 11)
/**
  * @}
  */

/**
  * @}
  */

/* Exported macro ------------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/

void CAN_DeInit(CAN_TypeDef* CANx);

void CAN_Init(CAN_TypeDef* CANx, CAN_InitTypeDef* CAN_InitStruct);
void CAN_StructInit(CAN_InitTypeDef* CAN_InitStruct);
void CAN_Transmit(CAN_TypeDef* CANx, CAN_Msg* Message, uint32_t AutoRetransmissionEnable);
void CAN_Receive(CAN_TypeDef* CANx, CAN_Msg* RxMessage);

void CANFD_Init(CAN_TypeDef* CANx, CANFD_InitTypeDef* CANFD_InitStruct);
void CANFD_StructInit(CANFD_InitTypeDef* CANFD_InitStruct);
void CANFD_Transmit(CAN_TypeDef* CANx, CANFD_Msg* TxMessage, uint32_t AutoRetransmissionEnable);
void CANFD_Receive(CAN_TypeDef* CANx, CANFD_Msg* RxMessage);
uint8_t CANFD_GetArbitrationErrorCount(CAN_TypeDef* CANx);
uint8_t CANFD_GetDataErrorCount(CAN_TypeDef* CANx);
FlagStatus CANFD_GetStatus(CAN_TypeDef* CANx, uint32_t CANFD_Status);
FlagStatus CANFD_GetError(CAN_TypeDef* CANx, uint32_t CANFD_Error);
uint8_t CANFD_GetSecondarySamplePosition(CAN_TypeDef* CANx);

void CAN_FilterInit(CAN_TypeDef* CANx, CAN_FilterInitTypeDef* CAN_FilterInitStruct);
void CAN_FilterStructInit(CAN_FilterInitTypeDef* CAN_FilterInitStruct);
void CAN_Reset(CAN_TypeDef* CANx, FunctionalState NewState);
void CAN_SendCmd(CAN_TypeDef* CANx, uint32_t CAN_Cmd);
void CAN_ITConfig(CAN_TypeDef* CANx, uint32_t CAN_IT, FunctionalState NewState);
void CAN_ClearITPendingBit(CAN_TypeDef* CANx, uint32_t CAN_Flag);

FlagStatus CAN_GetITStatus(CAN_TypeDef* CANx, uint32_t CAN_Flag);
FlagStatus CAN_GetFlagStatus(CAN_TypeDef* CANx, uint32_t CAN_Status);
uint8_t CAN_GetArbitrationLosePosition(CAN_TypeDef* CANx);
uint8_t CAN_GetMessageCount(CAN_TypeDef* CANx);
uint8_t CAN_GetTransmitErrorCount(CAN_TypeDef* CANx);
uint8_t CAN_GetReceiveErrorCount(CAN_TypeDef* CANx);

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __MG32F157_CAN_H */

