#include "mg32f157.h"
#include "bsp_usart1.h"
#include <stdio.h>

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define RTCClockOutput_Enable  /* RTC Clock/64 is output on tamper pin(PC.13) */

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
TIM_TimeBaseInitTypeDef TIM_TimeBaseStructure;
TIM_ICInitTypeDef TIM_ICInitStructure;
RCC_ClocksTypeDef RCC_Clocks;

__IO uint32_t PeriodValue = 0,  LsiFreq = 0;
__IO uint32_t OperationComplete = 0;
uint16_t tmpCC4[2] = {0, 0};

/* Private function prototypes -----------------------------------------------*/
void GPIO_Configuration(void);
void RTC_Configuration(void);
void NVIC_Configuration(void);

/* Private functions ---------------------------------------------------------*/

/**
 * @brief  Main program.
 * @param  None
 * @return None
 */
int main(void)
{
  /* Enable TIM5 APB1 clocks */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM5, ENABLE);
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_AFIO, ENABLE);
  
  SystemCoreClockUpdate();
  usart1_init(SystemCoreClock, 115200);
  
  GPIO_Configuration();
  
  RTC_Configuration();
  
  /* Get the Frequency value */
  RCC_GetClocksFreq(&RCC_Clocks);
  
  printf("test start\r\n");
  /* Get the Frequency value */
  RCC_GetClocksFreq(&RCC_Clocks);

  /* Connect internally the TM5_CH4 Input Capture to the LSI clock output */
  GPIO_PinRemapConfig(GPIO_Remap_TIM5CH4_LSI, ENABLE);

  /* NVIC configuration */
  NVIC_Configuration();
  
  /* TIM5 Time base configuration */
  TIM_TimeBaseStructure.TIM_Prescaler = 0;
  TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;
  TIM_TimeBaseStructure.TIM_Period = 0xFFFF;
  TIM_TimeBaseStructure.TIM_ClockDivision = TIM_CKD_DIV1;
  TIM_TimeBaseInit(TIM5, &TIM_TimeBaseStructure);

  /* TIM5 Channel4 Input capture Mode configuration */
  TIM_ICInitStructure.TIM_Channel = TIM_Channel_4;
  TIM_ICInitStructure.TIM_ICPolarity = TIM_ICPolarity_Rising;
  TIM_ICInitStructure.TIM_ICSelection = TIM_ICSelection_DirectTI;
  TIM_ICInitStructure.TIM_ICPrescaler = TIM_ICPSC_DIV1;
  TIM_ICInitStructure.TIM_ICFilter = 0;
  TIM_ICInit(TIM5, &TIM_ICInitStructure);

  /* Reinitialize the index for the interrupt */
  OperationComplete = 0;

  /* Reset all TIM5 flags */
  TIM5->SR = 0x00;
  /* Enable the TIM5 channel 4 */
  TIM_ITConfig(TIM5, TIM_IT_CC4, ENABLE);
  
  /* Enable the TIM5 Input Capture counter */
  TIM_Cmd(TIM5, ENABLE);

  /* Wait the TIM5 measuring operation to be completed */
  while (OperationComplete != 2)
  {
  }
  
  LsiFreq = (uint32_t)((uint32_t)(RCC_Clocks.PCLK1_Frequency * 2) / (uint32_t)PeriodValue);
  
  /* Adjust the RTC prescaler value */
  RTC_SetPrescaler(LsiFreq - 1);

  /* Wait until last write operation on RTC registers has finished */
  RTC_WaitForLastTask();  
  
  printf("Now the LSI Frequency is :%4d HZ\r\n",LsiFreq);
  printf("Now the frequency should be : %.2f HZ\r\n", LsiFreq * 1.0 / 64);
  
  /* Infinite loop */
  while (1)
  {
  }
}

/**
 * @brief  Configures the different GPIO ports.
 * @param  None
 * @return None
 */
void GPIO_Configuration(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;
  
  /* Enable GPIOC, GPIOB and AFIO clocks */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOC | RCC_APB2Periph_AFIO | RCC_APB2Periph_GPIOB, ENABLE);
  
  /* Configure PC13 pin as output alternatefunction pushpull mode */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_13;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_Init(GPIOC, &GPIO_InitStructure);

  /* Configure PB.10 and PB.11 pins as output pushpull mode */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_10 | GPIO_Pin_11;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GPIOB, &GPIO_InitStructure);
}

/**
 * @brief  Configures the RTC and TIM5 interrupt requests.
 * @param  None
 * @return None
 */
void NVIC_Configuration(void)
{
  NVIC_InitTypeDef NVIC_InitStructure;

  /* Configure one bit for preemption priority */
  NVIC_PriorityGroupConfig(NVIC_PriorityGroup_1);

  /* Enable the RTC Interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = RTC_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  /* Enable the TIM5 Interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = TIM5_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 2;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
}

/**
 * @brief  Configures the RTC.
 * @param  None
 * @return None
 */
void RTC_Configuration(void)
{
  /* Enable PWR and BKP clocks */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_PWR | RCC_APB1Periph_BKP, ENABLE);

  /* Enable write access to Backup domain */
  PWR_BackupAccessCmd(ENABLE);

  /* Reset Backup Domain */
  BKP_DeInit();

  /* Enable the LSI OSC */
  RCC_LSICmd(ENABLE);
  
  /* Wait till LSI is ready */
  while (RCC_GetFlagStatus(RCC_FLAG_LSIRDY) == RESET)
  {
  }

  /* Select the RTC Clock Source */
  RCC_RTCCLKConfig(RCC_RTCCLKSource_LSI);

  /* Enable RTC Clock */
  RCC_RTCCLKCmd(ENABLE);

  /* Wait for RTC registers synchronization */
  RTC_WaitForSynchro();

  /* Wait until last write operation on RTC registers has finished */
  RTC_WaitForLastTask();

  /* Enable the RTC Second */
  RTC_ITConfig(RTC_IT_SEC, ENABLE);

  /* Wait until last write operation on RTC registers has finished */
  RTC_WaitForLastTask();

  /* Set RTC prescaler: set RTC period to 1sic */
  RTC_SetPrescaler(40000);

  /* Wait until last write operation on RTC registers has finished */
  RTC_WaitForLastTask();

  /* To output second signal on Tamper pin, the tamper functionality
     must be disabled (by default this functionality is disabled) */
  BKP_TamperPinCmd(DISABLE);

  BKP_RTCOutputConfig(BKP_RTCOutputSource_CalibClock);
}

/**
 * @brief  Increments OperationComplete variable and return its value 
 *         before increment operation.
 * @param  None
 * @return OperationComplete value before increment
 */
uint32_t IncrementVar_OperationComplete(void)
{
  OperationComplete++;
  
  return (uint32_t)(OperationComplete -1);
}

/**
 * @brief  Returns OperationComplete value.
 * @param  None
 * @return OperationComplete value
 */
uint32_t GetVar_OperationComplete(void)
{
  return (uint32_t)OperationComplete;
}

/**
 * @brief  Sets the PeriodValue variable with input parameter.
 * @param  Value: Value of PeriodValue to be set.
 * @return None
 */
void SetVar_PeriodValue(uint32_t Value)
{
  PeriodValue = (uint32_t)(Value);
}

/**
 * @brief  This function handles RTC global interrupt request.
 * @param  None
 * @return None
 */
void RTC_IRQHandler(void)
{
  if (RTC_GetITStatus(RTC_IT_SEC) != RESET)
  {
    /* Toggle LED1 */
    GPIO_ToggleBits(GPIOB, GPIO_Pin_10);
    /* Clear Interrupt pending bit */
    RTC_ClearITPendingBit(RTC_FLAG_SEC);
  }
}

/**
 * @brief  This function handles TIM5 global interrupt request.
 * @param  None
 * @return None
 */
void TIM5_IRQHandler(void)
{
  uint32_t tmp = 0; 
  
  if (TIM_GetITStatus(TIM5, TIM_IT_CC4) == SET)
  {
    tmpCC4[IncrementVar_OperationComplete()] = (uint16_t)(TIM5->CCR4);

    TIM_ClearITPendingBit(TIM5, TIM_IT_CC4);
    
    if (GetVar_OperationComplete() >= 2)
    {
      /* Compute the period length */
      tmp = (uint16_t)(tmpCC4[1] - tmpCC4[0] + 1);
      SetVar_PeriodValue(tmp);
      
      /* Disable the TIM5 CCR4 */
      TIM_CCxCmd(TIM5, TIM_Channel_4, DISABLE);
      /* Disable the interrupt */
      TIM_ITConfig(TIM5, TIM_IT_CC4, DISABLE);
      TIM_Cmd(TIM5, DISABLE);
    }
  }
  TIM5->SR = 0;
}

#ifdef  USE_FULL_ASSERT
/**
 * @brief  Reports the name of the source file and the source line number
 *         where the assert_param error has occurred.
 * @param  file: pointer to the source file name
 * @param  line: assert_param error line source number
 * @return None
 */
void assert_failed(uint8_t* file, uint32_t line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif
