/**
 * @file    mg32f157_sdio.h
 * @author  MegawinTech Application Team
 * @version V0.0.4
 * @date    16-June-2023
 * @brief   This file contains all the functions prototypes for the SDIO firmware
 *          library.
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MG32F157_SDIO_H
#define __MG32F157_SDIO_H

#ifdef __cplusplus
 extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "mg32f157.h"

/** @addtogroup MG32F157_StdPeriph_Driver
  * @{
  */

/** @addtogroup SDIO
  * @{
  */

/* Exported types ------------------------------------------------------------*/
 
/**
 * @brief  SDIO Init structure definition
 */
typedef struct
{
  uint32_t ClockPowerSave;       /* Specifies whether SDIO Clock output is enabled or disabled when the bus is idle.
                                    This parameter can be a value of @ref SDIO_Clock_Power_Save */
  uint32_t BusWide;              /* Specifies the SDIO bus width.
                                    This parameter can be a value of @ref SDIO_Bus_Wide */
  uint32_t ClockDiv;             /* Specifies the clock frequency of the SDIO controller.
                                    This parameter can be a value between Min_Data = 0 and Max_Data = 255 */

} SDIO_InitTypeDef;

/**
 * @brief  SDIO Command Control structure
 */
typedef struct
{
  uint32_t Argument;             /* Specifies the SDIO command argument which is sent
                                    to a card as part of a command message. If a command
                                    contains an argument, it must be loaded into this register
                                    before writing the command to the command register. */
  uint32_t CmdIndex;             /* Specifies the SDIO command index. It must be Min_Data = 0 and
                                    Max_Data = 64 */
  uint32_t Response;             /* Specifies the SDIO response type.
                                    This parameter can be a value of @ref SDIO_Response_Type */
  uint32_t CPSM;                 /* Specifies whether SDIO Command path state machine (CPSM) is enabled or disabled.
                                    This parameter can be a value of @ref SDIO_CPSM_State */
} SDIO_CmdInitTypeDef;

/**
 * @brief  SDIO Data Control structure
 */
typedef struct
{
  uint32_t DataTimeOut;          /* Specifies the data timeout period in card bus clock periods. */

  uint32_t DataLength;           /* Specifies the number of data bytes to be transferred. */

  uint32_t DataBlockSize;        /* Specifies the data block size for block transfer.
                                    This parameter can be a value of @ref SDIO_Data_Block_Size */
  uint32_t TransferDir;          /* Specifies the data transfer direction, whether the transfer is a read or write.
                                    This parameter can be a value of @ref SDIO_Transfer_Direction */
  uint32_t TransferMode;         /* Specifies whether data transfer is in stream or block mode.
                                    This parameter can be a value of @ref SDIO_Transfer_Type */
  uint32_t DPSM;                 /* Specifies whether SDIO Data path state machine (DPSM) is enabled or disabled.
                                    This parameter can be a value of @ref SDIO_DPSM_State */
} SDIO_DataInitTypeDef;

/* Exported constants --------------------------------------------------------*/

/** @defgroup SDIO_Exported_Constants
  * @{
  */

#define IS_SDIO_ALL_PERIPH(SDIOX)    (((SDIOX) == SDIO))


/** @defgroup SDIO_Clock_Power_Save Clock Power Saving
 * @{
 */
#define SDIO_CLOCK_POWER_SAVE_DISABLE  0x00000000U
#define SDIO_CLOCK_POWER_SAVE_ENABLE   0x0000FFFFU // SDIO_CLKCR_PWRSAV
#define IS_SDIO_CLOCK_POWER_SAVE(SAVE) (((SAVE) == SDIO_CLOCK_POWER_SAVE_DISABLE) || \
                                        ((SAVE) == SDIO_CLOCK_POWER_SAVE_ENABLE))
/**
 * @}
 */


/** @defgroup SDIO_Bus_Wide Bus Width
 * @{
 */
#define SDIO_BUS_WIDE_1B       0x00000000U
#define SDIO_BUS_WIDE_4B       0x0000FFFFU // SDIO_CLKCR_WIDBUS_0
#define SDIO_BUS_WIDE_8B       0xFFFFFFFFU // SDIO_CLKCR_WIDBUS_1
#define IS_SDIO_BUS_WIDE(WIDE) (((WIDE) == SDIO_BUS_WIDE_1B) || \
                                ((WIDE) == SDIO_BUS_WIDE_4B) || \
                                ((WIDE) == SDIO_BUS_WIDE_8B))
/**
 * @}
 */


/** @defgroup SDIO_Clock_Division Clock Division
 * @{
 */
#define IS_SDIO_CLKDIV(DIV)        ((DIV) <= 0xFFU)
/**
 * @}
 */


/** @defgroup SDIO_Command_Index Command Index
 * @{
 */
#define IS_SDIO_CMD_INDEX(INDEX)   ((INDEX) < 0x40U)
/**
 * @}
 */


/** @defgroup SDIO_Response_Type Response Type
 * @{
 */
#define SDIO_Response_No           0x00000000U
#define SDIO_Response_Short        0x00000040U // SDIO_CMD_WAITRESP_0
#define SDIO_Response_Long         0x000000C0U // SDIO_CMD_WAITRESP
#define IS_SDIO_RESPONSE(RESPONSE) (((RESPONSE) == SDIO_Response_No) ||    \
                                    ((RESPONSE) == SDIO_Response_Short) || \
                                    ((RESPONSE) == SDIO_Response_Long))
/**
 * @}
 */


/** @defgroup SDIO_CPSM_State CPSM State
 * @{
 */
#define SDIO_CPSM_Disable  0x00000000U
#define SDIO_CPSM_Enable   0
#define IS_SDIO_CPSM(CPSM) 1
/**
 * @}
 */


/** @defgroup SDIO_Response_Registers Response Register
 * @{
 */
#define SDIO_RESP1         0x00000000U
#define SDIO_RESP2         0x00000004U
#define SDIO_RESP3         0x00000008U
#define SDIO_RESP4         0x0000000CU
#define IS_SDIO_RESP(RESP) (((RESP) == SDIO_RESP1) || \
                            ((RESP) == SDIO_RESP2) || \
                            ((RESP) == SDIO_RESP3) || \
                            ((RESP) == SDIO_RESP4))
/**
 * @}
 */


/** @defgroup SDIO_Data_Length Data Lenght
 * @{
 */
#define IS_SDIO_DATA_LENGTH(LENGTH) ((LENGTH) <= 0x01FFFFFFU)
/**
 * @}
 */


/** @defgroup SDIO_Data_Block_Size  Data Block Size
 * @{
 */
#define SDIO_DATABLOCK_SIZE_1B      0x00000000U
#define SDIO_DATABLOCK_SIZE_2B      0x00000002U
#define SDIO_DATABLOCK_SIZE_4B      0x00000004U
#define SDIO_DATABLOCK_SIZE_8B      0x00000008U
#define SDIO_DATABLOCK_SIZE_16B     0x00000010U
#define SDIO_DATABLOCK_SIZE_32B     0x00000020U
#define SDIO_DATABLOCK_SIZE_64B     0x00000040U
#define SDIO_DATABLOCK_SIZE_128B    0x00000080U
#define SDIO_DATABLOCK_SIZE_256B    0x00000100U
#define SDIO_DATABLOCK_SIZE_512B    0x00000200U
#define SDIO_DATABLOCK_SIZE_1024B   0x00000400U
#define SDIO_DATABLOCK_SIZE_2048B   0x00000800U
#define SDIO_DATABLOCK_SIZE_4096B   0x00001000U
#define SDIO_DATABLOCK_SIZE_8192B   0x00002000U
#define SDIO_DATABLOCK_SIZE_16384B  0x00004000U
#define SDIO_DATABLOCK_SIZE_32768B  0x00008000U
#define IS_SDIO_BLOCK_SIZE(SIZE) (((SIZE) == SDIO_DATABLOCK_SIZE_1B) ||  \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_2B) ||     \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_4B) ||     \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_8B) ||     \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_16B) ||    \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_32B) ||    \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_64B) ||    \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_128B) ||   \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_256B) ||   \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_512B) ||   \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_1024B) ||  \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_2048B) ||  \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_4096B) ||  \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_8192B) ||  \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_16384B) || \
                                  ((SIZE) == SDIO_DATABLOCK_SIZE_32768B))
/**
 * @}
 */


/** @defgroup SDIO_Transfer_Direction Transfer Direction
 * @{
 */
#define SDIO_TransferDir_ToCard   0x00000000U
#define SDIO_TransferDir_ToSDIO   0x00000001U
#define IS_SDIO_TRANSFER_DIR(DIR) (((DIR) == SDIO_TransferDir_ToCard) || \
                                   ((DIR) == SDIO_TransferDir_ToSDIO))
/**
 * @}
 */


/** @defgroup SDIO_Transfer_Type Transfer Type
 * @{
 */
#define SDIO_TransferMode_Block     0x00000000U
#define SDIO_TransferMode_Stream    SDIO_CMD_TRANSFER_MODE
#define IS_SDIO_TRANSFER_MODE(MODE) (((MODE) == SDIO_TransferMode_Block) || \
                                     ((MODE) == SDIO_TransferMode_Stream))
/**
 * @}
 */


/** @defgroup SDIO_DPSM_State DPSM State
 * @{
 */
#define SDIO_DPSM_Disable  0x00000000U
#define SDIO_DPSM_Enable   0x00000001U
#define IS_SDIO_DPSM(DPSM) (((DPSM) == SDIO_DPSM_Disable) || \
                            ((DPSM) == SDIO_DPSM_Enable))
/**
 * @}
 */


/** @defgroup SDIO_Interrupt_sources Interrupt Sources
 * @{
 */
#define SDIO_IT_CCRCFAIL    SDIO_RINTSTS_RESPCRCERR     // SDIO_STA_CCRCFAIL
#define SDIO_IT_DCRCFAIL    SDIO_RINTSTS_DATACRCERR     // SDIO_STA_DCRCFAIL
#define SDIO_IT_CTIMEOUT    SDIO_RINTSTS_RESPTIMEOUT    // SDIO_STA_CTIMEOUT
#define SDIO_IT_DTIMEOUT    SDIO_RINTSTS_DATATIMEOUT    // SDIO_STA_DTIMEOUT
#define SDIO_IT_TXUNDERR    SDIO_RINTSTS_UNDER_OVER_RUN // SDIO_STA_TXUNDERR
#define SDIO_IT_RXOVERR     SDIO_RINTSTS_UNDER_OVER_RUN // SDIO_STA_RXOVERR
#define SDIO_IT_CMDREND     SDIO_RINTSTS_CMDDONE        // SDIO_STA_CMDREND
#define SDIO_IT_CMDSENT     SDIO_RINTSTS_CMDDONE        // SDIO_STA_CMDSENT
#define SDIO_IT_DATAEND     SDIO_RINTSTS_DTRANSFEROVER  // SDIO_STA_DATAEND
#define SDIO_IT_STBITERR    SDIO_RINTSTS_STARTERROR     // SDIO_STA_STBITERR
#define SDIO_IT_DBCKEND     SDIO_RINTSTS_DTRANSFEROVER  // SDIO_STA_DBCKEND
#define SDIO_IT_TXFIFOF     SDIO_RINTSTS_UNDER_OVER_RUN // SDIO_STA_TXFIFOF
#define SDIO_IT_RXFIFOF     SDIO_RINTSTS_UNDER_OVER_RUN // SDIO_STA_RXFIFOF
#define SDIO_IT_TXDAVL      SDIO_RINTSTS_TXDATAREQ      // SDIO_STA_TXDAVL
#define SDIO_IT_RXDAVL      SDIO_RINTSTS_RXDATAREQ      // SDIO_STA_RXDAVL
#define SDIO_IT_SDIOIT      SDIO_RINTSTS_SDIOIT         // SDIO_STA_SDIOIT
#define SDIO_IT_ALL         0xFFFFFFFF                  // SDIO_STA_ALL
#define IS_SDIO_IT(IT) (((IT) == SDIO_IT_DCRCFAIL) || \
                        ((IT) == SDIO_IT_CTIMEOUT) || \
                        ((IT) == SDIO_IT_DTIMEOUT) || \
                        ((IT) == SDIO_IT_TXUNDERR) || \
                        ((IT) == SDIO_IT_RXOVERR) || \
                        ((IT) == SDIO_IT_CMDREND ) || \
                        ((IT) == SDIO_IT_CMDSENT ) || \
                        ((IT) == SDIO_IT_DATAEND ) || \
                        ((IT) == SDIO_IT_STBITERR) || \
                        ((IT) == SDIO_IT_DBCKEND ) || \
                        ((IT) == SDIO_IT_TXFIFOF ) || \
                        ((IT) == SDIO_IT_RXFIFOF ) || \
                        ((IT) == SDIO_IT_TXDAVL  ) || \
                        ((IT) == SDIO_IT_RXDAVL  ) || \
                        ((IT) == SDIO_IT_SDIOIT))
/**
 * @}
 */


/** @defgroup SDIO_Flags Flags
 * @{
 */
#define SDIO_FLAG_CCRCFAIL  SDIO_RINTSTS_RESPCRCERR     // SDIO_STA_CCRCFAIL
#define SDIO_FLAG_DCRCFAIL  SDIO_RINTSTS_DATACRCERR     // SDIO_STA_DCRCFAIL
#define SDIO_FLAG_CTIMEOUT  SDIO_RINTSTS_RESPTIMEOUT    // SDIO_STA_CTIMEOUT
#define SDIO_FLAG_DTIMEOUT  SDIO_RINTSTS_DATATIMEOUT    // SDIO_STA_DTIMEOUT
#define SDIO_FLAG_TXUNDERR  SDIO_RINTSTS_UNDER_OVER_RUN // SDIO_STA_TXUNDERR
#define SDIO_FLAG_RXOVERR   SDIO_RINTSTS_UNDER_OVER_RUN // SDIO_STA_RXOVERR
#define SDIO_FLAG_CMDREND   SDIO_RINTSTS_CMDDONE        // SDIO_STA_CMDREND
#define SDIO_FLAG_CMDSENT   SDIO_RINTSTS_CMDDONE        // SDIO_STA_CMDSENT
#define SDIO_FLAG_DATAEND   SDIO_RINTSTS_DTRANSFEROVER  // SDIO_STA_DATAEND
#define SDIO_FLAG_STBITERR  SDIO_RINTSTS_STARTERROR     // SDIO_STA_STBITERR
#define SDIO_FLAG_DBCKEND   SDIO_RINTSTS_DTRANSFEROVER  // SDIO_STA_DBCKEND
#define SDIO_FLAG_BUSYCLEAR SDIO_RINTSTS_BUSYCLEAR
#define SDIO_FLAG_TXFIFOF   SDIO_RINTSTS_UNDER_OVER_RUN // SDIO_STA_TXFIFOF
#define SDIO_FLAG_RXFIFOF   SDIO_RINTSTS_UNDER_OVER_RUN // SDIO_STA_RXFIFOF
#define SDIO_FLAG_TXDAVL    SDIO_RINTSTS_TXDATAREQ      // SDIO_STA_TXDAVL
#define SDIO_FLAG_RXDAVL    SDIO_RINTSTS_RXDATAREQ      // SDIO_STA_RXDAVL
#define SDIO_FLAG_SDIOIT    SDIO_RINTSTS_SDIOIT         // SDIO_STA_SDIOIT
#define SDIO_FLAG_ALL       0xFFFFFFFF
#define SDIO_STATIC_FLAGS   ((uint32_t)(SDIO_FLAG_TXDAVL |SDIO_FLAG_RXDAVL | SDIO_FLAG_CCRCFAIL | SDIO_FLAG_DCRCFAIL | \
                                        SDIO_FLAG_CTIMEOUT | SDIO_FLAG_DTIMEOUT | SDIO_FLAG_TXUNDERR | SDIO_FLAG_RXOVERR | \
                                        SDIO_FLAG_CMDREND | SDIO_FLAG_CMDSENT | SDIO_FLAG_DATAEND | SDIO_FLAG_DBCKEND))
/**
 * @}
 */


/** @defgroup SDIO_CMD
 * @{
 */
#define SDIO_CMD_CLK_UPDATE       0xA0202000
#define SDIO_CMD_COMPLATE_FLAG    0x80000000
/**
 * @}
 */


/** @defgroup SDIO_CTRL
 * @{
 */
#define SDIO_CTRL_RESET           0x00000007
/**
 * @}
 */


/** @defgroup SDIO_PowerState
 * @{
 */
#define SDIO_PowerState_OFF           0x01
#define SDIO_PowerState_ON            0x02
/**
 * @}
 */

/**
 * @}
 */

/* Exported macro ------------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/

void SDIO_DeInit(SDIO_TypeDef *SDIOx);
void SDIO_Init(SDIO_TypeDef *SDIOx, SDIO_InitTypeDef *SDIO_InitStruct);

void SDIO_ClockUpdate(SDIO_TypeDef *SDIOx);
void SDIO_ControlReset(SDIO_TypeDef *SDIOx);
void SDIO_PowerON(SDIO_TypeDef *SDIOx);
void SDIO_PowerOFF(SDIO_TypeDef *SDIOx);
uint32_t SDIO_GetPower(SDIO_TypeDef *SDIOx);
void SDIO_ClockCmd(SDIO_TypeDef *SDIOx, FunctionalState NewState);

uint8_t SDIO_GetCommandResponse(SDIO_TypeDef *SDIOx);
uint32_t SDIO_GetResponse(SDIO_TypeDef *SDIOx, uint32_t Response);
void SDIO_ConfigData(SDIO_TypeDef *SDIOx, SDIO_DataInitTypeDef *Data);
void SDIO_SendCommand(SDIO_TypeDef *SDIOx, SDIO_CmdInitTypeDef *Command);

uint32_t SDIO_ReadFIFO(SDIO_TypeDef *SDIOx);
void SDIO_WriteFIFO(SDIO_TypeDef *SDIOx, uint32_t *pWriteData);

FlagStatus SDIO_GetFlagStatus(SDIO_TypeDef *SDIOx, uint32_t SDIO_FLAG);
void SDIO_ClearFlagStatus(SDIO_TypeDef *SDIOx, uint32_t SDIO_FLAG);
ITStatus SDIO_GetITStatus(SDIO_TypeDef *SDIOx, uint32_t SDIO_IT);
void SDIO_ClearITPendingBit(SDIO_TypeDef *SDIOx, uint32_t SDIO_IT);

void SDIO_ConfigBusWidth(SDIO_TypeDef *SDIOx, uint32_t BusWidth);
void SDIO_DMACmd(SDIO_TypeDef *SDIOx, FunctionalState NewState);
void SDIO_ITConfig(SDIO_TypeDef *SDIOx, uint32_t SDIO_IT, FunctionalState NewState);

/**
 * @}
 */

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* __MG32F157_SDIO_H */
