/*
  EEPROM.h -originally written by Ivan Grokhotkov as part of the MG32x02z 
            core for Arduino environment.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "EEPROM.h"

#if defined(MG32F02U128)
#include "MG32x02z_MEM_MID.h"

#define  IAP_FLASH_SIZE          ((uint32_t)EEPROM_SIZE)
#define  IAP_FLASH_ADDR_BASE     ((uint32_t)0x1A000000)                      // оƬڲFLASHַ(޸ģ
#define  IAP_FLASH_SECTOR_SIZE   ((uint32_t)512)


uint8_t  IAP_FlashInit() 
{
    MID_MEM_ConfigIAPSize(IAP_FLASH_SIZE);
    return 0;
}

uint8_t  IAP_FlashErase(uint32_t PageAddress) 
{
    if( ( PageAddress < IAP_FLASH_ADDR_BASE )
     || ( PageAddress >= (IAP_FLASH_ADDR_BASE + IAP_FLASH_SIZE) ) )
    {
        return -1;
    }

    MID_MEM_FlashPageErase( (PageAddress & 0xFFFFFE00) );
    return 0;
}
uint8_t  IAP_FlashRead (uint32_t readAddr, void *ReadToBuffer, uint16_t numToRead) 
{     
    // оƬFLASHС
    // ȡоƬFLASHС;ĴֵΪоƬǰдFLASHСֻλByte
    //uint16_t flashSize = (IAP_FLASH_SIZE);

    __IO uint8_t*  pbuffer = (__IO uint8_t*)ReadToBuffer;

    // жϵַЧ                
    if(readAddr < IAP_FLASH_ADDR_BASE) 
    {
        // ĵַСFLASHСַ˳
        return 1; 
    }

    if(readAddr > (IAP_FLASH_ADDR_BASE+IAP_FLASH_SIZE))
    {
        // ĵַFLASHַ˳
        return 2;
    }
              
    // ʼ                
    while(numToRead--)
    {
        *pbuffer = *((__IO uint8_t*)readAddr);
        pbuffer++;        // ָһݳ
        readAddr++;       // ƫһλ
    }      
    
    return 0;             // ɹ0;    
}

uint8_t IAP_FlashWrite(uint32_t writeAddr, void *writeToBuffer, uint16_t numToWrite)               
{
    // оƬFLASHС
    // ȡоƬFLASHС;ĴֵΪоƬǰдFLASHСֻλByte

    int i;
    uint16_t     OutNumToWrite;
    __IO uint32_t *pBuffer;
    __IO uint8_t *pU8Buffer;
    __IO uint8_t   DataTemp[4] = {0};

    if(( numToWrite == 0 ) || (NULL == writeToBuffer) )
    {
        return -1;
    }

    if( ( ((uint32_t)writeToBuffer) & 0x00000003) != 0 )
    {
        return -1;
    }

    pBuffer = (uint32_t *)( ((uint32_t)writeToBuffer) & 0xFFFFFFFC );
                
    // жϵַЧ                
    if(writeAddr < IAP_FLASH_ADDR_BASE) 
    {
        // ĵַСFLASHСַ˳
        return 1; 
    }

    if( (writeAddr+numToWrite) > (IAP_FLASH_ADDR_BASE+IAP_FLASH_SIZE))
    {
        // ĵַFLASHַ˳
        return 2;
    }
    
    // ʼд ⲿ
    OutNumToWrite = numToWrite%4;
    numToWrite = numToWrite/4;
    while(  numToWrite-- )
    {
        MID_MEM_FlashProgram( writeAddr, *pBuffer );
        writeAddr = writeAddr+4;
        pBuffer++;
    }

    if(  OutNumToWrite > 0 )
    {
        pU8Buffer = (__IO uint8_t *)pBuffer;
        for(i=0; i<OutNumToWrite; i++)
        {
            DataTemp[i] = pU8Buffer[i];
        }
        MID_MEM_FlashProgram( writeAddr,  *((uint32_t*)(&DataTemp[0])) );
    }
    return 0;
}


#endif



EEPROMClass::EEPROMClass(void)
{
    _sector = 0;
    _restored = false;
    _size  = EEPROM_SIZE;
    memset( _data, 0, sizeof(_data) );
}

EEPROMClass::EEPROMClass(uint32_t sector)
{
    _sector = sector;
    _restored = false;
    _size  = EEPROM_SIZE;
    memset( _data, 0, sizeof(_data) );
}

EEPROMClass::~EEPROMClass() 
{
}

bool EEPROMClass::fromNVS()
{
    IAP_FlashRead( (IAP_FLASH_ADDR_BASE+IAP_FLASH_SECTOR_SIZE*_sector), &_data[0],  _size ) ;
    _restored = true;
    return true;
}

bool EEPROMClass::toNVS() 
{
    unsigned long iLen = 0;
    do
    {
        IAP_FlashErase( ((IAP_FLASH_ADDR_BASE+IAP_FLASH_SECTOR_SIZE*_sector) + iLen) ) ;
        iLen = iLen+512;
    }while( iLen < _size );

    IAP_FlashWrite( (IAP_FLASH_ADDR_BASE+IAP_FLASH_SECTOR_SIZE*_sector), (uint8_t *)&_data[0], _size );

    _restored = true;
    return true;
}

bool EEPROMClass::wasRestored() 
{
    return _restored;
}

bool EEPROMClass::begin(size_t size)
{
    if( (!size) 
     || (size > EEPROM_SIZE) )
    {
        return false;
    }
    _size = size;

    IAP_FlashInit();

 
    fromNVS();
    return true;
}

void EEPROMClass::end()
{
    commit();
}

uint8_t EEPROMClass::read(int address) 
{
    if (address < 0 || (size_t)address >= _size) 
    {
        return 0;
    }
    return _data[address];
}

void EEPROMClass::write(int address, uint8_t value) 
{
    if (address < 0 || (size_t)address >= _size)
    {
        return;
    }
    _data[address] = value;
    
    _restored = false;
}

bool EEPROMClass::commit() 
{
    toNVS();
    return true;
}

uint8_t * EEPROMClass::getDataPtr()
{
    return &_data[0];
}

/*
   Get EEPROM total size in byte defined by the user
*/
uint16_t EEPROMClass::length () 
{
    return _size;
}

/*
   Read 'value' from 'address'
*/
uint8_t EEPROMClass::readByte (int address)
{
    uint8_t value = 0;
    return EEPROMClass::readAll (address, value);
}

int8_t EEPROMClass::readChar (int address)
{
    int8_t value = 0;
    return EEPROMClass::readAll (address, value);
}

uint8_t EEPROMClass::readUChar (int address)
{
    uint8_t value = 0;
    return EEPROMClass::readAll (address, value);
}

int16_t EEPROMClass::readShort (int address)
{
    int16_t value = 0;
    return EEPROMClass::readAll (address, value);
}

uint16_t EEPROMClass::readUShort (int address)
{
    uint16_t value = 0;
    return EEPROMClass::readAll (address, value);
}

int32_t EEPROMClass::readInt (int address)
{
  int32_t value = 0;
  return EEPROMClass::readAll (address, value);
}

uint32_t EEPROMClass::readUInt (int address)
{
  uint32_t value = 0;
  return EEPROMClass::readAll (address, value);
}

int32_t EEPROMClass::readLong (int address)
{
  int32_t value = 0;
  return EEPROMClass::readAll (address, value);
}

uint32_t EEPROMClass::readULong (int address)
{
  uint32_t value = 0;
  return EEPROMClass::readAll (address, value);
}

int64_t EEPROMClass::readLong64 (int address)
{
  int64_t value = 0;
  return EEPROMClass::readAll (address, value);
}

uint64_t EEPROMClass::readULong64 (int address)
{
  uint64_t value = 0;
  return EEPROMClass::readAll (address, value);
}

float_t EEPROMClass::readFloat (int address)
{
    float_t value = 0;
    return EEPROMClass::readAll (address, value);
}

double_t EEPROMClass::readDouble (int address)
{
    double_t value = 0;
    return EEPROMClass::readAll (address, value);
}

bool EEPROMClass::readBool (int address)
{
    int8_t value = 0;
    return EEPROMClass::readAll (address, value) ? 1 : 0;
}

size_t EEPROMClass::readString (int address, char* value, size_t maxLen)
{
    if (!value)
    {
        return 0;
    }
    if (address < 0 || address + maxLen > _size)
    {
        return 0;
    }

    uint16_t len;
    for (len = 0; len <= _size; len++)
    {
        if (_data[address + len] == 0)
        {
            break;   
        }
    }

    if( ((size_t)address + (size_t)len) > _size)
    {
        return 0;
    }

    if (len > maxLen)
    {
        return 0; //Maybe return part of the string instead?
    }
    memcpy((uint8_t*) value, _data + address, len);
    value[len] = 0;
    return len;
}

String EEPROMClass::readString (int address)
{
    if( (address < 0) 
     || ((size_t)address > _size) )
    {
        return String();
    }
    
    uint16_t len;
    for (len = 0; len <= _size; len++)
    {
        if (_data[address + len] == 0)
        {
            break;
        }
    }
  
    if( ((size_t)address + (size_t)len) > _size)
    {
        return String();
    }
    
    char value[len+1];
    memcpy((uint8_t*) value, _data + address, len);
    value[len] = 0;
    return String(value);
}

size_t EEPROMClass::readBytes (int address, void* value, size_t maxLen)
{
    if( (!value) 
     || (!maxLen ))
    {
        return 0;
    }
    if( (address < 0)
     || (((size_t)address + (size_t)maxLen) > _size) )
    {
        return 0;
    } 

    memcpy((void*) value, _data + address, maxLen);
    return maxLen;
}

template <class T> T EEPROMClass::readAll (int address, T &value)
{
    if( (address < 0)
     || (((size_t)address + (size_t)sizeof(T)) > _size) )
    {
        return value;
    } 

    memcpy((uint8_t*) &value, _data + address, sizeof(T));
    return value;
}

/*
   Write 'value' to 'address'
*/
size_t EEPROMClass::writeByte (int address, uint8_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeChar (int address, int8_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeUChar (int address, uint8_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeShort (int address, int16_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeUShort (int address, uint16_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeInt (int address, int32_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeUInt (int address, uint32_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeLong (int address, int32_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeULong (int address, uint32_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeLong64 (int address, int64_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeULong64 (int address, uint64_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeFloat (int address, float_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeDouble (int address, double_t value)
{
    return EEPROMClass::writeAll (address, value);
}

size_t EEPROMClass::writeBool (int address, bool value)
{
    int8_t Bool;
    value ? Bool = 1 : Bool = 0;
    return EEPROMClass::writeAll (address, Bool);
}

size_t EEPROMClass::writeString (int address, const char* value)
{
    if (!value)
    {
        return 0;
    }
    
    if( (address < 0)
     || ((size_t)address > _size) )
    {
        return 0;
    }
    
    uint16_t len;
    for (len = 0; len <= _size; len++)
    {
        if (value[len] == 0)
        {
            break;
        }
    }

    if( ((size_t)address + (size_t)len) > _size)
    {
        return 0;
    }

    memcpy(_data + address, (const uint8_t*) value, len + 1);
    _restored = false;
    return strlen(value);
}

size_t EEPROMClass::writeString (int address, String value)
{
    return EEPROMClass::writeString (address, value.c_str());
}

size_t EEPROMClass::writeBytes (int address, const void* value, size_t len)
{
    if( (!value)
     || (!len) )
    {
        return 0;
    }
    
    if( (address < 0) 
     || (((size_t)address + (size_t)len) > _size) )
    {
        return 0;
    }

    memcpy(_data + address, (const void*) value, len);
    _restored = false;
    return len;
}

template <class T> T EEPROMClass::writeAll (int address, const T &value)
{
    if( (address < 0) 
    || (((size_t)address + (size_t)sizeof(T)) > _size) )
    {
        return value;
    }
    memcpy(_data + address, (const uint8_t*) &value, sizeof(T));
    _restored = false;
    return sizeof (value);
}

//#if !defined(NO_GLOBAL_INSTANCES) && !defined(NO_GLOBAL_EEPROM)
EEPROMClass EEPROM;
//#endif
