/*
 *  Interrupt and PWM utilities for 16 bit Timer1 on MG32x02z
 *  Original code by Cuijian May 2024
 *
 *  This is free software. You can redistribute it and/or modify it under
 *  the terms of Creative Commons Attribution 3.0 United States License. 
 *  To view a copy of this license, visit http://creativecommons.org/licenses/by/3.0/us/ 
 *  or send a letter to Creative Commons, 171 Second Street, Suite 300, San Francisco, California, 94105, USA.
 *
 */

#ifndef TimerOne_h_
#define TimerOne_h_


#include "Arduino.h"

#include "config/known_16bit_timers.h"

#include "timer.h"

//#define TIMER1_RESOLUTION 256UL  // Timer1 is 8 bit
#define TIMER1_RESOLUTION 65536UL  // Timer1 is 16 bit


#define TIMER1_PWM_MAX_DUTY     1024


// Placing nearly all the code in this .h file allows the functions to be
// inlined by the compiler.  In the very common case with constant values
// the compiler will perform all calculations and simply write constants
// to the hardware registers (for example, setPeriod).


class TimerOne
{
#if defined(MG32F02U128) 
public:
    //****************************
    //  Configuration
    //****************************
    void initialize(unsigned long microseconds=1000000);
    void setPeriod(unsigned long microseconds);

    //****************************
    //  Run Control
    //****************************	
    void start() __attribute__((always_inline)) { TimerHandleRest(USED_TIMERONE_ID); resume(); }
    void stop() __attribute__((always_inline)) { TimerHandleStop(USED_TIMERONE_ID); }
    void restart() __attribute__((always_inline)) {	start();    }
    void resume() __attribute__((always_inline)){	TimerHandleStart(USED_TIMERONE_ID);    }


    //****************************
    //  PWM outputs
    //****************************
    void setPwmDuty(char pin, unsigned int duty);
    //Duty:0~1023
    void pwm(char pin, unsigned int duty);
    void pwm(char pin, unsigned int duty, unsigned long microseconds) __attribute__((always_inline)) 
    {
        if (microseconds > 0)
        {
            setPeriod(microseconds);
        }
        pwm(pin, duty);
    }

    void disablePwm(char pin);

    //****************************
    //  Interrupt Function
    //****************************
    void attachInterrupt(void (*isr)())__attribute__((always_inline)) { m_isrCallback = isr;  }
    void attachInterrupt(void (*isr)(), unsigned long microseconds) __attribute__((always_inline))
    {
    	if(microseconds > 0) 
    	{
    	    setPeriod(microseconds);
    	}
    	attachInterrupt(isr);
    }
    void detachInterrupt()__attribute__((always_inline)) {m_isrCallback = TimerOne::isrDefaultUnused;}


    static void (*m_isrCallback)();
    static void isrDefaultUnused();

private:
    // properties
    uint32_t  m_microseconds = 1000;
    uint16_t  m_duty = 0;
    uint16_t  m_pin = 0xFFFF;
#endif
};

extern TimerOne Timer1;

#endif

