;******************************************************************************
; main.a51
; Demo code for using MPC82G516 as a simple Flash memory (with 64K bytes)
;******************************************************************************

;select Master MCU to access the 64K Flash memory implemented by MPC82G516
;
MPC89_series  EQU  0  ;1=select, 0=not_select
MPC82G516     EQU  1  ;1=select, 0=not_select

;* Common knowledge about the Flash memory operation:
;
;  (1) The Flash memory can only perform page erasing, not byte erasing.
;
;  (2) Before a byte can be written into some address of the Flash memory,
;      the page containing that address should have been erased previously,
;      otherwise the writing will fail.
;
;  (3) To modify only one byte of a page, the user should read out whole the
;      page contents and store in a buffer (may be XRAM or another Flash page).
;      Then, erase that page, and re-write the modified buffer data to that
;      page.
;
;  (4) For Megawin's Flash-based 8051 MCU, the page size is 512 bytes.
;      That also means the start address of a page is located at:
;      0000h, 0200h, 0400h, 0600h, .., F800h, FA00h, FC00h, FE00h.
;      The address increment is #0200h, which is equal to 512.

IF MPC89_series ;for 89-series SFR definition
$INCLUDE (REG_MPC89L51-515.inc)
ENDIF

IF MPC82G516    ;for MPC82G516 SFR definition
$INCLUDE (REG_MPC82G516.inc)
ENDIF

$INCLUDE (G516Flash.inc) ;for the 64K Flash memory implemented by MPC82G516

$INCLUDE (Debug.inc)     ;for the GNU Debugger

;==============================================================================
data_main SEGMENT DATA
         RSEG   data_main

Flash_addrH:    DS  1
Flash_addrL:    DS  1
Flash_data:     DS  1

stack_space:    DS  32

;==============================================================================

         CSEG   AT  0000h
         JMP    start

;==============================================================================
code_main SEGMENT CODE
         RSEG   code_main
         USING  0

;?-----------------------------------------------------------------------------
start:
         MOV    SP,#stack_space-1

;--------------------------------------------
; enable MPC82G516's Flash,
;   suppose the MCU speed is 12MHz
;--------------------------------------------

         G516Flash_enable #12   ;'12' means the master MCU speed

;-----------------------------------------------------------
; erase the page to which you will write data,
;   suppose the page used to store data is located at 0200h
;-----------------------------------------------------------

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#00h
         G516Flash_page_erase Flash_addrH,Flash_addrL
        ;or:
        ;G516Flash_page_erase #02h,#00h

;------------------------------------------------------------
; write data to MPC82G516's Flash
;   suppose write data to addresses: 0200h,0201h,0202h,0203h
;------------------------------------------------------------

     ;write #5Ah to address 0200h

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#00h
         MOV    Flash_data,#5Ah
         G516Flash_byte_program Flash_addrH,Flash_addrL,Flash_data
        ;or:
        ;G516Flash_byte_program #02h,#00h,#5Ah

     ;write #A5h to address 0201h

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#01h
         MOV    Flash_data,#0A5h
         G516Flash_byte_program Flash_addrH,Flash_addrL,Flash_data
        ;or:
        ;G516Flash_byte_program #02h,#01h,#0A5h

     ;write #55h to address 0202h

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#02h
         MOV    Flash_data,#55h
         G516Flash_byte_program Flash_addrH,Flash_addrL,Flash_data
        ;or:
        ;G516Flash_byte_program #02h,#02h,#55h

     ;write #AAh to address 0203h

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#03h
         MOV    Flash_data,#0AAh
         G516Flash_byte_program Flash_addrH,Flash_addrL,Flash_data
        ;or:
        ;G516Flash_byte_program #02h,#03h,#0AAh

;--------------------------------------------------------
; read data from MPC82G516's Flash
;   suppose data from addresses: 0200h,0201h,0202h,0203h
;--------------------------------------------------------

     ;read data from address 0200h

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#00h
         G516Flash_byte_read Flash_addrH,Flash_addrL,Flash_data
        ;or:
        ;G516Flash_byte_read #02h,#00h,Flash_data
         Debug_show_msg_to_LCD Flash_data  ;show data on the GNU Debugger

     ;read data from address 0201h

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#01h
         G516Flash_byte_read Flash_addrH,Flash_addrL,Flash_data
        ;or:
        ;G516Flash_byte_read #02h,#01h,Flash_data
         Debug_show_msg_to_LCD Flash_data  ;show data on the GNU Debugger

     ;read data from address 0202h

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#02h
         G516Flash_byte_read Flash_addrH,Flash_addrL,Flash_data
        ;or:
        ;G516Flash_byte_read #02h,#02h,Flash_data
         Debug_show_msg_to_LCD Flash_data  ;show data on the GNU Debugger

     ;read data from address 0203h

         MOV    Flash_addrH,#02h
         MOV    Flash_addrL,#03h
         G516Flash_byte_read Flash_addrH,Flash_addrL,Flash_data
        ;or:
        ;G516Flash_byte_read #02h,#03h,Flash_data
         Debug_show_msg_to_LCD Flash_data  ;show data on the GNU Debugger

;------------------------------------------------
; write data to MPC82G516's Flash
;   suppose write data to addresses: 0204h~03FFh
;------------------------------------------------

         MOV    DPTR,#0204h
  write_loop:
         MOV    Flash_data,DPL
         G516Flash_byte_program DPH,DPL,Flash_data
         INC    DPTR
         MOV    A,DPH
         CJNE   A,#04h,write_loop

;-------------------------------------------------
; read data from MPC82G516's Flash
;   suppose read data from addresses: 0204h~03FFh
;-------------------------------------------------

         MOV    DPTR,#0204h
  read_loop:
         G516Flash_byte_read DPH,DPL,Flash_data
         Debug_show_msg_to_LCD Flash_data  ;show data on the GNU Debugger
         MOV    A,Flash_data    ;verify Flash data
         CJNE   A,DPL,error     ;
         INC    DPTR
         MOV    A,DPH
         CJNE   A,#04h,read_loop

;------------------------------------------------------------------------------
         G516Flash_disable
         JMP    $

  error:
         JMP    $

;==============================================================================

         END

