/**
  ******************************************************************************
 *
 * @file        Sample_MID_TM36_IC1_Rising2Rising.c
 *
 * @brief       TM36 Input Capture channel-1 with trigger edge condition.
 *
 * @par         Project
 *              MG32
 * @version     V1.13
 * @date        2021/05/27
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
* @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_TM_MID.h"

/* Wizard menu ---------------------------------------------------------------*/
//*** <<< Use Configuration Wizard in Context Menu >>> ***

//  <h> simple function (TM36 IC1 from PD8 - for MG32F02A032) 
// 			<o0> IC1 capture type <1=> Rising to Rising edge <2=>Rising to Falling edge <3=> Falling to Falling edge <4=> Falling to Rising edge
//  </h>
#define TM36_IC1_TriggerSrc       1

//*** <<< end of configuration section >>>    ***

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static TM_HandleTypeDef    mTM36;

/* Private function prototypes -----------------------------------------------*/
void Sample_MID_TM36_IC1_Rising2Rising(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	    TM36 input capture channel1 with trigger edge condition.
 * @details     1.Initial TM36 for 32-bit of IC.
 *    \n        2.Clock initial.
 *    \n        3.Config TM slave mode for IC1.
 *    \n        4.Setup input capture parameters.
 *    \n        5.Start IC capture.
 * @return      None
 * @code
    uint32_t CaptureData;
    
    // initial IC1 (PD8) 
    // to do ...
    
    // config TM36 for IC1-32 bit mode (Keep capture data)
    // (Width) 'Rising to Rising edge' or 'Rising to Falling edge'
    // or 'Falling to Falling edge' or 'Falling to Rising edge'
    Sample_MID_TM36_IC1_Rising2Rising();
    
    // wait for Capture event (Only accept one capture event)
    while(__DRV_TM_GET_FLAG(&mTM36, TM_FLAG_CC1A) == 0);
    
    // Get IC0 capture data (32-bits)
    CaptureData = (__DRV_TM_GET_COMPARE(&mTM36,MID_TM_Channel1) << 16) | __DRV_TM_GET_COMPARE_B(&mTM36,MID_TM_Channel1);

    // clear TM36 flag
    __DRV_TM_CLEAR_FLAG(&mTM36, TM_FLAG_CC1A);
 * @endcode
 *******************************************************************************
 */
void Sample_MID_TM36_IC1_Rising2Rising(void)
{  
    TM_IC_InitTypeDef       iConfig;
    TM_ClockConfigTypeDef   CKConfig;
    TM_SlaveConfigTypeDef   TMSlave;
    
    // make sure :
    
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12M
    //  Select CK_PLLI Divider = CK_HS/2
    //  Select CK_PLL Multiplication factor = CK_PLLIx16
    //  Select CK_PLLO Divider = CK_PLL/4
    //Select CK_MAIN Source = CK_PLLO
    //Configure PLL->Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->TM36 = Enable
    //Configure Peripheral On Mode Clock->PortD = Enable
	
    //==Set GPIO init 
    //MG32_GPIO_Init.h(Configuration Wizard)->Use GPIOC->PD8
    //GPIO port initial is 0xFFFF
    //PD8 mode is "DIN" with "PU : Pull-up resister"
    //PD8's "AFS : Alternate function select" is "TM36_IC1"
    
    // ------------------------------------------------------------------------
    // 1.Initial TM36 for 32-bit of IC.
    // ------------------------------------------------------------------------
    mTM36.Instance                  = TM36;
    mTM36.Init.TM_CounterMode       = TM_FULLCOUNTER_UP;
    mTM36.Init.TM_Period            = 65535;
    mTM36.Init.TM_Prescaler         = 0;
    MID_TM_IC_Init(&mTM36);
    
    // ------------------------------------------------------------------------
    // 2.Clock initial.
    // ------------------------------------------------------------------------
    CKConfig.TM_ClockSource         = TM_INTERNAL_CLOCK;
    CKConfig.TM_ExternalClockSource = 0;
    CKConfig.TM_INTClockDivision    = TM_INTERNALCLOCK_DIVDER_DIV1;
    CKConfig.TM_InternalClockSource = TM_INTERNALCLOCK_PROC;
    
    MID_TM_ConfigClockSource(&mTM36, &CKConfig);  
    
    // ------------------------------------------------------------------------
    // 3.Config TM slave mode for IC1 
    //     first trigger edge : to startup timer 
    // ------------------------------------------------------------------------
    #if (TM36_IC1_TriggerSrc == 1)      
        TMSlave.SlaveMode       = TM_SLAVEMODE_TRIGGER_RISINGEDGE;      // Capture range : Rising to Rising edge 
                                                                        //                 ^^^^^^
    #elif (TM36_IC1_TriggerSrc == 2)    // Rising to Falling edge 
        TMSlave.SlaveMode       = TM_SLAVEMODE_TRIGGER_RISINGEDGE;      // Capture range : Rising to Falling edge 
                                                                        //                 ^^^^^^
    #elif (TM36_IC1_TriggerSrc == 3)    // Falling to Falling edge 
        TMSlave.SlaveMode       = TM_SLAVEMODE_TRIGGER_FALLINGEDGE;     // Capture range : Falling to Falling edge 
                                                                        //                 ^^^^^^^
    #elif (TM36_IC1_TriggerSrc == 4)    // Falling to Rising edge
        TMSlave.SlaveMode       = TM_SLAVEMODE_TRIGGER_FALLINGEDGE;     // Capture range : Falling to Rising edge 
                                                                        //                 ^^^^^^^
    #endif   
    TMSlave.InputTrigger    = TM_TS_IN1;
    
    MID_TM_SlaveConfigSynchronization(&mTM36, &TMSlave);
    

    // ------------------------------------------------------------------------
    // 4.Setup input capture parameters:
    //      1. Trigger edge
    //      2. Trigger source from PIN
    //      3. Capture data mode
    // ------------------------------------------------------------------------
    #if (TM36_IC1_TriggerSrc == 1)      // Rising to Rising edge 
        iConfig.ICPolarity              = TM_ICPOLARITY_RISING;         // Capture range : Rising to Rising
                                                                        //                           ^^^^^^
    #elif (TM36_IC1_TriggerSrc == 2)    // Rising to Falling edge 
        iConfig.ICPolarity              = TM_ICPOLARITY_FALLING;        // Capture range : Rising to Falling
                                                                        //                           ^^^^^^^
    #elif (TM36_IC1_TriggerSrc == 3)    // Falling to Falling edge 
        iConfig.ICPolarity              = TM_ICPOLARITY_FALLING;        // Capture range : Falling to Falling
                                                                        //                           ^^^^^^^
    #elif (TM36_IC1_TriggerSrc == 4)    // Falling to Rising edge
        iConfig.ICPolarity              = TM_ICPOLARITY_RISING;         // Capture range : Falling to Rising
                                                                        //                           ^^^^^^
    #endif   
    iConfig.ICSelection             = MID_TM_INPUTMUX_PIN; 
    iConfig.ICDataMode              = MID_TM_IC_KEEPDATA;
    
    MID_TM_IC_ConfigChannel(&mTM36, &iConfig, MID_TM_Channel1);
    
    
    // ------------------------------------------------------------------------
    // 5.Start IC capture.
    // ------------------------------------------------------------------------
    MID_TM_IC_Start(&mTM36, MID_TM_Channel1);                           // polling flag
    
}


