/**
 ******************************************************************************
 *
 * @file        Sample_IWDT_GeneralTimer.c
 *
 * @brief       IWDT general timer sample code, trigger window event interrupt.
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2025/06/11
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer
 *      The Demo software is provided "AS IS"  without any warranty, either
 *      expressed or implied, including, but not limited to, the implied warranties
 *      of merchantability and fitness for a particular purpose.  The author will
 *      not be liable for any special, incidental, consequential or indirect
 *      damages due to loss of data or any other reason.
 *      These statements agree with the world wide and local dictated laws about
 *      authorship and violence against these laws.
 ******************************************************************************
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_DRV.h"
/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
uint32_t Sample_IWDT_GeneralTimer(void);
void SYS_IRQHandler(void);
void IWDT_IRQ(void);

/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief  	    Sample_IWDT_GeneralTimer
 * @details     1. Enable CSC to WWDT clock
 *      \n      2. IWDT Clock Configure
 *      \n      3. IWDT Interrupt Configure
 *      \n      4. IWDT Reload Counter Refresh
 *      \n      5. IWDT Function Command
 * @return	    None	
 * @note        
 * @par         Example
 * @code
    Sample_IWDT_GeneralTimer();
 * @endcode
 *******************************************************************************
 */
uint32_t Sample_IWDT_GeneralTimer(void)
{
    //===Set CSC init====
    //MG32_CSC_Init.h(Configuration Wizard)
    //Select CK_HS source = CK_IHRCO
    //Select IHRCO = 12Mz
    //Select CK_MAIN Source = CK_HS
    //Select CK_LS Source = CK_ILRCO
    //Configure Select APB Prescaler = CK_MAIN/1
    //Configure Peripheral On Mode Clock->WWDT = Enable

    /*=== 1. Enable CSC to IWDT clock ===*/
    UnProtectModuleReg(CSCprotect);                                 // Unprotect CSC module.
    CSC_PeriphOnModeClock_Config(CSC_ON_IWDT, ENABLE);              // Enable IWDT module clock.
    ProtectModuleReg(CSCprotect);                                   // protect CSC module.
    
    // IWDT Module Protect Disable.
    UnProtectModuleReg(IWDTprotect);                                // IWDT module protect disable.
    
    /*=== 2. IWDT Clock Configure ===*/
    IWDT_Divider_Select(IWDT_DIV_1);                                    // IWDT clock source divider select.
    
    /*=== 3. IWDT Interrupt Configure ===*/
    IWDT_IT_Config(IWDT_INT_EW1, ENABLE);                           // IWDT early wake up even enable.
    SYS_ITEA_Cmd(ENABLE);                                           // IWDT all interrupt even enable.
    NVIC_EnableIRQ(SYS_IRQn);                                       // M0 NVIC configure.
    
    /*=== 4. IWDT Reload Counter Refresh ===*/
    IWDT_RefreshCounter();                                          // IWDT counter value reset to 0xFF.
    
    /*=== 5. IWDT Function Command ===*/
    IWDT_Cmd(ENABLE);                                               // IWDT module function enable.
    
    // WWDT Module Protect Enable.
    ProtectModuleReg(IWDTprotect);                                  // IWDT module protct disable.
    
    return DRV_Success;
}

/**
 *******************************************************************************
 * @brief       SYS: An interrupt, or IRQ, is an exception signalled by a
 *              peripheral, or generated by a software request. 
 * @details     SYS_IRQ hava IWDT, PW, RTC, CSC, APB,MEM or EMB interrupt source.
 * @return      No
 *******************************************************************************
 */
void SYS_IRQHandler(void)
{
    IWDT_IRQ();                                                     // IWDT interrupt service routine.
}

/**
 *******************************************************************************
 * @brief       IWDT module interrupt service routine.
 * @details     Refresh IWDT counter & clear interrup flag.
 * @return      No
 *******************************************************************************
 */
void IWDT_IRQ(void)
{
    // WWDT Counter Reload.
    IWDT_RefreshCounter();                                          // IWDT counter value reset(0xFF).
    // Clear WWDT Flag
    IWDT_ClearFlag(IWDT_EW1F);                                      // IWDT interrupt flag clear.
}

