

/**
 ******************************************************************************
 *
 * @file        Sample_ADC_TSO_Convert.c
 * @brief       Please reference line 402 - main routine. 
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2024/12/04
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */



/* Includes ------------------------------------------------------------------*/
#include "MG32_DRV.h"
#include "MG32_ChipInit.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
    
/**
 * @name    ADC_Conversion_MaxValue
 *          ADC conversion max value. (The range is 0~4095).
 */ 
//!@{
#define ADC_Max_Value           (4095)
//!@}

/**
 * @name    IAP_Page_Info
 *          IAP Size and Page size information.
 */ 
//!@{
#define IAP_Size                10      /*!<  IAP size (unit: 0.5K) / if IAP_Size=10 then IAP size will be 5K. */
#define IAP_Page_Size           512     /*!<  IAP page size (512bytes) */
//!@}
    
/**
 * @name    ADC_Temperature_AddressDef
 *          Declare address that ADC TSO conversion data.
 */ 
//!@{
#define ADC_T20_TSOValue_Addr   (0x1A000000 + (IAP_Size * IAP_Page_Size) - 8)
#define ADC_T60_TSOValue_Addr   (0x1A000000 + (IAP_Size * IAP_Page_Size) - 4)
//!@}

/**
 * @name    Convert_Celsius_degreeInfo
 *          Convert Celsius degree information.
 */ 
//!@{
#define T60_T20_Gap             (60 - 20)
//!@}


/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
static int32_t T60_T20_ADCGap;
static int16_t T20_ADC_Value, T60_ADC_Value;

/* Private function prototypes -----------------------------------------------*/
/**
 * @name    Function announce
 * @brief   addtogroup ADC_Exported_Functions_Group
 */ 
///@{
void ADC_TSO_Init(void);
int16_t Sample_ADC_TSO_Convert(void);
void MEM_IAP_Arrange(void);
void ADC_T20_TSO_SampleOnce(void);
void ADC_T60_TSO_SampleOnce(void);
//!@}

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/



/**
 *******************************************************************************
 * @brief	    ADC initial
 * @return 	    None
 *******************************************************************************
 */
void ADC_TSO_Init(void)
{
    ADC_InitTypeDef ADC_Base;
    
    // --------------------------------------------------------------
    // initial ADC base condition
    // --------------------------------------------------------------
    ADC_BaseStructure_Init(&ADC_Base);
    {   // modify parameter
        ADC_Base.ADCMainClockSelect = ADC_CKADC;    
            ADC_Base.ADC_IntCK_Div = ADC_IntDIV2;   // for internal clock divider
    
        // ADC data alignment mode (Right or Left)
        ADC_Base.ADC_DataAlign = ADC_RightJustified;
        
        // ADC conversion resolution 8, 10 or 12 bit
        ADC_Base.ADC_ResolutionSel = ADC_12BitData;
        
        // ADC overwritten data or keep data
        ADC_Base.ADC_DataOverrunEvent = ADC_DataOverWritten;
        
    }
    ADC_Base_Init(ADC0, &ADC_Base);
    
    ADC_Cmd(ADC0, ENABLE);
    
    // ------------------------------------------------------------------------
    // Enable TSO and ADC select TSO channel.
    // ADC Reference select IVR24 or VREF pin.
    // ------------------------------------------------------------------------
    ADC_TSO_Cmd(ADC0, ENABLE);
  
    #if defined(ADC_ANA_IVREF_SEL_ivr24_w)
        // ADC reference select : IVR24
        ADC_IVR24_Cmd(ADC0, ENABLE);
        ADC_TSOVref_Select(ADC0, ADC_IVR24);
    #endif
    
    // ------------------------------------------------------------------------
    // One shot conversion mode
    // ------------------------------------------------------------------------
    ADC_ConversionMode_Select(ADC0, ADCMode);
    ADC_SetExtendSampling(ADC0, 255);			// User can modify this value (255).
    
    // ------------------------------------------------------------------------
    // PGA disable
    // ------------------------------------------------------------------------
    #if defined(ADC_ANA_PGA_EN_mask_w)
        ADC_PGA_Cmd(ADC0, DISABLE);
    #endif
    // --------------------------------------------------------------
    // ADC calibration
    // --------------------------------------------------------------
    ADC_StartCalibration(ADC0, ENABLE);         // ADC calibration  
            
    
}


/**
 *******************************************************************************
 * @brief	    ADC sample TSO data in 20 degree. Then save this data into IAP area.
 * @return 	    None.
 *******************************************************************************
 */
void ADC_T20_TSO_SampleOnce(void)
{
    uint32_t i, ADC_Accumulation;
    uint32_t *ptr32;
    
    // ------------------------------------------------------------------------
    // Is T20 area is empty?
    ptr32 = (uint32_t *) ADC_T20_TSOValue_Addr;
    if (*ptr32 != 0xFFFFFFFF)
        return;
    
    // ------------------------------------------------------------------------
    ADC_ConversionMode_Select(ADC0, ADCMode);
    ADC_ClearFlag(ADC0, ADC_E1CNVF);
    ADC_InternalChannel_Select(ADC0, ADC_INT_TSO);
    
    // ------------------------------------------------------------------------
    for (i=ADC_Accumulation=0; i<4; i++)
    {
        ADC_SoftwareConversion_Cmd(ADC0, ENABLE);
        while(ADC_GetSingleFlagStatus(ADC0, ADC_E1CNVF) == DRV_UnHappened);
        ADC_ClearFlag(ADC0, ADC_E1CNVF);
        
        ADC_Accumulation += (uint32_t) ADC_GetDAT0Data(ADC0);
    }
    
    // Get average.
    ADC_Accumulation = ADC_Accumulation >> 2;
    
    // ------------------------------------------------------------------------
    UnProtectModuleReg(MEMprotect);
    
    // Config MEM into program state. 
    __MEM_Access_Enable(MEM_ACCESS_IAP_WRITE);
    __MEM_SetWriteMode(IAPProgram);
    
    // Write T20 TSO ADC conversion data into ADC_T20_TSOValue_Addr of IAP space.
    *ptr32 = ADC_Accumulation;

    // inactive MEM
    __MEM_SetWriteMode(None);
    
    ProtectModuleReg(MEMprotect);
    
    
    return;
}


/**
 *******************************************************************************
 * @brief	    ADC sample TSO data in 60 degree. Then save this data into IAP area.
 * @return 	    None.
 *******************************************************************************
 */
void ADC_T60_TSO_SampleOnce(void)
{
    uint32_t i, ADC_Accumulation;
    uint32_t *ptr32;
    
    // ------------------------------------------------------------------------
    // Is T60 area is empty?
    ptr32 = (uint32_t *) ADC_T60_TSOValue_Addr;
    if (*ptr32 != 0xFFFFFFFF)
        return;
    
    // ------------------------------------------------------------------------
    ADC_ConversionMode_Select(ADC0, ADCMode);
    ADC_ClearFlag(ADC0, ADC_E1CNVF);
    ADC_InternalChannel_Select(ADC0, ADC_INT_TSO);
    
    // ------------------------------------------------------------------------
    for (i=ADC_Accumulation=0; i<4; i++)
    {
        ADC_SoftwareConversion_Cmd(ADC0, ENABLE);
        while(ADC_GetSingleFlagStatus(ADC0, ADC_E1CNVF) == DRV_UnHappened);
        ADC_ClearFlag(ADC0, ADC_E1CNVF);
        
        ADC_Accumulation += (uint32_t) ADC_GetDAT0Data(ADC0);
    }
    
    // Get average.
    ADC_Accumulation = ADC_Accumulation >> 2;
    
    // ------------------------------------------------------------------------
    UnProtectModuleReg(MEMprotect);
    
    // Config MEM into program state. 
    __MEM_Access_Enable(MEM_ACCESS_IAP_WRITE);
    __MEM_SetWriteMode(IAPProgram);
    
    // Write T20 TSO ADC conversion data into ADC_T20_TSOValue_Addr of IAP space.
    ptr32 = (uint32_t *) ADC_T60_TSOValue_Addr;
    *ptr32 = ADC_Accumulation;

    // inactive MEM
    __MEM_SetWriteMode(None);
    
    ProtectModuleReg(MEMprotect);
        
    return;
}

/**
 *******************************************************************************
 * @brief	    Convert Celsius degree.
 * @return 	    Celsius degree (int16_t).
 *******************************************************************************
 */
int16_t Sample_ADC_TSO_Convert(void)
{
    int32_t     Celsius_degree;
    int16_t     TempValue;
    
    // ------------------------------------------------------------------------
    ADC_ConversionMode_Select(ADC0, ADCMode);
    ADC_ClearFlag(ADC0, ADC_E1CNVF);
    ADC_InternalChannel_Select(ADC0, ADC_INT_TSO);

    ADC_SoftwareConversion_Cmd(ADC0, ENABLE);
    while(ADC_GetSingleFlagStatus(ADC0, ADC_E1CNVF) == DRV_UnHappened);
    ADC_ClearFlag(ADC0, ADC_E1CNVF);
    
    // ------------------------------------------------------------------------
    TempValue = (int16_t) ADC_GetDAT0Data(ADC0) - T20_ADC_Value;
    
    // ------------------------------------------------------------------------
    Celsius_degree = (TempValue * 40) / (T60_T20_ADCGap);
    Celsius_degree += 20;
    
    return (int16_t) Celsius_degree;
}


/**
 *******************************************************************************
 * @brief	    MCU will set IAP size.
 * @return 	    None.
 *******************************************************************************
 */
void MEM_IAP_Arrange(void)
{
    uint32_t *ptr32;
    
    // ------------------------------------------------------------------------
    // Clear MEM-IAP Size Setting Error Flag
    MEM_ClearFlag(MEM_FLAG_IAPSEF);
    
    // ------------------------------------------------------------------------
    // Unproect Module - MEM
    MEM_SetIAPSize(IAP_Size * IAP_Page_Size);

    // ------------------------------------------------------------------------
    // Config MEM function.
    UnProtectModuleReg(MEMprotect);
    
    __MEM_Enable();
    __MEM_SetWriteMode(None);                   // None
    __MEM_MultipleWriteUnProtect();
    
    // Clear flags
    MEM->STA.B[0] = (MEM_STA_WPEF_mask_b0 | MEM_STA_EOPF_mask_b0);
    __ISB();

    // ------------------------------------------------------------------------
    // Check T20 TSO Value. If this code over ADC_Max_Value then clear this IAP page.
    // ------------------------------------------------------------------------
    ptr32 = (uint32_t *) ADC_T20_TSOValue_Addr;
    T20_ADC_Value = (int16_t) *ptr32;
    
    // Errror condition: 
    //  1. Over ADC TSO conversion range (0~4095). 
    //  2. T20_ADC_Value != 0xFFFFFFFF.
    if (((T20_ADC_Value > ADC_Max_Value) || (T20_ADC_Value == 0)) && ((uint32_t) T20_ADC_Value != 0xFFFFFFFF))
    {
        // Erase this page of IAP. (One page size is 512 byte)
        __MEM_Access_Enable(MEM_ACCESS_IAP_WRITE);
        __MEM_SetWriteMode(IAPErase);
        
        // Active erase the last page of IAP space.
        ptr32 = (uint32_t *) (0x1A000000 + ((IAP_Size - 1) * IAP_Page_Size));
        *ptr32 = 0xFFFFFFFF;
        
        // Protect MEM and exit.
        ProtectModuleReg(MEMprotect);
        return;
    }
    
    // ------------------------------------------------------------------------
    // Check T60 TSO Value. If this code over ADC_Max_Value then clear this IAP page.
    // ------------------------------------------------------------------------
    ptr32 = (uint32_t *) ADC_T60_TSOValue_Addr;
    T60_ADC_Value = (int16_t) *ptr32;
    
    // Errror condition: 
    //  1. Over ADC TSO conversion range (0~4095). 
    //  2. The T20_ADC_Value value biger than T60_ADC_Value.
    //  3. T60_ADC_Value != 0xFFFFFFFF.
    if (((T60_ADC_Value > ADC_Max_Value) || (T60_ADC_Value == 0) || (T20_ADC_Value > T60_ADC_Value)) && ((uint32_t) T60_ADC_Value != 0xFFFFFFFF))
    {
        // Erase this page of IAP. (One page size is 512 byte)
        __MEM_Access_Enable(MEM_ACCESS_IAP_WRITE);
        __MEM_SetWriteMode(IAPErase);
        
        // Active erase the last page of IAP space.
        ptr32 = (uint32_t *) (0x1A000000 + ((IAP_Size - 1) * IAP_Page_Size));
        *ptr32 = 0xFFFFFFFF;
    }
    
    // ------------------------------------------------------------------------
    // update T60_T20_ADCGap
    T60_T20_ADCGap = (int32_t) (T60_ADC_Value - T20_ADC_Value);
    
    // ------------------------------------------------------------------------
    // Protect MEM and exit.
    ProtectModuleReg(MEMprotect);
    return;
}



/**
 *******************************************************************************
 * @brief	    Sample main routine.
 * @return 	    None.
 *******************************************************************************
 */
//int main(void)
//{    
//    int16_t Temperature;
//    
//    // ------------------------------------------------------------------------
//    // chip initial (User can enable CSC, GPIO, TM, ADC, EXIC ... wizard)
//    // ------------------------------------------------------------------------
//    ChipInit();    

//    // ------------------------------------------------------------------------
//    // ADC initial (initial to sample INT_TSO : temperature sensor)
//    // ------------------------------------------------------------------------
//    ADC_TSO_Init();
//    
//    // ------------------------------------------------------------------------
//    // IAP arrange
//    // ------------------------------------------------------------------------
//    MEM_IAP_Arrange();
//    
//    
////    // ------------------------------------------------------------------------
////    // !!! Run 'ADC_T20_TSO_SampleOnce' function only once !!!
////    // ------------------------------------------------------------------------
////    // (20C) Record ADC TSO conversion data in last area of IAP space. 
////    // Only once, run this routine in chamber at 20C.
////    // ------------------------------------------------------------------------
////    ADC_T20_TSO_SampleOnce();

////    // ------------------------------------------------------------------------
////    // !!! Run 'ADC_T60_TSO_SampleOnce' function only once !!!
////    // ------------------------------------------------------------------------
////    // (60C) Record ADC TSO conversion data in last area of IAP space. 
////    // Only once, run this routine in chamber at 60C.
////    // ------------------------------------------------------------------------
////    ADC_T60_TSO_SampleOnce();
//    
//    
//    
//    // ------------------------------------------------------------------------
//    while(1)
//    {
//        // Convert ADC temperature
//        Temperature = Sample_ADC_TSO_Convert();
//        
//        // to do ...
//    }
//}




















