/**
 ******************************************************************************
 *
 * @file        MG32_OPA_MID.h
 *
 * @brief       This file provides firmware functions to manage the following 
 *              functionalities of the OPA peripheral:
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2025/06/13
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2016 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par         Disclaimer 
 *      The Demo software is provided "AS IS"  without any warranty, either 
 *      expressed or implied, including, but not limited to, the implied warranties 
 *      of merchantability and fitness for a particular purpose.  The author will 
 *      not be liable for any special, incidental, consequential or indirect 
 *      damages due to loss of data or any other reason. 
 *      These statements agree with the world wide and local dictated laws about 
 *      authorship and violence against these laws. 
 ******************************************************************************
 ******************************************************************************
 */ 


#include "MG32.h"

#if defined(ModuleExist_OPA)

#ifndef _MG32_OPA_MID_H

/*!< _MG32_OPA_DRV_H */ 
#define _MG32_OPA_MID_H

#include "MG32_Common_MID.h"
#include "MG32_PW_MID.h"
#include "MG32_ADC_MID.h"
#include "MG32_OPA.h"
#include "MG32__ExtraStruct.h"

/**
 * @name    OPx_InvertingInput
 *          OPA OPx InvertingInput
 */ 
//!@{
#define OPx_INVERTINGINPUT_VBUF                 OPA_CR0_OP0_NMUX_vbuf_w         /*!< OPx Analog input negative channel select internal VBUF. */
#define OPx_INVERTINGINPUT_I1                   OPA_CR0_OP0_NMUX_op0_i1_w       /*!< OPx Analog input negative channel select OP0_I1(PA5). */
#define OPx_INVERTINGINPUT_OPA_P0               OPA_CR0_OP0_NMUX_op0_p0_w       /*!< OPx Analog input negative channel select OPA output (UGB). */
//!@}

/**
 * @name    OPx_NonInvertingInput
 *          OPA OPx NonInvertingInput
 */ 
//!@{
#define OPx_NONINVERTINGINPUT_VBUF              OPA_CR0_OP0_PMUX_vbuf_w         /*!< OPx Analog input positive channel select internal VBUF. */
#define OPx_NONINVERTINGINPUT_I0                OPA_CR0_OP0_PMUX_op0_i0_w       /*!< OPx Analog input positive channel select OP0_I0(PA4). */
#define OPx_NONINVERTINGINPUT_PGA_P0            OPA_CR0_OP0_PMUX_pga_p0_w       /*!< OPx Analog input positive channel select PGA output. */
//!@}

/**
 * @name    OPx_ComparatorPolarity
 *          OPA OPx Comparator Polarity
 */ 
//!@{
#define OPx_POSITIVE_POLARITY                   (0x00000000U)                   /*!< OPx Positive polarity output control */
#define OPx_NEGATIVE_POLARITY                   OPA_CR0_OP0_INV_negative_w      /*!< OPx Negative polarity output control */
//!@}


/**
 * @name    OPA_Power_Mode
 *          OPA Power Mode
 */ 
//!@{
#define OPA_NORMAL                              (0x00000000U)                   /*!< OPA select normal power mode. */
#define OPA_LOWPOWER                            OPA_CR0_OP0_LPEN_enable_w       /*!< OPA select low power mode. */
//!@}




/** 
 * @struct  OPA_InitStructDef
 * @brief   Structure definition of OPA initialization and regular group 
 */        
typedef struct 
{ 
    uint32_t InvertingInput;        /*!< Selects the inverting input of the comparator.
                                         This parameter can be a value of reference OPx_InvertingInput. */
        
    uint32_t NonInvertingInput;     /*!< Selects the non inverting input of the comparator.
                                         This parameter can be a value of reference OPx_NonInvertingInput. */
        
    uint32_t PowerMode;             /*!< Selects the output polarity of the comparator. (Pin)
                                         This parameter can be a value of reference OPx_OutputPolarity. */
        
                                           

}OPA_InitStructDef;

/** 
 * @enum        MID_OPA_StateTypeDef
 * @brief       MID OPA State definition 
 */        
typedef enum 
{
    MID_OPA_STATE_RESET             = (0x00000000U),   /*!< OPA not yet initialized or disabled.             */
    MID_OPA_STATE_READY             = (0x00000001U),   /*!< OPA initialized and ready for use.               */
    MID_OPA_STATE_READY_LOCKED      = (0x00000011U),   /*!< OPA initialized but the configuration is locked. */
    MID_OPA_STATE_BUSY              = (0x00000002U),   /*!< OPA is running.                                  */
    MID_OPA_STATE_BUSY_LOCKED       = (0x00000012U)    /*!< OPA is running and the configuration is locked.  */
}MID_OPA_StateTypeDef;


/** 
 * @struct  OPA_HandleTypeDef
 * @brief   OPA handle Structure definition  
 */        
typedef struct __PACK_MG
{
    OPA_OP_Struct           *Instance;      /*!< Register base address */
                        
    OPA_InitStructDef       Init;           /*!< OPA required parameters */
        
    MID_LockTypeDef         Lock;           /*!< OPA locking object */
                        
    __IO uint32_t           State;          /*!< OPA communication state 
                                                This parameter can be a value of reference MID_OPA_StateTypeDef  */ 
                    
}OPA_HandleTypeDef;





/**
 *******************************************************************************
 * @brief       Reset OPA handle state
 * @param[in]   "__HANDLE__ : pointer to a OPA handle.
 * @return      None
 *******************************************************************************
 */
#define __DRV_OPA_RESET_HANDLE_STATE(__HANDLE__)                                \
  ((__HANDLE__)->State = MID_OPA_STATE_RESET)


/**
 *******************************************************************************
 * @brief       Enable the specified OPA.
 * @param[in]   "__HANDLE__ : pointer to a OPA_HandleTypeDef
 * @return      None
 *******************************************************************************
 */
#define __DRV_OPA_ENABLE(__HANDLE__)                                            \
    ((__HANDLE__)->Instance->CR.W |= OPA_CR0_OP0_EN_enable_w)                   \

/**
 *******************************************************************************
 * @brief       Disable the specified OPA.
 * @param[in]   "__HANDLE__ : pointer to a OPA_HandleTypeDef
 * @return      None
 *******************************************************************************
 */
#define __DRV_OPA_DISABLE(__HANDLE__)                                           \
    ((__HANDLE__)->Instance->CR.W &= ~OPA_CR0_OP0_EN_enable_w)                  \



/**
 *******************************************************************************
 * @brief       Select the OPA Inverting Input pin bits.
 * @param[in]   "__HANDLE__ : pointer to a OPA handle.
 * @param[in]   "__INVPIN__" : specifies the OPA invert pin.
 *  @arg\b      OPx_INVERTINGINPUT_VBUF : input channel select internal VBUF. 
 *  @arg\b      OPx_INVERTINGINPUT_I1 : input channel select OPA_I1. 
 *  @arg\b      OPx_INVERTINGINPUT_OPA_P0 : input channel select OPA output.       
 * @return      None
 * @note 
 * @par         Example
 * @code
    __DRV_OPA_NMUX_PIN(mOPA, OPx_INVERTINGINPUT_VBUF);
 * @endcode
 *******************************************************************************
 */
#define __DRV_OPA_NMUX_PIN(__HANDLE__,__INVPIN__)                               \
    ((__HANDLE__)->Instance->CR.W = ((__HANDLE__)->Instance->CR.W & ~(OPA_CR0_OP0_NMUX_mask_w)) | \
    								(__INVPIN__))


/**
 *******************************************************************************
 * @brief       Select OPA Non-Inverting Input pin.
 * @param[in]   "__HANDLE__ : pointer to a OPA handle.
 * @param[in]   "__NONINVPIN__" : specifies the OPA non-invert pin.
 *  @arg\b      OPx_NONINVERTINGINPUT_VBUF : input channel select internal VBUF. 
 *  @arg\b      OPx_NONINVERTINGINPUT_I0 : input channel select OPA_I1. 
 *  @arg\b      OPx_NONINVERTINGINPUT_PGA_P0 : input channel select PGA output.     
 * @return      None
 * @note 
 * @par         Example
 * @code
    __DRV_OPA_PMUX_PIN(mOPA, OPx_NONINVERTINGINPUT_VBUF);
 * @endcode
 *******************************************************************************
 */
#define __DRV_OPA_PMUX_PIN(__HANDLE__,__NONINVPIN__)                            \
    ((__HANDLE__)->Instance->CR.W = ((__HANDLE__)->Instance->CR.W & ~(OPA_CR0_OP0_PMUX_mask_w)) | \
    								(__NONINVPIN__))



/**
 * @name    Function announce
 * @brief   OPA base initial/Deinitial	
 */ 
///@{
/* Initialization and de-initialization functions  ****************************/
MID_StatusTypeDef MID_OPA_Init(OPA_HandleTypeDef* mOPA);
MID_StatusTypeDef MID_OPA_DeInit(OPA_HandleTypeDef* mOPA);
void MID_OPA_MspInit(OPA_HandleTypeDef* mOPA);
void MID_OPA_MspDeInit(OPA_HandleTypeDef* mOPA);

///@}

/**
 * @name    Function announce
 * @brief   addtogroup OPA_Exported_Functions_Group
 */ 
///@{
MID_StatusTypeDef MID_OPA_Calibration_Start(OPA_HandleTypeDef *mOPA);
///@}



#endif
#endif
