


/**
 ******************************************************************************
 *
 * @file        BSP_Init.c
 * @brief       This is BSP initial C file.
 
 * @par         Project
 *              MG32
 * @version     V1.03
 * @date        2023/07/03
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */
 
/* Includes ------------------------------------------------------------------*/
//Module
#include "MG32_GPIO_DRV.h"
#include "MG32_EXIC_DRV.h"
#include "MG32_IWDT_DRV.h"
#include "MG32_URT_DRV.h"
#include "MG32_CSC_Init.h"

//TH222x Control
#include "TH222x_Global.h"

//BSP
#if BSP_2_ARGB == 1
    #include "BSP_2_ARGB_WS2812.h"
#endif  
#if BSP_3_RGB == 1  
    #include "BSP_3_RGB.h"
#endif
#if BSP_7_STEP_MOTOR == 1 
    #include "BSP_7_StepMotor.h" 
#endif
#if BSP_8_VAR == 1
    #include "BSP_8_VariableResistor.h"
#endif
#if BSP_9_ROTARY_ENCODER == 1
    #include "BSP_9_RotaryEncoder.h"
#endif
#if BSP_10_BUZZER == 1
    #include "BSP_10_Buzzer.h"
#endif
#if BSP_11_RC_SERVO_MOTOR == 1
    #include "BSP_11_RCServoMotor.h"
#endif
#if BSP_12_BLE == 1
    #if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_MG126
        #include "BSP_12_BLE_MG126.h"
        #include "BSP_12_BLE_MG126_app.h"
    #endif
    #if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC06
        #include "BSP_12_BLE_HC06.h"
    #endif
    #if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC42
        #include "BSP_12_BLE_HC42.h"
    #endif
#endif
#if BSP_15_4X4_KEYBOARD == 1
    #include "BSP_15_4x4Keyboard.h"
#endif
#if BSP_17_2COLOR_DOTMATRIX_LED == 1
    #include "BSP_17_2ColorDotMatrixLED.h"
#endif
#if BSP_18_7SEGMENT_DISPLAY == 1
    #include "BSP_18_7Segment.h" 
#endif
#if BSP_19_LCD == 1
    #include "BSP_19_LCD.h"
#endif

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//GPIO
#define TH223x_LEDRYG_GPIOX     GPIOE
#define TH223x_LEDRYG_IOM       IOME
                                
#define TH223x_LEDR_PXPin       PX_Pin_13
#define TH223x_LEDY_PXPin       PX_Pin_14
#define TH223x_LEDG_PXPin       PX_Pin_15
                                
#define TH223x_LEDR_PINX        PINE(13)
#define TH223x_LEDY_PINX        PINE(14)
#define TH223x_LEDG_PINX        PINE(15)
                                
#define TH223x_SW34_GPIOX       GPIOB
#define TH223x_SW34_IOM         IOMB
                                
#define TH223x_SW3_PXPin        PX_Pin_10
#define TH223x_SW4_PXPin        PX_Pin_11
                                
#define TH223x_SW3_PINX         PINB(10)
#define TH223x_SW4_PINX         PINB(11)
                                
#define TH223x_SW3_PIN          PB10
#define TH223x_SW4_PIN          PB11

//EXIC
#define TH223x_SW34_EXIC_PX     EXIC_PB

#define TH223x_SW34_EXIC_IT     EXIC_PB_IT

#define TH223x_SW3_EXIC_TRGSPIN EXIC_TRGS_PIN10       
#define TH223x_SW4_EXIC_TRGSPIN EXIC_TRGS_PIN11

#define TH223x_SW3_EXIC_PINX    EXIC_PX_PIN10
#define TH223x_SW4_EXIC_PINX    EXIC_PX_PIN11

#define TH223x_SW34_IRQ         EXINT1_IRQn

//System Tick
#define SYSTICK_CLK             CONF_CK_AHB_FREQ


/* Private typedef -----------------------------------------------------------*/

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
TH222xCTR_TypeDef  TH222x_CTR;

#if BSP_12_BLE == 1 && BSP_2_ARGB == 1
    uint8_t TH222x_BLE_ARGDelay;
#endif

/* Private function prototypes -----------------------------------------------*/
void SysTick_Handler(void);
void EXINT1_IRQHandler(void);
void URT4x_IRQHandler(void);

static uint32_t TH222x_GetSysTickCount(void);
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	   
 * @details     
 * @return      
 * @note       No
 *******************************************************************************
 */
void SysTick_Handler(void)
{
    //======================================================
    TH222x_CTR.SysTickCount = TH222x_CTR.SysTickCount + 1; 

    //======================================================
    #if BSP_17_2COLOR_DOTMATRIX_LED == 1
        TH222x_2ColorMatrixLED_UpdateFlag = 1;
    #endif
    //======================================================
    switch( TH222x_CTR.DramaAct)
    {
        case 0:
                TH222x_CTR.UpdateFlag = 1;
        
                #if BSP_2_ARGB == 1
                    #if BSP_12_BLE == 1
                        if(TH222x_BLE_ARGDelay==0)
                        {
                            TH222x_ARGB_UpdateFlag = 1;
                        }
                        else
                        {
                            TH222x_BLE_ARGDelay = TH222x_BLE_ARGDelay - 1;
                        }
                    #else
                        TH222x_ARGB_UpdateFlag = 1;
                    #endif
                #endif
                #if BSP_3_RGB == 1 && BSP_2_ARGB == 1
                    TH222x_RGB_UpdateFlag  = TH222x_ARGB_UpdateFlag;
                #elif BSP_3_RGB == 1
                    TH222x_RGB_UpdateFlag  = 1;    
                #endif
                #if BSP_7_STEP_MOTOR == 1
                    TH222x_StepMotor_UpdateFlag = 1;
                #endif
                break;
        case 2: 
                #if BSP_8_VAR == 1
                    TH222x_VR_UpdateFlag = 1;
                #endif
                #if BSP_9_ROTARY_ENCODER == 1
                    TH222x_RotaryEncoder_UpdateFlag  = 1;
                #endif
                
                break;
        case 4: 
                #if BSP_11_RC_SERVO_MOTOR == 1
                    TH222x_RCServoMotor_UpdateFlag = 1;
                #endif
                break;
        case 6: 
                #if BSP_15_4X4_KEYBOARD == 1
                    TH222x_4x4Keyboard_UpdateFlag = 1;
                #endif
                #if BSP_18_7SEGMENT_DISPLAY == 1
                    TH222x_7Segment_UpdateFlag = 1;
                #endif 
                break;
        case 9: 
                #if BSP_12_BLE == 1
                    TH222x_BLE_UpdateFlag = 1;
                #endif 
                TH222x_CTR.DramaAct = 0;
                return;
        default:
                break;            
    }
    TH222x_CTR.DramaAct = TH222x_CTR.DramaAct + 1;
}
/**
 *******************************************************************************
 * @brief	   SW3 & SW4 IRQHandler function.
 * @details     
 * @return      
 * @note       No
 *******************************************************************************
 */
void EXINT1_IRQHandler(void)
{
    uint32_t EXINT1_IRQHandlerTmp;
    
    EXINT1_IRQHandlerTmp = EXIC_GetPxAllTriggerEventFlagStatus(EXIC_PB);
    
    if(EXINT1_IRQHandlerTmp & TH223x_SW3_EXIC_PINX)
    {
        EXIC_ClearPxTriggerEventFlag(EXIC_PB,TH223x_SW3_EXIC_PINX);
        
        TH222x_CTR.SW3_UpdateFlag = 1;
    }
    if(EXINT1_IRQHandlerTmp & TH223x_SW4_EXIC_PINX)
    {
        EXIC_ClearPxTriggerEventFlag(EXIC_PB,TH223x_SW4_EXIC_PINX);
        
        TH222x_CTR.SW4_UpdateFlag = 1;
    }
}
/**
 *******************************************************************************
 * @brief	 DEBUG & BLE URT IRQHandler function.  
 * @details     
 * @return      
 * @note       
 *******************************************************************************
 */
#if BSP_12_BLE == 1
void URT4x_IRQHandler(void)
{
    uint8_t URT4x_IDTmp;
    
    URT4x_IDTmp = __DRV_EXIC_GET_ID22_SOURCE();
    
    //=====================================================
    //BLE 
    //URT4
        #if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC06 || BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_HC42
            if( (URT4x_IDTmp & 0x01) == 0x01)
            {
                BSP_BLE_IRQHandler();
            }
        #endif
        //URT5
        if( (URT4x_IDTmp & 0x02) == 0x02)
        {
            TH222x_BLE_URTIRQHandler();
        }
    
}
#endif  

/**
 *******************************************************************************
 * @brief	   Delay X ms
 * @details     
 * @param[in]  
 * @return      
 * @note       
 *******************************************************************************
 */
void TH222x_Delay(__IO uint32_t Delay)
{
    uint32_t tickstart = 0;
    uint32_t ticknow = 0;
    uint32_t MID_Delay_wait = 0;
  
    MID_Delay_wait   = Delay;
    tickstart        = TH222x_GetSysTickCount();
    
    /* Add a period to guarantee minimum wait */
    if (MID_Delay_wait < 0xFFFFFFFF)
    {
       MID_Delay_wait++;
    }
  
    do{
        ticknow = TH222x_GetSysTickCount();
    }while((ticknow- tickstart) < MID_Delay_wait);
}
/**
 *******************************************************************************
 * @brief	   
 * @details     
 * @param[in]  
 * @return      
 * @note       
 *******************************************************************************
 */
void TH222x_Parameter_DeInit(void)
{
    /*Systick parameter initial*/
    TH222x_CTR.SysTickCount       = 0;
    
    /*TH222x common parameter initial*/    
    TH222x_CTR.UpdateFlag          = 0;
    TH222x_CTR.DramaAct            = TH222x_CTR.DramaAct + 1;
    

    /*TH223x Button software parameter initial*/    
    TH222x_CTR.SW3_UpdateFlag      = 0;
    TH222x_CTR.SW3_TRGlag          = 0;
    TH222x_CTR.SW3_DramaTime       = 0;
                               
    TH222x_CTR.SW4_UpdateFlag      = 0;
    TH222x_CTR.SW4_TRGlag          = 0;
    TH222x_CTR.SW4_DramaTime       = 0;
                               
    TH222x_CTR.SW_DramaTimeMax     = 7;
    TH222x_CTR.SW_DramaTimeDefault = 4;
    
    /*TH223x LED software parameter initial*/
    TH222x_CTR.LED_CountMax        = 200;
    TH222x_CTR.GLED_Count          = 0;
    TH222x_CTR.RLED_Count          = 0;
    TH222x_CTR.YLED_Count          = 0;
    
    TH222x_CTR.GLED_Lock           = 1;
    TH222x_CTR.RLED_Lock           = 1;
    TH222x_CTR.YLED_Lock           = 1;
    
    /**/
    #if BSP_12_BLE == 1 & BSP_2_ARGB == 1
        TH222x_BLE_ARGDelay = 0;
    #endif
    /*TH222x functions parameter initial.*/
    #if BSP_0_DEBUG == 1
        TH222x_DEBUGParameter_Init();
    #endif
    #if BSP_2_ARGB == 1
        TH222x_ARGBParameter_DeInit();
    #endif
    #if BSP_3_RGB == 1
        TH222x_RGBParameter_DeInit();
    #endif
    #if BSP_7_STEP_MOTOR == 1
        TH222x_StepMotorParameter_DeInit();
    #endif
    #if BSP_8_VAR == 1
        TH222x_VariableResistorParameter_DeInit();
    #endif
    #if BSP_9_ROTARY_ENCODER == 1
        TH222x_RotaryEncoderParameter_DeInit();
    #endif
    #if BSP_10_BUZZER == 1
        TH222x_BuzzerParameter_DeInit();
    #endif
    #if BSP_11_RC_SERVO_MOTOR == 1
        TH222x_RCServoMotorParameter_DeInit();
    #endif
    #if BSP_12_BLE == 1
        TH222x_BLEParameter_DeInit();
    #endif
    #if BSP_15_4X4_KEYBOARD == 1
        TH222x_4x4KeyboardParameter_DeInit();
    #endif
    #if BSP_17_2COLOR_DOTMATRIX_LED == 1
        TH222x_2ColorDotMatrixParameter_DeInit();
    #endif
    #if BSP_18_7SEGMENT_DISPLAY == 1
        TH222x_7SegmentParameter_DeInit();
    #endif
    #if BSP_19_LCD == 1
        TH222x_LCDParameter_DeInit();
    #endif
}
/**
 *******************************************************************************
 * @brief	   
 * @details     
 * @param[in]  
 * @return      
 * @note       
 *******************************************************************************
 */
void TH222x_Init(void)
{
    PIN_InitTypeDef    TH223x_PIN;
    EXIC_TRGSTypeDef   TH223x_EIXC;
    
    /*Disable IWDT*/
    UnProtectModuleReg(IWDTprotect);
    IWDT_Cmd(DISABLE);
    ProtectModuleReg(IWDTprotect);
    
    
    /*TH223A SW Control Pin and EXIC Initial.*/
    //GPIO Initial
    GPIO_SetPortBit( TH223x_SW34_GPIOX, (TH223x_SW3_PXPin | TH223x_SW4_PXPin ));
    
    TH223x_PIN.PINX_Mode               = PINX_Mode_OpenDrain_O;
    TH223x_PIN.PINX_PUResistant        = PINX_PUResistant_Enable;
    TH223x_PIN.PINX_Speed              = PINX_Speed_Low;
    TH223x_PIN.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    TH223x_PIN.PINX_FilterDivider      = PINX_FilterDivider_16;         // Digital input function filter is GPIO Clock Freq / 16.
    TH223x_PIN.PINX_Inverse            = PINX_Inverse_Disable;
    TH223x_PIN.PINX_Alternate_Function = 0;
    GPIO_PinMode_Config(TH223x_SW3_PINX,&TH223x_PIN);
    GPIO_PinMode_Config(TH223x_SW4_PINX,&TH223x_PIN);
    
    //EXIC Initial
    TH223x_EIXC.EXIC_Pin       = TH223x_SW3_EXIC_TRGSPIN | TH223x_SW4_EXIC_TRGSPIN;
    TH223x_EIXC.EXIC_TRGS_Mode = Edge;
    EXIC_PxTriggerMode_Select(TH223x_SW34_EXIC_PX,&TH223x_EIXC);
    
    EXIC_ClearPxTriggerEventFlag(TH223x_SW34_EXIC_PX,(TH223x_SW3_EXIC_PINX | TH223x_SW4_EXIC_PINX));
    EXIC_PxTriggerOrMask_Select(TH223x_SW34_EXIC_PX,(TH223x_SW3_EXIC_PINX | TH223x_SW4_EXIC_PINX));
    
    EXIC_PxTriggerITEA_Cmd(TH223x_SW34_EXIC_IT,ENABLE);
    NVIC_EnableIRQ( TH223x_SW34_IRQ);
    
    /*TH223A LED Pin Initial.*/
    GPIO_SetPortBit( TH223x_LEDRYG_GPIOX, (TH223x_LEDR_PXPin | TH223x_LEDY_PXPin | TH223x_LEDG_PXPin));

    TH223x_PIN.PINX_Mode               = PINX_Mode_PushPull_O;
    TH223x_PIN.PINX_PUResistant        = PINX_PUResistant_Disable;
    TH223x_PIN.PINX_Speed              = PINX_Speed_Low;
    TH223x_PIN.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    TH223x_PIN.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    TH223x_PIN.PINX_Inverse            = PINX_Inverse_Disable;
    TH223x_PIN.PINX_Alternate_Function = 0;
    GPIO_PinMode_Config(TH223x_LEDR_PINX,&TH223x_PIN);
    GPIO_PinMode_Config(TH223x_LEDY_PINX,&TH223x_PIN);
    GPIO_PinMode_Config(TH223x_LEDG_PINX,&TH223x_PIN);

    TH223x_LED_RED_OFF();
    TH223x_LED_YELLOW_OFF();
    TH223x_LED_GREEN_OFF();
    
    /*System Tick Initial.(The Project Time base)*/
    InitTick(SYSTICK_CLK/2,0);         // 500us
    
    /*TH222x Module Initial*/
    #if BSP_0_DEBUG == 1
        TH222x_DEBUG_Init();
    #endif
    #if BSP_2_ARGB == 1
        BSP_ARGB_Init();
    #endif        
    #if BSP_3_RGB                    == 1    
        BSP_RGB_Init();
    #endif               
    #if BSP_7_STEP_MOTOR             == 1
        BSP_StepMotor_Init();     
    #endif
    #if BSP_8_VAR                    == 1
        BSP_VR_Init();        
    #endif
    #if BSP_9_ROTARY_ENCODER         == 1
        BSP_RotaryEncoder_Init();        
    #endif
    #if BSP_10_BUZZER                == 1
        BSP_Buzzer_Init();        
    #endif
    #if BSP_11_RC_SERVO_MOTOR        == 1
        BSP_RCServoMotor_Init();        
    #endif
    #if BSP_12_BLE                   == 1
        TH222x_BLE_Init();     
        #if BSP_12_BLE_MODULE == BSP_12_BLE_MODULE_MG126
            SetBleIntRunningMode();
            radio_initBle(TXPWR_0DBM, &TH222x_CTR.ble_mac_addr);
            ble_run_interrupt_start(160*2);                       //320*0.625=200 ms    
        #endif
    #endif
    #if BSP_15_4X4_KEYBOARD          == 1
        BSP_4X4Keyboard_Init();
    #endif
    #if BSP_17_2COLOR_DOTMATRIX_LED  == 1
        BSP_2ColorDotMatrixLED_Init();
    #endif
    #if BSP_18_7SEGMENT_DISPLAY      == 1
        BSP_7Segment_Init();        
    #endif
    #if BSP_19_LCD                   == 1
        BSP_LCM_Init();        
    #endif
    
    TH222x_Delay(1);
}
/**
 *******************************************************************************
 * @brief	   
 * @details     
 * @param[in]  
 * @return      
 * @note       
 *******************************************************************************
 */ 
void TH222x_DisplayTestResult(void)
{
    uint8_t BSP_TestResultTmp;
    
    BSP_TestResultTmp = 0;
    #if BSP_13_SPIFLASH == 1
        BSP_TestResultTmp = BSP_TestResultTmp + TH222x_CTR.TestFlashResult;
    #endif
    #if BSP_14_EEPROM == 1
        BSP_TestResultTmp = BSP_TestResultTmp + TH222x_CTR.TestEEPROMResult;
    #endif
    
    TH222x_CTR.RLED_Count = TH222x_CTR.LED_CountMax;
    TH222x_CTR.GLED_Count = TH222x_CTR.LED_CountMax;
    TH222x_CTR.YLED_Count = TH222x_CTR.LED_CountMax;
    
    if( BSP_TestResultTmp !=TH222x_SUCCESS)
    {
        TH222x_CTR.RLED_Lock  = 1;
        TH222x_CTR.RLED_Count = 0;
    }
    else
    {
        TH222x_CTR.GLED_Lock  = 1;
        TH222x_CTR.GLED_Count = 0;
    }
}
/**
 *******************************************************************************
 * @brief	   
 * @details     
 * @param[in]  
 * @return      
 * @note       
 *******************************************************************************
 */ 
void TH222x_main(void)
{
    #if BSP_7_STEP_MOTOR == 1
        uint8_t TH222x_mainTmp;
    #endif
    
    
    if( TH222x_CTR.UpdateFlag == 0)
    {
        return;
        
    }
    TH222x_CTR.UpdateFlag = 0;
    
    
    /*====TH223A SW3 Button Detect.====*/
    if(TH222x_CTR.SW3_TRGlag == 1)
    {
        if( TH223x_SW3_PIN == 1)
        {
            TH222x_CTR.SW3_DramaTime = TH222x_CTR.SW3_DramaTime - 1;
        }
        else
        {
            TH222x_CTR.SW3_DramaTime = TH222x_CTR.SW3_DramaTime + 1;
        }
        
        if( TH222x_CTR.SW3_DramaTime == 0)
        {
            TH222x_CTR.SW3_TRGlag     = 0;
            TH222x_CTR.SW3_UpdateFlag = 0;
        }
        else if( TH222x_CTR.SW3_DramaTime > TH222x_CTR.SW_DramaTimeMax) 
        {
            TH222x_CTR.SW3_TRGlag     = 0;
            TH222x_CTR.SW3_UpdateFlag = 0;
    
            //-------------------------------------------------
            //Buzzer ON.
            #if BSP_10_BUZZER == 1
                TH222x_Buzzer_UpdateFlag = 1;
                TH222x_Buzzer_ONFlag     = 1;
                TH222x_Buzzer_ONCount    = BSP_BUZZER_TIME;
            #endif
            //-------------------------------------------------
            //Step Motor Status Control.
            #if BSP_7_STEP_MOTOR == 1
                TH222x_mainTmp = TH222x_StepMotor_GetStatus();
                if( (uint8_t)TH222x_mainTmp == STEPMOTOR_ANTICLOCKWISE)
                {
                    TH222x_StepMotor_SetStatus(STEPMOTOR_STOP);
                }
                else
                {
                    TH222x_StepMotor_SetStatus(((uint8_t)TH222x_mainTmp+1));
                }
            
            #endif
        }
    }
    else if( TH222x_CTR.SW3_UpdateFlag == 1)
    {
        TH222x_CTR.SW3_TRGlag    = 1;
        TH222x_CTR.SW3_DramaTime = TH222x_CTR.SW_DramaTimeDefault;
    }

    /*====TH223A SW4 Button Detect.====*/
    if(TH222x_CTR.SW4_TRGlag == 1)
    {
        if( TH223x_SW4_PIN == 1)
        {
            TH222x_CTR.SW4_DramaTime = TH222x_CTR.SW4_DramaTime - 1;
        }
        else
        {
            TH222x_CTR.SW4_DramaTime = TH222x_CTR.SW4_DramaTime + 1;
        }
        
        if( TH222x_CTR.SW4_DramaTime == 0)
        {
            TH222x_CTR.SW4_TRGlag     = 0;
            TH222x_CTR.SW4_UpdateFlag = 0;
        }
        else if( TH222x_CTR.SW4_DramaTime > TH222x_CTR.SW_DramaTimeMax) 
        {
            TH222x_CTR.SW4_TRGlag     = 0;
            TH222x_CTR.SW4_UpdateFlag = 0;
            #if (BSP_13_SPIFLASH==1) || (BSP_14_EEPROM==1)
                TH222x_DisplayTestResult();
            #endif
        }
    }
    else if( TH222x_CTR.SW4_UpdateFlag == 1)
    {
        TH222x_CTR.SW4_TRGlag    = 1;
        TH222x_CTR.SW4_DramaTime = TH222x_CTR.SW_DramaTimeDefault;
    }
    //=======================================================
    //TH223A LED Control
    if( TH222x_CTR.GLED_Count < TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_GREEN_ON();
        
        TH222x_CTR.GLED_Count = TH222x_CTR.GLED_Count + 1;
    }
    else if( TH222x_CTR.GLED_Count == TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_GREEN_OFF();
        TH222x_CTR.GLED_Lock = 0;
        TH222x_CTR.GLED_Count = TH222x_CTR.GLED_Count + 1;
    }
    
    if( TH222x_CTR.RLED_Count < TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_RED_ON();
        
        TH222x_CTR.RLED_Count = TH222x_CTR.RLED_Count + 1;
    }
    else if( TH222x_CTR.RLED_Count == TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_RED_OFF();
        TH222x_CTR.RLED_Lock = 0;
        TH222x_CTR.RLED_Count = TH222x_CTR.RLED_Count + 1;
    }
    
    if( TH222x_CTR.YLED_Count < TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_YELLOW_ON();
        
        TH222x_CTR.YLED_Count = TH222x_CTR.YLED_Count + 1;
    }
    else if( TH222x_CTR.YLED_Count == TH222x_CTR.LED_CountMax)
    {
        TH223x_LED_YELLOW_OFF();
        TH222x_CTR.YLED_Lock = 0;
        TH222x_CTR.YLED_Count = TH222x_CTR.YLED_Count + 1;
    }
    //===================================================
    //Buzzer Control
    #if BSP_10_BUZZER == 1
        if( TH222x_Buzzer_ONFlag == 1)
        {
            if( TH222x_Buzzer_ONCount == 0)
            {
                TH222x_Buzzer_UpdateFlag = 1;
                TH222x_Buzzer_ONFlag     = 0;
            }
            else
            {
                TH222x_Buzzer_ONCount = TH222x_Buzzer_ONCount - 1;
            }
            
        }
    #endif
}
 
/**
 *******************************************************************************
 * @brief	   Get SystemTick counter value.
 * @details     
 * @return      
 * @note       No
 *******************************************************************************
 */
static uint32_t TH222x_GetSysTickCount(void) 
{
    return( TH222x_CTR.SysTickCount);
} 
 
 
 
 
 
 
 
 
 
 
 

