
/**
 ******************************************************************************
 *
 * @file        TH222x_2_ARGB.c
 * @brief       This is to test TH222x ARGB C file.
 
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2023/02/08
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "MG32_IWDT_DRV.h"
#include "TH222x_Global.h"
#include "BSP_2_ARGB_WS2812.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define TH222x_ARGB_COLOR       4

#define TH222x_ARGB_PIECE       14

#define TH222x_ARGB_G           0                           // Array offset for RGB-R
#define TH222x_ARGB_R           1                           // Array offset for RGB-G
#define TH222x_ARGB_B           2                           // Array offset for RGB-B



/* Private typedef -----------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
// ----------------------------------------------------------------------------
uint8_t TH222x_ARGB_UpdateFlag;                             // UpdateFlag=0 (Nothing)    
                                                            // UpdateFlag=1 (Needs to do ...)
                                                            
// ----------------------------------------------------------------------------
static uint8_t  TH222x_ARGB_Status;                         // There are 2 status for TH222x_ARGB_Status.
                                                            //      1. TH222x_READY    
                                                            //      2. TH222x_BUSY     

// ----------------------------------------------------------------------------
static uint16_t TH222x_ARGB_DramaTime;                      // Drama Time counter : (0~TH222x_ARGB_DramaTimeMax)
static uint16_t TH222x_ARGB_DramaTimeMax;                   // The maximum of TH222x_ARGB_DramaTime.

// ----------------------------------------------------------------------------
static uint8_t  TH222x_ARGB_DramaAct;                       // Drama Axt index : (0~TH222x_ARGB_DramaActMax)
static uint8_t  TH222x_ARGB_DramaActMax;                    // The maximum of TH222x_ARGB_DramaAct.
                                          
// ----------------------------------------------------------------------------
static uint8_t TH222x_ARGB_DramaBUF[TH222x_ARGB_PIECE * 3]; /*!<.The buffer is use for ARGB calculate.*/
                                                            // TH222x_ARGB_PIECE=14
// ----------------------------------------------------------------------------
const static uint8_t TH222x_ARGBDefault_TABLE[TH222x_ARGB_COLOR][3] =   // TH222x_ARGB_COLOR=4
                                    {
                                        {0 ,   5 ,   0},    // 1st Color : [G : R : B ] = [  0, 50,  0]
                                        {5 ,   0 ,   0},    // 2nd Color : [G : R : B ] = [ 50,  0,  0]
                                        {0 ,   0 ,   5},    // 3rd Color : [G : R : B ] = [  0,  0, 50] 
                                        {5 ,   5 ,   5},    // 4th Color : [G : R : B ] = [ 50, 50, 50]
                                    };

/* Private function prototypes -----------------------------------------------*/                                                                   
/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/

                                                                      
/**
 *******************************************************************************
 * @brief	    TH222x ARGB parameter default initial.
 * @details     Deinitial ARGB's parameters.
 * @return      None      
 * @note        No
 *******************************************************************************
 */
void TH222x_ARGBParameter_DeInit(void)
{
    uint8_t TH222x_ARGB_DeInitTmp;
    
    // ------------------------------------------------------------------------
    TH222x_ARGB_UpdateFlag      = 0;                        // No update event
    
    TH222x_ARGB_DramaTime       = 0;                        // Drama time counter    
    TH222x_ARGB_DramaTimeMax    = 400;
        
    TH222x_ARGB_DramaAct        = 0;                        // Drama index 
    TH222x_ARGB_DramaActMax     = (TH222x_ARGB_COLOR - 1);
    
    // ------------------------------------------------------------------------
    // Clear ARGB data 
    // ------------------------------------------------------------------------
    for(TH222x_ARGB_DeInitTmp=0; TH222x_ARGB_DeInitTmp<(TH222x_ARGB_PIECE*3); TH222x_ARGB_DeInitTmp++)
    {
        TH222x_ARGB_DramaBUF[TH222x_ARGB_DeInitTmp] = 0;
    }
    
    // ------------------------------------------------------------------------
    TH222x_ARGB_Status          = TH222x_READY;
}
/**
 *******************************************************************************
 * @brief	    TH222x ARGB main function.
 * @details     Drive ARGB with pattern index.
 * @return      None
 * @note        No
 *******************************************************************************
 */
void TH222x_ARGB(void)
{
    uint8_t TH222x_ARGB_Tmp;
    uint8_t TH222x_ARGB_Adr;
    
    // ------------------------------------------------------------------------
    // Condition is TH222x_ARGB_UpdateFlag=1.
    // ------------------------------------------------------------------------
    if(TH222x_ARGB_UpdateFlag == 0)
    {
        return;
    }
    
    // ------------------------------------------------------------------------
    TH222x_ARGB_UpdateFlag = 0;
    
    // ------------------------------------------------------------------------
    // Check status == Ready?
    //      1. Change status to Busy.
    //      2. Change EXINT3 interrupt priority.
    //      3. Update ARGB with URT.    
    // ------------------------------------------------------------------------
    if(TH222x_ARGB_Status == TH222x_READY)
    {
        TH222x_ARGB_Status  = TH222x_BUSY;
        
        NVIC_SetPriority(EXINT3_IRQn, 1);                   // the lower priority
        
        // Update ARGB status.
        BSP_ARGB_SendData(TH222x_ARGB_DramaBUF, TH222x_ARGB_PIECE);
    }
    
    // ------------------------------------------------------------------------
    // Drama Time range is 0~TH222x_ARGB_DramaTimeMax
    // ------------------------------------------------------------------------
    if(TH222x_ARGB_DramaTime < TH222x_ARGB_DramaTimeMax)
    {
        TH222x_ARGB_DramaTime = TH222x_ARGB_DramaTime + 1;
        return;
    }
    
    TH222x_ARGB_DramaTime   = 0;
    
    // ------------------------------------------------------------------------
    // Update TH222x_ARGB_DramaBUF array
    // The buffer is use for ARGB calculate.
    // ------------------------------------------------------------------------
    TH222x_ARGB_Adr         = 0;
    for(TH222x_ARGB_Tmp=0; TH222x_ARGB_Tmp<TH222x_ARGB_PIECE; TH222x_ARGB_Tmp++)    // TH222x_ARGB_PIECE=14
    {
        TH222x_ARGB_Adr = (TH222x_ARGB_Tmp * 3);
        
        // --------------------------------------------------------------------
        // Source data from TH222x_ARGBDefault_TABLE array.
        TH222x_ARGB_DramaBUF[TH222x_ARGB_Adr + TH222x_ARGB_G] = TH222x_ARGBDefault_TABLE[TH222x_ARGB_DramaAct][TH222x_ARGB_G];
        TH222x_ARGB_DramaBUF[TH222x_ARGB_Adr + TH222x_ARGB_R] = TH222x_ARGBDefault_TABLE[TH222x_ARGB_DramaAct][TH222x_ARGB_R];
        TH222x_ARGB_DramaBUF[TH222x_ARGB_Adr + TH222x_ARGB_B] = TH222x_ARGBDefault_TABLE[TH222x_ARGB_DramaAct][TH222x_ARGB_B];
    }
    
    // ------------------------------------------------------------------------
    TH222x_ARGB_Status      = TH222x_ARGB_Status | TH222x_READY;
    
    // ------------------------------------------------------------------------
    // TH222x_ARGB_DramaAct range is 0~TH222x_ARGB_DramaActMax
    // ------------------------------------------------------------------------
    TH222x_ARGB_DramaAct    = TH222x_ARGB_DramaAct + 1;
    
    if (TH222x_ARGB_DramaAct > TH222x_ARGB_DramaActMax)
    {
        TH222x_ARGB_DramaAct = 0;
    }
    
}
/**
 *******************************************************************************
 * @brief	    Release ARGB BUSY state & Refresh IWDT
 * @return      None      
 *******************************************************************************
 */
void BSP_ARGB_CpltCallback(void) 
{
    NVIC_SetPriority(EXINT3_IRQn, 0);       // The highest priority - EXINT3
    
    TH222x_ARGB_Status = TH222x_ARGB_Status & (uint8_t)(~TH222x_BUSY);  // Release BUSY state
    
    IWDT_RefreshCounter();                  // Refresh IWDT counter
}





