
/**
 ******************************************************************************
 *
 * @file        BSP_5_4x4Keyboard.c
 * @brief       The 4 X 4 KEYBOARD Code's c file.
 *
 * @par         Project
 *              MG32
 * @version     V1.03
 * @date        2023/02/08
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/*==============================================================================
                                 User Notes
How To use this function:
-----------------------
    1. Use BSP_4X4Keyboard_Init() to initial.
    2. Periodic call BSP_4X4Keyboard_main() function to update 4x4 Keyboard status. 
    3. If KB_CTR.KeybufferChangeFlag == 1 represent button status change.
       (1). From KB_CTR.Keybuffer parameter can get which function buttion is pressed.    
    4. Change Default_KeyCode_Table[] to set customize's key function.         
        
Driver architecture:
--------------------
    + MG32_GPIO_DRV

Known Limitations:
------------------
    1. The most detect 3 key functions.

Require parameter
------------------
    Require module : CSC / GPIO 
    
    GPIO pin configuration : 
        Pin  / IO mode / AFS
        ---  --------  -----
        PC0  / ODO     / GPIO
        PC1  / ODO     / GPIO
        PC2  / ODO     / GPIO
        PC3  / ODO     / GPIO
        PC7  / ODO     / GPIO
        PC12 / ODO     / GPIO
        PC13 / ODO     / GPIO
        PC14 / ODO     / GPIO
        
Example codes:
------------------

==============================================================================*/



/* Includes ------------------------------------------------------------------*/
#include "BSP_15_4x4Keyboard.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//GPIO
#define KB_ROW0_PIN                PX_Pin_0          /*!< SCAN_ROW0 IOM pin. */
#define KB_ROW1_PIN                PX_Pin_1          /*!< SCAN_ROW1 IOM pin. */
#define KB_ROW2_PIN                PX_Pin_2          /*!< SCAN_ROW2 IOM pin. */
#define KB_ROW3_PIN                PX_Pin_3          /*!< SCAN_ROW3 IOM pin. */
#define KB_ROW_IOM                 IOMC              /*!< SCAN_ROW IOM port. */
#define KB_ROW_PORT                GPIOC             /*!< SCAN_ROW IO port. */
#define KB_ROW_MSK                 0x000F            /*!< SCAN_ROW IO status mask. */


#define KB_COL0_PIN                PX_Pin_7          /*!< SCAN_COL0 IOM pin. */          
#define KB_COL1_PIN                PX_Pin_12         /*!< SCAN_COL1 IOM pin. */
#define KB_COL2_PIN                PX_Pin_13         /*!< SCAN_COL2 IOM pin. */
#define KB_COL3_PIN                PX_Pin_14         /*!< SCAN_COL3 IOM pin. */
#define KB_COL_IOM                 IOMC              /*!< SCAN_COL IOM port. */
#define KB_COL_PORT                GPIOC             /*!< SCAN_COL IO port. */
#define KB_COLUMN_MSK              0x7080  

//KB control
#define KB_SCAN_COLUMN             4                 /*!< Total key colume. */            
#define KB_SCAN_ROW                4                 /*!< Total key row. */
#define KB_TOTAOL_KEY              (4 * 4)           /*!< Total key that can define . */

#define KB_MAKE_DELAY_TIME         0                 /*!< Control buttion make debounce time. */
#define KB_BREAK_DELAY_TIME        6                 /*!< Control buttion break debounce time. */

#define KB_SCAN_MAKE_STATUS_MASK   0x30             // KB_SCAN_MAKE_STATUS_MASK handles { KB_SCAN_MAKE_NOW, KB_SCAN_MAKE_CONFIRM}
    #define KB_SCAN_MAKE_NOW           0x10                            
    #define KB_SCAN_MAKE_CONFIRM       0x20                            
#define KD_SCAN_Code_Update        0x40     
#define KB_SCAN_BREAK_CONFIRM      0x80
#define KB_SCAN_MAKE_COUNT_MASK    0x07

#define KEY_RECORD_MAX             16

/* Private typedef -----------------------------------------------------------*/
/** 
 * @struct	KEYBOARD
 * @brief	Control Keyboard.
 */
typedef struct
{
    // ------------------------------------------------------------------------
    // For Scan Key Relationship Control
    uint8_t    Scan_Column;                                 // column index (0~3)
//    uint8_t    Scan_Flag_Start;                             // Unused
    uint8_t    Scan_Flag_Matrix_CheckFull;                  // CLR=0 or SET=1 (When press key counter > 16) 
    uint8_t    Scan_Flag_Matrix_Checked;                    // 0 or SET 
    uint8_t    Scan_Count_CheckMatrix;                      // just counter (press key counter)
    uint8_t    Scan_Count_BuffOut;                          // The same as column index. (The index range is 0~3)
    uint16_t   Scan_Buffer_Matrix[KB_SCAN_COLUMN];          // Save press key state depend on column (0~3)
    uint8_t    Scan_Buffer_Matrix_Flag[KEY_RECORD_MAX];     // 16 array element
    uint8_t    Scan_Buffer_Matrix_Address[KEY_RECORD_MAX];  // 0 or KB.Tmp_Addr    
    // ------------------------------------------------------------------------
    // handle Tmp;
    uint8_t    Tmp_Flag;                                    // Element from 'KB.Scan_Buffer_Matrix_Flag[KB.Tmp_Index]'
                                                            // content may KB_SCAN_BREAK_CONFIRM=0x80 or
                                                            // (KB_SCAN_MAKE_CONFIRM | KB_SCAN_BREAK_CONFIRM | KD_SCAN_Code_Update)=0xE0
    
    uint8_t    Tmp_Addr;                                    // Press key index with (column, row)
    uint8_t    Tmp_Code;                                    // Key code ('0', '1' ... 'F')
    uint8_t    Tmp_Index;                                   // Just Array index temporary

}KEYBOARD_TypeDef;

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

// ----------------------------------------------------------------------------
static KEYBOARD_TypeDef  KB;   

// ----------------------------------------------------------------------------
KBCTR_TypeDef            KB_CTR;       
       
// ----------------------------------------------------------------------------
// Key scan table.
static uint16_t const KeyScan1_Table[KB_SCAN_COLUMN] = { 0x7000,0x6080,0x5080,0x3080};
static uint16_t const KeyScan0_Table[KB_SCAN_COLUMN] = { 0x0080,0x1000,0x2000,0x4000};
static uint16_t const KeyScanMak_Table[KB_SCAN_ROW]  = { 0x0001,0x0002,0x0004,0x0008};

// Key code mapping table.
static const uint8_t Default_KeyCode_Table[KB_TOTAOL_KEY]= 			
{
//Column       0     1     2     3  // Row   |  ASCII code :    
            0x30, 0x31, 0x32, 0x33, // 0     |      '0', '1', '2', '3',
            0x34, 0x35, 0x36, 0x37, // 1     |      '4', '5', '6', '7',    
            0x38, 0x39, 0x41, 0x42, // 2     |      '8', '9', 'A', 'B',
            0x43, 0x44, 0x45, 0x46, // 3     |      'C', 'D', 'E', 'F'
};

/* Private function prototypes -----------------------------------------------*/
static void BSP_KB_TranslateToKeyCode(void);
static void BSP_4x4Keyboard_Parameter_DeInit(void);
static uint32_t BSP_4x4Keyboard_Scaning(uint8_t Scan_Index);
static void BSP_4x4Keyboard_ScanKeyStatus(void);
static void BSP_4x4Keyboard_ClearKeyRecord(void);
static void BSP_4x4Keyboard_RecordKeySend(void);
static void BSP_4x4Keyboard_CheckUpdateKeyMatrix(void);
static void BSP_4x4Keyboard_CheckScanInMatrix(void);
static void BSP_4x4Keyboard_CheckScanIn(void);
static void BSP_4x4Keyboard_CheckUpdate(void);
static void BSP_4x4Keyboard_TranslateToKeyCode_Flow(void);
static void BSP_4x4Keyboard_Keyboard_Function(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	    4x4 keyboard relationship control initial.
 * @details     GPIO initial & 
 * @return      None
 *******************************************************************************
 */
void BSP_4X4Keyboard_Init(void)
{
    PIN_InitTypeDef  KB_Pin;
    
    // ------------------------------------------------------------------------
    // GPIO initial. (Open drain + Pull-up Enable)
    //  ROW0 : ROW3  = PB12 ~ PB15
    //  COL0 : COL3  = PC7, PC12 ~ PC14
    // --------------------------------------------------------------------
    // Scan Row
    KB_Pin.PINX_Pin                 = (KB_ROW0_PIN | KB_ROW1_PIN | KB_ROW2_PIN | KB_ROW3_PIN );
    KB_Pin.PINX_Mode                = PINX_Mode_OpenDrain_O;
    KB_Pin.PINX_PUResistant         = PINX_PUResistant_Enable;
    KB_Pin.PINX_Speed               = PINX_Speed_Low;
    KB_Pin.PINX_Inverse             = PINX_Inverse_Disable;
    KB_Pin.PINX_OUTDrive            = PINX_OUTDrive_Level0;
    KB_Pin.PINX_FilterDivider       = PINX_FilterDivider_Bypass;
    KB_Pin.PINX_Alternate_Function  = 0;
    
    GPIO_PortMode_Config(KB_ROW_IOM, &KB_Pin);
    
    // Scan Column
    KB_Pin.PINX_Pin                 = (KB_COL0_PIN | KB_COL1_PIN | KB_COL2_PIN | KB_COL3_PIN );
    
    GPIO_PortMode_Config(KB_COL_IOM, &KB_Pin);
    
    // ------------------------------------------------------------------------
    // all GPIO pins output 'High'
    // ------------------------------------------------------------------------
    KB_ROW_PORT->SC.H[0]            = KB_ROW_MSK;       // PC0~PC3
    KB_COL_PORT->SC.H[0]            = KB_COLUMN_MSK;    // PC7, PC12~14

    // ------------------------------------------------------------------------
    // Keyboard control parameter initial
    // ------------------------------------------------------------------------
    BSP_4x4Keyboard_Parameter_DeInit(); 
    
}
/**
 *******************************************************************************
 * @brief	    4x4 Keyboard main function.
 * @details     Scan Key and key function.
 * @return      None
 *******************************************************************************
 */
void BSP_4X4Keyboard_main(void)
{
    // ------------------------------------------------------------------------
    // Scan Key status
    // ------------------------------------------------------------------------
    BSP_4x4Keyboard_ScanKeyStatus();    
    
    // ------------------------------------------------------------------------
    // According to Scan result to handle. 
    // ------------------------------------------------------------------------
    BSP_4x4Keyboard_Keyboard_Function();     
    
}
/**
 *******************************************************************************
 * @brief	    4x4 Keyboard software control parameter initial.
 * @details     Deinitial KB & KB_CTR
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_4x4Keyboard_Parameter_DeInit(void)
{
    // ------------------------------------------------------------------------
    KB.Scan_Count_BuffOut           = 0;                    // column index
    KB.Scan_Count_CheckMatrix       = 0;                    // just counter

    KB_CTR.KeybufferChangeFlag      = 0;                    // Clear=0
    
    // ------------------------------------------------------------------------
    // Clear KB & KB_CTR array data
    // ------------------------------------------------------------------------
    memset(KB.Scan_Buffer_Matrix, 0, KB_SCAN_COLUMN * 2);   // Scan_Buffer_Matrix    
    memset(KB.Scan_Buffer_Matrix_Address, 0, KEY_RECORD_MAX);// Scan_Buffer_Matrix_Address
    memset(KB.Scan_Buffer_Matrix_Flag, 0, KEY_RECORD_MAX);  // Scan_Buffer_Matrix_Flag
    
    memset(KB_CTR.Keybuffer, 0, KEY_BUFFER_MAX);            // Keybuffer
    
}
/**
 *******************************************************************************
 * @brief	    Clear status of Key record. 
 * @details        
 * @return      None
 *******************************************************************************
 */
static void BSP_4x4Keyboard_ClearKeyRecord(void)
{
    KB.Scan_Buffer_Matrix_Address[KB.Tmp_Index] = 0;
    KB.Scan_Buffer_Matrix_Flag[KB.Tmp_Index]    = 0;
}
/**
 *******************************************************************************
 * @brief	    Record the key code already send.
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_4x4Keyboard_RecordKeySend(void)
{
    // 'Scan_Buffer_Matrix_Flag |= 0x40'
    KB.Scan_Buffer_Matrix_Flag[KB.Tmp_Index] |= KD_SCAN_Code_Update;    /*!< Record the key already send. */
                                                                        // KD_SCAN_Code_Update=0x40
}
/**
 *******************************************************************************
 * @brief	    Update key code to tmporary buffer.
 * @details     It check status of tmporary buffer at the same time.
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_4x4Keyboard_CheckUpdateKeyMatrix(void)
{
    uint8_t KB_CK_UpdateTmp;
    
    // ------------------------------------------------------------------------
    // Check the code whether already exist in buffer or not.
    // ------------------------------------------------------------------------
    for(KB_CK_UpdateTmp=0; KB_CK_UpdateTmp<KEY_RECORD_MAX; KB_CK_UpdateTmp++)   // KEY_RECORD_MAX=16
    {
        // --------------------------------------------------------------------
        // condition: 
        //  1. KB.Scan_Buffer_Matrix_Flag[KB_CK_UpdateTmp] != 0
        //  2. KB.Scan_Buffer_Matrix_Address[KB_CK_UpdateTmp] == KB.Tmp_Addr
        // --------------------------------------------------------------------
        if((KB.Scan_Buffer_Matrix_Flag[KB_CK_UpdateTmp] != 0) && (KB.Scan_Buffer_Matrix_Address[KB_CK_UpdateTmp] == KB.Tmp_Addr))
        {
            KB.Scan_Buffer_Matrix_Flag[KB_CK_UpdateTmp] |= KB_SCAN_MAKE_NOW;    // KB_SCAN_MAKE_NOW=0x10
            
            KB.Scan_Count_CheckMatrix ++;                                       // press key counter
            
            // ----------------------------------------------------------------
            if(KB.Scan_Count_CheckMatrix > (KEY_RECORD_MAX - 1))                /*!< Key event temporary buffer still is full . */
            {
                KB.Scan_Flag_Matrix_Checked = SET;
            }
            return;
        }
    }
    
    // ------------------------------------------------------------------------
    // Save the code to the buffer if the buffer no full.
    // ------------------------------------------------------------------------
    if(KB.Scan_Flag_Matrix_CheckFull == CLR)
    {
        
        for(KB_CK_UpdateTmp=0; KB_CK_UpdateTmp<KEY_RECORD_MAX; KB_CK_UpdateTmp++)// KEY_RECORD_MAX=16
        {
            // ----------------------------------------------------------------
            // condition : KB.Scan_Buffer_Matrix_Flag[KB_CK_UpdateTmp] is empty
            // ----------------------------------------------------------------
            if(KB.Scan_Buffer_Matrix_Flag[KB_CK_UpdateTmp] == 0)
            {                    
                KB.Scan_Buffer_Matrix_Address[KB_CK_UpdateTmp] =  KB.Tmp_Addr;
                KB.Scan_Buffer_Matrix_Flag[KB_CK_UpdateTmp]   |= KB_SCAN_MAKE_NOW;// KB_SCAN_MAKE_NOW=0x30
                
                KB.Scan_Count_CheckMatrix ++;
                
                if(KB.Scan_Count_CheckMatrix > (KEY_RECORD_MAX - 1))
                {
                    KB.Scan_Flag_Matrix_Checked = SET;   
                }
                return;
            }
        }
        
        KB.Scan_Flag_Matrix_CheckFull = SET;                /*!< Key event temporary buffer is full. */
    }
}
/**
 *******************************************************************************
 * @brief	    Scan the column status.
 * @details     
 * @param[in]   Scan_Index : Now Scan line.
 * @return      Scan Result.
 * @exception   No
 * @note        
 *******************************************************************************
 */
static uint32_t BSP_4x4Keyboard_Scaning(uint8_t Scan_Index)
{
    uint32_t KB_SCANTmp = 0;
    uint32_t KB_SCANTDelay;
    
    // ------------------------------------------------------------------------
    // Start Scan the column. (PC7, PC12, PC13, PC14)
    //                                          PC7     PC12    PC13    PC14
    //                                         ======  ======  ======  ======
    //      KeyScan0_Table[KB_SCAN_COLUMN] = { 0x0080, 0x1000, 0x2000, 0x4000}; 
    //      KeyScan1_Table[KB_SCAN_COLUMN] = { 0x7000, 0x6080, 0x5080, 0x3080};
    //
    //      If Scan_Index=1 then MCU will output 'High' on PC12 pin. Anothers pin 
    //      output 'Low'.
    // ------------------------------------------------------------------------
    GPIO_SetClearPortBit(KB_COL_PORT , KeyScan1_Table[Scan_Index] , KeyScan0_Table[Scan_Index]);   

    // for stable delay
    for(KB_SCANTDelay= 0; KB_SCANTDelay<30;KB_SCANTDelay++)
    {
        __NOP();
    }
    
    // ------------------------------------------------------------------------
    // Read Row pin status. (PC0, PC1, PC2, PC3)
    // Get keyboard event from GPIO_ReadPort routine. 
    // ------------------------------------------------------------------------
    KB_SCANTmp = ((GPIO_ReadPort(KB_ROW_PORT) & KB_ROW_MSK));// Only accept PC0~PC3. (KB_ROW_MSK = 0x000F)    
    KB_SCANTmp = ((~KB_SCANTmp) & KB_ROW_MSK);              // Invert key state and mask with KB_ROW_MSK
                                                            // Only '1' indicates press the key.
        
    // ------------------------------------------------------------------------
    // End Scan the column.
    // ------------------------------------------------------------------------
    GPIO_SetPortBit(KB_COL_PORT, KB_COLUMN_MSK);            // KB_COLUMN_MSK=0x7080 (PC7, PC12, PC13, PC14 output high)
        
    // until release press state.
    while((GPIO_ReadPort(KB_COL_PORT) & KB_COLUMN_MSK)!=KB_COLUMN_MSK)
    {
        __NOP();
    }
    
    // ------------------------------------------------------------------------
    // return press key
    return(KB_SCANTmp);
} 
/**
 *******************************************************************************
 * @brief	    Scan all key status.
 * @details     Get press key when column from 0 to 3. (Total 4 columns of pins)
 * @return      None
 *******************************************************************************
 */
static void BSP_4x4Keyboard_ScanKeyStatus(void)
{
    KB.Scan_Column = 0;                                     // column index (0~3)

    do{
        // Get press key when KB.Scan_Column in scanning state.
        KB.Scan_Buffer_Matrix[KB.Scan_Column] = (uint16_t)BSP_4x4Keyboard_Scaning(KB.Scan_Column);
        
        // for next column
        KB.Scan_Column = KB.Scan_Column + 1;
        
    }while(KB.Scan_Column != KB_SCAN_COLUMN);               // KB_SCAN_COLUMN=4
}
/**
 *******************************************************************************
 * @brief	    Check event happen in which row  
 * @details     
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_4x4Keyboard_CheckScanInMatrix(void)
{
    uint8_t  KB_CKSCanIn_RowTmp;

    for(KB_CKSCanIn_RowTmp=0; KB_CKSCanIn_RowTmp<KB_SCAN_ROW ; KB_CKSCanIn_RowTmp++)            // KB_SCAN_ROW=4
    {
        // is press key ?
        if(KB.Scan_Buffer_Matrix[KB.Scan_Count_BuffOut] & KeyScanMak_Table[KB_CKSCanIn_RowTmp]) // the row of Keyboard have key event happen.
        {                                                                                       // KeyScanMak_Table[x] = { 0x0001,0x0002,0x0004,0x0008};
                                                                                                // Means press Key1 or Key2 or Key3 or Key4 when column index=KB.Scan_Count_BuffOut
            // ----------------------------------------------------------------
            KB.Tmp_Addr = (KB_CKSCanIn_RowTmp * KB_SCAN_COLUMN) + KB.Scan_Count_BuffOut;        // (KB_SCAN_COLUMN=4) Get index number
            
            KB.Tmp_Code = Default_KeyCode_Table[KB.Tmp_Addr];                                   /*!< Mapping key function. */
                                                                                                //  static const uint8_t Default_KeyCode_Table[KB_TOTAOL_KEY]= 			
                                                                                                //  {
                                                                                                //  //Column       0     1     2     3  // Row   |  ASCII code :    
                                                                                                //              0x30, 0x31, 0x32, 0x33, // 0     |      '0', '1', '2', '3',
                                                                                                //              0x34, 0x35, 0x36, 0x37, // 1     |      '4', '5', '6', '7',    
                                                                                                //              0x38, 0x39, 0x41, 0x42, // 2     |      '8', '9', 'A', 'B',
                                                                                                //              0x43, 0x44, 0x45, 0x46, // 3     |      'C', 'D', 'E', 'F'
                                                                                                //  };

            // ----------------------------------------------------------------
            if(KB.Tmp_Code != 0)                                                                /*!< If the key code is significant update the key code to tmporary buffer of send. */
            {
                BSP_4x4Keyboard_CheckUpdateKeyMatrix();
                
                if(KB.Scan_Flag_Matrix_Checked == SET)                                          /*!< If tmporary buffer is already full jump away the check flow. */
                {
                    return;
                }
            }
        }
    }
}
/**
 *******************************************************************************
 * @brief	    Check keyboard scan result.
 * @details     
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_4x4Keyboard_CheckScanIn(void)
{
    uint8_t KB_CKScanIn_Tmp;

    // ------------------------------------------------------------------------
    KB.Scan_Flag_Matrix_Checked   = 0;
    KB.Scan_Flag_Matrix_CheckFull = 0;
    KB.Scan_Count_CheckMatrix     = 0;
    
    // ------------------------------------------------------------------------
    for(KB_CKScanIn_Tmp=0; KB_CKScanIn_Tmp<KB_SCAN_COLUMN; KB_CKScanIn_Tmp++)   // KB_SCAN_COLUMN=4
    {
        // --------------------------------------------------------------------
        // Press any key in Scan_Buffer_Matrix[0] when column index is 0.
        // Press any key in Scan_Buffer_Matrix[1] when column index is 1. 
        // ...
        // --------------------------------------------------------------------
        // BSP_4x4Keyboard_ScanKeyStatus routine handles Scan_Buffer_Matrix.
        // --------------------------------------------------------------------
        if(KB.Scan_Buffer_Matrix[KB.Scan_Count_BuffOut] != 0)                   /*!< Keyboard have any key event happen. */
        {
            BSP_4x4Keyboard_CheckScanInMatrix();                                /*!< Check keyboard event in whcich { column , row}. */
            
            if(KB.Scan_Flag_Matrix_Checked == SET)                                                  
            {
                if(KB.Scan_Count_BuffOut < KB_SCAN_COLUMN)                      // KB_SCAN_COLUMN=4                              
                {
                    KB.Scan_Count_BuffOut = KB_SCAN_COLUMN;                     // KB.Scan_Count_BuffOut=4 
                }
                else
                {
                    KB.Scan_Count_BuffOut = 0;                                  // KB.Scan_Count_BuffOut=0
                }
                return;
            }
        }
        KB.Scan_Count_BuffOut++;
    }
    
    // ------------------------------------------------------------------------
    if(KB.Scan_Count_BuffOut > (KB_SCAN_COLUMN-1))
    {
        KB.Scan_Count_BuffOut = 0;
    }
}
/**
 *******************************************************************************
 * @brief	    Check status of tmporary buffer.
 * @details     
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_4x4Keyboard_CheckUpdate(void)
{
    uint8_t KB_CKUP_Tmp;
    uint8_t KB_CKUP_Status;
    
    for(KB_CKUP_Tmp=0; KB_CKUP_Tmp<KEY_RECORD_MAX; KB_CKUP_Tmp++)       // KEY_RECORD_MAX=16
    {
        KB_CKUP_Status = KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp];       // KB.Scan_Buffer_Matrix_Flag[0],[1],[2]...[15]
        
        switch(KB_CKUP_Status & KB_SCAN_MAKE_STATUS_MASK )              // KB_SCAN_MAKE_STATUS_MASK=0x30
        {
        // -------------------------------------------------------------------- 
        case (KB_SCAN_MAKE_NOW | KB_SCAN_MAKE_CONFIRM):                 // (0x10 | 0x20) = 0x30
            KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] &= (~KB_SCAN_MAKE_NOW);
            KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] &= (~KB_SCAN_MAKE_COUNT_MASK);
            break;
        
        // -------------------------------------------------------------------- 
        case (KB_SCAN_MAKE_NOW):                                        // 0x10
            KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] &= (~KB_SCAN_MAKE_NOW);
            
            if((KB_CKUP_Status & KB_SCAN_MAKE_COUNT_MASK) > KB_MAKE_DELAY_TIME)     /*!< Check make debounce. */
            {
                KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] |= KB_SCAN_MAKE_CONFIRM;    /*!< Confirm the key status is make. */
                KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] &= (~KB_SCAN_MAKE_COUNT_MASK);
            }
            else
            {
                KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] ++;
            }
            break;
                                                        
        // -------------------------------------------------------------------- 
        case (KB_SCAN_MAKE_CONFIRM) :                                   // 0x20
            if(KB_CKUP_Status & KB_SCAN_BREAK_CONFIRM)                                                         
            {
                break;
            }
            else if((KB_CKUP_Status & KB_SCAN_MAKE_COUNT_MASK) > KB_BREAK_DELAY_TIME)                         
            {
                if(KB_CKUP_Status & KD_SCAN_Code_Update)
                {
                    KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] |= KB_SCAN_BREAK_CONFIRM;
                    KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] &= (~KB_SCAN_MAKE_COUNT_MASK);
                }
                else
                {
                    KB.Scan_Buffer_Matrix_Address[KB_CKUP_Tmp] = 0;
                    KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp]    = 0;
                }
            }
            else
            {
                KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp] ++;
            }
            break;
                                                        
        // -------------------------------------------------------------------- 
        case 0:                                                         // 0x00
        default:
            KB.Scan_Buffer_Matrix_Address[KB_CKUP_Tmp] = 0;
            KB.Scan_Buffer_Matrix_Flag[KB_CKUP_Tmp]    = 0;
            break;                

        }
    }
}
/**
 *******************************************************************************
 * @brief	    According to species classify Key code. 
 * @details     
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_4x4Keyboard_TranslateToKeyCode_Flow(void)
{
    uint8_t KB_TraNKeyCode_Tmp;
    uint8_t KB_TraNKeyCode_Tmp2;
    
    // ------------------------------------------------------------------------
    KB.Tmp_Code = Default_KeyCode_Table[KB.Tmp_Addr];
    
    // ------------------------------------------------------------------------
    if(KB.Tmp_Flag & KB_SCAN_BREAK_CONFIRM)                 // KB_SCAN_BREAK_CONFIRM=0x80
    {
        
        for(KB_TraNKeyCode_Tmp=0; KB_TraNKeyCode_Tmp<KEY_BUFFER_MAX; KB_TraNKeyCode_Tmp++)      // KEY_BUFFER_MAX=3
        {
            
            if(KB_CTR.Keybuffer[KB_TraNKeyCode_Tmp] == KB.Tmp_Code)
            {
                for(KB_TraNKeyCode_Tmp2 = KB_TraNKeyCode_Tmp; KB_TraNKeyCode_Tmp2 < (KEY_BUFFER_MAX-1); KB_TraNKeyCode_Tmp2++)
                {
                    KB_CTR.Keybuffer[KB_TraNKeyCode_Tmp2] = KB_CTR.Keybuffer[KB_TraNKeyCode_Tmp2 + 1];
                }
                KB_CTR.Keybuffer[(KEY_BUFFER_MAX-1)] = 0;

                KB_CTR.KeybufferChangeFlag = 1;
                
                BSP_4x4Keyboard_ClearKeyRecord();
                return;
            }
        }
    }
    else
    {
        for(KB_TraNKeyCode_Tmp=0; KB_TraNKeyCode_Tmp<KEY_BUFFER_MAX; KB_TraNKeyCode_Tmp++)
        {
            if(KB_CTR.Keybuffer[KB_TraNKeyCode_Tmp] == KB.Tmp_Code)
            {
                BSP_4x4Keyboard_RecordKeySend();
                return;
            }
            else if(KB_CTR.Keybuffer[KB_TraNKeyCode_Tmp ] == 0)
            {
                KB_CTR.Keybuffer[KB_TraNKeyCode_Tmp] = KB.Tmp_Code;   
                
                KB_CTR.KeybufferChangeFlag = 1;
                BSP_4x4Keyboard_RecordKeySend();
                return;
            }
        }
    }
}
/**
 *******************************************************************************
 * @brief	    Translate to key code data package.
 * @details     
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_KB_TranslateToKeyCode(void)
{
    for(KB.Tmp_Index=0; KB.Tmp_Index<KEY_RECORD_MAX; KB.Tmp_Index++)    // KEY_RECORD_MAX=16
    {
        KB.Tmp_Addr = KB.Scan_Buffer_Matrix_Address[KB.Tmp_Index];
        
        KB.Tmp_Flag = KB.Scan_Buffer_Matrix_Flag[KB.Tmp_Index];
        
        if(KB.Tmp_Flag != 0)
        {
            // ----------------------------------------------------------------
            // KB_SCAN_BREAK_CONFIRM=0x80
            // KD_SCAN_Code_Update=0x40
            // KB_SCAN_MAKE_CONFIRM=0x20
            // ----------------------------------------------------------------
            // (KB_SCAN_MAKE_CONFIRM | KB_SCAN_BREAK_CONFIRM | KD_SCAN_Code_Update)=0xE0
            // Run  BSP_4x4Keyboard_TranslateToKeyCode_Flow when KB.Tmp_Flag=0x20 or =0xE0.
            // ----------------------------------------------------------------
            switch((KB.Tmp_Flag & ((KB_SCAN_MAKE_CONFIRM | KB_SCAN_BREAK_CONFIRM | KD_SCAN_Code_Update))))
            {
            case (KB_SCAN_MAKE_CONFIRM):                                // KB_SCAN_MAKE_CONFIRM=0x20
            case (KB_SCAN_MAKE_CONFIRM | KB_SCAN_BREAK_CONFIRM | KD_SCAN_Code_Update):  // (...|...|...)=0xE0
                BSP_4x4Keyboard_TranslateToKeyCode_Flow();
                break;
            default:
                break;
            }
        }
    }
}
/**
 *******************************************************************************
 * @brief	    Handle Keyboard status according to Key scan result.
 * @details     
 * @return      None
 * @exception   No
 * @note        
 *******************************************************************************
 */
static void BSP_4x4Keyboard_Keyboard_Function(void)
{
    // ------------------------------------------------------------------------
    // Decode Keyboard status.
    // ------------------------------------------------------------------------
    BSP_4x4Keyboard_CheckScanIn();      // Check event happen in which row       
    BSP_4x4Keyboard_CheckUpdate();      // Update key code to tmporary buffer.
    BSP_KB_TranslateToKeyCode();        // Translate to key code data package.
    
}



