/**
 ******************************************************************************
 *
 * @file        MG32_IEC60730_RAM_RunTime.c
 *
 * @brief       IEC60730 Class-C : RAM test in Run Time state.
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2025/07/10
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par     Disclaimer
 *    The Demo software is provided "AS IS"  without any warranty, either
 *    expressed or implied, including, but not limited to, the implied warranties
 *    of merchantability and fitness for a particular purpose.  The author will
 *    not be liable for any special, incidental, consequential or indirect
 *    damages due to loss of data or any other reason.
 *    These statements agree with the world wide and local dictated laws about
 *    authorship and violence against these laws.
 ******************************************************************************
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32_IEC60730_Common.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

// ----------------------------------------------------------------------------
// IEC60730_RAM_Array :
//      1. Total 128 bytes.
//      2. Check pattern
//           ... 0 0 0 ...
//           ... 0 1 0 ...   !!! Only one bit is '1'. !!!
//           ... 0 0 0 ...
//           (It can't affect others data bits!)
//      3. If the '1' bit position is equal to the calculate data, shift the '1' bit to next position.
// ----------------------------------------------------------------------------
static uint8_t IEC60730_RAM_Array[128] = {
    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,    // In initial state, the first byte is 0x80.
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,    // In initial state, the remaining bytes are 0x00.
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

// The '1' bit position in IEC60730_RAM_Verify_IDX. (Range : 1~1024)
static uint16_t IEC60730_RAM_Verify_IDX = 1;

/* Private function prototypes -----------------------------------------------*/

// IEC60730 Class-C : Test RAM function call, in run-time state.
IEC60730_Ret IEC60730_RAM_RunTime(void);

// IEC60730 Class-C : Initial RAM test parameters. 
//                    When an error occurs in the RAM test function. 
//                    User can execute this function call to restart the RAM test.
void IEC60730_RAM_RunTime_Init(void);

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief       Test the RAM area in the 'Run-Time' state.
 * @details     1. Check the '1' bit position.
 *              2. Shift the '1' bit to next position.
 * @return      IEC60730_Ret: return state of the IEC60730_Ret.
 *              This parameter can be: IEC60730_TEST_SUCCESS or IEC60730_TEST_FAILURE.
 *******************************************************************************
 */
IEC60730_Ret IEC60730_RAM_RunTime(void)
{
    uint16_t IDX;    // The '1' bit position.
    uint16_t CNT;    // Just counter index.
    uint8_t *ptr8 = (uint8_t *)&IEC60730_RAM_Array;

    // ------------------------------------------------------------------------
    // Check IEC60730_RAM_Array pattern (Only accept one bits is '1')
    // ------------------------------------------------------------------------
    for (IDX = CNT = 0; CNT < 128; CNT++)   // Total 128 bytes.
    {
        // --------------------------------------------------------------------
        // Check IEC60730_RAM_Array[CNT] is 0x00 ?
        if (*ptr8 == 0x00)
        {
            IDX += 8;    // Calculate the '1' bit position. (8 bits)
            ptr8++;      // Move to next byte.
        }
        else
        {
            // Check this byte data (Only accept 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01)
            switch (*ptr8)
            {
                // ------------------------------------------------------------
                case 0x80:              // 0x80 : The '1' bit is in MSB bit.
                    *ptr8++  = 0x40;    // This '1' bit must shift to next position.
                                        // e.g: 0x80 -> 0x40 -> 0x20 -> 0x10 ->
                                        //      0x08 -> 0x04 -> 0x02 -> 0x01 -> Next byte...

                    IDX     += 1;    // Calculate the '1' bit position.
                    break;
                // ------------------------------------------------------------
                case 0x40:              // 0x40 : The '1' bit is in 6th bit.
                    *ptr8++  = 0x20;    // Shift to next position.
                    IDX     += 2;
                    break;
                // ------------------------------------------------------------
                case 0x20:              // 0x20 : The '1' bit is in 5th bit.
                    *ptr8++  = 0x10;    // Shift to next position.
                    IDX     += 3;
                    break;
                // ------------------------------------------------------------
                case 0x10:              // 0x10 : The '1' bit is in 4th bit.
                    *ptr8++  = 0x08;    // Shift to next position.
                    IDX     += 4;
                    break;
                // ------------------------------------------------------------
                case 0x08:              // 0x08 : The '1' bit is in 3th bit.
                    *ptr8++  = 0x04;    // Shift to next position.
                    IDX     += 5;
                    break;
                // ------------------------------------------------------------
                case 0x04:              // 0x04 : The '1' bit is in 2th bit.
                    *ptr8++  = 0x02;    // Shift to next position.
                    IDX     += 6;
                    break;
                // ------------------------------------------------------------
                case 0x02:              // 0x02 : The '1' bit is in 1th bit.
                    *ptr8++  = 0x01;    // Shift to next position.
                    IDX     += 7;
                    break;
                // ------------------------------------------------------------
                case 0x01:              // 0x01 : The '1' bit is in LSB bit.
                    *ptr8++  = 0x00;    // 1. Clear this byte
                                        // 2. Move to next byte

                    IDX     += 8;       // Calculate the '1' bit position.

                    // Shift to next position. --------------------------------
                    if (CNT == 127)                      // End byte reached ?
                        IEC60730_RAM_Array[0] = 0x80;    // !!! Start position !!!
                    else
                    {
                        *ptr8++ = 0x80;    // 1. Fill '0x80' to this byte.
                                           // 2. Move to next byte.

                        CNT++;    // for check the remaining data index.
                    }
                    break;
                // ------------------------------------------------------------
                // This byte data is not equal to 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02 or 0x01.
                default:
                    return IEC60730_TEST_FAILURE; 
            }
            break;
        }
    }

    // ------------------------------------------------------------------------
    // Check remaining 'IEC60730_RAM_Array' (Must be reserved as 0x00.)
    // ------------------------------------------------------------------------
    for (CNT++; CNT < 128; CNT++)
    {
        if (*ptr8++ != 0x00)
            return IEC60730_TEST_FAILURE;    // The remaining data is not equal to '0x00'.
    }

    // ------------------------------------------------------------------------
    // Check if IDX is equal to 'IEC60730_RAM_Verify_IDX'.
    // ------------------------------------------------------------------------
    if (IDX != IEC60730_RAM_Verify_IDX++)   // 1. Not equal check
                                            // 2. IEC60730_RAM_Verify_IDX ++ (Next Position)
        return IEC60730_TEST_FAILURE;

    // ------------------------------------------------------------------------
    // The range of IEC60730_RAM_Verify_IDX is 1~1024.
    // ------------------------------------------------------------------------
    if (IEC60730_RAM_Verify_IDX == (1024 + 1))
        IEC60730_RAM_Verify_IDX = 1;

    return IEC60730_TEST_SUCCESS;
}

/**
 *******************************************************************************
 * @brief       Initial parameters of RAM runtime function. (Foe restart test.)
 * @details     1. Set the '1' bit data and position.
 *              2. Clear IEC60730_RAM_Array data.
 * @return      None.
 *******************************************************************************
 */
void IEC60730_RAM_RunTime_Init(void)
{
    uint16_t CNT;    // Just counter index.
    
    // ------------------------------------------------------------------------
    // Initial RAM test parameter to default state.
    // ------------------------------------------------------------------------
    IEC60730_RAM_Array[0] = 0x80;
    IEC60730_RAM_Verify_IDX = 0x0001;
    
    // ------------------------------------------------------------------------
    // Clear the remaining data
    // ------------------------------------------------------------------------
    for (CNT=1; CNT < 128; CNT++)
        IEC60730_RAM_Array[CNT] = 0x00;
    
    return;
}



