/**
 ******************************************************************************
 *
 * @file        MG32_IEC60730_Common.h
 *
 * @brief       This is the C code format driver head file.
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2025/07/11
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2018 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************
 * @par 		Disclaimer
 *		The Demo software is provided "AS IS"  without any warranty, either
 *		expressed or implied, including, but not limited to, the implied warranties
 *		of merchantability and fitness for a particular purpose.  The author will
 *		not be liable for any special, incidental, consequential or indirect
 *		damages due to loss of data or any other reason.
 *		These statements agree with the world wide and local dictated laws about
 *		authorship and violence against these laws.
 ******************************************************************************
 ******************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MG32_IEC60730_COMMON_H
#define __MG32_IEC60730_COMMON_H

#ifdef __cplusplus
extern "C"
{
#endif

/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_IEC60730_Manager.h"
#include "MG32_DRV.h"

/* Exported types ------------------------------------------------------------*/

/**
 * @enum		IEC60730_Ret
 * @brief		General IEC60730 return status
 */
typedef enum
{
    IEC60730_TEST_SUCCESS,
    IEC60730_TEST_FAILURE
} IEC60730_Ret;

#ifdef __cplusplus
}
#endif

/* Exported constants -------------------------------------------------------*/

/* These are the direct and inverted data (pattern) used during the RAM
test, performed using March C- Algorithm */
#define BCKGRND                  ((uint32_t)0x00000000uL)
#define INV_BCKGRND              ((uint32_t)0xFFFFFFFFuL)
#define RT_RAM_BLOCKSIZE         ((uint32_t)6u) /* Number of RAM words tested at once */

/* These are the direct used during the interrupt test */
#define Int_freq_lower           98
#define Int_freq_upper           102

#define Int_Freq_Start           48
#define Int_Freq_Get             05
#define Int_Freq_End             30

#define IEC60370_IntFreq_Success 0
#define IEC60370_IntFreq_Fail    1

/* Exported macro -----------------------------------------------------------*/

/* Exported functions -------------------------------------------------------*/

/**
 * @name    Function announce
 * @brief   IEC60730 CPU Register test functions
 */
///@{
IEC60730_Ret IEC60730_CPURegister_StartUp(void);
IEC60730_Ret IEC60730_CPURegister_RunTime(void);
IEC60730_Ret IEC60730_CPU_PC(void);
///@}

/**
 * @name    Function announce
 * @brief   IEC60730 RAM test functions
 */
///@{
IEC60730_Ret IEC60730_RAM_Full_ASM(void);
IEC60730_Ret IEC60730_RAM_RunTime(void);
void         IEC60730_RAM_RunTime_Init(void);
///@}

/**
 * @name    Function announce
 * @brief   IEC60730 ADC peripheral test functions
 */
///@{
IEC60730_Ret
    IEC60730_ADC(ADC_ResolutionDef ADC_Resol, uint8_t ADC_ChannelSel, uint16_t HThreshold, uint16_t LThreshold);
///@}

/**
 * @name    Function announce
 * @brief   IEC60730 Clock(CSC) test functions
 */
///@{
IEC60730_Ret IEC60730_Clock_Test(void);
IEC60730_Ret IEC60730_Clock_RunTime(void);
///@}

/**
 * @name    Function announce
 * @brief   IEC60730 Flash (AP/IAP) test functions
 */
///@{
/* Start Up functions ********************************************************/
IEC60730_Ret IEC60730_Flash_Initial_IAP(void);
IEC60730_Ret IEC60730_Flash_Initial(void);
IEC60730_Ret IEC60730_Flash_StartUp_IAP(void);
IEC60730_Ret IEC60730_Flash_StartUp(void);
/* Run Time functions ********************************************************/
IEC60730_Ret IEC60730_Flash_RunTime_AP(void);
IEC60730_Ret IEC60730_Flash_RunTime_IAP(void);
/* Software CRC calculation functions ****************************************/
uint16_t IEC60730_SWCRC16_8Bit(uint16_t InitVlaue, uint8_t *StartAddr, uint32_t Length);
uint32_t IEC60730_SWCRC32_8Bit(uint32_t InitVlaue, uint8_t *StartAddr, uint32_t Length);
/* Hardware CRC calculation functions ****************************************/
uint16_t IEC60730_CRC16_8Bit(uint16_t InitVlaue, uint8_t *StartAddr, uint32_t Length);
uint16_t IEC60730_CRC16_32Bit(uint16_t InitVlaue, uint32_t *StartAddr, uint32_t Length);
uint32_t IEC60730_CRC32_8Bit(uint32_t InitVlaue, uint8_t *StartAddr, uint32_t Length);
uint32_t IEC60730_CRC32_32Bit(uint32_t InitVlaue, uint32_t *StartAddr, uint32_t Length);
/* DMA with CRC functions ****************************************************/
#if !defined(MG32_1ST) && !defined(MG32_2ND) && !defined(MG32_3RD)
uint32_t IEC60730_CRC32_32Bit_DMA(uint32_t InitVlaue, uint32_t *StartAddr, uint32_t Length);
uint32_t IEC60730_CRC32_8Bit_DMA(uint32_t InitVlaue, uint8_t *StartAddr, uint32_t Length);
#endif
/* Compare hardware and software CRC functions ********************************/
IEC60730_Ret IEC60730_Flash_CRC16_8Bit(uint32_t nStartAddr, uint32_t nLength, uint16_t nCRCValue);
IEC60730_Ret IEC60730_Flash_CRC16_32Bit(uint32_t nStartAddr, uint32_t nLength, uint16_t nCRCValue);
IEC60730_Ret IEC60730_Flash_CRC32_8Bit(uint32_t nStartAddr, uint32_t nLength, uint32_t nCRCValue);
IEC60730_Ret IEC60730_Flash_CRC32_32Bit(uint32_t nStartAddr, uint32_t nLength, uint32_t nCRCValue);
///@}

/**
 * @name    Function announce
 * @brief   IEC60730 Interrrupt test functions
 */
///@{
void MG32_TestInterruptCount_Set(void);
void MG32_TestInterruptResult_Get(void);
void MG32_TestInterrupt_Init(void);
///@}

/**
 * @name    Function announce
 * @brief   IEC60730 GPIO test functions
 */
///@{
IEC60730_Ret MG32_GPIOutput_Test(void);
IEC60730_Ret MG32_GPIOInput_Test(void);
IEC60730_Ret IEC60730_TestGPIO(void);
///@}

/**
 * @name    Function announce
 * @brief   IEC60730 IWDT test functions
 */
///@{
IEC60730_Ret IEC60730_IWDT_Test(void);
IEC60730_Ret IEC60730_IWDT_RunTime(void);
///@}

/**
 * @name    Function announce
 * @brief   IEC60730 Fail Safe POR routine
 */
///@{
void FailSafePOR(void) __attribute__((noreturn));
///@}

#endif
