
/**
 ******************************************************************************
 *
 * @file        Sample_LCD_InternalChargePump_Trim.c
 * @brief       LCD internal charge pump C file
 *
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2025/11/10
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_DRV.h"


/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/**
  *****************************************************************************
  * @brief  The application entry point.
  * @retval int
  *
  *****************************************************************************
  */
typedef enum{   // Vref form 2.4V
                       /* VT Voltage / 12BIAS ADCv / 13BIAS ADCv / 14BIAS ADCv  */
    Lv236v = 2013,     /* VT 2.36V   / 2013.867    / 1342.578    / 1006.933     */
    Lv244v = 2082,     /* VT 2.44V   / 2082.133    / 1388.089    / 1041.067     */
    Lv252v = 2150,     /* VT 2.52V   / 2150.400    / 1433.600    / 1075.200     */
    Lv260v = 2218,     /* VT 2.60V   / 2218.667    / 1479.111    / 1109.333     */
    Lv268v = 2286,     /* VT 2.68V   / 2286.933    / 1524.622    / 1146.467     */
    Lv276v = 2355,     /* VT 2.76V   / 2355.200    / 1570.133    / 1177.600     */
    Lv284v = 2423,     /* VT 2.84V   / 2423.467    / 1615.644    / 1211.733     */
    Lv292v = 2491,     /* VT 2.92V   / 2491.733    / 1661.156    / 1245.867     */
    Lv300v = 2560,     /* VT 3.00V   / 2560.000    / 1706.667    / 1280.000     */
    Lv308v = 2628,     /* VT 3.08V   / 2628.267    / 1752.178    / 1314.133     */
    Lv316v = 2696,     /* VT 3.16V   / 2696.533    / 1797.689    / 1348.267     */
    Lv324v = 2764,     /* VT 3.24V   / 2764.800    / 1843.200    / 1382.400     */
    Lv332v = 2833,     /* VT 3.32V   / 2833.067    / 1888.711    / 1416.533     */
    Lv340v = 2901,     /* VT 3.40V   / 2901.333    / 1979.222    / 1450.667     */
    Lv348v = 2969,     /* VT 3.48V   / 2969.600    / 1979.733    / 1484.800     */
    Lv356v = 3037,     /* VT 3.56V   / 3037.867    / 2025.244    / 1518.933     */
}SMP_LCD_ChargePumpOutputVoltageTypeDef;


/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
uint32_t Sample_LCD_InternalChargePumpTrim(SMP_LCD_ChargePumpOutputVoltageTypeDef TargetVoltage);


/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
uint32_t Sample_LCD_InternalChargePumpTrimX(void);


/* External vairables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief       LCD Internal Charge Pump Trim
 * @details     1. BackUp Register Config.
 *      \n      2. ADC Config, Input source internal LCD V1, Internal reference top voltage from IVR24.
 *      \n      3. LCD Config 1/2 BIAS, Charge Pump Enable, High Driver 290Kohm, V1 / VT Output Pin Enable.
 *      \n      4. Trim Start.
 *      \n      5. Recovery ADC / LCD Register Config, and Reserve charge pump trim value.
 * @exception   None
 * @param[in]   TargetVoltage
 *   @arg\b     Lv236v
 *   @arg\b     Lv244v
 *   @arg\b     Lv252v
 *   @arg\b     Lv260v
 *   @arg\b     Lv268v
 *   @arg\b     Lv276v
 *   @arg\b     Lv284v
 *   @arg\b     Lv292v
 *   @arg\b     Lv300v
 *   @arg\b     Lv308v
 *   @arg\b     Lv316v
 *   @arg\b     Lv324v
 *   @arg\b     Lv332v
 *   @arg\b     Lv340v
 *   @arg\b     Lv348v
 *   @arg\b     Lv356v
 * @return      LCD power top voltage select.
 * @note        
 * @par         Example
 * @code        
                Sample_LCD_InternalChargePumpTrim(Lv300v);
                Sample_LCD_InternalChargePumpTrim(Lv332v);
 * @endcode     
 * @par         Modify
 *              uint32_t Sample_LCD_InternalChargePumpTrim(SMP_LCD_ChargePumpOutputVoltageTypeDef TargetVoltage)
 *******************************************************************************
 */
uint32_t Sample_LCD_InternalChargePumpTrim(SMP_LCD_ChargePumpOutputVoltageTypeDef TargetVoltage)
{
    // Story Register Value.
    uint32_t lCSC_APB0_Tmp = CSC->APB0.W;
    uint32_t lADC_CLK_Tmp = ADC0->CLK.W;
    uint32_t lADC_CR0_Tmp = ADC0->CR0.W;
    uint32_t lADC_CR1_Tmp = ADC0->CR1.W;
    uint32_t lADC_START_Tmp = ADC0->START.W;
    uint32_t lADC_ANA_Tmp = ADC0->ANA.W;
    uint32_t lLCD_CR0_Tmp = LCD->CR0.W; 
    uint32_t lLCD_CR1_Tmp = LCD->CR1.W;

    uint32_t lStore[16];
    uint32_t lReg_Tmp;
    uint32_t lADCSum, lCnt32;
    uint16_t lTemp16;

    // get Vref Voltage
    CSC->KEY.W = 0xA217;
    CSC->APB0.W = CSC->APB0.W | CSC_APB0_ADC0_EN_enable_w;
    CSC->APB1.W = CSC->APB1.W | CSC_APB1_LCD_EN_enable_w;
    CSC->KEY.W = 0;

    //-----------------------------------------------------------------------------
    // ADC Config 
    ADC0->CR0.W = 0;
    ADC0->CLK.W = (0 << 24) | \
                  ADC_CLK_CK_SDIV_div1_w | \
                  ADC_CLK_CK_DLY_no_w | \
                  ADC_CLK_CK_DIV2_div6_w | \
                  ADC_CLK_CK_SEL2_ck_adc_w | \
                  ADC_CLK_CK_DIV_div4_w;

    ADC0->CR1.W = 0 << ADC_CR1_DOS_VAL_shift_w | \
                  0 << ADC_CR1_SUM_NUM_shift_w | \
                  ADC_CR1_SUM_MDS_single_w | \
                  ADC_CR1_SOVR_MDS_overwritten_w | \
                  ADC_CR1_OVR_MDS_overwritten_w | \
                  ADC_CR1_OUT_SEL_wdl_w | \
                  ADC_CR1_ALIGN_SEL_right_w | \
                  ADC_CR1_WIND_MDS_single_w | \
                  ADC_CR1_WIND_EN_disable_w;

    ADC0->START.W = ADC_START_CONV_MDS_one_w | \
                    ADC_START_TRG_SEL_disable_w | \
                    ADC_START_TRG_CONT_disable_w | \
                    ADC_START_START_SEL_sw_w | \
                    ADC_START_CH_SEL_int_w | \
                    3UL << 8 | \
                    ADC_START_HOLD_disable_w;

    ADC0->ANA.W = ADC_ANA_CONV_TIME_30adck_w | ADC_ANA_DISCHR_EN_disable_w | ADC_ANA_BIAS_lvl3_w | ADC_ANA_IVREF_SEL_ivr24_w | ADC_ANA_IVR_EN_enable_w;

    ADC0->CR0.W = ADC_CR0_DMA_EN_disable_w | \
                  ADC_CR0_DMA_DSIZE_16bit_w | \
                  ADC_CR0_DMA_MDS_disable_w | \
                  2 << ADC_CR0_SMP_SEL_shift_w | \
                  ADC_CR0_LIM_MDS_no_operation_w | \
                  ADC_CR0_CH_CHG_conv_w | \
                  ADC_CR0_RES_SEL_12_bit_w | \
                  ADC_CR0_WAIT_EN_disable_w | \
                  ADC_CR0_AUTOFF_EN_disable_w | \
                  ADC_CR0_EN_enable_w;

    //-----------------------------------------------------------------------------
    ADC0->START.W = (ADC0->START.W & (~(ADC_START_CH_MUX_mask_w | ADC_START_CH_SEL_mask_w))) | 0x00001400UL;

    if((LCD->CR0.W & LCD_CR0_BIAS_mask_w) ==  LCD_CR0_BIAS_static_w)
        LCD->CR0.W = (LCD->CR0.W & (~(LCD_CR0_VS_SEL_mask_w))) | (LCD_CR0_BIAS_b12_w | LCD_CR0_VS_SEL_cp_w);
    else
        LCD->CR0.W = (LCD->CR0.W & (~(LCD_CR0_BIAS_mask_w | LCD_CR0_VS_SEL_mask_w))) | (LCD_CR0_BIAS_b12_w | LCD_CR0_VS_SEL_cp_w);

    LCD->CR1.W = (LCD->CR1.W & (~LCD_CR1_VT_SEL_mask_w)) | (LCD_CR1_CP_EN_enable_w | LCD_CR1_DRV_MDS_high_w | LCD_CR1_VT_CTL_pin1_w | LCD_CR1_V1_CTL_pin1_w | LCD_CR1_VT_SEL_lvl0_w);

    do{
        Delay(30);  // Wait LCD Charge Pump Ready

        lADCSum = 0;
        lCnt32 = 0;
        do{
            ADC0->START.W = ADC0->START.W | ADC_START_START_mask_w; // Start ADC Conver.
            while((ADC0->STA.W & ADC_STA_E1CNVF_mask_w) == 0);      // Wait ADC Conver ready.
            ADC0->STA.W = ADC_STA_E1CNVF_mask_w;                    // Clear Flag.
            lTemp16 = ADC0->DAT0.H[0];                              // Read ADC Conver Data.
            lStore[lCnt32] = lTemp16;
            lADCSum = lADCSum + lTemp16;                            // Accumulation ADC Conver Value.
            Delay(1);                                               // ADC Sample Time.
            lCnt32 ++;
        }while(lCnt32 < 16);

        lADCSum = lADCSum / 16;

        if(lADCSum < TargetVoltage)
        {
            lReg_Tmp = LCD->CR1.W & LCD_CR1_VT_SEL_mask_w;
            if(lReg_Tmp < LCD_CR1_VT_SEL_mask_w)
                LCD->CR1.W = ((LCD->CR1.W & (~LCD_CR1_VT_SEL_mask_w)) | (lReg_Tmp + LCD_CR1_VT_SEL_lvl1_w));
        }
        lReg_Tmp = LCD->CR1.W & LCD_CR1_VT_SEL_mask_w;
    }while(lADCSum < TargetVoltage);

    //-----------------------------------------------------------------------------
    // Register Recovery Config
    LCD->CR0.W = 0;
    LCD->CR1.W = (lLCD_CR1_Tmp & (~LCD_CR1_VT_SEL_mask_w)) | lReg_Tmp;
    LCD->CR0.W = lLCD_CR0_Tmp; 

    ADC0->CR0.W = 0;
    ADC0->CLK.W = lADC_CLK_Tmp;
    ADC0->START.W = lADC_START_Tmp;
    ADC0->ANA.W = lADC_ANA_Tmp;
    ADC0->CR1.W = lADC_CR1_Tmp;
    ADC0->CR0.W = lADC_CR0_Tmp;

    CSC->KEY.W = 0xA217;
    CSC->APB0.W = lCSC_APB0_Tmp;
    CSC->KEY.W = 0;

    return(lReg_Tmp >> 16);
}


/**
 *******************************************************************************
 * @brief       LCD Internal Charge Pump Trim X
 * @details     None
 * @exception   None
 * @return      LCD power top voltage select.
 * @note        
 * @par         Example
 * @code        
                Sample_LCD_InternalChargePumpTrimX();
 * @endcode     
 * @par         Modify
 *              uint32_t Sample_LCD_InternalChargePumpTrimX(void)
 *******************************************************************************
 */
uint32_t Sample_LCD_InternalChargePumpTrimX(void)
{
    uint32_t lTmp;

    lTmp = Sample_LCD_InternalChargePumpTrim(Lv300v);
    printf("    LCD VT Level %d\n", lTmp);

    lTmp = Sample_LCD_InternalChargePumpTrim(Lv332v);
    printf("    LCD VT Level %d\n", lTmp);

    return lTmp;
}

