

/**
 ******************************************************************************
 *
 * @file        MG32_Smartcard_API.h
 *
 * @brief       The code is Smart Card H file.
 *
 * @par         Project
 *              Smart Card
 * @version     V1.00
 * @date        2022/11/23
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 ******************************************************************************
 */ 

#ifndef __MG32_SmartCard_H

#define __MG32_SmartCard_H


/* Includes ------------------------------------------------------------------*/
#include "MG32_URT_DRV.h"
#include "MG32_EXIC_DRV.h"
#include "MG32_TM_DRV.h"


/* Define ------------------------------------------------------------*/

#define SmartCard_SUCCESS                      1
#define SmartCard_FAILURE                      0

#define SmartCard_TRUE                         1
#define SmartCard_FALSE                        0


#define SmartCard_STATUS_INSERTED              0x00000001
#define SmartCard_STATUS_ATR_RECEIVECOMPLETE   0x00000002
#define SmartCard_STATUS_PPSEXCHANGE           0x00000004
#define SmartCard_STATUS_TRANSMISSION_ACTIVE   0x00000008
#define SmartCard_STATUS_ERROR                 0xFFFF0000

#define SmartCard_ATR_MAX_HISTORICAL           15            /*!< Maximum number of historical bytes.*/
#define SmartCard_ATR_MAX_PROTOCOLS            7             /*!< Maximum number of data level in the protocol.*/
#define SmartCard_ATR_MAX_INTERFACEBYTE        4             /*!< Maximum number of interface bytes per protocol.*/
#define SmartCard_ATR_MAX_TOTAOL_INTERFACEBYTE 16            /*!< Maximum number of interface bytes.*/
#define SmartCard_ATR_MAX_T_TYPE               15

#define SmartCard_ATR_SPIT_TD                  0             /*!< TD i - 1.*/
#define SmartCard_ATR_SPIT_TA                  1             /*!< TA i.*/
#define SmartCard_ATR_SPIT_TB                  2             /*!< TB i.*/
#define SmartCard_ATR_SPIT_TC                  3             /*!< TC i.*/

#define SmartCard_PPS_MAX_REQUEST              6
#define SmartCard_PPS_MAX_RESPONSE             6

#define SmartCard_PPS_STATUS_IDLE              0x00          /*!< PPS exchange no execute.*/
#define SmartCard_PPS_STATUS_TRG               0x01          /*!< Trigger PPS exchange.*/
#define SmartCard_PPS_STATUS_BUSY              0x02          /*!< In sending PPS request.*/
#define SmartCard_PPS_STATUS_RESPONSE          0x04          /*!< PPS response receive completing.*/

#define SmartCard_ERROR_ATRTIMEOUT      0x00010000
#define SmartCard_ERROR_PARITYERROR     0x00020000
#define SmartCard_ERROR_ATRERROR        0x00040000
#define SmartCard_ERROR_PPSERROR        0x00080000
#define SmartCard_ERROR_WTERROR         0x00100000


// ATR TAi ~ TBi present or abset mask.
#define SmartCard_ATR_Y_MASK            0xF0
#define SmartCard_ATR_Y_TA              0x10
#define SmartCard_ATR_Y_TB              0x20
#define SmartCard_ATR_Y_TC              0x40
#define SmartCard_ATR_Y_TD              0x80

// ATR type T transmission protocol mask
#define SmartCard_ATR_T_MASK            0x0F


/* Typedef -----------------------------------------------------------*/

/*! @struct  Answer to Reset packet
    @brief   
*/
typedef struct
{
    uint8_t TS;                      /*!< Initial character.*/
    uint8_t T0;                      /*!< Format byte.*/
    uint8_t Interface[16];           /*!< Interface bytes.*/
    uint8_t InterfaceLength;         /*!< Length of Interface bytes .*/
    uint8_t Historical[15];          /*!< Historical bytes.*/
    uint8_t HistoricalLength;        /*!< Length of Historical bytes.*/
    uint8_t TCK_present;             /*!< TCK byte whether present or not.*/
    uint8_t TCK;                     /*!< Check byte.*/
}SmartCard_ATR_TypeDef;

/*! @union  
    @brief   
*/
typedef union
{
    __IO uint32_t  W;
    struct
    {
        __IO uint8_t  Inserted                 : 1;     /*!< Card inserted.*/
        __IO uint8_t  ATRReceiveComplete       : 1;     /*!< ATR aready receive but not yet decode.*/
        __IO uint8_t  PPSExchange              : 1;     /*!< PPS exchange.*/     
        __IO uint8_t  Transmission_Init        : 1;     /*!< Determine transmisson protocol and parameter.*/        
        __IO uint8_t  TransmissionActive       : 1;     /*!< Transmission protocol active.*/
        __IO uint8_t                           : 3;
        __IO uint8_t                           : 8;
        __IO uint8_t  ATRTimeout               : 1;     /*!< Wait ATR timeout in cold reset.*/
        __IO uint8_t  ParityError              : 1;     /*!< Parity error.*/
        __IO uint8_t  ATRError                 : 1;     /*!< ATR error.*/
        __IO uint8_t  PPSError                 : 1;     /*!< PPS error.*/
        __IO uint8_t  WTError                  : 1;     /*!< RX over WT.*/
        __IO uint32_t                          :11;
    }MBIT;
}SmartCard_CTRStatus_TypeDef;
   



/*! @struct Global interface bytes   
    @brief   
*/
typedef struct
{
    uint8_t        Fi;                     /*!< Clock rate conversion integer*/
    uint8_t        Di;                     /*!< Baud rate adjustment integer*/
    uint8_t        T;                       
    uint8_t        N;                      /*!< Extra guard time integer.*/
    uint8_t        TA2_present;            /*!< TA2 whether present in the ATR.*/
    uint8_t        TA2;                    /*!< TA2 byte.*/
    uint8_t        T_15_present;           /*!< T = 15 whether present in the ATR.*/
    uint8_t        T15_Data[4];            /*!< The first TA for T15.*/
}SmartCard_GITByte_TypeDef;


/*! @struct Specific interface bytes   
    @brief   
*/
typedef struct
{
    uint8_t         SpData[SmartCard_ATR_MAX_PROTOCOLS][SmartCard_ATR_MAX_INTERFACEBYTE];  /*!< The type T interface byte.*/
    uint8_t         Total_SPinterface;
    uint8_t         WI;                                                        /*!< waiting time integer.*/  
    uint8_t         __PADDING[2];                                              /*!< Reserved for future.*/
    uint32_t        WT;                                                        /*!< Guard time.*/

}SmartCard_SPITByte_TypeDef;

/*! @struct Specific interface bytes   
    @brief   
*/
typedef struct
{
    uint32_t         Status;
    uint8_t          Request[SmartCard_PPS_MAX_REQUEST];        /*!<PPS request buffer.*/
    uint8_t          Response[SmartCard_PPS_MAX_RESPONSE];      /*!<PPS response buffer.*/
}SmartCard_PPS_TypeDef;

typedef struct
{

    SmartCard_CTRStatus_TypeDef   Status;
           
    /*ATR related*/
    SmartCard_ATR_TypeDef        ATR;             /*!< ATR source data.*/
    SmartCard_GITByte_TypeDef    GITByte;         /*!< Global interface bytes.*/
    SmartCard_SPITByte_TypeDef   SPITByte;        /*!< Specific interface bytes.*/
    SmartCard_PPS_TypeDef        PPS;             /*!< PPS exchange status.*/
    
    /*Transmit related*/
    uint8_t                      *TxBuf;
    uint32_t                     TXLength;
    /*Receive related*/          
    uint8_t                      *RxBuf;
    uint32_t                     RXLength;
    
}SmartCard_CTRTypeDef;
    

          
/* Macro -------------------------------------------------------------*/
#define __API_SmartCard_GetHistoricalByteLength()  (SmartCard_CTR.ATRData[0] & 0x0F)


/* Exported variables ------------------------------------------------*/
extern SmartCard_CTRTypeDef SmartCard_CTR;


/* Exported functions -----------------------------------------------*/
void API_SmartCard_ATR_DeInit(FunctionalState CLK_fDefault);
void API_SmartCard_Init(void);


uint8_t API_SmartCard_Transmit(uint8_t *pBuf, uint16_t Length);
uint8_t API_SmartCard_Receive(uint8_t *pBuf, uint16_t Length);



void API_SmartCard_CardInsertCallback(void);
void API_SmartCard_CardRemoveCallback(void);
void API_SmartCard_ATRCpltCallback(void);
void API_SmartCard_UserTransmission(void);
void API_SmartCard_TxCpltCallback(void);
void API_SmartCard_RxCpltCallback(void);
void API_SmartCard_ErrorCallback(void);

void API_SmartCard_main(void);


#endif

