

/**
 ******************************************************************************
 *
 * @file        MG32_CAN_MID.c
 * @brief       The CAN module middleware C file.
 *
 * @par         Project
 *              MG32
 * @version     V1.03
 * @date        2025/06/12
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "MG32.h" 

#if defined(ModuleExist_CAN0) && defined(CAN_TYPE)
#include "MG32_CAN_MID.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define CAN_TIMEOUT_VALUE   10U

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* External variables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief     Initializes the CAN peripheral according to the specified parameter
 * @details     
 * @param[in] mCAN : Pointer to CAN_HandleTypeDef structure that contains the 
 *                   configuration information for the specified CAN. 
 * @return    Middleware status.  
 * @exception No
 * @note      No
 * @par       Example
 * @code
       CAN_HandleTypeDef mCAN;
    
       mCAN.Instance                  = CAN0;
       mCAN.Init.Mode                 = CAN_MODE_NORMAL; // Operating mode
                                      
       mCAN.Init.Prescaler            = 2;               // TQ = CAN Module clock source / 2
       mCAN.Init.TimeSeg1             = CAN_BS1_2TQ;     // Prop_Seg + Phase_Seg1 = 2TQ.
       mCAN.Init.TimeSeg2             = CAN_BS2_2TQ;     // Phase_Seg2 + Information processing time = 2TQ 
       mCAN.Init.SyncJumpWidth        = CAN_SJW_4TQ;     // SJW = 4TQ
       
       mCAN.Init.TransmitFifoPriority = ENABLE;          // Priority driven by the request order(chronologically)
       mCAN.Init.ReceiveFifoLocked    = ENABLE;          // FIFO message keep when overrun             
       mCAN.Init.AutoRetransmission   = ENABLE;          // Auto retransmission enable
       mCAN.Init.AutoWakeup           = ENABLE;          // Auto wakeup enable
       mCAN.Init.FDTolerant           = DISABLE;         // CAN_FD tolerant disable
       mCAN.Init.PinSwap              = DISABLE;         // CAN_TX / CAN_RX pin swap
       
       MID_CAN_Init(&mCAN);
 * @endcode 
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_Init( CAN_HandleTypeDef *mCAN)
{
    /*Parameter declare*/
    uint32_t mCAN_Init_tickstart;
    
    /*Check the CAN handle allocation.*/
    if( mCAN == NULL)
    {
        return(MID_FAILURE);
    }

    /*Check the parameter*/
    assert_param(IS_CAN_ALL_INSTANCE(hcan->Instance));
    assert_param(IS_FUNCTIONAL_STATE(mCAN->Init.AutoWakeup));
    assert_param(IS_FUNCTIONAL_STATE(mCAN->Init.AutoRetransmission));
    assert_param(IS_FUNCTIONAL_STATE(mCAN->Init.ReceiveFifoLocked));
    assert_param(IS_FUNCTIONAL_STATE(mCAN->Init.TransmitFifoPriority));
    assert_param(IS_FUNCTIONAL_STATE(mCAN->Init.PinSwap));
    assert_param(IS_FUNCTIONAL_STATE(mCAN->Init.FDTolerant));
    assert_param(IS_CAN_MODE(mCAN->Init.Mode));
    assert_param(IS_CAN_SJW(mCAN->Init.SyncJumpWidth));
    assert_param(IS_CAN_BS1(mCAN->Init.TimeSeg1));
    assert_param(IS_CAN_BS2(mCAN->Init.TimeSeg2));
    assert_param(IS_CAN_PRESCALER(mCAN->Init.Prescaler));
    
    if( mCAN->State == MID_CAN_STATE_RESET)
    {
        /*Init the low level hardware*/
        MID_CAN_MspInit(mCAN);
    }

    /*Request initialisation*/
    mCAN->Instance->CR0.W = ( CAN_CR0_EN_enable_w | CAN_CR0_INIT_EN_enable_w | CAN_CR0_LP_EN_disable_w);
    
    /*Get tick*/
    mCAN_Init_tickstart = MID_GetTick();
    
    /*Wait initialisation acknowledge*/
    while((mCAN->Instance->STA.W & (CAN_STA_LP_STA_mask_w | CAN_STA_INIT_STA_mask_w))!=CAN_STA_INIT_STA_init_w)
    {
        if((MID_GetTick()- mCAN_Init_tickstart)>CAN_TIMEOUT_VALUE)
        {
            /*Update error code*/
            mCAN->ErrorCode |= MID_CAN_ERROR_TIMEOUT;
            
            /*Change CAN state*/
            mCAN->State = MID_CAN_STATE_ERROR;
            
            return(MID_FAILURE);
        }
    }
    /*Set the automatic wake-up mode*/
    if( mCAN->Init.AutoWakeup == ENABLE)
    {
        SET_BIT(mCAN->Instance->CR0.W, CAN_CR0_WUP_MDS_mask_w);
    }
    else
    {
        CLEAR_BIT(mCAN->Instance->CR0.W, CAN_CR0_WUP_MDS_mask_w);
    }
    /*Set the automatic retransmission*/
    if( mCAN->Init.AutoRetransmission == ENABLE)
    {
        CLEAR_BIT(mCAN->Instance->CR0.W, CAN_CR0_TXE_MDS_mask_w);
    }
    else
    {
        SET_BIT(mCAN->Instance->CR0.W, CAN_CR0_TXE_MDS_mask_w);
    }
    /*Set the receive FIFO locked mode*/
    if( mCAN->Init.ReceiveFifoLocked == ENABLE)
    {
        SET_BIT(mCAN->Instance->CR0.W, CAN_CR0_ROVR_MDS_mask_w);
    }
    else
    {
        CLEAR_BIT(mCAN->Instance->CR0.W, CAN_CR0_ROVR_MDS_mask_w);
    }
    /*Set the transmit FIFO priority*/
    if( mCAN->Init.TransmitFifoPriority == ENABLE)
    {
        SET_BIT(mCAN->Instance->CR0.W, CAN_CR0_TX_PRI_mask_w);
    }
    else
    {
        CLEAR_BIT(mCAN->Instance->CR0.W, CAN_CR0_TX_PRI_mask_w);
    }
    /*FD Tolerant*/
    if( mCAN->Init.FDTolerant == ENABLE)
    {
        SET_BIT(mCAN->Instance->CR0.W, CAN_CR0_FDT_MDS_mask_w);
    }
    else
    {
        CLEAR_BIT(mCAN->Instance->CR0.W, CAN_CR0_FDT_MDS_mask_w);
    }
    /*CAN_TX / CAN_RX Swap*/
    if( mCAN->Init.PinSwap == ENABLE)
    {
        SET_BIT(mCAN->Instance->CR0.W, CAN_CR0_IO_SWP_mask_w);
    }
    else
    {
        CLEAR_BIT(mCAN->Instance->CR0.W, CAN_CR0_IO_SWP_mask_w);
    }
    /*Set the bit timing*/
    WRITE_REG( mCAN->Instance->CLK.H[1], (mCAN->Init.Prescaler - 1));
    WRITE_REG( mCAN->Instance->CR1.W   , (mCAN->Init.TimeSeg1  | 
                                          mCAN->Init.TimeSeg2  |
                                          mCAN->Init.SyncJumpWidth));
    /*Set the operating mode*/
    WRITE_REG( mCAN->Instance->CR0.MBIT.TST_MDS, (uint8_t)mCAN->Init.Mode);
    
    
    /*Initialize the error code*/
    mCAN->ErrorCode          = MID_CAN_ERROR_NONE;
    mCAN->BusErrorCode       = 0;
    mCAN->ArbitrationLostBit = 0;
    
    /*Initialize the CAN status*/
    mCAN->State = MID_CAN_STATE_READY;
    
    /*Return function status*/
    return(MID_SUCCESS);
}

/**
 *******************************************************************************
 * @brief     Deinitializes the CAN peripheral registers to their default 
 *            reset values.
 * @details     
 * @param[in] mCAN : Pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuration information for the specified CAN.
 * @return    Middleware status.  
 * @exception No
 * @note      No
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_DeInit( CAN_HandleTypeDef *mCAN)
{
    /*Check CAN handle*/
    if(mCAN == NULL)
    {
        return(MID_FAILURE);
    }
    /* Check the parameters */
    assert_param(IS_CAN_ALL_INSTANCE(mCAN->Instance));
    
    /*Stop the CAN module*/
    (void)MID_CAN_Stop(mCAN);
    
    /*DeInit the low level hardware*/
    MID_CAN_MspDeInit(mCAN);
    
    /*Reset the CAN module*/
    RST->KEY.W  = 0x0000A217;
    RST->APB0.W = RST_APB0_CAN0_EN_reset_w;
    __NOP();
    __NOP();
    RST->APB0.W = RST_APB0_CAN0_EN_no_reset_w;
    RST->KEY.W  = 0x00000000;
    
    /*Reset the CAN Error Code*/
    mCAN->ErrorCode          = MID_CAN_ERROR_NONE;
    mCAN->BusErrorCode       = 0;
    mCAN->ArbitrationLostBit = 0;
    
    /*Change CAN state*/
    mCAN->State = MID_CAN_STATE_RESET;
    
    /*Return function status*/
    return(MID_SUCCESS);
}

/**
 *******************************************************************************
 * @brief     Configures the CAN reception filter according to the specified
 *            parameters in the CAN_FilterInitStruct.
 * @details     
 * @param[in] mCAN : Pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuration information for the specified CAN.
 * @param[in] FilterConfig: Pointer to a CAN_FilterTypeDef structure that contains
 *                          the filter configuration information.
 * @return    Middleware status.      
 * @exception No
 * @note      No
 * @par       Example
 * @code
       // Instance of mCAN has to pointer to CAN module before configure filter 
       CAN_HandleTypeDef mCAN;
    
       mCAN.Instance  = CAN0;        

       // Example 1 : 32 bit ID mask mode
       CAN_FilterTypeDef   mCAN_FL0;
    
       mCAN_FL0.FilterBank                        = 0;                      // Configure Filter 0
       mCAN_FL0.FilterMode                        = CAN_FILTERMODE_IDMASK;  // Filter mode is ID mask mode
       mCAN_FL0.FilterScale                       = CAN_FILTERSCALE_32BIT;  // Filter scaler is 32bit

       mCAN_FL0.FilterIdMask.AFnR.Mask32.ID.SID   = 0x7FF;                  // Filter base ID is 0x7FF
       mCAN_FL0.FilterIdMask.AFnR.Mask32.ID.EID   = 0x3FFFF;                // Filter ID extension is 0x3FFFF
       mCAN_FL0.FilterIdMask.AFnR.Mask32.ID.IDE   = 1;                      // Filter IDE is 1
       mCAN_FL0.FilterIdMask.AFnR.Mask32.ID.RTR   = 0;                      // Filter RTR is 0
       
       mCAN_FL0.FilterIdMask.AFnR.Mask32.Mask.SID = 0x7FF;                  // Filter base ID mask is 0x7FF ( 0 : don't care, 1 : must match)
       mCAN_FL0.FilterIdMask.AFnR.Mask32.Mask.EID = 0x3FFFF;                // Filter ID extension mask is 0x3FFFF ( 0 : don't care, 1 : must match)
       mCAN_FL0.FilterIdMask.AFnR.Mask32.Mask.IDE = 1;                      // Filter IDE mask is 1 ( 0 : don't care, 1 : must match)
       mCAN_FL0.FilterIdMask.AFnR.Mask32.Mask.RTR = 1;                      // Filter RTR mask is 1 ( 0 : don't care, 1 : must match)
       mCAN_FL0.FilterFIFOAssignment              = CAN_FILTER_FIFO0;       // The filter use FIFO0
       mCAN_FL0.FilterActivation                  = CAN_FILTER_ENABLE;      // The filter enable
       MID_CAN_ConfigFilter(&mCAN,&mCAN_FL0);
       
       // Example 2 : 16 bit ID list mode
       CAN_FilterTypeDef   mCAN_FL0;
    
       mCAN_FL0.FilterBank                               = 1;                      // Configure Filter 1
       mCAN_FL0.FilterMode                               = CAN_FILTERMODE_IDLIST;  // Filter mode is ID list mode
       mCAN_FL0.FilterScale                              = CAN_FILTERSCALE_16BIT;  // Filter scaler is 16bit
       
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.SID       = 0x7FF;                  // List0 : base ID is 0x7FF
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.EID_17_15 = 0x0;                    //         ID extension bit17 ~ bit15 is 0x0 
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.IDE       = 0;                      //         IDE is 0
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.RTR       = 0;                      //         RTR is 0
       
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.SID       = 0x7FF;                  // List1 : base ID is 0x7FF
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.EID_17_15 = 0x1;                    //         ID extension bit17 ~ bit15 is 0x1 
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.IDE       = 1;                      //         IDE is 1
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.RTR       = 0;                      //         RTR is 0
       
       mCAN_FL0.FilterIdMask.AFnR.List16.List2.SID       = 0x7FF;                  // List2 : base ID is 0x7FF
       mCAN_FL0.FilterIdMask.AFnR.List16.List2.EID_17_15 = 0x2;                    //         ID extension bit17 ~ bit15 is 0x2 
       mCAN_FL0.FilterIdMask.AFnR.List16.List2.IDE       = 1;                      //         IDE is 1
       mCAN_FL0.FilterIdMask.AFnR.List16.List2.RTR       = 0;                      //         RTR is 0
       
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.SID       = 0x7FF;                  // List3 : base ID is 0x7FF
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.EID_17_15 = 0x3;                    //         ID extension bit17 ~ bit15 is 0x3 
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.IDE       = 1;                      //         IDE is 1
       mCAN_FL0.FilterIdMask.AFnR.List16.List0.RTR       = 0;                      //         RTR is 0
       
       mCAN_FL0.FilterFIFOAssignment              = CAN_FILTER_FIFO0;              // The filter use FIFO1
       mCAN_FL0.FilterActivation                  = CAN_FILTER_ENABLE;             // The filter enable
       MID_CAN_ConfigFilter(&mCAN,&mCAN_FL0); 
 * @endcode
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_ConfigFilter(CAN_HandleTypeDef *mCAN, CAN_FilterTypeDef *FilterConfig)
{
    CAN_FilterBankTypeDef *mCAN_FilterBank;
    uint32_t mCAN_cFilterState = mCAN->State;
    uint16_t mCAN_cFilterBitpos;
    
    if((mCAN_cFilterState == MID_CAN_STATE_READY) ||
       (mCAN_cFilterState == MID_CAN_STATE_LISTENING))
    {
        /* Check the parameters */
        assert_param(IS_CAN_FILTER_MODE(FilterConfig->FilterMode));
        assert_param(IS_CAN_FILTER_SCALE(FilterConfig->FilterScale));
        assert_param(IS_CAN_FILTER_FIFO(FilterConfig->FilterFIFOAssignment));
        assert_param(IS_CAN_FILTER_ACTIVATION(FilterConfig->FilterActivation));
        assert_param(IS_CAN_FILTER_BANK(FilterConfig->FilterBank));
        
        
        /*Covert filter number into bit position*/
        if( FilterConfig->FilterBank > CAN_FILTERBANK_MAX)
        {
            mCAN->ErrorCode |= MID_CAN_ERROR_PARAM;
            return(MID_FAILURE);   
        }
        else
        {
            mCAN_cFilterBitpos = (uint16_t)(1 << FilterConfig->FilterBank);
            mCAN_FilterBank    = (CAN_FilterBankTypeDef *)(((uint32_t)mCAN->Instance        + (uint32_t)CAN_FILTERBANK_SHIFT) + \
                                                           ((uint32_t)CAN_FILTERBANKX_SHIFT * (uint32_t)FilterConfig->FilterBank));
        }
        /*Filter Deactivation*/
        CLEAR_BIT( mCAN->Instance->AFC0.H[0],mCAN_cFilterBitpos);
        
        /*Filter Scale*/
        if( FilterConfig->FilterScale == CAN_FILTERSCALE_32BIT)
        {
            CLEAR_BIT( mCAN->Instance->AFC1.H[1],mCAN_cFilterBitpos);
        }
        else
        {
            SET_BIT( mCAN->Instance->AFC1.H[1],mCAN_cFilterBitpos);
        }
        /*Filter ID , Mask */
        mCAN_FilterBank->AFnR0 = FilterConfig->FilterIdMask.AFnR.W[0];
        mCAN_FilterBank->AFnR1 = FilterConfig->FilterIdMask.AFnR.W[1];
        
        /*Filter Mode*/
        if( FilterConfig->FilterMode ==  CAN_FILTERMODE_IDMASK)
        {
            CLEAR_BIT( mCAN->Instance->AFC1.H[0],mCAN_cFilterBitpos); 
        }
        else
        {
            SET_BIT( mCAN->Instance->AFC1.H[0],mCAN_cFilterBitpos);
        }
        /*Filter FIFO assigment*/
        if( FilterConfig->FilterFIFOAssignment == CAN_FILTER_FIFO0)
        {
            CLEAR_BIT( mCAN->Instance->AFC0.H[1],mCAN_cFilterBitpos);
        }
        else
        {
            SET_BIT( mCAN->Instance->AFC0.H[1],mCAN_cFilterBitpos);
        }
        /*Filter FIFO activation*/
        if( FilterConfig->FilterActivation == CAN_FILTER_ENABLE)
        {
            SET_BIT( mCAN->Instance->AFC0.H[0],mCAN_cFilterBitpos);
        }

        /*Return function status*/
        return(MID_SUCCESS);
    }
    else
    {
        /*Update error cod*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
        
        return(MID_FAILURE);
    }        
}

/**
 *******************************************************************************
 * @brief     Start the CAN module. 
 * @details   
 * @param[in] mCAN : mCAN pointer to an CAN_HandleTypeDef structure that contains
 *       \n          the configuration information for the specified CAN.
 * @return    Middleware status
 * @exception No
 * @note      No
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_Start( CAN_HandleTypeDef *mCAN)
{
    /*Parameter declare*/
    uint32_t mCAN_Start_tickstart;
    
    if( mCAN->State != MID_CAN_STATE_READY)
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_READY;
        return(MID_FAILURE);
    }
    
    /*Change CAN peripheral state*/
    mCAN->State = MID_CAN_STATE_LISTENING;
    
    /*Request leave initialisation*/
    CLEAR_BIT(mCAN->Instance->CR0.W, (CAN_CR0_LP_EN_mask_w | CAN_CR0_INIT_EN_mask_w));

    /*Get tick*/
    mCAN_Start_tickstart = MID_GetTick();
    
    /*Wait the acknowledge*/
    while((mCAN->Instance->STA.W & (CAN_STA_INIT_STA_mask_w | CAN_STA_LP_STA_mask_w))!= 
                                   (CAN_STA_LP_STA_not_w | CAN_STA_INIT_STA_not_w))
    {
        /*Check for the Timeout*/
        if((MID_GetTick() - mCAN_Start_tickstart) > CAN_TIMEOUT_VALUE)
        {
            /*Update error code*/
            mCAN->ErrorCode |= MID_CAN_ERROR_TIMEOUT;
            
            /*Change CAN state*/
            mCAN->State = MID_CAN_STATE_ERROR;
            
            return(MID_FAILURE);
        }
    }
    
    /*Reset the CAN ErrorCode*/
    mCAN->ErrorCode = MID_CAN_ERROR_NONE;
    
    /*Return function status*/
    return(MID_SUCCESS);   
}

/**
 *******************************************************************************
 * @brief     Stop the CAN module and enable access to configuration registers.
 * @details     
 * @param[in] mCAN : mCAN pointer to an CAN_HandleTypeDef struct that contains
 *       \n          the configuration information for the specified CAN. 
 * @return    Middleware status  
 * @exception No
 * @note      No
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_Stop( CAN_HandleTypeDef *mCAN)
{
    uint32_t mCAN_Stop_tickstart;
    
    if(mCAN->State != MID_CAN_STATE_LISTENING)
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_STARTED;
        
        return(MID_FAILURE);
    }
    /*Request initialisation*/
    SET_BIT( mCAN->Instance->CR0.W , CAN_CR0_INIT_EN_mask_w);

    /*Get tick*/
    mCAN_Stop_tickstart = MID_GetTick();
    
    /*Wait the acknowledge*/
    while((mCAN->Instance->STA.W & (CAN_STA_INIT_STA_mask_w | CAN_STA_LP_STA_mask_w))!= 
                                   (CAN_STA_LP_STA_not_w | CAN_STA_INIT_STA_init_w))
    {
        if((MID_GetTick() - mCAN_Stop_tickstart) > CAN_TIMEOUT_VALUE)
        {
            /*Update error code*/
            mCAN->ErrorCode |= MID_CAN_ERROR_TIMEOUT;
            /*Change CAN state*/
            mCAN->State      = MID_CAN_STATE_ERROR;
            return(MID_FAILURE);
        }
    }
    /*Change CAN peripheral state*/
    mCAN->State = MID_CAN_STATE_READY;
    
    return(MID_SUCCESS);
}


/**
 *******************************************************************************
 * @brief     Request the sleep mode (low power) entry.
 * @details   When returning from this function. Sleep mode will be entered
 *       \n   as soon as the current CAN activity (transmission or reception of 
 *       \n   a CAN frame) has been completed.
 * @param[in] mCAN : mCAN pointer to an CAN_HandleTypeDef struct that contains
 *       \n          the configuration information for the specified CAN.   
 * @return    Middleware status  
 * @exception No
 * @note      No
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_RequestSleep( CAN_HandleTypeDef *mCAN)
{
    if((mCAN->State == MID_CAN_STATE_READY) || \
       (mCAN->State == MID_CAN_STATE_LISTENING))
    {
        /*Request sleep mode*/
        MODIFY_REG( mCAN->Instance->CR0.W , 
                (CAN_CR0_INIT_EN_mask_w    | CAN_CR0_LP_EN_mask_w),
                (CAN_CR0_INIT_EN_disable_w | CAN_CR0_LP_EN_enable_w));
        
        /*Return function status*/
        return( MID_SUCCESS);
    }
    else
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
        
        /*Return function status*/
        return(MID_ERROR);
    }
}

/**
 *******************************************************************************
 * @brief     Wake up from sleep mode. 
 * @details   When returning with MID_SUCCESS status from this function,
 *       \n   Sleep mode is exited.
 * @param[in] mCAN : mCAN pointer to an CAN_HandleTypeDef struct that contains
 *       \n          the configuration information for the specified CAN.    
 * @return    Middleware status
 * @exception No
 * @note      No
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_WakeUp( CAN_HandleTypeDef *mCAN)
{
    uint32_t mCAN_Wakeupcount;
    uint32_t mCAN_WakeupTimeout;
    uint32_t mCAN_WakeupTmp;
    uint32_t mCAN_WakeupNextMode;
    
    /*Wake-up to pre-sleep status.*/
    if( mCAN->State == MID_CAN_STATE_READY)
    {
        mCAN_WakeupNextMode   = (CAN_CR0_LP_EN_disable_w | CAN_CR0_INIT_EN_enable_w);
    }
    else if( mCAN->State == MID_CAN_STATE_LISTENING)
    {
        mCAN_WakeupNextMode   = (CAN_CR0_LP_EN_disable_w | CAN_CR0_INIT_EN_disable_w);
    }
    else
    {
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
       return(MID_FAILURE);
    }
    
    /*Wake up request.*/
    mCAN_WakeupTmp = mCAN->Instance->CR0.W;
    mCAN_WakeupTmp = ((mCAN_WakeupTmp & (~(CAN_CR0_LP_EN_mask_w | CAN_CR0_INIT_EN_mask_w))) | \
                      (mCAN_WakeupNextMode));
    mCAN->Instance->CR0.W = mCAN_WakeupTmp;
    
    /*Wait sleep mode is exited*/
    mCAN_Wakeupcount   = 0;
    mCAN_WakeupTimeout = 1000000U;
    do
    {
        mCAN_Wakeupcount++;
        
        if(mCAN_Wakeupcount > mCAN_WakeupTimeout)
        {
            /*Update error code*/
            mCAN->ErrorCode |= MID_CAN_ERROR_TIMEOUT;
            
            return(MID_FAILURE);
        }
    }while((mCAN->Instance->STA.W & CAN_STA_LP_STA_mask_w)!=0);
    return(MID_SUCCESS);
}
/**
 *******************************************************************************
 * @brief     Check is whether in sleep mode.
 * @details     
 * @param[in] mCAN : mCAN pointer to an CAN_HandleTypeDef struct that contains
 *       \n          the configuration information for the specified CAN.     
 * @return    Status
 *            - 0 : No in sleep mode now.
 *            - 1 : In sleep mode now.
 * @exception No
 * @note      No
 *******************************************************************************
 */
uint32_t MID_CAN_IsSleepActive( CAN_HandleTypeDef *mCAN)
{
    uint32_t   mCAN_IsSleepActiveStatus = 0U;
    uint32_t   mCAN_IsSleepActiveState;
    
    mCAN_IsSleepActiveState = mCAN->State;
    
    if((mCAN_IsSleepActiveState == MID_CAN_STATE_READY) || \
       (mCAN_IsSleepActiveState == MID_CAN_STATE_LISTENING))
    {
        /*Check low power mode status*/
        if((mCAN->Instance->STA.W & CAN_STA_LP_STA_mask_w)!=0)
        {
            mCAN_IsSleepActiveStatus = 1U;
        }
    }
    /*Return function status.*/
    return(mCAN_IsSleepActiveStatus);
}

/**
 *******************************************************************************
 * @brief      Add a message to the free mailbox and activate the corresponding 
 *             transmission request.
 * @details     
 * @param[in]  mCAN: pointer to a CAN_HandleTypeDef structure.
 * @param[in]  pHeader: pointer to a CAN_TxHeaderTypeDef structure.
 * @param[in]  aData[]: The array containing the payload of the TX frame.
 * @param[out] pTxMailbox: pointer to a variable where the function will return
 *                        the TxMailbox used to store the Tx message.
 *   @arg\b    CAN_TX_MAILBOX0: Tx Mailbox 0
 *   @arg\b    CAN_TX_MAILBOX1: Tx Mailbox 1
 *   @arg\b    CAN_TX_MAILBOX2: Tx Mailbox 2
 * @return     Middleware status 
 * @exception  No
 * @note       No
 * @par       Example
 * @code
       CAN_HandleTypeDef    mCAN;
       CAN_TxHeaderTypeDef  Msg0_Header;         // TX frame header
       uint8_t              Msg0_Data[8];        // TX frame data
       uint32_t             Msg_UseTxMailbox;    // Receive CAN module use which Tx mailbox 
       
       Msg0_Header.Format.Field.BaseID = 0x7FF;
       Msg0_Header.Format.Field.IDE    = 0;
       Msg0_Header.Format.Field.RTR    = 0;
       Msg0_Header.Format.Field.DLC    = 8;
       
       Msg0_Data[0] = 0x00;
       Msg0_Data[1] = 0x01;
       Msg0_Data[2] = 0x02;
       Msg0_Data[3] = 0x03;
       Msg0_Data[4] = 0x04;
       Msg0_Data[5] = 0x05;
       Msg0_Data[6] = 0x06;
       Msg0_Data[7] = 0x07;
       
       MID_CAN_AddTxMessage(&mCAN,&Msg0_Header,Msg0_Data,&Msg_UseTxMailbox);
 * @endcode 
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_AddTxMessage( CAN_HandleTypeDef *mCAN, CAN_TxHeaderTypeDef *pHeader, uint8_t aData[], uint32_t *pTxMailbox)
{
    CAN_TxMailBox_TypeDef *mCAN_TxMailBox;
    
    uint32_t mCAN_AddTxMailIndex;
    uint32_t mCAN_AddTxMsgState  = mCAN->State;
    uint32_t mCAN_AddTxMsgStatus = mCAN->Instance->STA2.W; 
    
    if((mCAN_AddTxMsgState == MID_CAN_STATE_READY) ||
       (mCAN_AddTxMsgState == MID_CAN_STATE_LISTENING))
    {
        if((mCAN_AddTxMsgStatus & ( CAN_STA2_TB0_STA_mask_w | CAN_STA2_TB1_STA_mask_w | CAN_STA2_TB2_STA_mask_w))!=0)
        {
            /*Select an empty transmit mailbox*/
            if( mCAN_AddTxMsgStatus & CAN_STA2_TB0_STA_mask_w)
            {
                mCAN_AddTxMailIndex = 0;
            }
            else if( mCAN_AddTxMsgStatus & CAN_STA2_TB1_STA_mask_w) 
            {
                mCAN_AddTxMailIndex = 1;
            }
            else
            {
                mCAN_AddTxMailIndex = 2;
            }
            mCAN_TxMailBox = (CAN_TxMailBox_TypeDef *)(((uint32_t)mCAN->Instance + (uint32_t)CAN_TXBUF_SHIFT) + (CAN_TXBUFX_SHIFT * mCAN_AddTxMailIndex));
            
            /*Store the TX mailbox*/
            *pTxMailbox = (uint32_t)(1<<mCAN_AddTxMailIndex); 
            
            /*Set up the Id and DLC*/
            WRITE_REG(mCAN_TxMailBox->TDATx1,pHeader->Format.W[0]);
            WRITE_REG(mCAN_TxMailBox->TDATx2,pHeader->Format.W[1]);              
           
            /*Set up the data field*/
            WRITE_REG(mCAN_TxMailBox->TDATx3.B[0],aData[0]);
            WRITE_REG(mCAN_TxMailBox->TDATx3.B[1],aData[1]);
            WRITE_REG(mCAN_TxMailBox->TDATx3.B[2],aData[2]);
            WRITE_REG(mCAN_TxMailBox->TDATx3.B[3],aData[3]);
            WRITE_REG(mCAN_TxMailBox->TDATx4.B[0],aData[4]);
            WRITE_REG(mCAN_TxMailBox->TDATx4.B[1],aData[5]);
            WRITE_REG(mCAN_TxMailBox->TDATx4.B[2],aData[6]);
            WRITE_REG(mCAN_TxMailBox->TDATx4.B[3],aData[7]);
            /*Request transmission*/
            SET_BIT(mCAN_TxMailBox->TDATx2 ,CAN_TDAT01_TX0_REQ_enable_w); 
            
            /*Return function status*/
            return(MID_SUCCESS);
        }
        else
        {
            /*Update error code*/
            mCAN->ErrorCode |= MID_CAN_ERROR_PARAM;
            return( MID_FAILURE);
        }            
    }
    else
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
        return( MID_FAILURE);
    }
}

/**
 *******************************************************************************
 * @brief     Abort transmission request
 * @details     
 * @param[in] mCAN: pointer to a CAN_HandleTypeDef structure that contains 
 *                  the configuration information for the specified CAN.
 * @param[in] TxMailboxes: Tx mailbox list of the TX mailboxes to abort.
 *                       This parameter can be any combination of following define.
 *   @arg\b    CAN_TX_MAILBOX0: Tx Mailbox 0
 *   @arg\b    CAN_TX_MAILBOX1: Tx Mailbox 1
 *   @arg\b    CAN_TX_MAILBOX2: Tx Mailbox 2
 * @return      
 * @exception No
 * @note      No
 * @par       Example
 * @code
       CAN_HandleTypeDef    mCAN;

       MID_CAN_AbortTxRequest(&mCAN, CAN_TX_MAILBOX0);   // Abort TX mailbox0

 * @endcode 
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_AbortTxRequest(CAN_HandleTypeDef *mCAN, uint32_t TxMailboxes)
{
    uint32_t mCAN_AbortTxMailBoxState = mCAN->State;

    /* Check function parameters */
    assert_param(IS_CAN_TX_MAILBOX_LIST(TxMailboxes));
    
    if( mCAN_AbortTxMailBoxState == MID_CAN_STATE_READY || 
        mCAN_AbortTxMailBoxState == MID_CAN_STATE_LISTENING)
    {
        /*Check Tx Mailbox 0*/
        if((TxMailboxes & CAN_TX_MAILBOX0) != 0U)
        {
            /*Abort request for Tx Mailbox 0*/
            SET_BIT(mCAN->Instance->TDAT01.W, CAN_TDAT01_TX0_STOP_enable_w); 
        }
        /*Check Tx Mailbox 1*/
        if((TxMailboxes & CAN_TX_MAILBOX1) != 0U)
        {
            /*Abort request for Tx Mailbox 1*/
            SET_BIT(mCAN->Instance->TDAT11.W, CAN_TDAT01_TX0_STOP_enable_w); 
        }
        /*Check Tx Mailbox 2*/
        if((TxMailboxes & CAN_TX_MAILBOX2) != 0U)
        {
            /*Abort request for Tx Mailbox 2*/
            SET_BIT(mCAN->Instance->TDAT21.W, CAN_TDAT01_TX0_STOP_enable_w);  
        }
        
        /*Return function status*/
        return(MID_SUCCESS);
    }
    else
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
        
        return(MID_ERROR);
    }
}
/**
 *******************************************************************************
 * @brief     Return Tx Mailboxes free level : number of free Tx Mailboxes.
 * @details     
 * @param[in] mCAN: pointer to a CAN_HandleTypeDef structure that contains 
 *                  the configuration information for the specified CAN.
 * @return    Number of free TX mailbox. 
 * @exception No
 * @note      No
 *******************************************************************************
 */
uint32_t MID_CAN_GetTxMailboxesFreeLevel( CAN_HandleTypeDef *mCAN)
{
    uint32_t mCAN_TxMailboxFreeLevel      = 0U;
    uint32_t mCAN_TxMailboxFreeLevelState = mCAN->State; 
    
    if((mCAN_TxMailboxFreeLevelState == MID_CAN_STATE_READY) ||   
       (mCAN_TxMailboxFreeLevelState == MID_CAN_STATE_LISTENING))
    {
        if( mCAN->Instance->STA2.W & CAN_STA2_TB0_STA_mask_w)
        {
            mCAN_TxMailboxFreeLevel++;
        }
        if( mCAN->Instance->STA2.W & CAN_STA2_TB1_STA_mask_w)
        {
            mCAN_TxMailboxFreeLevel++;
        }
        if( mCAN->Instance->STA2.W & CAN_STA2_TB2_STA_mask_w)
        {
            mCAN_TxMailboxFreeLevel++;
        }
    }
    return(mCAN_TxMailboxFreeLevel);
}

/**
 *******************************************************************************
 * @brief     Check if a transmission nrrequest is pending on the select TX 
 *            Mailboxes.
 * @details     
 * @param[in] mCAN: pointer to a CAN_HandleTypeDef structure that contains 
 *                  the configuration information for the specified CAN.
 * @param[in] TxMailbox: Tx mailbox list of Tx mailboxes to check.
 *                       This parameter can be any combination of following define.
 *   @arg\b    CAN_TX_MAILBOX0: Tx Mailbox 0
 *   @arg\b    CAN_TX_MAILBOX1: Tx Mailbox 1
 *   @arg\b    CAN_TX_MAILBOX2: Tx Mailbox 2
 * @return    Tx mailboxes status
 *            - 0 : No pending transmission request on any selected TX mailboxes.
 *            - 1 : Pending transmission request on at least one of the selected
 *                  TX mailbox.
 * @exception No
 * @note      No
 *******************************************************************************
 */
uint32_t MID_CAN_IsTxMessagePending( CAN_HandleTypeDef *mCAN, uint32_t TxMailbox)
{
    uint32_t mCAN_IsTxMsgPendingStatus = 0U;
    uint32_t mCAN_IsTxMsgPendingState  = mCAN->State;
    
    /* Check function parameters */
    assert_param(IS_CAN_TX_MAILBOX_LIST(TxMailboxes));
    
    if((mCAN_IsTxMsgPendingState == MID_CAN_STATE_READY) ||
       (mCAN_IsTxMsgPendingState == MID_CAN_STATE_LISTENING))
    {
        /*Check pending transmission request on the selected Tx Mailbox*/
        if((mCAN->Instance->STA2.H[1] & TxMailbox)==0)
        {
            mCAN_IsTxMsgPendingStatus = 1U;
        }
    }
    /*Return status*/
    return(mCAN_IsTxMsgPendingStatus);
}

/**
 *******************************************************************************
 * @brief     Get an CAN frame the Rx FIFO into the message RAM.
 * @details     
 * @param[in] mCAN: pointer to a CAN_HandleTypeDef structure that contains 
 *                  the configuration information for the specified CAN.
 * @param[in] RxFIFO: FIFO number of the received message to be read.
 *   @arg\b   CAN_RX_FIFO0 : Receive FIFO 0
 *   @arg\b   CAN_RX_FIFO1 : Receive FIFO 1
 * @param[in] pHeader: pointer to a CAN_RxHeaderTypeDef structure where the header
 *                     of the RX frame will be stored.
 * @param[in] aData[] : Data array where the payload of the RX frame will be stored.
 * @return    Middleware status
 * @exception No
 * @note      No
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_GetRxMessage(CAN_HandleTypeDef* mCAN, uint32_t RxFIFO, CAN_RxHeaderTypeDef *pHeader, uint8_t aData[])
{
    CAN_RxFIFO_TypeDef *mCAN_RxMsg;
    uint32_t mCAN_GetMsgState = mCAN->State;
    
    /*Check function parameters*/
    assert_param(IS_CAN_RX_FIFO(RxFifo));

    if((mCAN_GetMsgState == MID_CAN_STATE_READY)||
       (mCAN_GetMsgState == MID_CAN_STATE_LISTENING))
    {
        /*Check the the Rx FIFO*/
        switch(RxFIFO)
        {
            case CAN_RX_FIFO0:
                              /*Check the Rx FIFO 0 is not empty*/
                              if((mCAN->Instance->STA2.W & CAN_STA2_RX0_NUM_mask_w)==0U)
                              {
                                  mCAN->ErrorCode |= MID_CAN_ERROR_PARAM;
                                  return(MID_FAILURE);
                              }
                              break; 
            case CAN_RX_FIFO1:
                              /*Check the Rx FIFO 0 is not empty*/
                              if((mCAN->Instance->STA2.W & CAN_STA2_RX1_NUM_mask_w)==0U)
                              {
                                  mCAN->ErrorCode |= MID_CAN_ERROR_PARAM;
                                  return(MID_FAILURE);
                              }
                              break;
            default:
                              mCAN->ErrorCode |= MID_CAN_ERROR_PARAM;
                              return(MID_FAILURE);
        }
        /*Pointer to Rx FIFO interface*/
        mCAN_RxMsg = (CAN_RxFIFO_TypeDef *)(((uint32_t)mCAN->Instance + (uint32_t)CAN_RXBUF_SHIFT) + (CAN_RXBUFX_SHIFT * RxFIFO));
        
        /*Get the header*/
        pHeader->Format.W[0] = mCAN_RxMsg->RDATx1;
        pHeader->Format.W[1] = mCAN_RxMsg->RDATx2;
        
        /*Get the data*/
        aData[0] = mCAN_RxMsg->RDATx3.B[0];
        aData[1] = mCAN_RxMsg->RDATx3.B[1];
        aData[2] = mCAN_RxMsg->RDATx3.B[2];
        aData[3] = mCAN_RxMsg->RDATx3.B[3];
        aData[4] = mCAN_RxMsg->RDATx4.B[0];
        aData[5] = mCAN_RxMsg->RDATx4.B[1];
        aData[6] = mCAN_RxMsg->RDATx4.B[2];
        aData[7] = mCAN_RxMsg->RDATx4.B[3];
        
        /*Release the Rx FIFO and clear RXnF (n = 0,1,..)*/
        mCAN_RxMsg->RDATx2 |= CAN_RDAT01_RX0_CLR_enable_w;
        
        
        /*Return function status*/
        return(MID_SUCCESS);
    }
    else
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
        
        return(MID_FAILURE);
    }
}
/**
 *******************************************************************************
 * @brief     Return Rx FIFO fill level
 * @details     
 * @param[in] mCAN: pointer to a CAN_HandleTypeDef structure that contains 
 *                  the configuration information for the specified CAN.
 * @param[in] RxFifo: RX FIFO
 *   @arg\b   CAN_RX_FIFO0 : Receive FIFO 0
 *   @arg\b   CAN_RX_FIFO1 : Receive FIFO 1
 * @return    Number of message available in RX FIFO
 * @exception No
 * @note      No
 *******************************************************************************
 */
uint32_t MID_CAN_GetRxFifoFillLevel( CAN_HandleTypeDef *mCAN, uint32_t RxFifo)
{
    uint32_t mCAN_FIFOFillLevel = 0U;
    uint32_t mCAN_FIFOFillLevelState = mCAN->State;
    
    /*Check function parameters*/
    assert_param(IS_CAN_RX_FIFO(RxFifo));
    
    if((mCAN_FIFOFillLevelState == MID_CAN_STATE_READY) ||
       (mCAN_FIFOFillLevelState == MID_CAN_STATE_LISTENING))
    {
        switch(RxFifo)
        {
            case CAN_RX_FIFO0:
                              mCAN_FIFOFillLevel = mCAN->Instance->STA2.MBIT.RX0_NUM;
                              break;
            case CAN_RX_FIFO1:
                              mCAN_FIFOFillLevel = mCAN->Instance->STA2.MBIT.RX1_NUM;
                              break;
            default:
                              break;
        }
    }
    /*Return Rx FIFO fill level*/
    return(mCAN_FIFOFillLevel);
}

/**
 *******************************************************************************
 * @brief     Enable interrupt
 * @details     
 * @param[in] mCAN: pointer to a CAN_HandleTypeDef structure that contains 
 *                  the configuration information for the specified CAN.
 * @param[in] ActiveITs: indicates which interrupts will be enable.
 *                       This parameter can be any combination of the following interrupt.
 *   @arg CAN_IT_TX_MAILBOX0_EMPTY    : Transmit mailbox 0 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX1_EMPTY    : Transmit mailbox 1 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX2_EMPTY    : Transmit mailbox 2 empty interrupt.
 *   @arg CAN_IT_RX_FIFO0_OVERRUN     : FIFO 0 overrun interrupt
 *   @arg CAN_IT_RX_FIFO0_FULL        : FIFO 0 full interrupt
 *   @arg CAN_IT_RX_FIFO0_PENDING     : FIFO 0 message pending interrupt
 *   @arg CAN_IT_RX_FIFO0_MSG_PENDING : FIFO 0 receive interrupt
 *   @arg CAN_IT_RX_FIFO1_OVERRUN     : FIFO 1 overrun interrupt
 *   @arg CAN_IT_RX_FIFO1_FULL        : FIFO 1 full interrupt
 *   @arg CAN_IT_RX_FIFO1_PENDING     : FIFO 1 message pending interrupt
 *   @arg CAN_IT_RX_FIFO1_MSG_PENDING : FIFO 1 receive interrupt
 *   @arg CAN_IT_WAKEUP               : Wake-up interrupt
 *   @arg CAN_IT_ERROR_WARNING        : Error warning interrupt
 *   @arg CAN_IT_ERROR_PASSIVE        : Error passive interrupt
 *   @arg CAN_IT_BUSOFF               : Bus-off interrupt
 *   @arg CAN_IT_BUS_ERROR            : Bus error interrupt
 *   @arg CAN_IT_ALST                 : Arbitration lost interrupt
 *   @arg CAN_IT_IEA                  : CAN interrupt all enable (CAN module interrupt main enable)
 * @return    Middleware status
 * @exception No
 * @note      No
 * @par       Example
 * @code
       CAN_HandleTypeDef    mCAN;

       MID_CAN_ActivateNotification(&mCAN,( CAN_IT_IEA   | \
                                            CAN_IT_TX_MAILBOX0_EMPTY   | \
                                           (CAN_IT_RX_FIFO0_OVERRUN | CAN_IT_RX_FIFO0_FULL | CAN_IT_RX_FIFO0_MSG_PENDING) | \
                                           (CAN_IT_ERROR_PASSIVE    | CAN_IT_BUS_ERROR)));
 * @endcode 
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_ActivateNotification( CAN_HandleTypeDef *mCAN, uint32_t ActiveITs)
{
    uint32_t mCAN_ActivateNotificationState = mCAN->State;
    
    /*Check function parameters*/
    assert_param(IS_CAN_IT(ActiveITs));
    
    if((mCAN_ActivateNotificationState == MID_CAN_STATE_READY) ||
       (mCAN_ActivateNotificationState == MID_CAN_STATE_LISTENING))
    {
        /*Enable the selected interrupts*/
        __DRV_CAN_ENABLE_IT(mCAN, ActiveITs);
        
        /*Return function status*/
        return(MID_SUCCESS);
    }
    else
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
        
        return(MID_ERROR);
    }
}
/**
 *******************************************************************************
 * @brief     Disable interrupt
 * @details     
 * @param[in] mCAN: pointer to a CAN_HandleTypeDef structure that contains 
 *                  the configuration information for the specified CAN.
 * @param[in] InactiveITs: Indicates which interrupts will be disable.
 *                         This parameter can be any combination of the following interrupt.
 *   @arg CAN_IT_TX_MAILBOX0_EMPTY    : Transmit mailbox 0 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX1_EMPTY    : Transmit mailbox 1 empty interrupt.
 *   @arg CAN_IT_TX_MAILBOX2_EMPTY    : Transmit mailbox 2 empty interrupt.
 *   @arg CAN_IT_RX_FIFO0_OVERRUN     : FIFO 0 overrun interrupt
 *   @arg CAN_IT_RX_FIFO0_FULL        : FIFO 0 full interrupt
 *   @arg CAN_IT_RX_FIFO0_PENDING     : FIFO 0 message pending interrupt
 *   @arg CAN_IT_RX_FIFO0_MSG_PENDING : FIFO 0 receive interrupt
 *   @arg CAN_IT_RX_FIFO1_OVERRUN     : FIFO 1 overrun interrupt
 *   @arg CAN_IT_RX_FIFO1_FULL        : FIFO 1 full interrupt
 *   @arg CAN_IT_RX_FIFO1_PENDING     : FIFO 1 message pending interrupt
 *   @arg CAN_IT_RX_FIFO1_MSG_PENDING : FIFO 1 receive interrupt
 *   @arg CAN_IT_WAKEUP               : Wake-up interrupt
 *   @arg CAN_IT_ERROR_WARNING        : Error warning interrupt
 *   @arg CAN_IT_ERROR_PASSIVE        : Error passive interrupt
 *   @arg CAN_IT_BUSOFF               : Bus-off interrupt
 *   @arg CAN_IT_BUS_ERROR            : Bus error interrupt
 *   @arg CAN_IT_ALST                 : Arbitration lost interrupt
 *   @arg CAN_IT_IEA                  : CAN interrupt all enable (CAN module interrupt main enable)
 * @return    Middleware status  
 * @exception No
 * @note      No
 * @par       Example
 * @code
       CAN_HandleTypeDef    mCAN;

       MID_CAN_DeactivateNotification(&mCAN,( CAN_IT_IEA   | \
                                            CAN_IT_TX_MAILBOX0_EMPTY   | \
                                           (CAN_IT_RX_FIFO0_OVERRUN | CAN_IT_RX_FIFO0_FULL | CAN_IT_RX_FIFO0_MSG_PENDING) | \
                                           (CAN_IT_ERROR_PASSIVE    | CAN_IT_BUS_ERROR)));
 * @endcode  
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_DeactivateNotification( CAN_HandleTypeDef *mCAN, uint32_t InactiveITs)
{
    uint32_t mCAN_DeactivateNotificationState = mCAN->State;
    
    /*Check function parameters*/
    assert_param(IS_CAN_IT(InactiveITs));
    
    if((mCAN_DeactivateNotificationState == MID_CAN_STATE_READY) ||
       (mCAN_DeactivateNotificationState == MID_CAN_STATE_LISTENING))
    {
        /*Enable the selected interrupts*/
        __DRV_CAN_DISABLE_IT(mCAN, InactiveITs);
        
        /*Return function status*/
        return(MID_SUCCESS);
    }
    else
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
        
        return(MID_ERROR);
    }
}

/**
 *******************************************************************************
 * @brief     Handler CAN interrupt request
 * @details     
 * @param[in] mCAN: pointer to a CAN_HandleTypeDef structure that contains 
 *                  the configuration information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
void MID_CAN_IRQHandler( CAN_HandleTypeDef *mCAN)
{
    uint32_t mCAN_IRQHandler_ErrorCode = MID_CAN_ERROR_NONE;
    uint32_t mCAN_IRQHandler_IT;
    uint32_t mCAN_IRQHandler_ITFlag;
    uint32_t mCAN_IRQHandler_EventFlag;
    ctype    mCAN_IRQHandler_ErrorInfo;
    
    mCAN_IRQHandler_ITFlag    = READ_REG(mCAN->Instance->STA.W);
    mCAN_IRQHandler_IT        = READ_REG(mCAN->Instance->INT.W) & mCAN_IRQHandler_ITFlag;
    mCAN_IRQHandler_EventFlag = READ_REG(mCAN->Instance->STA2.W);
    
    
    /*Transmit Mailbox 0 interrupt managment-----------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_TX0) != 0U)
    {
        /*Clear the transmission interrupt flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_TX0);
        
        if((mCAN_IRQHandler_EventFlag & (CAN_FALG_TB0 | CAN_FLAG_TC0)) == (CAN_FALG_TB0 | CAN_FLAG_TC0))
        {
            /*Transmission Mailbox 0 transmit success callback*/
            MID_CAN_TxMailbox0CompleteCallback(mCAN);
        }
        else
        {
            /*Transmission Mailbox 0 abort callback*/
            MID_CAN_TxMailbox0FailCallback(mCAN);
        }
    }
    /*Transmit Mailbox 1 interrupt managment----------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_TX1) != 0U)
    {
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_TX1);
        
        if((mCAN_IRQHandler_EventFlag & (CAN_FALG_TB1 | CAN_FLAG_TC1)) == (CAN_FALG_TB1 | CAN_FLAG_TC1))
        {
            /*Transmission Mailbox 0 transmit success callback*/
            MID_CAN_TxMailbox1CompleteCallback(mCAN);
        }
        else
        {
            MID_CAN_TxMailbox1FailCallback(mCAN);
        }
    }
    /*Transmit Mailbox 2 interrupt managment-----------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_TX2) != 0U)
    {
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_TX2);
        
        if((mCAN_IRQHandler_EventFlag & (CAN_FALG_TB2 | CAN_FLAG_TC2)) == (CAN_FALG_TB2 | CAN_FLAG_TC2))
        {
            /*Transmission Mailbox 0 transmit success callback*/
            MID_CAN_TxMailbox2CompleteCallback(mCAN);
        }
        else
        {
            MID_CAN_TxMailbox2FailCallback(mCAN);
        }
    }
    /*Receive FIFO 0 managment-------------------------------------*/
    /*Receive FIFO 0 overrun interrupt managment*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_ROVR0) != 0U)
    {
        /*Clear Receive FIF0 0 overrun interrupt flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_ROVR0);
        
        /*Update error code*/
        mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_RX_FOV0;
    }
    /*Receive FIFO 0 full interrupt managment*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_RFUL0) != 0U)
    {
        /*Clear Receive FIFO 0 full flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_RFUL0);
        
        /*Receive FIFO 0 full callback*/
        MID_CAN_RxFifo0FullCallback(mCAN);
    }
    /*Receive FIF0 0 message interrupt management*/
    if((mCAN_IRQHandler_IT & (CAN_ITFLAG_RPEND0 | CAN_ITFLAG_RX0)) != 0U)
    {
        /*Receive FIFO 0 message callback*/
        MID_CAN_RxFifo0MsgPendingCallback(mCAN);
    }
    
    /*Receive FIFO 1 managment-------------------------------------*/
    /*Receive FIFO 1 overrun interrupt managment*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_ROVR1) != 0U)
    {
        /*Clear Receive FIF0 1 overrun interrupt flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_ROVR1);
        
        /*Update error code*/
        mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_RX_FOV1;
    }
    /*Receive FIFO 1 full interrupt managment*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_RFUL1) != 0U)
    {
        /*Clear Receive FIFO 1 full flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_RFUL1);
        
        /*Receive FIFO 1 full callback*/
        MID_CAN_RxFifo1FullCallback(mCAN);
    }
    /*Receive FIF0 1 message interrupt management*/
    if((mCAN_IRQHandler_IT & (CAN_ITFLAG_RPEND1 | CAN_ITFLAG_RX1)) != 0U)
    {
        /*Receive FIFO 1 message callback*/
        MID_CAN_RxFifo1MsgPendingCallback(mCAN);
    }
    /*Wakeup interrupt management----------------------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_WKU) != 0U)
    {
        /*Clear wakeup flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_WKU);
        
        /*Wakeup callback*/
        MID_CAN_WakeupFromRxMsgCallback(mCAN);
    }
    /*Check error warning interrupt management---------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_EWG) != 0U)
    {
        /*Clear error warning flag ( */
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_EWG);
                   
        /*To check error count is lower than warning limit or higher than warning limit*/
        if((mCAN_IRQHandler_ITFlag & CAN_FLAG_EWG) != 0U)
        {
            /*Set CAN error code to error warning*/
            mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_EWG;
        }
        else
        {
            /*Re-below warn limit*/
            MID_CAN_ReBelowWarnLimitCallback(mCAN);
        }
    }
    /*Check error passive interrupt management---------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_EPV) != 0)
    {
        /*Clear error passive flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_EPV);
        
        /*Check is Error active --> Error passive or Error passive --> Error active*/
        if((mCAN_IRQHandler_ITFlag & CAN_FLAG_EPV) != 0U)
        {
            /*Set CAN error code to error passive*/
            mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_EPV;
        }
        else
        {
            /*Re-enter error active callback*/
            MID_CAN_ReEnterErrorActiveCallback(mCAN);
        }
    }
    /*Check bus-off interrupt management---------------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_BOF) != 0U)
    {
        /*Clear error bus-off flag ( */
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_BOF);
        
        /*Set CAN error code to bus-off*/
        mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_BOF;
    }
    /*Arbitration lost interrupt management------------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_ALST) != 0U)
    {
        /*Clear arbitration lost flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_ALST);
        
        /*Get last arbitration lost bit*/
        mCAN->ArbitrationLostBit = READ_REG(mCAN->Instance->STA3.B[2]);
        
        /*Update error code*/
        mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_TX_ALST;
    }
    /*Bus Error interrupts management------------------------------*/
    if((mCAN_IRQHandler_IT & CAN_ITFLAG_BERR) != 0U)
    {
        /*Get bus error code*/
        mCAN_IRQHandler_ErrorInfo.H[0]  = READ_REG(mCAN->Instance->STA3.H[0]);
      
        /*Clear bus error interrupt flag*/
        __DRV_CAN_CLEAR_FLAG(mCAN, CAN_ITFLAG_BERR);
        
        /*Check error direct*/
        if((mCAN_IRQHandler_ErrorInfo.H[0] & CAN_ERROR_DIR_MASK) != 0U)
        {
            mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_RX_RERR;
        }
        else
        {
            mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_TX_TERR;
        }
        /*Check hardware error code*/
        switch((mCAN_IRQHandler_ErrorInfo.H[0] & CAN_ERROR_CODE_MASK))
        {
            case CAN_ERR_CODE_BIT:
                                    /*Set CAN error code to bit error*/
                                    mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_BIT;
                                    break;
            case CAN_ERR_CODE_FORM:
                                    /*Set CAN error code to form error*/
                                    mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_FOR;
                                    break;
            case CAN_ERR_CODE_STUFF:
                                    /*Set CAN error code to stuff error*/
                                    mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_STF;
                                    break;
            case CAN_ERR_CODE_CRC:
                                    /*Set CAN error code to CRC error*/
                                    mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_CRC;
                                    break;
            case CAN_ERR_CODE_OTHER:
                                    /*Set CAN error code to other error*/
                                    mCAN_IRQHandler_ErrorCode |= MID_CAN_ERROR_OTHER;
                                    break;
            default:
                                    break;
        }
    }
    
    /*Call the Error call back in case of errors-------------------*/
    if(mCAN_IRQHandler_ErrorCode != MID_CAN_ERROR_NONE)
    {
        /*Update error code in handle*/
        mCAN->ErrorCode |= mCAN_IRQHandler_ErrorCode;

        /*Call error callback function*/
        MID_CAN_ErrorCallback(mCAN);
    }
    
}

/**
 *******************************************************************************
 * @brief     Return the CAN state
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuration information for the specified CAN
 * @return    CAN middleware software state  
 * @exception No
 * @note      No
 *******************************************************************************
 */
MID_CAN_StateTypeDef MID_CAN_GetState(CAN_HandleTypeDef *mCAN)
{
    MID_CAN_StateTypeDef gmCAN_state = mCAN->State;
    
    if((gmCAN_state == MID_CAN_STATE_READY) ||
       (gmCAN_state == MID_CAN_STATE_LISTENING))
    {
        if((mCAN->Instance->STA.W & CAN_FLAG_LP) != 0U)
        {
            gmCAN_state = MID_CAN_STATE_SLEEP_ACTIVE;
        }
        else if((mCAN->Instance->CR0.W &  CAN_CR0_INIT_EN_mask_w) != 0U)
        {
            gmCAN_state = MID_CAN_STATE_SLEEP_PENDING;
        }
    }
    return(gmCAN_state);
}

/**
 *******************************************************************************
 * @brief     Return the CAN error code.
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuration information for the specified CAN
 * @return    CAN error code  
 * @exception No
 * @note      No
 *******************************************************************************
 */
uint32_t MID_CAN_GetError( CAN_HandleTypeDef *mCAN)
{
    return(mCAN->ErrorCode);
}


/**
 *******************************************************************************
 * @brief     Reset the CAN error code
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuration information for the specified CAN
 * @return    Middleware status
 * @exception No
 * @note      No
 *******************************************************************************
 */
MID_StatusTypeDef MID_CAN_ResetError( CAN_HandleTypeDef *mCAN)
{
    MID_CAN_StateTypeDef rmCAN_State = mCAN->State;
    
    if((rmCAN_State == MID_CAN_STATE_READY) ||
       (rmCAN_State == MID_CAN_STATE_LISTENING))
    {
        /*Reset CAN error code*/
        mCAN->ErrorCode = 0U;
        
        return(MID_SUCCESS);
    }
    else
    {
        /*Update error code*/
        mCAN->ErrorCode |= MID_CAN_ERROR_NOT_INITIALIZED;
        
        return(MID_FAILURE);
    }
}



/**
 *******************************************************************************
 * @brief     Initializes the CAN MSP
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuration information for the specified CAN
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_MspInit( CAN_HandleTypeDef *mCAN)
{
    /*Prevent unsed argument(s) compilation warning*/
    UNUSED(mCAN);
    
    /*Note : This function should not be modified , when the callback is needed,
             the MID_CAN_MspInit can be implemented in the user file.*/
}

/**
 *******************************************************************************
 * @brief     DeInitializes the CAN MSP 
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_MspDeInit( CAN_HandleTypeDef *mCAN)
{
    /*Prevent unsed argument(s) compilation warning*/
    UNUSED(mCAN);
    
    /*Note : This function should not be modified , when the callback is needed,
             the MID_CAN_MspDeInit can be implemented in the user file.*/
}


/**
 *******************************************************************************
 * @brief     Transmission Mailbox 0 complete callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_TxMailbox0CompleteCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_TxMailbox0CompleteCallback could be implemented in the
            user file
   */
}
/**
 *******************************************************************************
 * @brief     Transmission Mailbox 0 fail callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_TxMailbox0FailCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_TxMailbox0FailCallback could be implemented in the
            user file
   */
}

/**
 *******************************************************************************
 * @brief     Transmission Mailbox 1 complete callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_TxMailbox1CompleteCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_TxMailbox1CompleteCallback could be implemented in the
            user file
   */
}
/**
 *******************************************************************************
 * @brief     Transmission Mailbox 1 fail callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_TxMailbox1FailCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_TxMailbox1FailCallback could be implemented in the
            user file
   */
}
/**
 *******************************************************************************
 * @brief     Transmission Mailbox 2 complete callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_TxMailbox2CompleteCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_TxMailbox2CompleteCallback could be implemented in the
            user file
   */
}
/**
 *******************************************************************************
 * @brief     Transmission Mailbox 2 fail callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_TxMailbox2FailCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_TxMailbox2FailCallback could be implemented in the
            user file
   */
}


/**
 *******************************************************************************
 * @brief     RX FIFO 0 full callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_RxFifo0FullCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_RxFifo0FullCallback could be implemented in the
            user file
   */
}

/**
 *******************************************************************************
 * @brief     RX FIFO 0 message pending callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_RxFifo0MsgPendingCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_RxFifo0MsgPendingCallback could be implemented in the
            user file
   */
}

/**
 *******************************************************************************
 * @brief     RX FIFO 1 full callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_RxFifo1FullCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_RxFifo1FullCallback could be implemented in the
            user file
   */
}

/**
 *******************************************************************************
 * @brief     RX FIFO 1 message pending callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_RxFifo1MsgPendingCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_RxFifo1MsgPendingCallback could be implemented in the
            user file
   */
}

/**
 *******************************************************************************
 * @brief     Wakeup from RX message callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_WakeupFromRxMsgCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_WakeupFromRxMsgCallback could be implemented in the
            user file
   */
}

/**
 *******************************************************************************
 * @brief     Error count rebelow warning limit callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_ReBelowWarnLimitCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_ReBelowWarnLimitCallback could be implemented in the
            user file
   */    
    
}

/**
 *******************************************************************************
 * @brief     Error count re-enter error active callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_ReEnterErrorActiveCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_ReEnterErrorActiveCallback could be implemented in the
            user file
   */     
}

/**
 *******************************************************************************
 * @brief     Error CAN callback
 * @details     
 * @param[in] mCAN : pointer to a CAN_HandleTypeDef structure that contains
 *                   the configuraton information for the specified CAN.
 * @return      
 * @exception No
 * @note      No
 *******************************************************************************
 */
__WEAK void MID_CAN_ErrorCallback(CAN_HandleTypeDef *mCAN)
{
    /* Prevent unused argument(s) compilation warning */
    UNUSED(mCAN);
    
    /* NOTE : This function Should not be modified, when the callback is needed,
            the MID_CAN_ErrorCallback could be implemented in the
            user file
   */    
}

#endif







