
/**
 ******************************************************************************
 *
 * @file        BSP_0_DEBUG.c
 * @brief       BSP_0_DEBUG Code's c file.
 *
 * @par         Project
 *              MG32
 * @version     V1.01
 * @date        2023/02/08
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 * 
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "TH222x_Global.h"
#include "MG32_GPIO_DRV.h"
#include "MG32_URT_DRV.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//GPIO
#define DEBUG_TX_PINX    PINB(8)
#define DEBUG_TX_AFS     11

//URT
#define DEBUG_URTn       URT4 

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
int fputc(int ch, FILE *f __attribute__((unused)));

/* Exported variables --------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/
/* External variables --------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief	    Debug com printf default string. 
 * @details     User can use printf to output data via TH223x's COM port.
 *              (J3 short enable COM port). 
 * @return      None
 * @note        No
 *******************************************************************************
 */
void TH222x_DEBUGParameter_Init(void)
{
    printf("Megawin MLink Virtual Com Port\n");
}

/**
 *******************************************************************************
 * @brief	    TH222 debug initial.   
 * @details     Use TH223x COM output data to help debug.    
 * @return      None  
 * @note        No
 *******************************************************************************
 */
void TH222x_DEBUG_Init(void)
{
    PIN_InitTypeDef    TH222x_DBUG_URT_PINX;
    URT_Data_TypeDef   TH222x_DBUG_URT_DataType;
    URT_BRG_TypeDef    TH222x_DBUG_URT_BRG;
    
    // ------------------------------------------------------------------------
    // GPIO Initial
    // ------------------------------------------------------------------------
    // URTn_TX pin initial.
    TH222x_DBUG_URT_PINX.PINX_Mode				 = PINX_Mode_PushPull_O;
    TH222x_DBUG_URT_PINX.PINX_PUResistant        = PINX_PUResistant_Enable;
    TH222x_DBUG_URT_PINX.PINX_Speed              = PINX_Speed_Low;
    TH222x_DBUG_URT_PINX.PINX_OUTDrive           = PINX_OUTDrive_Level0;
    TH222x_DBUG_URT_PINX.PINX_FilterDivider      = PINX_FilterDivider_Bypass;
    TH222x_DBUG_URT_PINX.PINX_Inverse            = PINX_Inverse_Disable;
    TH222x_DBUG_URT_PINX.PINX_Alternate_Function = DEBUG_TX_AFS;
    
    GPIO_PinMode_Config(DEBUG_TX_PINX, &TH222x_DBUG_URT_PINX);
    
    
    // ------------------------------------------------------------------------
    // URT Initial
    // ------------------------------------------------------------------------
    // Baud-Rate Initial
    // Baud-Rate = (URT clock source freq.) / (([PSR:RLR]+1)*(TX_OS_NUM + 1))           in Baud-Rate Generator mode is combined mode.
    // Baud-Rate = (URT clock source freq.) / ((PSR + 1) * (RLR + 1) *(TX_OS_NUM + 1)) in Baud-Rate Generator mode is separated mode.
    //  1. Set Baud-Rate Generaotr.
    TH222x_DBUG_URT_BRG.URT_InternalClockSource    = URT_BDClock_PROC;
    TH222x_DBUG_URT_BRG.URT_BaudRateMode           = URT_BDMode_Combined;
    TH222x_DBUG_URT_BRG.URT_PrescalerCounterReload = 0;	                //Set PSR
    TH222x_DBUG_URT_BRG.URT_BaudRateCounterReload  = 31;	            //Set RLR
    
    URT_BaudRateGenerator_Config(DEBUG_URTn, &TH222x_DBUG_URT_BRG);   	
    URT_BaudRateGenerator_Cmd(DEBUG_URTn, ENABLE);	                    //Enable BaudRateGenerator
    
    //  2. Set TX / RX clock source and oversampling.
    URT_TXClockSource_Select(DEBUG_URTn, URT_TXClock_Internal);	        //URT_TX use BaudRateGenerator
    URT_RXClockSource_Select(DEBUG_URTn, URT_RXClock_Internal);	        //URT_RX use BaudRateGenerator
    URT_RXOverSamplingMode_Select(DEBUG_URTn, URT_RXSMP_3TIME);         
    URT_TXOverSamplingSampleNumber_Select(DEBUG_URTn, 12);	            //Set TX OS_NUM
    URT_RXOverSamplingSampleNumber_Select(DEBUG_URTn, 12);	            //Set RX OS_NUM
    
    // ------------------------------------------------------------------------
    // Mode Initial
    // ------------------------------------------------------------------------
    //  1. Set URT Character Format.
    TH222x_DBUG_URT_DataType.URT_TX_DataLength  = URT_DataLength_8;
    TH222x_DBUG_URT_DataType.URT_RX_DataLength  = URT_DataLength_8;
    TH222x_DBUG_URT_DataType.URT_TX_DataOrder   = URT_DataTyped_LSB;
    TH222x_DBUG_URT_DataType.URT_RX_DataOrder   = URT_DataTyped_LSB;
    TH222x_DBUG_URT_DataType.URT_TX_Parity      = URT_Parity_No;
    TH222x_DBUG_URT_DataType.URT_RX_Parity      = URT_Parity_No;
    TH222x_DBUG_URT_DataType.URT_TX_StopBits    = URT_StopBits_1_0;
    TH222x_DBUG_URT_DataType.URT_RX_StopBits    = URT_StopBits_1_0;
    TH222x_DBUG_URT_DataType.URT_TX_DataInverse = DISABLE;
    TH222x_DBUG_URT_DataType.URT_RX_DataInverse = DISABLE;
    
    URT_DataCharacter_Config(DEBUG_URTn, &TH222x_DBUG_URT_DataType);

    //  2. Set URT mode (URT4 ~ URT7 only URT mode)
    URT_Mode_Select(DEBUG_URTn, URT_URT_mode);
    
    // ---Set DataLine Select---
    URT_DataLine_Select(DEBUG_URTn, URT_DataLine_2);
    
    //  3. Set URT buffer status (URT4 ~ URT7 can skip the step)
    URT_RXShadowBufferThreshold_Select(DEBUG_URTn, URT_RXTH_1BYTE);
    URT_IdlehandleMode_Select(DEBUG_URTn, URT_IDLEMode_No);
    URT_TXGuardTime_Select(DEBUG_URTn, 0);
    
    //  4. Enable URT ,URT_TX and URT_RX. 
    URT_TX_Cmd(DEBUG_URTn, ENABLE);	                                    //Enable TX
    URT_RX_Cmd(DEBUG_URTn, ENABLE);	                                    //Enable RX
    URT_Cmd(DEBUG_URTn, ENABLE);
    
}

/**
 *******************************************************************************
 * @brief	    Replace printf fputc function.
 * @details     Printf output data to  this routine. (URT4 RS232 output)
 * @return      None  
 * @note        No
 *******************************************************************************
 */
int fputc(int ch, FILE *f __attribute__((unused)))
{
    DEBUG_URTn->TDAT.B[0] = (uint8_t)ch;
    while((DEBUG_URTn->STA.W & URT_STA_TXF_mask_w)==0);
    
    return(ch);
}


