
/**
 ******************************************************************************
 *
 * @file        EPD_2in13V3_Function.c
 * @brief       Control WaveShare 2.13" E-Paper routine code.
 *              
 * @par         Project
 *              MG32
 * @version     V1.00
 * @date        2025/07/14
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 *******************************************************************************
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 */

/*==============================================================================
                                 User NOTES
How To use this function:
-----------------------
   + EPD_2in13_V3_Full_Init() - Initial E-Paper with all screen.
   + EPD_2IN13_V3_Draw_All_Screen() - Display an all screen image on E-Paper.
   + EPD_2IN13_V3_Draw_Partial_Pattern() - Initial WaveShare 2.13" E-Paper for window display.
   + EPD_2in13_V3_EntrySLEEP() - E-Paper enter sleep mode.
   
   Update E-Paper by this functions.
   
Driver architecture:
--------------------
   + MG32_SPI_DRV
   + MG32_GPIO_DRV

Known Limitations:
------------------
   1- This E-Paper is SPI interface.
   2- There are two LUT table for all screen / windows display.
   3- Use Command/Data to control or Update screen.
==============================================================================*/

/*==============================================================================
                                 Require parameter
Require module : SPI / GPIO

SPI Module : 
    Clock source        : CK_APB 
    Clock frequency     : 48MHz
    CPOL select         : '0'
    CPHA select         : '0'
    Bit-time            : ~12.5MHz
                                
GPIO pin configuration : 
    Pin / IO mode       / AFS
    ---  --------------  -----------------
    PB0 / PPO           / GPB0 (EPD_CS_Pin)
    PB2 / PPO           / SPI0_CLK 
    PB3 / PPO           / SPI0_MOSI 
    PB4 / PPO           / GPB4 (EPD_CMD_DAT_Pin)
    PB5 / PPO           / GPB5 (EPD_RESET_Pin)
    PB6 / DIN+PullUp    / GPB6 (EPD_BUSY_Pin)
    
==============================================================================*/





/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_SPI_DRV.h"
#include "EPD_2in13V3_Function.h"

/* External function prototypes ----------------------------------------------*/

/* Private includes ----------------------------------------------------------*/

/* Private typedef -----------------------------------------------------------*/

/* Private define ------------------------------------------------------------*/
// uintptr_t: Conversion to 'void *'
#define UN_CONSTIFY(_t, _v) ((_t)(uintptr_t)(_v))

/* Private macro -------------------------------------------------------------*/

/* Private variables ---------------------------------------------------------*/
static const uint8_t EPD_2IN13_V3_LUT_full_update[]= {
    // Total 159 bytes
	0x80,	0x4A,	0x40,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x40,	0x4A,	0x80,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x80,	0x4A,	0x40,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x40,	0x4A,	0x80,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,
    
	0x08,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x08,	0x00,	0x00,	0x0F,	0x00,	0x00,	0x02,					
	0x08,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x01,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,					
	
    0x22,	0x22,	0x22,	0x22,	0x22,	0x22,	0x00,	0x00,	0x00,			
	0x22,	0x17,	0x41,	0x00,	0x32,	0x36						
};

static const uint8_t EPD_2IN13_V3_LUT_partial_update[]= { 
    // Total 159 bytes
	0x00,	0x40,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x80,	0x80,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x40,	0x40,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x80,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
    
	0x05,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	  
	0x05,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x05,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	0x00,	
	
    0x22,	0x22,	0x22,	0x22,	0x22,	0x22,	0x00,	0x00,	0x00,	
	0x22,	0x17,	0x41,	0x00,	0x32,	0x36,	

};
  
/* Private function prototypes -----------------------------------------------*/

/* Private user code ---------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief       Write data to WaveShare 2.13" E-Paper.
 * @param[in] 	Datavalue: Write data value. 
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_IO_V3_WriteData(0x22);
 * @endcode
 *******************************************************************************
 */
void EPD_IO_V3_WriteData(uint8_t Datavalue) 
{
    /* Reset EPD control line CS */
    EPD_CS_LOW();
    
    /* Set EPD data/command line DC to High (Data state) */
    EPD_DAT_HIGH();
    
    /* Send Data (Wait for SPI Complete flag) */
    SPI_ClearFlag(SPI0, SPI_TCF);
    SPI_SetTxData(SPI0, SPI_1Byte, Datavalue);
    while(SPI_GetSingleFlagStatus(SPI0, SPI_TCF) == DRV_Normal);

    /* Deselect: Chip Select high */
    EPD_CS_HIGH();
}


/**
 *******************************************************************************
 * @brief       Write command to WaveShare 2.13" E-Paper.
 * @param[in] 	Reg: Command Register.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_IO_V3_WriteReg(0x20);
 * @endcode
 *******************************************************************************
 */
void EPD_IO_V3_WriteReg(uint8_t Reg)
{
    /* Reset EPD control line CS */
    EPD_CS_LOW();
    
    /* Set EPD data/command line DC to Low (Command State) */
    EPD_CMD_LOW();
    
    /* Send Command (Wait for SPI Complete flag) */
    SPI_ClearFlag(SPI0, SPI_TCF);
    SPI_SetTxData(SPI0, SPI_1Byte, Reg);
    while(SPI_GetSingleFlagStatus(SPI0, SPI_TCF) == DRV_Normal);

    /* Deselect: Chip Select high */
    EPD_CS_HIGH();
}

/**
 *******************************************************************************
 * @brief       Software Reset WAVESHARE 2.13" E-Paper device. 
 * @details     Control EPD's RESET pin to reset EPD.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2in13_V3_Reset();
 * @endcode
 *******************************************************************************
 */
void EPD_2in13_V3_Reset(void)
{
    // ------------------------------------------------------------------------
    EPD_RESET_INACTIVE();                       // Reset inactive
    MID_Delay(20);                              // Delay 20ms
    
    // ------------------------------------------------------------------------
    EPD_RESET_ACTIVE();                         // Reset E-Ppaer device
    MID_Delay(2);                               // Delay 2ms
    
    // ------------------------------------------------------------------------
    EPD_RESET_INACTIVE();
    MID_Delay(20);                              
}

/**
 *******************************************************************************
 * @brief       Setting the EPD 2.13" display window. 
 * @details     Set display area with x,y position and size.
 * @param[in] 	Xstart : X-axis starting position (multiples of 8).
 * @param[in] 	Ystart : Y-axis starting position (multiples of 8).
 * @param[in] 	Xend : End position of X-axis
 * @param[in] 	Yend : End position of Y-axis
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2in13_SetWindoes(0, 0, 121, 249);
 * @endcode
 *******************************************************************************
 */
void EPD_2in13_V3_SetWindoes(uint16_t Xstart, uint16_t Ystart, uint16_t Xend, uint16_t Yend)
{
    // --------------------------------------------------------------------
    // Set RAM X - address Start / End position (0x44)
    // XStart=0, XEnd=122
    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x44);                   // set Ram-X address start/end position - Specify the start/end position o fthe window address in the X direction by an address unit.
    EPD_IO_V3_WriteData((Xstart >> 3) & 0x1F);  // (Xstart >> 3) & 0x1F
    EPD_IO_V3_WriteData((Xend>>3) & 0x1F);      // (Xend>>3) & 0x1F

    // --------------------------------------------------------------------
    // Set RAM Y - address Start / End position (0x45)
    // YStart=0, YEnd=249
    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x45);                   // set Ram-Y address start/end position - Specify the start/end position o fthe window address in the Y direction by an address unit.
    EPD_IO_V3_WriteData(Ystart & 0xFF);         // Ystart & 0xFF
    EPD_IO_V3_WriteData((Ystart >> 8) & 0x01);  // (Ystart >> 8) & 0xFF
    EPD_IO_V3_WriteData(Yend & 0xFF);           // Yend & 0xFF
    EPD_IO_V3_WriteData((Yend >> 8) & 0x01);    // (Yend >> 8) & 0xFF
    
}

/**
 *******************************************************************************
 * @brief       Set the EPD 2.13" course. 
 * @details     Set course.
 * @param[in] 	Xstart : X-axis starting position (multiples of 8).
 *                                                ^^^^^^^^^^^^^^^^
 * @param[in] 	Ystart : Y-axis starting position.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2in13_SetCursor(0, 0);
 * @endcode
 *******************************************************************************
 */
void EPD_2in13_V3_SetCursor(uint16_t Xstart, uint16_t Ystart)
{

    // --------------------------------------------------------------------
    // Set RAM X address X address (0x4E)
    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x4E);                   // set RAM x address count to (0~122);
    EPD_IO_V3_WriteData((Xstart >> 3) & 0x1F);  // Xstart & 0xFF
    
    // --------------------------------------------------------------------
    // Set RAM Y address counter (0x4F)
    // YStart=0
    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x4F);                   // set RAM y address count to (0~250);
    EPD_IO_V3_WriteData(Ystart & 0xFF);         // Ystart & 0xFF
    EPD_IO_V3_WriteData((Ystart >> 8) & 0xFF);  // (Ystart >> 8) & 0xFF

}


/**
 *******************************************************************************
 * @brief       Update LUT table. 
 * @details     Update LUT and config another parameters.
 * @param[in] 	LUT : LUT table array.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2in13_SetCursor(0, 0);
 * @endcode
 *******************************************************************************
 */
void EPD_2in13_V3_UpdateLUT(uint8_t *LUT)
{
    uint16_t nb_bytes = 0;

    // --------------------------------------------------------------------
    // Write LUT register (0x32)
    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x32);                   // Write LUT register
    for(nb_bytes=0; nb_bytes<153; nb_bytes++)
    {
        EPD_IO_V3_WriteData(LUT[nb_bytes]);
    }
    // --------------------------------------------------------------------
    EPD_WAIT_BUSY_LOW();

    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x3F);                      
    EPD_IO_V3_WriteData(LUT[153]);                                                   
                                    

    // --------------------------------------------------------------------
    // Gate Driving voltage Control (0x03)
    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x03);                   // Gate Driving voltage Control
    EPD_IO_V3_WriteData(LUT[154]);              // A[4:0]=00h [POR]
                                                // VGH setting from 12V to 20V (15h=VGH=19)
                                    
    // --------------------------------------------------------------------
    // Source Driving voltage Control (0x04)
    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x04);                   // Gate Driving voltage Control
    EPD_IO_V3_WriteData(LUT[155]);              // A[7:0]=41h [POR]
    EPD_IO_V3_WriteData(LUT[156]);              // B[7:0]=A8h [POR]
    EPD_IO_V3_WriteData(LUT[157]);              // C[7:0]=32h [POR]

    // --------------------------------------------------------------------
    // Write VCOM register (0x2C)
    // --------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x2C);                   // VCOM Voltage
    EPD_IO_V3_WriteData(LUT[158]);              // A[7:0]=00h [POR]
                                                // 54h=-2.1, 58h=-2.2

}

/**
 *******************************************************************************
 * @brief       Initial WaveShare 2.13" E-Paper with all screen.
 * @details     Run this routine every time when user wants update all screen.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2in13_Full_Init();
 * @endcode
 *******************************************************************************
 */
void EPD_2in13_V3_Full_Init(void)
{    
    
    // ------------------------------------------------------------------------
    // Reset driver - Control EPD RESET pin
    // ------------------------------------------------------------------------
    EPD_2in13_V3_Reset();
    MID_Delay(100);                             // Delay 100ms

    // ------------------------------------------------------------------------
    // WAVESHARE 122x250 E-Paper
    // ------------------------------------------------------------------------
    //  X handle width area of EPD (0~121)
    //  Y handle higth area of EPD (0~249)
    // ------------------------------------------------------------------------
    // Check Busy?
    EPD_WAIT_BUSY_LOW();                        // Poll on the BUSY signal and wait for the EPD to be ready 

    EPD_IO_V3_WriteReg(0x12);                      // (SWRESET) soft reset
                                                // It resets the commands and patameters to their S/W
                                                // Reset defailt valies except R10h-Deep Sleep Mode
                                                // Note: RAM are unaffacted by this command.
    // Check Busy?
    EPD_WAIT_BUSY_LOW();                        // Poll on the BUSY signal and wait for the EPD to be ready 

    MID_Delay(10);                              // Delay 10ms

    
    // ------------------------------------------------------------------------
    // Driver Output control (0x01)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x01);                   // Driver output control - Set the number of gate. Setting for 212 gates is:
    EPD_IO_V3_WriteData(0xF9);                  // A[8:0]= 127h [POR], 296 MUX - MUX Gate lines setting as (A[8:0] + 1).
    EPD_IO_V3_WriteData(0x00);                  // A8[0]: 
    EPD_IO_V3_WriteData(0x00);                  // B[2:0]: Gate scanning sequence and direction
    
    // ------------------------------------------------------------------------
    // Data Entry mode setting (0x11)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x11);                   // (Data Entry Mode Setting) data entry mode
    EPD_IO_V3_WriteData(0x03);                  // A[1:0]: Address automatic increment / decrement setting -> (11-Y increment, X increment [POR])
                                                // A[2]: =0, the address counter is updated in thhe X direction,

    // ------------------------------------------------------------------------
    // SetWindoes
    // ------------------------------------------------------------------------
    EPD_2in13_V3_SetWindoes(0, 0, 121, 249);

    // ------------------------------------------------------------------------
    // SetCourse
    // ------------------------------------------------------------------------
    EPD_2in13_V3_SetCursor(0, 0);

    // ------------------------------------------------------------------------
    // Border Waveform Control (0x3C)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x3C);                   // (Select Border waveform for VBD) BorderWaveform
    EPD_IO_V3_WriteData(0x05);                  // A[7:6]:Select VBD option (A[7:0] = C0h [POR], set VBD as HIZ.)
                                                // A[5:4]:Fix Level Setting for VBD
                                                // A[1:0]:GS Transition setting for VBD

    // ------------------------------------------------------------------------
    // Display Update Control 1 (0x21) 
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x21);                   // Display Update Control 1 (0x21)
    EPD_IO_V3_WriteData(0x00);                  // A[7:0]=00h [POR]
    EPD_IO_V3_WriteData(0x80);                  // A[7:0]=00h [POR]


    // ------------------------------------------------------------------------
    // Temperature Sensor Control (0x18)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x18);                   // Temperature Sensor Control 
    EPD_IO_V3_WriteData(0x80);                  // A[7:0] = 80h Internal temperature

    // ------------------------------------------------------------------------
    EPD_WAIT_BUSY_LOW();                        // Poll on the BUSY signal and wait for the EPD to be ready 


    // ------------------------------------------------------------------------
    // Update LUT
    // ------------------------------------------------------------------------
    EPD_2in13_V3_UpdateLUT(UN_CONSTIFY(uint8_t *, &EPD_2IN13_V3_LUT_full_update));


}

/**
 *******************************************************************************
 * @brief       WaveShare 2.13" E-Paper enter sleep mode
 * @details     E-Paper needs to call this function to prevent damage this device.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2in13_V3_EntrySLEEP();
 * @endcode
 *******************************************************************************
 */
void EPD_2in13_V3_EntrySLEEP(void)
{
    // ------------------------------------------------------------------------
    // Deep Sleep mode (0x10)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x10);                   // Deep Sleep mode Control:
    EPD_IO_V3_WriteData(0x01);                  // A[1:0]=01, Enter Deep Sleep Mode 1
                                                // A[1:0]=11, Enter Deep Sleep Mode 2

}

/**
 *******************************************************************************
 * @brief       Turn On E-Paper Display.
 * @details     Needs this function to update screen for all screen.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_TurnOnDisplay();
 * @endcode
 *******************************************************************************
 */
static void EPD_TurnOnDisplay(void)
{
    // ------------------------------------------------------------------------
    // Display Update Control 2 (0x22)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x22);                   // Display Update Control 2
    EPD_IO_V3_WriteData(0xC7);                  // Enable Clock Signal, Then Enable ANALOG 
                                                // Then DISPLAY with DISPLAY Mode 1 
                                                // Then Disable ANALOG 
                                                // Then Disable OSC
    
    // ------------------------------------------------------------------------
    // Master Activation (0x20)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x20);                   // Master Activation (0x20)
                                                // Activate Display Update Sequence
                                                // The Display Update Sequence Option
                                                // is located at R22h.
    
    // ------------------------------------------------------------------------
    EPD_WAIT_BUSY_LOW();
    

}

/**
 *******************************************************************************
 * @brief       Turn On E-Paper Display (Partial).
 * @details     Run this routine every time when user wants update partial screen.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_TurnOnDisplayPart();
 * @endcode
 *******************************************************************************
 */
static void EPD_TurnOnDisplayPart(void)
{
    // ------------------------------------------------------------------------
    // Display Update Control 2 (0x22)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x22);                   // Display Update Control 2
    EPD_IO_V3_WriteData(0x0F);                  // fast:0x0c, quality:0x0f, 0xcf
    
    // ------------------------------------------------------------------------
    // Master Activation (0x20)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x20);                   // Master Activation (0x20)
                                                // Activate Display Update Sequence
                                                // The Display Update Sequence Option
                                                // is located at R22h.
    
    // ------------------------------------------------------------------------
    EPD_WAIT_BUSY_LOW();
}

/**
 *******************************************************************************
 * @brief       Refresh E-Paper display when active in partial update function.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_RefreshDisplay();
 * @endcode
 *******************************************************************************
 */
void EPD_RefreshDisplay(void)
{
    // ------------------------------------------------------------------------
    // Display Update Control 2 (0x22)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x22);                   // Display Update Control 2
    EPD_IO_V3_WriteData(0xB1);                  // Enable clock signal
                                                //      Load temperature value
                                                //      Load LUT with DISPLAY Mode 2
                                                //      Disable clock signal
//    EPD_IO_V3_WriteData(0x91);                  // Enable clock signal
                                                //      Load LUT with DISPLAY Mode 2
                                                //      Disable clock signal
    
    // ------------------------------------------------------------------------
    // Master Activation (0x20)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x20);                   // Master Activation (0x20)
                                                // Activate Display Update Sequence
                                                // The Display Update Sequence Option
                                                // is located at R22h.
    
    // ------------------------------------------------------------------------
    EPD_WAIT_BUSY_LOW();
    

}

/**
 *******************************************************************************
 * @brief       Display an all screen image on WaveShare 2.13" E-Paper.
 * @param[in]   ptr8: The image raw data of 122x250 pixel.
 *              There are total 4000 bytes data size. (16 bytes x 250 lines)
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2IN13_V3_Draw_All_Screen(&image);
 * @endcode
 *******************************************************************************
 */
void EPD_2IN13_V3_Draw_All_Screen(uint8_t *ptr8)
{
    // ------------------------------------------------------------------------
    // Write RAM (BW) - (0x24)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x24);
        
    // ------------------------------------------------------------------------
    for (uint16_t i = 0; i < 4000; i++) 
    {
        EPD_IO_V3_WriteData(*ptr8);
        ptr8 ++;
    }

    // --------------------------------------------------------------------
    EPD_TurnOnDisplay();
}


/**
 *******************************************************************************
 * @brief       Display an all screen image on WaveShare 2.13" E-Paper.
 * @param[in]   ptr8: The image raw data of 122x250 pixel.
 *              There are total 4000 bytes data size. (16 bytes x 250 lines)
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2IN13_V3_Draw_All_Screen(&image);
 * @endcode
 *******************************************************************************
 */
void EPD_2IN13_V3_Clear(void)
{
    // ------------------------------------------------------------------------
    // Write RAM (BW) - (0x24)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x24);
        
    // ------------------------------------------------------------------------
    for (uint16_t i = 0; i < 4000; i++) 
    {
        EPD_IO_V3_WriteData(0xFF);
    }

    // --------------------------------------------------------------------
    EPD_TurnOnDisplay();
}

/**
 *******************************************************************************
 * @brief       Initial WaveShare 2.13" E-Paper for window display.
 * @param[in] 	x: X-axis starting position (multiples of 8).
 * @param[in] 	y: Y-axis starting position.
 * @param[in] 	Image_Width: The width of this image. 
 * @param[in] 	Image_Hight: The hright of this image.
 * @param[in] 	ptr8: The pointer of image raw data.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2IN13_V3_Draw_Partial_Pattern(0, 160, 80-1, 70-1, ImagePtr8);
 * @endcode
 *******************************************************************************
 */
void EPD_2IN13_V3_Draw_Partial_Pattern(uint8_t x, uint8_t y, uint8_t Image_Width, uint8_t Image_Hight, uint8_t *ptr8)
{
    uint16_t Width, Height;
    
    // ------------------------------------------------------------------------
    // Reset EPD device
    // ------------------------------------------------------------------------
    EPD_RESET_ACTIVE();
    MID_Delay(2);                               // Delay 2ms
    EPD_RESET_INACTIVE();
                                    
    // ------------------------------------------------------------------------
    // Update LUT
    // ------------------------------------------------------------------------
//    EPD_2in13_UpdateLUT((uint8_t *) &EPD_2IN13_V2_LUT_partial_update);
    EPD_2in13_V3_UpdateLUT(UN_CONSTIFY(uint8_t *, &EPD_2IN13_V3_LUT_partial_update));

    // ------------------------------------------------------------------------
    // Write Register for Display Option (0x37)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x37);                   // Write Register for Display Option
    EPD_IO_V3_WriteData(0x00);                  // Keep 0x00
    EPD_IO_V3_WriteData(0x00);                  // B[7:0] Display Mode for WS[7:0]
    EPD_IO_V3_WriteData(0x00);                  // C[7:0] Display Mode for WS[15:8]
    EPD_IO_V3_WriteData(0x00);                  // D[7:0] Display Mode forWS[23:16]
    EPD_IO_V3_WriteData(0x00);                  // E[7:0] Display Mode for WS[31:24]
    EPD_IO_V3_WriteData(0x40);                  // F[3:0] Display Mode for WS[35:32]
    EPD_IO_V3_WriteData(0x00);                  //  0:Display Mode 1 [POR]
    EPD_IO_V3_WriteData(0x00);                  //  1: Display Mode2
    EPD_IO_V3_WriteData(0x00);                  // F[6]: PingPong for Display Mode 2
    EPD_IO_V3_WriteData(0x00);                  //  0: RAM ping-pong disable [POR]
                                                //  1: RAM ping-pong enable
                                                // G[7:0]~J[7:0] module ID /waveform
                                                // version.
                                                // Remarks:
                                                //   1) A[7:0]~J[7:0] can be stored in OTP
                                                //   2) RAM ping-pong function is not support 
                                                //      for Display Mode 1
    
    // ------------------------------------------------------------------------
    // Border Waveform Control (0x3C)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x3C);                   // (Select Border waveform for VBD) BorderWaveform
    EPD_IO_V3_WriteData(0x80);                  // A[7:6]:Select VBD, 00-GS Transition Define A[1:0]
                                                // A[5:4]:Fix Level Setting for VBD,00-VSS
                                                // A[1:0]:LUT1

    // ------------------------------------------------------------------------
    // Display Update Control 2 (0x22)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x22);                   // Display Update Control 2
    EPD_IO_V3_WriteData(0xC0);                  // To Enable Clock Signal, then Enable 
                                                // ANALOG (CLKEN=1, ANALOGEN=1)

    // ------------------------------------------------------------------------
    // Master Activation (0x20)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x20);                   // Master Activation
                                                // The Display Update Sequence Option
                                                // is located at R22h.
                                                // BUSY pad will output high during
                                                // operation. User should not interrupt
                                                // this operation to avoid corruption of
                                                // panel images.
                                    

    // ------------------------------------------------------------------------
    EPD_WAIT_BUSY_LOW();

    // ------------------------------------------------------------------------
    // SetWindoes
    // ------------------------------------------------------------------------
    EPD_2in13_V3_SetWindoes(x, y, x+Image_Width, y+Image_Hight);

    // ------------------------------------------------------------------------
    // SetCourse
    // ------------------------------------------------------------------------
    EPD_2in13_V3_SetCursor(x, y);
    
    // ------------------------------------------------------------------------
    // Recalaulate the width / Hight or image
    // ------------------------------------------------------------------------
    Width = (Image_Width % 8 == 0)? (Image_Width / 8): (Image_Width / 8 + 1);
    Height = Image_Hight;
    
    // ------------------------------------------------------------------------
    // Send raw data to E-Paper
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x24);
    for (uint16_t j = 0; j < Height; j++) {
        for (uint16_t i = 0; i < Width; i++) {
            EPD_IO_V3_WriteData(ptr8[i + j * Width]);
        }
    }

    // ------------------------------------------------------------------------
    EPD_TurnOnDisplayPart();
    
}


/**
 *******************************************************************************
 * @brief       Show 17x24 character on WaveShare 2.13" E-Paper. (Turn right)
 * @param[in] 	x: X-axis starting position (multiples of 8).
 *                                          ^^^^^^^^^^^^^^^^
 * @param[in] 	y: Y-axis starting position.
 * @param[in] 	ptr8: The pointer of image raw data.
 * @return		No
 * @note 
 * @par         Example
 * @code
    EPD_2IN13_V3_Draw_Partial_Pattern(0, 160, 80-1, 70-1, ImagePtr8);
 * @endcode
 *******************************************************************************
 */
void EPD_2IN13_V3_Draw_Font17x24(uint8_t x, uint8_t y, uint8_t *ptr8)
{
    uint8_t j, CheckBit, Dat;
    
    // ------------------------------------------------------------------------
    // Reset EPD device
    // ------------------------------------------------------------------------
    EPD_RESET_ACTIVE();
    MID_Delay(2);                               // Delay 2ms
    EPD_RESET_INACTIVE();
                                    
    // ------------------------------------------------------------------------
    // Update LUT
    // ------------------------------------------------------------------------
    EPD_2in13_V3_UpdateLUT(UN_CONSTIFY(uint8_t *, &EPD_2IN13_V3_LUT_partial_update));

    // ------------------------------------------------------------------------
    // Write Register for Display Option (0x37)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x37);                   // Write Register for Display Option
    EPD_IO_V3_WriteData(0x00);                  // Keep 0x00
    EPD_IO_V3_WriteData(0x00);                  // B[7:0] Display Mode for WS[7:0]
    EPD_IO_V3_WriteData(0x00);                  // C[7:0] Display Mode for WS[15:8]
    EPD_IO_V3_WriteData(0x00);                  // D[7:0] Display Mode forWS[23:16]
    EPD_IO_V3_WriteData(0x00);                  // E[7:0] Display Mode for WS[31:24]
    EPD_IO_V3_WriteData(0x40);                  // F[3:0] Display Mode for WS[35:32] 0:
    EPD_IO_V3_WriteData(0x00);                  // Display Mode 1 [POR]
    EPD_IO_V3_WriteData(0x00);                  // 1: Display Mode2
    EPD_IO_V3_WriteData(0x00);                  // F[6]: PingPong for Display Mode 2
    EPD_IO_V3_WriteData(0x00);                  // 0: RAM ping-pong disable [POR]
                                                // 1: RAM ping-pong enable
                                                // G[7:0]~J[7:0] module ID /waveform
                                                // version.
                                                // Remarks:
                                                // 1) A[7:0]~J[7:0] can be stored in
                                                // OTP
                                                // 2) RAM ping-pong function is not
                                                // support for Display Mode 1
    
    // ------------------------------------------------------------------------
    // Border Waveform Control (0x3C)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x3C);                   // (Select Border waveform for VBD) BorderWaveform
    EPD_IO_V3_WriteData(0x80);                  // A[7:6]:Select VBD, 00-GS Transition Define A[1:0]
                                                // A[5:4]:Fix Level Setting for VBD,00-VSS
                                                // A[1:0]:LUT1

    // ------------------------------------------------------------------------
    // Display Update Control 2 (0x22)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x22);                   // Display Update Control 2
    EPD_IO_V3_WriteData(0xC0);                  // To Enable Clock Signal, then Enable 
                                                // ANALOG (CLKEN=1, ANALOGEN=1)

    // ------------------------------------------------------------------------
    // Master Activation (0x20)
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x20);                   // Master Activation
                                                // The Display Update Sequence Option
                                                // is located at R22h.
                                                // BUSY pad will output high during
                                                // operation. User should not interrupt
                                                // this operation to avoid corruption of
                                                // panel images.
                                    

    // ------------------------------------------------------------------------
    EPD_WAIT_BUSY_LOW();

    // ------------------------------------------------------------------------
    // SetCourse
    // ------------------------------------------------------------------------
    EPD_2in13_V3_SetCursor(x, y);
    
    // ------------------------------------------------------------------------
    // SetWindoes
    // ------------------------------------------------------------------------
    EPD_2in13_V3_SetWindoes(x, y, x+23, y+16);  

    // ------------------------------------------------------------------------
    // SetCourse
    // ------------------------------------------------------------------------
    EPD_2in13_V3_SetCursor(x, y);
  


    // ------------------------------------------------------------------------
    // Write RAM
    // ------------------------------------------------------------------------
    EPD_IO_V3_WriteReg(0x24);
    
    // ------------------------------------------------------------------------
    // Handles Line 1~8
    // ------------------------------------------------------------------------
    j = 7;
    do
    {
        CheckBit = (uint8_t) (0x01 << j);
        
        // --------------------------------------------------------------------
        // First byte start from the bottom points.
        // First Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[69] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[66] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[63] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[60] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[57] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[54] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[51] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[48] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
        // --------------------------------------------------------------------
        // Second Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[45] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[42] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[39] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[36] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[33] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[30] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[27] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[24] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
        // --------------------------------------------------------------------
        // Third Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[21] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[18] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[15] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[12] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[9] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[6] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[3] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[0] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
        j --;
    } while(j != 0xFF);

    // ------------------------------------------------------------------------
    // Handles Line 9~16
    // ------------------------------------------------------------------------
    j = 7;
    do
    {
        CheckBit = (uint8_t) (0x01 << j);
        
        // --------------------------------------------------------------------
        // First byte start from the bottom points.
        // First Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[70] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[67] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[64] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[61] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[58] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[55] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[52] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[49] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
        // --------------------------------------------------------------------
        // Second Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[46] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[43] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[40] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[37] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[34] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[31] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[28] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[25] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
        // --------------------------------------------------------------------
        // Third Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[22] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[19] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[16] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[13] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[10] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[7] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[4] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[1] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
        j --;
    } while(j != 0xFF);

    // ------------------------------------------------------------------------
    // Handles Line 17
    // ------------------------------------------------------------------------
    {
//        CheckBit = (uint8_t) (0x01 << (uint8_t) (7-0));
        CheckBit = 0x80;
        
        // --------------------------------------------------------------------
        // First byte start from the bottom points.
        // First Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[71] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[68] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[65] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[62] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[59] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[56] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[53] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[50] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
        // --------------------------------------------------------------------
        // Second Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[47] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[44] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[41] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[38] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[35] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[32] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[29] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[26] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
        // --------------------------------------------------------------------
        // Third Byte:
        // --------------------------------------------------------------------
        Dat = 0x00;
        if ((ptr8[23] & CheckBit) != 0) Dat |= 0x80;
        if ((ptr8[20] & CheckBit) != 0) Dat |= 0x40;
        if ((ptr8[17] & CheckBit) != 0) Dat |= 0x20;
        if ((ptr8[14] & CheckBit) != 0) Dat |= 0x10;
        if ((ptr8[11] & CheckBit) != 0) Dat |= 0x08;
        if ((ptr8[8] & CheckBit) != 0) Dat |= 0x04;
        if ((ptr8[5] & CheckBit) != 0) Dat |= 0x02;
        if ((ptr8[2] & CheckBit) != 0) Dat |= 0x01;

        EPD_IO_V3_WriteData((uint8_t)~Dat);
            
    }
    
    // ------------------------------------------------------------------------
    EPD_TurnOnDisplayPart();
        
}


